package at.hannibal2.skyhanni.utils

import at.hannibal2.skyhanni.utils.LocationUtils.calculateEdges
import at.hannibal2.skyhanni.utils.NumberUtil.roundTo
import com.google.gson.annotations.Expose
import net.minecraft.entity.Entity
import net.minecraft.network.packet.s2c.play.ParticleS2CPacket
import net.minecraft.util.math.Box
import net.minecraft.util.math.BlockPos
import net.minecraft.util.math.EulerAngle
import net.minecraft.util.math.Vec3d
import kotlin.math.abs
import kotlin.math.absoluteValue
import kotlin.math.acos
import kotlin.math.ceil
import kotlin.math.cos
import kotlin.math.floor
import kotlin.math.max
import kotlin.math.min
import kotlin.math.pow
import kotlin.math.round
import kotlin.math.sin
import kotlin.math.sqrt

@Suppress("TooManyFunctions", "MemberVisibilityCanBePrivate")
data class LorenzVec(
    val x: Double,
    val y: Double,
    val z: Double,
) {
    val edges by lazy { boundingToOffset(1.0, 1.0, 1.0).expand(0.0001, 0.0001, 0.0001).calculateEdges() }

    constructor() : this(0.0, 0.0, 0.0)

    constructor(x: Int, y: Int, z: Int) : this(x.toDouble(), y.toDouble(), z.toDouble())

    constructor(x: Float, y: Float, z: Float) : this(x.toDouble(), y.toDouble(), z.toDouble())

    fun toBlockPos(): BlockPos = BlockPos(floor(x).toInt(), floor(y).toInt(), floor(z).toInt())

    fun toVec3(): Vec3d = Vec3d(x, y, z)

    fun distanceIgnoreY(other: LorenzVec): Double = distanceSqIgnoreY(other).pow(0.5)

    fun distance(other: LorenzVec): Double = distanceSq(other).pow(0.5)

    fun distanceSq(x: Double, y: Double, z: Double): Double = distanceSq(LorenzVec(x, y, z))

    fun distance(x: Double, y: Double, z: Double): Double = distance(LorenzVec(x, y, z))

    fun distanceChebyshevIgnoreY(other: LorenzVec) = max(abs(x - other.x), abs(z - other.z))

    fun distanceSq(other: LorenzVec): Double {
        val dx = other.x - x
        val dy = other.y - y
        val dz = other.z - z
        return (dx * dx + dy * dy + dz * dz)
    }

    fun distanceSqIgnoreY(other: LorenzVec): Double {
        val dx = other.x - x
        val dz = other.z - z
        return (dx * dx + dz * dz)
    }

    operator fun plus(other: LorenzVec) = LorenzVec(x + other.x, y + other.y, z + other.z)

    operator fun minus(other: LorenzVec) = LorenzVec(x - other.x, y - other.y, z - other.z)

    operator fun times(other: LorenzVec) = LorenzVec(x * other.x, y * other.y, z * other.z)
    operator fun times(other: Double) = LorenzVec(x * other, y * other, z * other)
    operator fun times(other: Int) = LorenzVec(x * other, y * other, z * other)

    operator fun div(other: LorenzVec) = LorenzVec(x / other.x, y / other.y, z / other.z)
    operator fun div(other: Double) = LorenzVec(x / other, y / other, z / other)

    fun add(x: Double = 0.0, y: Double = 0.0, z: Double = 0.0): LorenzVec =
        LorenzVec(this.x + x, this.y + y, this.z + z)

    fun add(x: Int = 0, y: Int = 0, z: Int = 0): LorenzVec = LorenzVec(this.x + x, this.y + y, this.z + z)

    fun dotProduct(other: LorenzVec): Double = (x * other.x) + (y * other.y) + (z * other.z)

    fun angleAsCos(other: LorenzVec) = normalize().dotProduct(other.normalize())

    fun angleInRad(other: LorenzVec) = acos(angleAsCos(other))

    fun angleInDeg(other: LorenzVec) = Math.toDegrees(angleInRad(other))

    fun crossProduct(other: LorenzVec): LorenzVec = LorenzVec(
        this.y * other.z - this.z * other.y,
        this.z * other.x - this.x * other.z,
        this.x * other.y - this.y * other.x,
    )

    fun scaledTo(other: LorenzVec) = this.normalize().times(other.length())

    fun normalize() = length().let { LorenzVec(x / it, y / it, z / it) }

    fun inverse() = LorenzVec(1.0 / x, 1.0 / y, 1.0 / z)

    fun min() = min(x, min(y, z))
    fun max() = max(x, max(y, z))

    fun minOfEachElement(other: LorenzVec) = LorenzVec(min(x, other.x), min(y, other.y), min(z, other.z))
    fun maxOfEachElement(other: LorenzVec) = LorenzVec(max(x, other.x), max(y, other.y), max(z, other.z))

    fun printWithAccuracy(accuracy: Int, splitChar: String = " "): String {
        return if (accuracy == 0) {
            val x = round(x).toInt()
            val y = round(y).toInt()
            val z = round(z).toInt()
            "$x$splitChar$y$splitChar$z"
        } else {
            val x = (round(x * accuracy) / accuracy)
            val y = (round(y * accuracy) / accuracy)
            val z = (round(z * accuracy) / accuracy)
            "$x$splitChar$y$splitChar$z"
        }
    }

    fun toCleanString(separator: String = ", "): String = listOf(x, y, z).joinToString(separator)

    fun asStoredString(): String = "$x:$y:$z"

    fun lengthSquared(): Double = x * x + y * y + z * z
    fun length(): Double = sqrt(lengthSquared())

    fun isNormalized(tolerance: Double = 0.01) = (lengthSquared() - 1.0).absoluteValue < tolerance

    fun isZero(): Boolean = x == 0.0 && y == 0.0 && z == 0.0

    fun clone(): LorenzVec = LorenzVec(x, y, z)

    fun toDoubleArray(): Array<Double> = arrayOf(x, y, z)
    fun toFloatArray(): Array<Float> = arrayOf(x.toFloat(), y.toFloat(), z.toFloat())

    fun equalsIgnoreY(other: LorenzVec) = x == other.x && z == other.z

    fun roundTo(precision: Int) = LorenzVec(x.roundTo(precision), y.roundTo(precision), z.roundTo(precision))

    fun roundToBlock() = LorenzVec(floor(x), floor(y), floor(z))

    fun blockCenter() = roundToBlock().add(0.5, 0.5, 0.5)

    fun slope(other: LorenzVec, factor: Double) = this + (other - this).scale(factor)

    // TODO better name. dont confuse with roundTo()
    fun roundLocation(): LorenzVec {
        val x = if (x < 0) x.toInt() - 1 else x.toInt()
        val y = y.toInt() - 1
        val z = if (z < 0) z.toInt() - 1 else z.toInt()
        return LorenzVec(x, y, z)
    }

    fun floor(): LorenzVec {
        val x = floor(x).toInt()
        val y = floor(y).toInt()
        val z = floor(z).toInt()
        return LorenzVec(x, y, z)
    }

    fun ceil(): LorenzVec {
        val x = ceil(x).toInt()
        val y = ceil(y).toInt()
        val z = ceil(z).toInt()
        return LorenzVec(x, y, z)
    }

    fun boundingCenter(expand: Double): Box {
        return Box(x - expand, y - expand, z - expand, x + expand, y + expand, z + expand)
    }

    fun boundingToOffset(offX: Double, offY: Double, offZ: Double) =
        Box(x, y, z, x + offX, y + offY, z + offZ)

    fun scale(scalar: Double): LorenzVec = LorenzVec(scalar * x, scalar * y, scalar * z)

    fun axisAlignedTo(other: LorenzVec) = Box(x, y, z, other.x, other.y, other.z)

    fun up(offset: Number = 1): LorenzVec = copy(y = y + offset.toDouble())

    fun down(offset: Number = 1): LorenzVec = copy(y = y - offset.toDouble())

    fun interpolate(other: LorenzVec, factor: Double): LorenzVec {
        require(factor in 0.0..1.0) { "Percentage must be between 0 and 1: $factor" }

        val x = (1 - factor) * x + factor * other.x
        val y = (1 - factor) * y + factor * other.y
        val z = (1 - factor) * z + factor * other.z

        return LorenzVec(x, y, z)
    }

    fun negated() = LorenzVec(-x, -y, -z)

    fun rotateXY(theta: Double) = LorenzVec(x * cos(theta) - y * sin(theta), x * sin(theta) + y * cos(theta), z)
    fun rotateXZ(theta: Double) = LorenzVec(x * cos(theta) + z * sin(theta), y, -x * sin(theta) + z * cos(theta))
    fun rotateYZ(theta: Double) = LorenzVec(x, y * cos(theta) - z * sin(theta), y * sin(theta) + z * cos(theta))

    fun nearestPointOnLine(startPos: LorenzVec, endPos: LorenzVec): LorenzVec {
        var d = endPos - startPos
        val w = this - startPos

        val dp = d.lengthSquared()
        var dt = 0.0

        if (dp != dt) dt = (w.dotProduct(d) / dp).coerceIn(0.0, 1.0)

        d *= dt
        d += startPos
        return d
    }

    fun distanceToLine(startPos: LorenzVec, endPos: LorenzVec): Double {
        return (nearestPointOnLine(startPos, endPos) - this).lengthSquared()
    }

    fun middle(other: LorenzVec): LorenzVec = this + ((other - this) / 2)

    private operator fun div(i: Number): LorenzVec = LorenzVec(x / i.toDouble(), y / i.toDouble(), z / i.toDouble())

    private val normX = if (x == 0.0) 0.0 else x
    private val normY = if (y == 0.0) 0.0 else y
    private val normZ = if (z == 0.0) 0.0 else z

    override fun equals(other: Any?): Boolean {
        if (other is LorenzVec) {
            val v2: LorenzVec = other
            if (this.x == v2.x && this.y == v2.y && this.z == v2.z) {
                return true
            }
        }
        return false
    }

    override fun hashCode() = 31 * (31 * normX.hashCode() + normY.hashCode()) + normZ.hashCode()

    companion object {

        val directions = setOf(
            LorenzVec(1, 0, 0),
            LorenzVec(-1, 0, 0),
            LorenzVec(0, 1, 0),
            LorenzVec(0, -1, 0),
            LorenzVec(0, 0, 1),
            LorenzVec(0, 0, -1),
        )

        fun getFromYawPitch(yawDegrees: Double, pitchDegrees: Double): LorenzVec {
            val yawRad: Double = (yawDegrees + 90) * Math.PI / 180
            val pitchRad: Double = (pitchDegrees + 90) * Math.PI / 180

            val x = sin(pitchRad) * cos(yawRad)
            val y = sin(pitchRad) * sin(yawRad)
            val z = cos(pitchRad)
            return LorenzVec(x, z, y)
        }

        // Format: "x:y:z"
        fun decodeFromString(string: String): LorenzVec {
            val (x, y, z) = string.split(":").map { it.toDouble() }
            return LorenzVec(x, y, z)
        }

        fun List<Double>.toLorenzVec(): LorenzVec {
            if (size != 3) error("Can not transform a list of size $size to LorenzVec")

            return LorenzVec(this[0], this[1], this[2])
        }

        val expandVector = LorenzVec(0.0020000000949949026, 0.0020000000949949026, 0.0020000000949949026)
    }
}

fun BlockPos.toLorenzVec(): LorenzVec = LorenzVec(x, y, z)

fun Entity.getLorenzVec(): LorenzVec = LorenzVec(entityPos.x, entityPos.y, entityPos.z)
fun Entity.getPrevLorenzVec(): LorenzVec = LorenzVec(lastRenderX, lastRenderY, lastRenderZ)
fun Entity.getServerLorenzVec(): LorenzVec = LorenzVec(trackedPosition.getPos().x, trackedPosition.getPos().y, trackedPosition.getPos().z)

fun Entity.getMotionLorenzVec(): LorenzVec = LorenzVec(velocity.x, velocity.y, velocity.z)

fun Entity.getPositionLog() = PositionLog(
    tick = age,
    position = getLorenzVec(),
    prev = getPrevLorenzVec(),
    server = getServerLorenzVec(),
    motion = getMotionLorenzVec(),
    yaw = yaw,
    pitch = pitch,
)

data class PositionLog(
    @Expose val tick: Int,
    @Expose val position: LorenzVec,
    @Expose val prev: LorenzVec,
    @Expose val server: LorenzVec,
    @Expose val motion: LorenzVec,
    @Expose val yaw: Float,
    @Expose val pitch: Float,
)

fun Vec3d.toLorenzVec(): LorenzVec = LorenzVec(x, y, z)

fun EulerAngle.toLorenzVec(): LorenzVec = LorenzVec(pitch, yaw, roll)

fun ParticleS2CPacket.toLorenzVec() = LorenzVec(x, y, z)

fun Array<Double>.toLorenzVec(): LorenzVec {
    return LorenzVec(this[0], this[1], this[2])
}

fun Box.expand(vec: LorenzVec): Box = expand(vec.x, vec.y, vec.z)

fun Box.expand(amount: Double): Box = expand(amount, amount, amount)
