package at.hannibal2.skyhanni.data

import at.hannibal2.skyhanni.api.event.HandleEvent
import at.hannibal2.skyhanni.api.event.HandleEvent.Companion.HIGHEST
import at.hannibal2.skyhanni.data.jsonobjects.repo.IslandTypeJson
import at.hannibal2.skyhanni.events.RepositoryReloadEvent
import at.hannibal2.skyhanni.skyhannimodule.SkyHanniModule
import at.hannibal2.skyhanni.utils.LocationUtils.isInside
import at.hannibal2.skyhanni.utils.LorenzVec
import at.hannibal2.skyhanni.utils.SkyBlockUtils
import de.hype.bingonet.shared.constants.Islands
import net.minecraft.util.AxisAlignedBB

enum class IslandType(private val nameFallback: String) {
    PRIVATE_ISLAND("Private Island"),
    PRIVATE_ISLAND_GUEST("Private Island Guest"),
    THE_END("The End"),
    KUUDRA_ARENA("Kuudra"),
    CRIMSON_ISLE("Crimson Isle"),
    DWARVEN_MINES("Dwarven Mines"),
    DUNGEON_HUB("Dungeon Hub"),
    CATACOMBS("Catacombs"),

    HUB("Hub"),
    DARK_AUCTION("Dark Auction"),
    THE_FARMING_ISLANDS("The Farming Islands"),
    CRYSTAL_HOLLOWS("Crystal Hollows"),
    THE_PARK("The Park"),
    DEEP_CAVERNS("Deep Caverns"),
    GOLD_MINES("Gold Mine"),
    GARDEN("Garden"),
    GARDEN_GUEST("Garden Guest"),
    SPIDER_DEN("Spider's Den"),
    WINTER("Jerry's Workshop"),
    THE_RIFT("The Rift"),
    MINESHAFT("Mineshaft"),
    BACKWATER_BAYOU("Backwater Bayou"),
    GALATEA("Galatea"),

    NONE(""),
    ANY(""),
    UNKNOWN("???"),
    ;

    fun isValidIsland(): Boolean = when (this) {
        NONE,
        ANY,
        UNKNOWN,
        -> false

        else -> true
    }

    fun guestVariant(): IslandType = when (this) {
        PRIVATE_ISLAND -> PRIVATE_ISLAND_GUEST
        GARDEN -> GARDEN_GUEST
        else -> this
    }

    // TODO: IslandTags
    fun hasGuestVariant(): Boolean = when (this) {
        PRIVATE_ISLAND, GARDEN -> true
        else -> false
    }

    var islandData: IslandData? = null
        private set

    val displayName: String get() = islandData?.name ?: nameFallback

    fun isInBounds(vec: LorenzVec): Boolean = islandData?.boundingBox?.isInside(vec) ?: true

    @SkyHanniModule
    companion object {
        /**
         * The maximum amount of players that can be on an island.
         */
        var maxPlayers = 24
            private set

        /**
         * The maximum amount of players that can be on a mega hub.
         */
        var maxPlayersMega = 60
            private set

        fun getByName(name: String): IslandType = getByNameOrNull(name) ?: error("IslandType not found: '$name'")
        fun getByNameOrUnknown(name: String): IslandType = getByNameOrNull(name) ?: UNKNOWN
        fun getByNameOrNull(name: String): IslandType? = entries.find { it.displayName == name }

        fun getByIdOrNull(id: String): IslandType? = entries.find { it.islandData?.apiName == id }
        fun getByIdOrUnknown(id: String): IslandType = getByIdOrNull(id) ?: UNKNOWN

        @HandleEvent(priority = HIGHEST)
        fun onRepoReload(event: RepositoryReloadEvent) {
            val data = event.getConstant<IslandTypeJson>("misc/IslandType")

            val islandDataMap = data.islands.mapValues {
                val island = it.value
                val boundingBox = island.bounds?.let { bounds ->
                    AxisAlignedBB(
                        bounds.minX.toDouble(), 0.0, bounds.minZ.toDouble(),
                        bounds.maxX.toDouble(), 256.0, bounds.maxZ.toDouble(),
                    )
                }

                IslandData(island.name, island.apiName, island.maxPlayers ?: data.maxPlayers, boundingBox)
            }

            entries.forEach { islandType ->
                islandType.islandData = islandDataMap[islandType.name]
            }

            maxPlayers = data.maxPlayers
            maxPlayersMega = data.maxPlayersMega
        }
    }

    // TODO rename to isInIsland once the funciton in lorenz utils is gone
    fun isCurrent() = SkyBlockUtils.inSkyBlock && SkyBlockUtils.currentIsland == this
}

data class IslandData(
    val name: String,
    val apiName: String?,
    val maxPlayers: Int,
    val boundingBox: AxisAlignedBB?,
)
@Suppress("CyclomaticComplexMethod")
fun IslandType.toBNIsland(): Islands? {
    return when (this) {
        IslandType.PRIVATE_ISLAND -> Islands.PRIVATE_ISLAND
        IslandType.PRIVATE_ISLAND_GUEST -> Islands.PRIVATE_ISLAND
        IslandType.THE_END -> Islands.THE_END
        IslandType.KUUDRA_ARENA -> Islands.KUUDRA
        IslandType.CRIMSON_ISLE -> Islands.CRIMSON_ISLE
        IslandType.DWARVEN_MINES -> Islands.DWARVEN_MINES
        IslandType.DUNGEON_HUB -> Islands.DUNGEON_HUB
        IslandType.CATACOMBS -> Islands.DUNGEON
        IslandType.HUB -> Islands.HUB
        IslandType.DARK_AUCTION -> Islands.DARK_AUCTION
        IslandType.THE_FARMING_ISLANDS -> Islands.THE_FARMING_ISLANDS
        IslandType.CRYSTAL_HOLLOWS -> Islands.CRYSTAL_HOLLOWS
        IslandType.THE_PARK -> Islands.THE_PARK
        IslandType.DEEP_CAVERNS -> Islands.DEEP_CAVERNS
        IslandType.GOLD_MINES -> Islands.GOLD_MINE
        IslandType.GARDEN -> Islands.GARDEN
        IslandType.GARDEN_GUEST -> Islands.GARDEN
        IslandType.SPIDER_DEN -> Islands.SPIDERS_DEN
        IslandType.WINTER -> Islands.JERRYS_WORKSHOP
        IslandType.THE_RIFT -> Islands.THE_RIFT
        IslandType.MINESHAFT -> Islands.GLACITE_TUNNEL
        IslandType.BACKWATER_BAYOU -> Islands.BAYOU
        IslandType.GALATEA -> Islands.GALATEA
        IslandType.NONE -> null
        IslandType.ANY -> null
        IslandType.UNKNOWN -> null
    }
}
