package at.hannibal2.skyhanni.utils

import at.hannibal2.skyhanni.data.mob.Mob
import at.hannibal2.skyhanni.data.mob.MobData
import at.hannibal2.skyhanni.skyhannimodule.SkyHanniModule
import at.hannibal2.skyhanni.utils.EntityUtils.cleanName
import at.hannibal2.skyhanni.utils.LocationUtils.distanceTo
import at.hannibal2.skyhanni.utils.LocationUtils.rayIntersects
import at.hannibal2.skyhanni.utils.compat.InventoryCompat.isNotEmpty
import at.hannibal2.skyhanni.utils.compat.getInventoryItems
import net.minecraft.client.resources.I18n
import net.minecraft.entity.Entity
import net.minecraft.entity.EntityLivingBase
import net.minecraft.entity.item.EntityArmorStand
import net.minecraft.entity.player.EntityPlayer

@SkyHanniModule
object MobUtils {

    private val defaultArmorStandName
        get() =
            //#if MC < 1.21
            I18n.format("entity.ArmorStand.name")
    //#else
    //$$ I18n.translate("entity.minecraft.armor_stand")
    //#endif

    // The corresponding ArmorStand for a mob has always the ID + 1 (with some exceptions)
    fun getArmorStand(entity: Entity, offset: Int = 1) = getNextEntity(entity, offset) as? EntityArmorStand

    fun getNextEntity(entity: Entity, offset: Int): Entity? = EntityUtils.getEntityByID(entity.entityId + offset)

    fun getArmorStandByRangeAll(entity: Entity, range: Double) =
        EntityUtils.getEntitiesNearby<EntityArmorStand>(entity.getLorenzVec(), range)

    fun getClosestArmorStand(entity: Entity, range: Double) =
        getArmorStandByRangeAll(entity, range).minByOrNull { it.distanceTo(entity) }

    fun getClosestArmorStandWithName(entity: Entity, range: Double, name: String) =
        getArmorStandByRangeAll(entity, range).filter { it.cleanName().startsWith(name) }.minByOrNull { it.distanceTo(entity) }

    fun EntityArmorStand.isDefaultValue() = this.name == defaultArmorStandName

    fun EntityArmorStand?.takeNonDefault() = this?.takeIf { !it.isDefaultValue() }

    fun EntityArmorStand.hasEmptyInventory() = getInventoryItems().none { it.isNotEmpty() }

    fun EntityArmorStand.isCompletelyDefault() = isDefaultValue() && hasEmptyInventory()

    class OwnerShip(val ownerName: String) {
        val ownerPlayer = MobData.players.firstOrNull { it.name == ownerName }
        override fun equals(other: Any?): Boolean {
            if (other is EntityPlayer) return ownerPlayer == other || ownerName == other.name
            if (other is String) return ownerName == other
            return false
        }

        override fun hashCode(): Int {
            return ownerName.hashCode()
        }
    }

    fun rayTraceForMob(entity: Entity, distance: Double, partialTicks: Float, offset: LorenzVec = LorenzVec()) =
        rayTraceForMob(entity, partialTicks, offset)?.takeIf {
            it.baseEntity.distanceTo(entity.getLorenzVec()) <= distance
        }

    fun rayTraceForMobs(
        entity: Entity,
        distance: Double,
        partialTicks: Float,
        offset: LorenzVec = LorenzVec(),
    ) =
        rayTraceForMobs(entity, partialTicks, offset)?.filter {
            it.baseEntity.distanceTo(entity.getLorenzVec()) <= distance
        }.takeIf { it?.isNotEmpty() ?: false }

    fun rayTraceForMob(entity: Entity, partialTicks: Float, offset: LorenzVec = LorenzVec()) =
        rayTraceForMobs(entity, partialTicks, offset)?.firstOrNull()

    fun rayTraceForMobs(entity: Entity, partialTicks: Float, offset: LorenzVec = LorenzVec()): List<Mob>? {
        //#if MC < 1.21
        val pos = entity.getPositionEyes(partialTicks).toLorenzVec() + offset
        val look = entity.getLook(partialTicks).toLorenzVec().normalize()
        //#else
        //$$ val look = entity.rotationVector.toLorenzVec().normalize()
        //$$ val pos = entity.eyePos.toLorenzVec() + offset
        //#endif
        val possibleEntities = MobData.entityToMob.filterKeys {
            it !is EntityArmorStand &&
                it.entityBoundingBox.rayIntersects(
                    pos, look,
                )
        }.values
        if (possibleEntities.isEmpty()) return null
        return possibleEntities.distinct().sortedBy { it.baseEntity.distanceTo(pos) }
    }

    val EntityLivingBase.mob: Mob? get() = MobData.entityToMob[this]

    val Entity.mob: Mob? get() = (this as? EntityLivingBase)?.mob

}
