#version 150

layout(std140) uniform LightmapInfo {
    float AmbientLightFactor;
    float SkyFactor;
    float BlockFactor;
    float NightVisionFactor;
    float DarknessScale;
    float DarkenWorldFactor;
    float BrightnessFactor;
    vec3 SkyLightColor;
    vec3 AmbientColor;
} lightmapInfo;

float getAdjustedAmbientLightFactor() {
    return (lightmapInfo.AmbientLightFactor > 0.24 && lightmapInfo.AmbientLightFactor < 0.26) ? 0.0 : lightmapInfo.AmbientLightFactor;
}

int toint(vec3 col) {
  ivec3 icol = ivec3(col*255.);
  return (icol.r << 16) + (icol.g << 8) + icol.b;
}
bool isInEnd() {
    int skyCol = toint(lightmapInfo.SkyLightColor);
    return skyCol == 0xe57fff;
}


in vec2 texCoord;

out vec4 fragColor;

float get_brightness(float level) {
    float curved_level = level / (4.0 - 3.0 * level);
    return mix(curved_level, 1.0, getAdjustedAmbientLightFactor());
}

vec3 notGamma(vec3 color) {
    if(isInEnd()) {
        // this is the 1.21.9 function
        float maxComponent = max(max(color.x, color.y), color.z);
        float maxInverted = 1.0f - maxComponent;
        float maxScaled = 1.0f - maxInverted * maxInverted * maxInverted * maxInverted;
        return color * (maxScaled / maxComponent);
    }
    // otherwise 1.21.8 function
    vec3 nx = 1.0 - color;
    return 1.0 - nx * nx * nx * nx;
}

void main() {
    float block_brightness = get_brightness(floor(texCoord.x * 16) / 15) * lightmapInfo.BlockFactor;
    float sky_brightness = get_brightness(floor(texCoord.y * 16) / 15) * lightmapInfo.SkyFactor;

    // cubic nonsense, dips to yellowish in the middle, white when fully saturated
    vec3 color = vec3(
        block_brightness,
        block_brightness * ((block_brightness * 0.6 + 0.4) * 0.6 + 0.4),
        block_brightness * (block_brightness * block_brightness * 0.6 + 0.4)
    );

    if(isInEnd()) {
        color = mix(mix(color, lightmapInfo.AmbientColor, lightmapInfo.AmbientLightFactor), color, sky_brightness);

        color += (lightmapInfo.SkyLightColor * 1.5) * sky_brightness;
        color = mix(color, vec3(0.75), 0.04);
    } else {
        color += lightmapInfo.SkyLightColor * sky_brightness;
        color = mix(color, vec3(0.75), 0.04);

        vec3 darkened_color = color * vec3(0.7, 0.6, 0.6);
        color = mix(color, darkened_color, lightmapInfo.DarkenWorldFactor);
    }

    if (lightmapInfo.NightVisionFactor > 0.0) {
        // scale up uniformly until 1.0 is hit by one of the colors
        float max_component = max(color.r, max(color.g, color.b));
        if (max_component < 1.0) {
            vec3 bright_color = color / max_component;
            color = mix(color, bright_color, lightmapInfo.NightVisionFactor);
        }
    }

    color = clamp(color - (isInEnd() ? vec3(lightmapInfo.DarknessScale) : vec3(0.0)), 0.0, 1.0);
    
    vec3 notGamma = notGamma(color);
    color = mix(color, notGamma, lightmapInfo.BrightnessFactor);
    color = mix(color, vec3(0.75), 0.04);
    color = clamp(color, 0.0, 1.0);

    fragColor = vec4(color, 1.0);
}
