/*
 * Decompiled with CFR 0.152.
 */
package MPP.marketPlacePlus.database;

import MPP.marketPlacePlus.MarketPlacePlus;
import MPP.marketPlacePlus.models.Bid;
import MPP.marketPlacePlus.models.CollectableItem;
import MPP.marketPlacePlus.models.MarketItem;
import MPP.marketPlacePlus.models.PlayerShop;
import MPP.marketPlacePlus.models.PriceHistory;
import MPP.marketPlacePlus.models.ShopAdvertisement;
import MPP.marketPlacePlus.models.TradeHistory;
import MPP.marketPlacePlus.utils.ItemUtils;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.UUID;
import org.bukkit.Bukkit;
import org.bukkit.Location;
import org.bukkit.World;
import org.bukkit.inventory.ItemStack;

public class DatabaseManager {
    private final MarketPlacePlus plugin;
    private Connection connection;
    private final String dbType;

    public DatabaseManager(MarketPlacePlus plugin) {
        this.plugin = plugin;
        this.dbType = plugin.getConfig().getString("database.type", "SQLITE");
    }

    public void initialize() {
        try {
            if (this.dbType.equalsIgnoreCase("MYSQL")) {
                String host = this.plugin.getConfig().getString("database.mysql.host");
                int port = this.plugin.getConfig().getInt("database.mysql.port");
                String database = this.plugin.getConfig().getString("database.mysql.database");
                String username = this.plugin.getConfig().getString("database.mysql.username");
                String password = this.plugin.getConfig().getString("database.mysql.password");
                Properties props = new Properties();
                props.setProperty("user", username);
                props.setProperty("password", password);
                props.setProperty("useSSL", "false");
                props.setProperty("autoReconnect", "true");
                props.setProperty("maxReconnects", "3");
                props.setProperty("initialTimeout", "2");
                this.connection = DriverManager.getConnection("jdbc:mysql://" + host + ":" + port + "/" + database, props);
            } else {
                if (!this.plugin.getDataFolder().exists()) {
                    this.plugin.getDataFolder().mkdirs();
                }
                try {
                    Class.forName("org.sqlite.JDBC");
                }
                catch (ClassNotFoundException e) {
                    this.plugin.getLogger().severe("SQLite JDBC driver not found!");
                    throw new RuntimeException("SQLite driver not available", e);
                }
                String dbPath = this.plugin.getDataFolder().getAbsolutePath() + "/marketplace.db";
                this.connection = DriverManager.getConnection("jdbc:sqlite:" + dbPath);
            }
            this.createTables();
            this.migrateDatabase();
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Failed to initialize database: " + e.getMessage());
            e.printStackTrace();
            throw new RuntimeException("Database initialization failed", e);
        }
    }

    private void createTables() throws SQLException {
        try (Statement stmt = this.getConnection().createStatement();){
            stmt.executeUpdate("CREATE TABLE IF NOT EXISTS market_listings (id VARCHAR(36) PRIMARY KEY,seller_id VARCHAR(36) NOT NULL,seller_name VARCHAR(16) NOT NULL,item_data TEXT NOT NULL,price DOUBLE NOT NULL,listed_time BIGINT NOT NULL,expiry_time BIGINT NOT NULL,category VARCHAR(32) NOT NULL,sold BOOLEAN DEFAULT FALSE,buyer_id VARCHAR(36),buyer_name VARCHAR(16),sold_time BIGINT)");
            stmt.executeUpdate("CREATE TABLE IF NOT EXISTS player_shops (id VARCHAR(36) PRIMARY KEY,owner_id VARCHAR(36) NOT NULL,owner_name VARCHAR(16) NOT NULL,shop_name VARCHAR(32) NOT NULL,description VARCHAR(128) DEFAULT '',created_time BIGINT NOT NULL,total_earnings DOUBLE DEFAULT 0,visit_count INT DEFAULT 0,active BOOLEAN DEFAULT TRUE,world_name VARCHAR(64),loc_x DOUBLE,loc_y DOUBLE,loc_z DOUBLE,loc_yaw FLOAT,loc_pitch FLOAT)");
            stmt.executeUpdate("CREATE TABLE IF NOT EXISTS shop_items (shop_id VARCHAR(36) NOT NULL,slot INT NOT NULL,item_data TEXT NOT NULL,price DOUBLE NOT NULL,stock INT NOT NULL,PRIMARY KEY (shop_id, slot))");
            this.migrateShopTable(stmt);
            stmt.executeUpdate("CREATE TABLE IF NOT EXISTS trade_history (id VARCHAR(36) PRIMARY KEY,seller_id VARCHAR(36) NOT NULL,seller_name VARCHAR(16) NOT NULL,buyer_id VARCHAR(36) NOT NULL,buyer_name VARCHAR(16) NOT NULL,item_data TEXT NOT NULL,price DOUBLE NOT NULL,quantity INT NOT NULL,trade_time BIGINT NOT NULL,trade_type VARCHAR(16) NOT NULL)");
            stmt.executeUpdate("CREATE TABLE IF NOT EXISTS price_history (item_key VARCHAR(128) NOT NULL,price DOUBLE NOT NULL,quantity INT NOT NULL,time BIGINT NOT NULL)");
            stmt.executeUpdate("CREATE TABLE IF NOT EXISTS player_balances (player_id VARCHAR(36) PRIMARY KEY,balance DOUBLE DEFAULT " + this.plugin.getConfig().getDouble("economy.starting-balance") + ")");
            stmt.executeUpdate("CREATE INDEX IF NOT EXISTS idx_listings_seller ON market_listings(seller_id)");
            stmt.executeUpdate("CREATE INDEX IF NOT EXISTS idx_shops_owner ON player_shops(owner_id)");
            stmt.executeUpdate("CREATE INDEX IF NOT EXISTS idx_history_time ON trade_history(trade_time)");
            stmt.executeUpdate("CREATE INDEX IF NOT EXISTS idx_price_history ON price_history(item_key, time)");
            stmt.executeUpdate("CREATE TABLE IF NOT EXISTS shop_ratings (shop_id VARCHAR(36) NOT NULL,rater_id VARCHAR(36) NOT NULL,rating INT NOT NULL,rating_time BIGINT NOT NULL,PRIMARY KEY (shop_id, rater_id))");
            stmt.executeUpdate("CREATE TABLE IF NOT EXISTS shop_advertisements (shop_id VARCHAR(36) NOT NULL,owner_id VARCHAR(36) NOT NULL,shop_name VARCHAR(32) NOT NULL,message VARCHAR(128) NOT NULL,start_time BIGINT NOT NULL,end_time BIGINT NOT NULL,cost DOUBLE NOT NULL,PRIMARY KEY (shop_id, start_time))");
            stmt.executeUpdate("CREATE INDEX IF NOT EXISTS idx_ads_end_time ON shop_advertisements(end_time)");
            stmt.executeUpdate("CREATE TABLE IF NOT EXISTS auction_bids (id VARCHAR(36) PRIMARY KEY,auction_id VARCHAR(36) NOT NULL,bidder_id VARCHAR(36) NOT NULL,bidder_name VARCHAR(16) NOT NULL,amount DOUBLE NOT NULL,bid_time BIGINT NOT NULL,is_winning BOOLEAN DEFAULT FALSE,refunded BOOLEAN DEFAULT FALSE)");
            stmt.executeUpdate("CREATE INDEX IF NOT EXISTS idx_bids_auction ON auction_bids(auction_id)");
            stmt.executeUpdate("CREATE INDEX IF NOT EXISTS idx_bids_bidder ON auction_bids(bidder_id)");
            stmt.executeUpdate("CREATE TABLE IF NOT EXISTS market_bans (player_id VARCHAR(36) PRIMARY KEY,ban_reason VARCHAR(256) NOT NULL,ban_time BIGINT NOT NULL,banned_by VARCHAR(36) NOT NULL)");
            stmt.executeUpdate("CREATE TABLE IF NOT EXISTS market_reports (id VARCHAR(36) PRIMARY KEY,reporter_id VARCHAR(36) NOT NULL,reporter_name VARCHAR(16) NOT NULL,reported_id VARCHAR(36) NOT NULL,reported_name VARCHAR(16) NOT NULL,report_type VARCHAR(16) NOT NULL,target_id VARCHAR(36) NOT NULL,reason VARCHAR(256) NOT NULL,details TEXT,report_time BIGINT NOT NULL,status VARCHAR(16) DEFAULT 'PENDING',resolved_by VARCHAR(36),resolved_time BIGINT,resolution VARCHAR(256))");
            stmt.executeUpdate("CREATE INDEX IF NOT EXISTS idx_reports_status ON market_reports(status)");
            stmt.executeUpdate("CREATE INDEX IF NOT EXISTS idx_reports_reported ON market_reports(reported_id)");
            stmt.executeUpdate("CREATE INDEX IF NOT EXISTS idx_reports_time ON market_reports(report_time)");
            stmt.executeUpdate("CREATE TABLE IF NOT EXISTS collection_items (id VARCHAR(36) PRIMARY KEY,player_id VARCHAR(36) NOT NULL,item_data TEXT NOT NULL,type VARCHAR(32) NOT NULL,price DOUBLE NOT NULL,added_time VARCHAR(32) NOT NULL,expiry_time VARCHAR(32) NOT NULL)");
            stmt.executeUpdate("CREATE INDEX IF NOT EXISTS idx_collection_player ON collection_items(player_id)");
            stmt.executeUpdate("CREATE INDEX IF NOT EXISTS idx_collection_expiry ON collection_items(expiry_time)");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void migrateDatabase() throws SQLException {
        block36: {
            try (Statement stmt = this.getConnection().createStatement();){
                if (this.dbType.equalsIgnoreCase("SQLITE")) {
                    ResultSet rs = stmt.executeQuery("PRAGMA table_info(player_shops)");
                    HashSet<String> existingColumns = new HashSet<String>();
                    boolean hasWorldColumn = false;
                    try {
                        while (rs.next()) {
                            String columnName = rs.getString("name").toLowerCase();
                            existingColumns.add(columnName);
                            if (!columnName.equals("world")) continue;
                            hasWorldColumn = true;
                        }
                    }
                    finally {
                        rs.close();
                    }
                    if (hasWorldColumn) {
                        this.plugin.getLogger().info("Detected old player_shops table schema. Migrating to new schema...");
                        stmt.executeUpdate("ALTER TABLE player_shops RENAME TO player_shops_old");
                        stmt.executeUpdate("CREATE TABLE IF NOT EXISTS player_shops (id VARCHAR(36) PRIMARY KEY,owner_id VARCHAR(36) NOT NULL,owner_name VARCHAR(16) NOT NULL,shop_name VARCHAR(32) NOT NULL,description VARCHAR(128) DEFAULT '',created_time BIGINT NOT NULL,total_earnings DOUBLE DEFAULT 0,visit_count INT DEFAULT 0,active BOOLEAN DEFAULT TRUE)");
                        try {
                            stmt.executeUpdate("INSERT INTO player_shops (id, owner_id, owner_name, shop_name, description, created_time, total_earnings, active) SELECT id, owner_id, owner_name, shop_name, '', created_time, total_earnings, active FROM player_shops_old");
                            stmt.executeUpdate("DROP TABLE player_shops_old");
                            this.plugin.getLogger().info("Successfully migrated player_shops table to new schema");
                        }
                        catch (SQLException e) {
                            this.plugin.getLogger().warning("Failed to migrate data from old table: " + e.getMessage());
                        }
                    } else {
                        HashMap<String, String> requiredColumns = new HashMap<String, String>();
                        requiredColumns.put("description", "VARCHAR(128) DEFAULT ''");
                        requiredColumns.put("visit_count", "INT DEFAULT 0");
                        requiredColumns.put("average_rating", "DOUBLE DEFAULT 0.0");
                        requiredColumns.put("total_ratings", "INT DEFAULT 0");
                        for (Map.Entry column : requiredColumns.entrySet()) {
                            if (existingColumns.contains(((String)column.getKey()).toLowerCase())) continue;
                            try {
                                this.plugin.getLogger().info("Migrating database: Adding " + (String)column.getKey() + " column to player_shops table");
                                stmt.executeUpdate("ALTER TABLE player_shops ADD COLUMN " + (String)column.getKey() + " " + (String)column.getValue());
                                this.plugin.getLogger().info("Added " + (String)column.getKey() + " column successfully");
                            }
                            catch (SQLException sQLException) {
                                this.plugin.getLogger().warning("Failed to add " + (String)column.getKey() + " column: " + sQLException.getMessage());
                            }
                        }
                    }
                    rs = stmt.executeQuery("PRAGMA table_info(market_listings)");
                    HashSet<String> listingColumns = new HashSet<String>();
                    try {
                        while (rs.next()) {
                            listingColumns.add(rs.getString("name").toLowerCase());
                        }
                    }
                    finally {
                        rs.close();
                    }
                    HashMap<String, String> auctionColumns = new HashMap<String, String>();
                    auctionColumns.put("is_auction", "BOOLEAN DEFAULT FALSE");
                    auctionColumns.put("starting_bid", "DOUBLE DEFAULT 0");
                    auctionColumns.put("current_bid", "DOUBLE DEFAULT 0");
                    auctionColumns.put("bid_increment", "DOUBLE DEFAULT 0");
                    auctionColumns.put("highest_bidder_id", "VARCHAR(36)");
                    auctionColumns.put("highest_bidder_name", "VARCHAR(16)");
                    auctionColumns.put("total_bids", "INT DEFAULT 0");
                    for (Map.Entry entry : auctionColumns.entrySet()) {
                        if (listingColumns.contains(((String)entry.getKey()).toLowerCase())) continue;
                        try {
                            this.plugin.getLogger().info("Adding " + (String)entry.getKey() + " column to market_listings table");
                            stmt.executeUpdate("ALTER TABLE market_listings ADD COLUMN " + (String)entry.getKey() + " " + (String)entry.getValue());
                        }
                        catch (SQLException e) {
                            this.plugin.getLogger().warning("Failed to add " + (String)entry.getKey() + " column: " + e.getMessage());
                        }
                    }
                    break block36;
                }
                if (!this.dbType.equalsIgnoreCase("MYSQL")) break block36;
                String dbName = this.plugin.getConfig().getString("database.mysql.database");
                HashSet<String> existingColumns = new HashSet<String>();
                try (ResultSet rs = stmt.executeQuery("SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = '" + dbName + "' AND TABLE_NAME = 'player_shops'");){
                    while (rs.next()) {
                        existingColumns.add(rs.getString("COLUMN_NAME").toLowerCase());
                    }
                }
                HashMap<String, String> requiredColumns = new HashMap<String, String>();
                requiredColumns.put("description", "VARCHAR(128) DEFAULT ''");
                requiredColumns.put("visit_count", "INT DEFAULT 0");
                requiredColumns.put("average_rating", "DOUBLE DEFAULT 0.0");
                requiredColumns.put("total_ratings", "INT DEFAULT 0");
                for (Map.Entry column : requiredColumns.entrySet()) {
                    if (existingColumns.contains(((String)column.getKey()).toLowerCase())) continue;
                    try {
                        this.plugin.getLogger().info("Migrating database: Adding " + (String)column.getKey() + " column to player_shops table");
                        stmt.executeUpdate("ALTER TABLE player_shops ADD COLUMN " + (String)column.getKey() + " " + (String)column.getValue());
                        this.plugin.getLogger().info("Added " + (String)column.getKey() + " column successfully");
                    }
                    catch (SQLException sQLException) {
                        this.plugin.getLogger().warning("Failed to add " + (String)column.getKey() + " column: " + sQLException.getMessage());
                    }
                }
            }
            catch (SQLException e) {
                this.plugin.getLogger().warning("Failed to migrate database: " + e.getMessage());
            }
        }
    }

    public void saveListing(MarketItem item) {
        if (this.connection == null) {
            this.plugin.getLogger().severe("Database connection is null, cannot save listing");
            return;
        }
        String sql = this.dbType.equalsIgnoreCase("MYSQL") ? "INSERT INTO market_listings (id, seller_id, seller_name, item_data, price, listed_time, expiry_time, category, sold, buyer_id, buyer_name, sold_time, is_auction, starting_bid, current_bid, bid_increment, highest_bidder_id, highest_bidder_name, total_bids) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE seller_id=VALUES(seller_id), seller_name=VALUES(seller_name), item_data=VALUES(item_data), price=VALUES(price), listed_time=VALUES(listed_time), expiry_time=VALUES(expiry_time), category=VALUES(category), sold=VALUES(sold), buyer_id=VALUES(buyer_id), buyer_name=VALUES(buyer_name), sold_time=VALUES(sold_time), is_auction=VALUES(is_auction), starting_bid=VALUES(starting_bid), current_bid=VALUES(current_bid), bid_increment=VALUES(bid_increment), highest_bidder_id=VALUES(highest_bidder_id), highest_bidder_name=VALUES(highest_bidder_name), total_bids=VALUES(total_bids)" : "INSERT OR REPLACE INTO market_listings (id, seller_id, seller_name, item_data, price, listed_time, expiry_time, category, sold, buyer_id, buyer_name, sold_time, is_auction, starting_bid, current_bid, bid_increment, highest_bidder_id, highest_bidder_name, total_bids) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        try (PreparedStatement ps = this.getConnection().prepareStatement(sql);){
            ps.setString(1, item.getId());
            ps.setString(2, item.getSellerId().toString());
            ps.setString(3, item.getSellerName());
            ps.setString(4, item.getSerializedItem());
            ps.setDouble(5, item.getPrice());
            ps.setLong(6, this.toTimestamp(item.getListedTime()));
            ps.setLong(7, this.toTimestamp(item.getExpiryTime()));
            ps.setString(8, item.getCategory());
            ps.setBoolean(9, item.isSold());
            ps.setString(10, item.getBuyerId() != null ? item.getBuyerId().toString() : null);
            ps.setString(11, item.getBuyerName());
            ps.setLong(12, item.getSoldTime() != null ? this.toTimestamp(item.getSoldTime()) : 0L);
            ps.setBoolean(13, item.isAuction());
            ps.setDouble(14, item.getStartingBid());
            ps.setDouble(15, item.getCurrentBid());
            ps.setDouble(16, item.getBidIncrement());
            ps.setString(17, item.getHighestBidderId() != null ? item.getHighestBidderId().toString() : null);
            ps.setString(18, item.getHighestBidderName());
            ps.setInt(19, item.getTotalBids());
            int rowsAffected = ps.executeUpdate();
            if (rowsAffected == 0) {
                this.plugin.getLogger().warning("No rows affected when saving listing " + item.getId());
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to save listing " + item.getId() + ": " + e.getMessage());
            e.printStackTrace();
        }
    }

    public void updateListing(MarketItem item) {
        String sql = item.isAuction() ? "UPDATE market_listings SET sold = ?, buyer_id = ?, buyer_name = ?, sold_time = ?, current_bid = ?, highest_bidder_id = ?, highest_bidder_name = ?, total_bids = ? WHERE id = ?" : "UPDATE market_listings SET sold = ?, buyer_id = ?, buyer_name = ?, sold_time = ? WHERE id = ?";
        try (PreparedStatement ps = this.getConnection().prepareStatement(sql);){
            ps.setBoolean(1, item.isSold());
            ps.setString(2, item.getBuyerId() != null ? item.getBuyerId().toString() : null);
            ps.setString(3, item.getBuyerName());
            ps.setLong(4, item.getSoldTime() != null ? this.toTimestamp(item.getSoldTime()) : 0L);
            if (item.isAuction()) {
                ps.setDouble(5, item.getCurrentBid());
                ps.setString(6, item.getHighestBidderId() != null ? item.getHighestBidderId().toString() : null);
                ps.setString(7, item.getHighestBidderName());
                ps.setInt(8, item.getTotalBids());
                ps.setString(9, item.getId());
            } else {
                ps.setString(5, item.getId());
            }
            ps.executeUpdate();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    public void deleteListing(String listingId) {
        try (PreparedStatement ps = this.getConnection().prepareStatement("DELETE FROM market_listings WHERE id = ?");){
            ps.setString(1, listingId);
            ps.executeUpdate();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    public void markListingAsProcessed(String listingId) {
        try (PreparedStatement ps = this.getConnection().prepareStatement("UPDATE market_listings SET sold = 1 WHERE id = ?");){
            ps.setString(1, listingId);
            int updated = ps.executeUpdate();
            if (updated == 0) {
                this.plugin.getLogger().warning("No listing found to mark as processed: " + listingId);
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to mark listing as processed: " + e.getMessage());
            e.printStackTrace();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Loose catch block
     * Enabled aggressive exception aggregation
     */
    public boolean processExpiredListingsTransaction(List<String> listingIds, List<CollectableItem> collectionItems) {
        try (Connection conn = this.getConnection();){
            boolean oldAutoCommit = conn.getAutoCommit();
            conn.setAutoCommit(false);
            try (PreparedStatement markProcessed = conn.prepareStatement("UPDATE market_listings SET sold = 1 WHERE id = ?");){
                for (String string : listingIds) {
                    markProcessed.setString(1, string);
                    markProcessed.addBatch();
                }
                markProcessed.executeBatch();
            }
            String insertSql = "INSERT INTO collection_items (id, player_id, item_data, type, price, added_time, expiry_time) VALUES (?, ?, ?, ?, ?, ?, ?)";
            try (PreparedStatement insertItems = conn.prepareStatement(insertSql);){
                for (CollectableItem item : collectionItems) {
                    insertItems.setString(1, item.getId().toString());
                    insertItems.setString(2, item.getPlayerId().toString());
                    insertItems.setString(3, ItemUtils.serializeItem(item.getItemStack()));
                    insertItems.setString(4, item.getType().name());
                    insertItems.setDouble(5, item.getPrice());
                    insertItems.setString(6, item.getAddedTime().toString());
                    insertItems.setString(7, item.getExpiryTime().toString());
                    insertItems.addBatch();
                }
                insertItems.executeBatch();
            }
            try (PreparedStatement deleteListings2 = conn.prepareStatement("DELETE FROM market_listings WHERE id = ?");){
                for (String listingId : listingIds) {
                    deleteListings2.setString(1, listingId);
                    deleteListings2.addBatch();
                }
                deleteListings2.executeBatch();
            }
            conn.commit();
            boolean deleteListings2 = true;
            try {
                conn.setAutoCommit(oldAutoCommit);
            }
            catch (SQLException sQLException) {
                this.plugin.getLogger().severe("Failed to reset auto-commit: " + sQLException.getMessage());
            }
            return deleteListings2;
            catch (SQLException e) {
                boolean bl;
                block46: {
                    this.plugin.getLogger().severe("Transaction failed for processing expired listings: " + e.getMessage());
                    e.printStackTrace();
                    try {
                        conn.rollback();
                    }
                    catch (SQLException rollbackEx) {
                        this.plugin.getLogger().severe("Failed to rollback transaction: " + rollbackEx.getMessage());
                    }
                    bl = false;
                    try {
                        conn.setAutoCommit(oldAutoCommit);
                    }
                    catch (SQLException sQLException) {
                        this.plugin.getLogger().severe("Failed to reset auto-commit: " + sQLException.getMessage());
                    }
                    if (conn == null) break block46;
                    conn.close();
                }
                return bl;
                {
                    catch (Throwable throwable) {
                        try {
                            conn.setAutoCommit(oldAutoCommit);
                        }
                        catch (SQLException e2) {
                            this.plugin.getLogger().severe("Failed to reset auto-commit: " + e2.getMessage());
                        }
                        throw throwable;
                    }
                }
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to get database connection: " + e.getMessage());
            return false;
        }
    }

    public void deleteListings(List<String> listingIds) {
        if (listingIds == null || listingIds.isEmpty()) {
            return;
        }
        try (PreparedStatement ps = this.getConnection().prepareStatement("DELETE FROM market_listings WHERE id = ?");){
            for (String id : listingIds) {
                ps.setString(1, id);
                ps.addBatch();
            }
            ps.executeBatch();
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to batch delete listings: " + e.getMessage());
            e.printStackTrace();
        }
    }

    public List<MarketItem> loadAllListings() {
        ArrayList<MarketItem> listings = new ArrayList<MarketItem>();
        if (this.connection == null) {
            this.plugin.getLogger().severe("Database connection is null, cannot load listings");
            return listings;
        }
        try (Statement stmt = this.getConnection().createStatement();
             ResultSet rs = stmt.executeQuery("SELECT * FROM market_listings");){
            int loadedCount = 0;
            int failedCount = 0;
            while (rs.next()) {
                try {
                    String buyerIdStr;
                    String itemData = rs.getString("item_data");
                    ItemStack item = ItemUtils.deserializeItem(itemData);
                    if (item == null) {
                        this.plugin.getLogger().warning("Failed to deserialize item for listing " + rs.getString("id"));
                        ++failedCount;
                        continue;
                    }
                    MarketItem marketItem = new MarketItem(rs.getString("id"), UUID.fromString(rs.getString("seller_id")), rs.getString("seller_name"), item, rs.getDouble("price"), this.fromTimestamp(rs.getLong("listed_time")), this.fromTimestamp(rs.getLong("expiry_time")), rs.getString("category"), itemData);
                    if (rs.getBoolean("sold") && (buyerIdStr = rs.getString("buyer_id")) != null) {
                        marketItem.markAsSold(UUID.fromString(buyerIdStr), rs.getString("buyer_name"));
                    }
                    try {
                        if (rs.getBoolean("is_auction")) {
                            marketItem.enableAuction(rs.getDouble("starting_bid"), rs.getDouble("bid_increment"));
                            String highestBidderId = rs.getString("highest_bidder_id");
                            if (highestBidderId != null) {
                                marketItem.placeBid(UUID.fromString(highestBidderId), rs.getString("highest_bidder_name"), rs.getDouble("current_bid"));
                            }
                        }
                    }
                    catch (SQLException sQLException) {
                        // empty catch block
                    }
                    listings.add(marketItem);
                    ++loadedCount;
                }
                catch (Exception e) {
                    this.plugin.getLogger().warning("Error loading listing: " + e.getMessage());
                    ++failedCount;
                }
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to load listings from database: " + e.getMessage());
            e.printStackTrace();
        }
        return listings;
    }

    public void saveTradeHistory(TradeHistory trade) {
        try (PreparedStatement ps = this.getConnection().prepareStatement("INSERT INTO trade_history VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");){
            ps.setString(1, trade.getId());
            ps.setString(2, trade.getSellerId().toString());
            ps.setString(3, trade.getSellerName());
            ps.setString(4, trade.getBuyerId().toString());
            ps.setString(5, trade.getBuyerName());
            ps.setString(6, trade.getSerializedItem());
            ps.setDouble(7, trade.getPrice());
            ps.setInt(8, trade.getQuantity());
            ps.setLong(9, this.toTimestamp(trade.getTradeTime()));
            ps.setString(10, trade.getType().name());
            ps.executeUpdate();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    public void savePriceHistory(PriceHistory history) {
        for (PriceHistory.PricePoint point : history.getPricePoints()) {
            try {
                PreparedStatement ps = this.getConnection().prepareStatement("INSERT INTO price_history VALUES (?, ?, ?, ?)");
                try {
                    ps.setString(1, history.getItemKey());
                    ps.setDouble(2, point.getPrice());
                    ps.setInt(3, point.getQuantity());
                    ps.setLong(4, this.toTimestamp(point.getTime()));
                    ps.executeUpdate();
                }
                finally {
                    if (ps == null) continue;
                    ps.close();
                }
            }
            catch (SQLException e) {
                e.printStackTrace();
            }
        }
    }

    public Map<String, PriceHistory> loadAllPriceHistories() {
        HashMap<String, PriceHistory> histories = new HashMap<String, PriceHistory>();
        try (Statement stmt = this.getConnection().createStatement();
             ResultSet rs = stmt.executeQuery("SELECT * FROM price_history ORDER BY time DESC");){
            while (rs.next()) {
                String itemKey = rs.getString("item_key");
                PriceHistory history = histories.computeIfAbsent(itemKey, PriceHistory::new);
                history.addPricePoint(rs.getDouble("price"), rs.getInt("quantity"), this.fromTimestamp(rs.getLong("time")));
            }
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
        return histories;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public double getPlayerBalance(UUID playerId) {
        try {
            ResultSet rs;
            block14: {
                double d;
                PreparedStatement ps;
                block15: {
                    ps = this.getConnection().prepareStatement("SELECT balance FROM player_balances WHERE player_id = ?");
                    ps.setString(1, playerId.toString());
                    rs = ps.executeQuery();
                    try {
                        if (!rs.next()) break block14;
                        d = rs.getDouble("balance");
                        if (rs == null) break block15;
                    }
                    catch (Throwable throwable) {
                        if (rs == null) throw throwable;
                        try {
                            rs.close();
                            throw throwable;
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                        throw throwable;
                    }
                    rs.close();
                }
                if (ps == null) return d;
                ps.close();
                return d;
            }
            if (rs != null) {
                rs.close();
            }
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
        double startingBalance = this.plugin.getConfig().getDouble("economy.starting-balance");
        this.setPlayerBalance(playerId, startingBalance);
        return startingBalance;
    }

    public void setPlayerBalance(UUID playerId, double balance) {
        String query = this.dbType.equalsIgnoreCase("MYSQL") ? "INSERT INTO player_balances VALUES (?, ?) ON DUPLICATE KEY UPDATE balance = ?" : "INSERT OR REPLACE INTO player_balances VALUES (?, ?)";
        try (PreparedStatement ps = this.getConnection().prepareStatement(query);){
            ps.setString(1, playerId.toString());
            ps.setDouble(2, balance);
            if (this.dbType.equalsIgnoreCase("MYSQL")) {
                ps.setDouble(3, balance);
            }
            ps.executeUpdate();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    public void close() {
        try {
            if (this.connection != null && !this.connection.isClosed()) {
                this.connection.close();
            }
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    public boolean isConnected() {
        try {
            return this.connection != null && !this.connection.isClosed() && this.connection.isValid(2);
        }
        catch (SQLException e) {
            return false;
        }
    }

    public void testConnection() {
        if (!this.isConnected()) {
            this.plugin.getLogger().severe("Database connection is not valid!");
            return;
        }
        try (Statement stmt = this.getConnection().createStatement();
             ResultSet rs = stmt.executeQuery("SELECT COUNT(*) FROM market_listings");){
            if (rs.next()) {
                int n = rs.getInt(1);
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to query database: " + e.getMessage());
        }
    }

    private long toTimestamp(LocalDateTime dateTime) {
        return dateTime.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
    }

    private LocalDateTime fromTimestamp(long timestamp) {
        return LocalDateTime.ofInstant(Instant.ofEpochMilli(timestamp), ZoneId.systemDefault());
    }

    public List<TradeHistory> getPlayerTradeHistory(UUID playerId, int limit) {
        ArrayList<TradeHistory> trades = new ArrayList<TradeHistory>();
        String sql = "SELECT * FROM trade_history WHERE (buyer_id = ? OR seller_id = ?) ORDER BY trade_time DESC LIMIT ?";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(sql);){
            stmt.setString(1, playerId.toString());
            stmt.setString(2, playerId.toString());
            stmt.setInt(3, limit);
            try (ResultSet rs = stmt.executeQuery();){
                while (rs.next()) {
                    String serializedItem = rs.getString("item_data");
                    ItemStack itemStack = ItemUtils.deserializeItem(serializedItem);
                    if (itemStack == null) continue;
                    TradeHistory trade = new TradeHistory(rs.getString("id"), UUID.fromString(rs.getString("seller_id")), rs.getString("seller_name"), UUID.fromString(rs.getString("buyer_id")), rs.getString("buyer_name"), itemStack, serializedItem, rs.getDouble("price"), rs.getInt("quantity"), this.fromTimestamp(rs.getLong("trade_time")), TradeHistory.TradeType.valueOf(rs.getString("trade_type")));
                    trades.add(trade);
                }
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to load trade history: " + e.getMessage());
            e.printStackTrace();
        }
        return trades;
    }

    public void saveShop(PlayerShop shop) {
        String sql = this.dbType.equalsIgnoreCase("MYSQL") ? "INSERT INTO player_shops (id, owner_id, owner_name, shop_name, description, created_time, total_earnings, visit_count, active, average_rating, total_ratings, world_name, loc_x, loc_y, loc_z, loc_yaw, loc_pitch) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE owner_id=VALUES(owner_id), owner_name=VALUES(owner_name), shop_name=VALUES(shop_name), description=VALUES(description), created_time=VALUES(created_time), total_earnings=VALUES(total_earnings), visit_count=VALUES(visit_count), active=VALUES(active), average_rating=VALUES(average_rating), total_ratings=VALUES(total_ratings), world_name=VALUES(world_name), loc_x=VALUES(loc_x), loc_y=VALUES(loc_y), loc_z=VALUES(loc_z), loc_yaw=VALUES(loc_yaw), loc_pitch=VALUES(loc_pitch)" : "INSERT OR REPLACE INTO player_shops (id, owner_id, owner_name, shop_name, description, created_time, total_earnings, visit_count, active, average_rating, total_ratings, world_name, loc_x, loc_y, loc_z, loc_yaw, loc_pitch) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(sql);){
            stmt.setString(1, shop.getId());
            stmt.setString(2, shop.getOwnerId().toString());
            stmt.setString(3, shop.getOwnerName());
            stmt.setString(4, shop.getShopName());
            stmt.setString(5, shop.getDescription());
            stmt.setLong(6, this.toTimestamp(shop.getCreatedTime()));
            stmt.setDouble(7, shop.getTotalEarnings());
            stmt.setInt(8, shop.getVisitCount());
            stmt.setBoolean(9, shop.isActive());
            stmt.setDouble(10, shop.getAverageRating());
            stmt.setInt(11, shop.getTotalRatings());
            if (shop.getLocation() != null) {
                stmt.setString(12, shop.getLocation().getWorld().getName());
                stmt.setDouble(13, shop.getLocation().getX());
                stmt.setDouble(14, shop.getLocation().getY());
                stmt.setDouble(15, shop.getLocation().getZ());
                stmt.setFloat(16, shop.getLocation().getYaw());
                stmt.setFloat(17, shop.getLocation().getPitch());
            } else {
                stmt.setNull(12, 12);
                stmt.setNull(13, 8);
                stmt.setNull(14, 8);
                stmt.setNull(15, 8);
                stmt.setNull(16, 6);
                stmt.setNull(17, 6);
            }
            stmt.executeUpdate();
            this.saveShopItems(shop);
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to save shop: " + e.getMessage());
            e.printStackTrace();
        }
    }

    public void updateShop(PlayerShop shop) {
        this.saveShop(shop);
    }

    public void deleteShop(String shopId) {
        try {
            String deleteItems = "DELETE FROM shop_items WHERE shop_id = ?";
            try (PreparedStatement stmt = this.getConnection().prepareStatement(deleteItems);){
                stmt.setString(1, shopId);
                stmt.executeUpdate();
            }
            String deleteShop = "DELETE FROM player_shops WHERE id = ?";
            try (PreparedStatement stmt = this.getConnection().prepareStatement(deleteShop);){
                stmt.setString(1, shopId);
                stmt.executeUpdate();
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to delete shop: " + e.getMessage());
            e.printStackTrace();
        }
    }

    private void saveShopItems(PlayerShop shop) {
        String deleteSql = "DELETE FROM shop_items WHERE shop_id = ?";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(deleteSql);){
            stmt.setString(1, shop.getId());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to clear shop items: " + e.getMessage());
        }
        String insertSql = "INSERT INTO shop_items (shop_id, slot, item_data, price, stock) VALUES (?, ?, ?, ?, ?)";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(insertSql);){
            for (Map.Entry<Integer, PlayerShop.ShopItem> entry : shop.getItems().entrySet()) {
                PlayerShop.ShopItem item = entry.getValue();
                stmt.setString(1, shop.getId());
                stmt.setInt(2, entry.getKey());
                stmt.setString(3, ItemUtils.serializeItem(item.getItem()));
                stmt.setDouble(4, item.getPrice());
                stmt.setInt(5, item.getStock());
                stmt.addBatch();
            }
            stmt.executeBatch();
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to save shop items: " + e.getMessage());
            e.printStackTrace();
        }
    }

    public List<PlayerShop> loadPlayerShops(UUID playerId) {
        ArrayList<PlayerShop> shops = new ArrayList<PlayerShop>();
        String sql = "SELECT * FROM player_shops WHERE owner_id = ?";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(sql);){
            stmt.setString(1, playerId.toString());
            try (ResultSet rs = stmt.executeQuery();){
                while (rs.next()) {
                    PlayerShop shop = this.loadShopFromResultSet(rs);
                    if (shop == null) continue;
                    this.loadShopItems(shop);
                    shops.add(shop);
                }
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to load player shops: " + e.getMessage());
            e.printStackTrace();
        }
        return shops;
    }

    public List<PlayerShop> loadAllShops() {
        ArrayList<PlayerShop> shops = new ArrayList<PlayerShop>();
        String sql = "SELECT * FROM player_shops";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(sql);
             ResultSet rs = stmt.executeQuery();){
            while (rs.next()) {
                PlayerShop shop = this.loadShopFromResultSet(rs);
                if (shop == null) continue;
                this.loadShopItems(shop);
                shops.add(shop);
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to load all shops: " + e.getMessage());
            e.printStackTrace();
        }
        return shops;
    }

    private PlayerShop loadShopFromResultSet(ResultSet rs) throws SQLException {
        String id = rs.getString("id");
        UUID ownerId = UUID.fromString(rs.getString("owner_id"));
        String ownerName = rs.getString("owner_name");
        String shopName = rs.getString("shop_name");
        String description = rs.getString("description");
        LocalDateTime createdTime = this.fromTimestamp(rs.getLong("created_time"));
        PlayerShop shop = new PlayerShop(id, ownerId, ownerName, shopName, description, createdTime);
        shop.setTotalEarnings(rs.getDouble("total_earnings"));
        shop.setActive(rs.getBoolean("active"));
        try {
            shop.setAverageRating(rs.getDouble("average_rating"));
            shop.setTotalRatings(rs.getInt("total_ratings"));
        }
        catch (SQLException sQLException) {
            // empty catch block
        }
        try {
            String worldName = rs.getString("world_name");
            if (worldName != null && !rs.wasNull()) {
                double x = rs.getDouble("loc_x");
                double y = rs.getDouble("loc_y");
                double z = rs.getDouble("loc_z");
                float yaw = rs.getFloat("loc_yaw");
                float pitch = rs.getFloat("loc_pitch");
                World world = Bukkit.getWorld((String)worldName);
                if (world != null) {
                    shop.setLocation(new Location(world, x, y, z, yaw, pitch));
                }
            }
        }
        catch (SQLException sQLException) {
            // empty catch block
        }
        return shop;
    }

    private void loadShopItems(PlayerShop shop) {
        String sql = "SELECT * FROM shop_items WHERE shop_id = ?";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(sql);){
            stmt.setString(1, shop.getId());
            try (ResultSet rs = stmt.executeQuery();){
                while (rs.next()) {
                    int slot = rs.getInt("slot");
                    String itemData = rs.getString("item_data");
                    double price = rs.getDouble("price");
                    int stock = rs.getInt("stock");
                    ItemStack item = ItemUtils.deserializeItem(itemData);
                    if (item == null) continue;
                    shop.addItem(slot, item, price, stock);
                }
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to load shop items: " + e.getMessage());
            e.printStackTrace();
        }
    }

    public Connection getConnection() {
        block8: {
            try {
                if (this.connection != null && !this.connection.isClosed()) break block8;
                if (this.connection != null) {
                    try {
                        this.connection.close();
                    }
                    catch (SQLException sQLException) {
                        // empty catch block
                    }
                }
                this.initialize();
            }
            catch (SQLException e) {
                this.plugin.getLogger().severe("Connection check failed, reinitializing: " + e.getMessage());
                if (this.connection != null) {
                    try {
                        this.connection.close();
                    }
                    catch (SQLException sQLException) {
                        // empty catch block
                    }
                }
                this.initialize();
            }
        }
        return this.connection;
    }

    public void saveShopRating(String shopId, UUID raterId, int rating) {
        String sql = this.dbType.equalsIgnoreCase("MYSQL") ? "INSERT INTO shop_ratings (shop_id, rater_id, rating, rating_time) VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE rating=VALUES(rating), rating_time=VALUES(rating_time)" : "INSERT OR REPLACE INTO shop_ratings (shop_id, rater_id, rating, rating_time) VALUES (?, ?, ?, ?)";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(sql);){
            stmt.setString(1, shopId);
            stmt.setString(2, raterId.toString());
            stmt.setInt(3, rating);
            stmt.setLong(4, this.toTimestamp(LocalDateTime.now()));
            stmt.executeUpdate();
            this.updateShopRatingStats(shopId);
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to save shop rating: " + e.getMessage());
            e.printStackTrace();
        }
    }

    private void updateShopRatingStats(String shopId) {
        block20: {
            String sql = "SELECT AVG(rating) as avg_rating, COUNT(*) as total_ratings FROM shop_ratings WHERE shop_id = ?";
            try (PreparedStatement stmt = this.getConnection().prepareStatement(sql);){
                stmt.setString(1, shopId);
                try (ResultSet rs = stmt.executeQuery();){
                    if (!rs.next()) break block20;
                    double avgRating = rs.getDouble("avg_rating");
                    int totalRatings = rs.getInt("total_ratings");
                    String updateSql = "UPDATE player_shops SET average_rating = ?, total_ratings = ? WHERE id = ?";
                    try (PreparedStatement updateStmt = this.getConnection().prepareStatement(updateSql);){
                        updateStmt.setDouble(1, avgRating);
                        updateStmt.setInt(2, totalRatings);
                        updateStmt.setString(3, shopId);
                        updateStmt.executeUpdate();
                    }
                }
            }
            catch (SQLException e) {
                this.plugin.getLogger().severe("Failed to update shop rating stats: " + e.getMessage());
                e.printStackTrace();
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public Integer getPlayerRating(String shopId, UUID playerId) {
        String sql = "SELECT rating FROM shop_ratings WHERE shop_id = ? AND rater_id = ?";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(sql);){
            stmt.setString(1, shopId);
            stmt.setString(2, playerId.toString());
            try (ResultSet rs = stmt.executeQuery();){
                if (!rs.next()) return null;
                Integer n = rs.getInt("rating");
                return n;
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to get player rating: " + e.getMessage());
            e.printStackTrace();
        }
        return null;
    }

    public void saveAdvertisement(ShopAdvertisement ad) {
        String sql = "INSERT INTO shop_advertisements (shop_id, owner_id, shop_name, message, start_time, end_time, cost) VALUES (?, ?, ?, ?, ?, ?, ?)";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(sql);){
            stmt.setString(1, ad.getShopId());
            stmt.setString(2, ad.getOwnerId().toString());
            stmt.setString(3, ad.getShopName());
            stmt.setString(4, ad.getMessage());
            stmt.setLong(5, this.toTimestamp(ad.getStartTime()));
            stmt.setLong(6, this.toTimestamp(ad.getEndTime()));
            stmt.setDouble(7, ad.getCost());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to save advertisement: " + e.getMessage());
            e.printStackTrace();
        }
    }

    public List<ShopAdvertisement> loadActiveAdvertisements() {
        ArrayList<ShopAdvertisement> ads = new ArrayList<ShopAdvertisement>();
        String sql = "SELECT * FROM shop_advertisements WHERE end_time > ?";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(sql);){
            stmt.setLong(1, this.toTimestamp(LocalDateTime.now()));
            try (ResultSet rs = stmt.executeQuery();){
                while (rs.next()) {
                    ShopAdvertisement ad = new ShopAdvertisement(rs.getString("shop_id"), UUID.fromString(rs.getString("owner_id")), rs.getString("shop_name"), rs.getString("message"), this.fromTimestamp(rs.getLong("start_time")), this.fromTimestamp(rs.getLong("end_time")), rs.getDouble("cost"));
                    ads.add(ad);
                }
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to load advertisements: " + e.getMessage());
            e.printStackTrace();
        }
        return ads;
    }

    public void saveBid(Bid bid) {
        String sql = "INSERT INTO auction_bids (id, auction_id, bidder_id, bidder_name, amount, bid_time, is_winning, refunded) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(sql);){
            stmt.setString(1, bid.getId());
            stmt.setString(2, bid.getAuctionId());
            stmt.setString(3, bid.getBidderId().toString());
            stmt.setString(4, bid.getBidderName());
            stmt.setDouble(5, bid.getAmount());
            stmt.setLong(6, this.toTimestamp(bid.getBidTime()));
            stmt.setBoolean(7, bid.isWinning());
            stmt.setBoolean(8, bid.isRefunded());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to save bid: " + e.getMessage());
            e.printStackTrace();
        }
    }

    public List<Bid> loadAuctionBids(String auctionId) {
        ArrayList<Bid> bids = new ArrayList<Bid>();
        String sql = "SELECT * FROM auction_bids WHERE auction_id = ? ORDER BY bid_time DESC";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(sql);){
            stmt.setString(1, auctionId);
            try (ResultSet rs = stmt.executeQuery();){
                while (rs.next()) {
                    Bid bid = new Bid(rs.getString("id"), rs.getString("auction_id"), UUID.fromString(rs.getString("bidder_id")), rs.getString("bidder_name"), rs.getDouble("amount"), this.fromTimestamp(rs.getLong("bid_time")));
                    bid.setWinning(rs.getBoolean("is_winning"));
                    bid.setRefunded(rs.getBoolean("refunded"));
                    bids.add(bid);
                }
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to load bids: " + e.getMessage());
            e.printStackTrace();
        }
        return bids;
    }

    public void banPlayer(UUID playerId, String reason) {
        String query = this.dbType.equalsIgnoreCase("MYSQL") ? "INSERT INTO market_bans (player_id, ban_reason, ban_time, banned_by) VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE ban_reason = VALUES(ban_reason), ban_time = VALUES(ban_time), banned_by = VALUES(banned_by)" : "INSERT OR REPLACE INTO market_bans (player_id, ban_reason, ban_time, banned_by) VALUES (?, ?, ?, ?)";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(query);){
            stmt.setString(1, playerId.toString());
            stmt.setString(2, reason);
            stmt.setLong(3, System.currentTimeMillis());
            stmt.setString(4, "ADMIN");
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to ban player: " + e.getMessage());
            e.printStackTrace();
        }
    }

    public void unbanPlayer(UUID playerId) {
        String query = "DELETE FROM market_bans WHERE player_id = ?";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(query);){
            stmt.setString(1, playerId.toString());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to unban player: " + e.getMessage());
            e.printStackTrace();
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    public boolean isPlayerBanned(UUID playerId) {
        String query = "SELECT 1 FROM market_bans WHERE player_id = ?";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(query);){
            boolean bl;
            block14: {
                stmt.setString(1, playerId.toString());
                ResultSet rs = stmt.executeQuery();
                try {
                    bl = rs.next();
                    if (rs == null) break block14;
                }
                catch (Throwable throwable) {
                    if (rs != null) {
                        try {
                            rs.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                rs.close();
            }
            return bl;
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to check ban status: " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public String getPlayerBanReason(UUID playerId) {
        String query = "SELECT ban_reason FROM market_bans WHERE player_id = ?";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(query);){
            stmt.setString(1, playerId.toString());
            try (ResultSet rs = stmt.executeQuery();){
                if (!rs.next()) return "Unknown";
                String string = rs.getString("ban_reason");
                return string;
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to get ban reason: " + e.getMessage());
            e.printStackTrace();
        }
        return "Unknown";
    }

    public List<UUID> getBannedPlayers() {
        ArrayList<UUID> bannedPlayers = new ArrayList<UUID>();
        String query = "SELECT player_id FROM market_bans";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(query);
             ResultSet rs = stmt.executeQuery();){
            while (rs.next()) {
                bannedPlayers.add(UUID.fromString(rs.getString("player_id")));
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to get banned players: " + e.getMessage());
            e.printStackTrace();
        }
        return bannedPlayers;
    }

    public void clearPlayerTradeHistory(UUID playerId) {
        String query = "DELETE FROM trade_history WHERE seller_id = ? OR buyer_id = ?";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(query);){
            stmt.setString(1, playerId.toString());
            stmt.setString(2, playerId.toString());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to clear player trade history: " + e.getMessage());
            e.printStackTrace();
        }
    }

    public List<TradeHistory> getPlayerTradeHistory(UUID playerId) {
        ArrayList<TradeHistory> history = new ArrayList<TradeHistory>();
        String query = "SELECT * FROM trade_history WHERE seller_id = ? OR buyer_id = ? ORDER BY trade_time DESC";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(query);){
            stmt.setString(1, playerId.toString());
            stmt.setString(2, playerId.toString());
            try (ResultSet rs = stmt.executeQuery();){
                while (rs.next()) {
                    String itemData = rs.getString("item_data");
                    ItemStack item = ItemUtils.deserializeItem(itemData);
                    TradeHistory trade = new TradeHistory(rs.getString("id"), UUID.fromString(rs.getString("seller_id")), rs.getString("seller_name"), UUID.fromString(rs.getString("buyer_id")), rs.getString("buyer_name"), item, itemData, rs.getDouble("price"), rs.getInt("quantity"), this.fromTimestamp(rs.getLong("trade_time")), TradeHistory.TradeType.valueOf(rs.getString("trade_type")));
                    history.add(trade);
                }
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to get player trade history: " + e.getMessage());
            e.printStackTrace();
        }
        return history;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public int getTotalTransactionCount() {
        String query = "SELECT COUNT(*) FROM trade_history";
        try (PreparedStatement stmt = this.getConnection().prepareStatement(query);
             ResultSet rs = stmt.executeQuery();){
            if (!rs.next()) return 0;
            int n = rs.getInt(1);
            return n;
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to get transaction count: " + e.getMessage());
            e.printStackTrace();
        }
        return 0;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public double getTotalTransactionVolume() {
        ResultSet rs;
        block13: {
            String query = "SELECT SUM(price) FROM trade_history";
            try {
                double d;
                PreparedStatement stmt;
                block14: {
                    stmt = this.getConnection().prepareStatement(query);
                    rs = stmt.executeQuery();
                    try {
                        if (!rs.next()) break block13;
                        d = rs.getDouble(1);
                        if (rs == null) break block14;
                    }
                    catch (Throwable throwable) {
                        if (rs == null) throw throwable;
                        try {
                            rs.close();
                            throw throwable;
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                        throw throwable;
                    }
                    rs.close();
                }
                if (stmt == null) return d;
                stmt.close();
                return d;
            }
            catch (SQLException e) {
                this.plugin.getLogger().severe("Failed to get transaction volume: " + e.getMessage());
                e.printStackTrace();
                return 0.0;
            }
        }
        if (rs == null) return 0.0;
        rs.close();
        return 0.0;
    }

    private void migrateShopTable(Statement stmt) {
        try {
            Connection conn = this.getConnection();
            DatabaseMetaData meta = conn.getMetaData();
            ResultSet rs = meta.getColumns(null, null, "player_shops", null);
            boolean hasWorldName = false;
            boolean hasLocX = false;
            boolean hasLocY = false;
            boolean hasLocZ = false;
            boolean hasLocYaw = false;
            boolean hasLocPitch = false;
            boolean hasAverageRating = false;
            boolean hasTotalRatings = false;
            while (rs.next()) {
                String columnName;
                switch (columnName = rs.getString("COLUMN_NAME").toLowerCase()) {
                    case "world_name": {
                        hasWorldName = true;
                        break;
                    }
                    case "loc_x": {
                        hasLocX = true;
                        break;
                    }
                    case "loc_y": {
                        hasLocY = true;
                        break;
                    }
                    case "loc_z": {
                        hasLocZ = true;
                        break;
                    }
                    case "loc_yaw": {
                        hasLocYaw = true;
                        break;
                    }
                    case "loc_pitch": {
                        hasLocPitch = true;
                        break;
                    }
                    case "average_rating": {
                        hasAverageRating = true;
                        break;
                    }
                    case "total_ratings": {
                        hasTotalRatings = true;
                    }
                }
            }
            rs.close();
            if (!hasWorldName) {
                stmt.executeUpdate("ALTER TABLE player_shops ADD COLUMN world_name VARCHAR(64)");
                this.plugin.getLogger().info("Added world_name column to player_shops table");
            }
            if (!hasLocX) {
                stmt.executeUpdate("ALTER TABLE player_shops ADD COLUMN loc_x DOUBLE");
                this.plugin.getLogger().info("Added loc_x column to player_shops table");
            }
            if (!hasLocY) {
                stmt.executeUpdate("ALTER TABLE player_shops ADD COLUMN loc_y DOUBLE");
                this.plugin.getLogger().info("Added loc_y column to player_shops table");
            }
            if (!hasLocZ) {
                stmt.executeUpdate("ALTER TABLE player_shops ADD COLUMN loc_z DOUBLE");
                this.plugin.getLogger().info("Added loc_z column to player_shops table");
            }
            if (!hasLocYaw) {
                stmt.executeUpdate("ALTER TABLE player_shops ADD COLUMN loc_yaw FLOAT");
                this.plugin.getLogger().info("Added loc_yaw column to player_shops table");
            }
            if (!hasLocPitch) {
                stmt.executeUpdate("ALTER TABLE player_shops ADD COLUMN loc_pitch FLOAT");
                this.plugin.getLogger().info("Added loc_pitch column to player_shops table");
            }
            if (!hasAverageRating) {
                stmt.executeUpdate("ALTER TABLE player_shops ADD COLUMN average_rating DOUBLE DEFAULT 0");
                this.plugin.getLogger().info("Added average_rating column to player_shops table");
            }
            if (!hasTotalRatings) {
                stmt.executeUpdate("ALTER TABLE player_shops ADD COLUMN total_ratings INT DEFAULT 0");
                this.plugin.getLogger().info("Added total_ratings column to player_shops table");
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().warning("Failed to migrate shop table: " + e.getMessage());
        }
    }
}

