/*
 * Decompiled with CFR 0.152.
 */
package MPP.marketPlacePlus.managers;

import MPP.marketPlacePlus.MarketPlacePlus;
import MPP.marketPlacePlus.managers.PlaceholderDataManager;
import MPP.marketPlacePlus.models.MarketItem;
import MPP.marketPlacePlus.models.PlayerShop;
import java.time.Duration;
import java.time.Instant;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import org.bukkit.Bukkit;
import org.bukkit.entity.Player;
import org.bukkit.plugin.Plugin;

public class CacheManager {
    private final MarketPlacePlus plugin;
    private final Map<String, CacheEntry<?>> cache;
    private final ScheduledExecutorService executor;
    private final Duration defaultTTL;
    private final PlaceholderDataManager placeholderData;

    public CacheManager(MarketPlacePlus plugin) {
        this.plugin = plugin;
        this.cache = new ConcurrentHashMap();
        this.executor = new ScheduledThreadPoolExecutor(1);
        this.defaultTTL = Duration.ofMinutes(plugin.getConfig().getInt("cache.default-ttl-minutes", 5));
        this.placeholderData = new PlaceholderDataManager(plugin);
        this.startCleanupTask();
    }

    private void startCleanupTask() {
        int cleanupInterval = this.plugin.getConfig().getInt("cache.cleanup-interval-minutes", 10);
        this.executor.scheduleAtFixedRate(() -> {
            int removed = 0;
            Iterator<Map.Entry<String, CacheEntry<?>>> iterator = this.cache.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<String, CacheEntry<?>> entry = iterator.next();
                if (!entry.getValue().isExpired()) continue;
                iterator.remove();
                ++removed;
            }
        }, cleanupInterval, cleanupInterval, TimeUnit.MINUTES);
    }

    public <T> T get(String key, Supplier<T> loader) {
        return this.get(key, loader, this.defaultTTL);
    }

    public <T> T get(String key, Supplier<T> loader, Duration ttl) {
        T value;
        CacheEntry<?> entry = this.cache.get(key);
        if (entry != null && !entry.isExpired()) {
            return (T)entry.getValue();
        }
        if (entry != null) {
            entry.recordMiss();
        }
        if ((value = loader.get()) != null) {
            this.cache.put(key, new CacheEntry<T>(key, value, ttl));
        }
        return value;
    }

    public <T> void put(String key, T value) {
        this.put(key, value, this.defaultTTL);
    }

    public <T> void put(String key, T value, Duration ttl) {
        this.cache.put(key, new CacheEntry<T>(key, value, ttl));
    }

    public void invalidate(String key) {
        this.cache.remove(key);
    }

    public void invalidatePattern(String pattern) {
        this.cache.entrySet().removeIf(entry -> ((String)entry.getKey()).matches(pattern));
    }

    public void invalidateAll() {
        this.cache.clear();
        this.plugin.getLogger().info("Cache cleared: all entries invalidated");
    }

    public List<MarketItem> getCachedActiveListings() {
        return this.get("listings:active", () -> this.plugin.getAuctionManager().getActiveListings(), Duration.ofMinutes(1L));
    }

    public List<MarketItem> getCachedPlayerListings(UUID playerId) {
        return this.get("listings:player:" + String.valueOf(playerId), () -> this.plugin.getAuctionManager().getPlayerListings(playerId), Duration.ofMinutes(2L));
    }

    public List<PlayerShop> getCachedActiveShops() {
        return this.get("shops:active", () -> this.plugin.getShopManager().getActiveShops(), Duration.ofMinutes(3L));
    }

    public PlayerShop getCachedShop(String shopId) {
        return this.get("shop:" + shopId, () -> this.plugin.getShopManager().getShop(shopId), Duration.ofMinutes(5L));
    }

    public List<PlayerShop> getCachedPlayerShops(UUID playerId) {
        return this.get("shops:player:" + String.valueOf(playerId), () -> this.plugin.getShopManager().getPlayerShops(playerId), Duration.ofMinutes(5L));
    }

    public Double getCachedPlayerBalance(UUID playerId) {
        return this.get("balance:" + String.valueOf(playerId), () -> {
            Player player = Bukkit.getPlayer((UUID)playerId);
            if (player != null) {
                return this.plugin.getEconomyManager().getBalance(player);
            }
            return this.plugin.getDatabaseManager().getPlayerBalance(playerId);
        }, Duration.ofSeconds(30L));
    }

    public void invalidateListingCache() {
        this.invalidatePattern("listings:.*");
    }

    public void invalidateShopCache() {
        this.invalidatePattern("shops:.*");
        this.invalidatePattern("shop:.*");
    }

    public void invalidatePlayerCache(UUID playerId) {
        this.invalidate("listings:player:" + String.valueOf(playerId));
        this.invalidate("shops:player:" + String.valueOf(playerId));
        this.invalidate("balance:" + String.valueOf(playerId));
    }

    public CacheStatistics getStatistics() {
        long totalHits = 0L;
        long totalMisses = 0L;
        int entryCount = this.cache.size();
        for (CacheEntry<?> entry : this.cache.values()) {
            totalHits += entry.hits;
            totalMisses += entry.misses;
        }
        return new CacheStatistics(totalHits, totalMisses, entryCount);
    }

    public void warmupCache() {
        this.plugin.getServer().getScheduler().runTaskAsynchronously((Plugin)this.plugin, () -> {
            this.getCachedActiveListings();
            this.getCachedActiveShops();
            this.plugin.getLogger().info("Cache warmed up with initial data");
        });
    }

    public void shutdown() {
        this.executor.shutdown();
        try {
            if (!this.executor.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.executor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.executor.shutdownNow();
        }
        this.cache.clear();
    }

    public double getTotalMarketVolume() {
        return this.get("market_volume", () -> this.placeholderData.getTotalMarketVolume(), Duration.ofMinutes(5L));
    }

    public double getAverageItemPrice() {
        return this.get("avg_price", () -> this.placeholderData.getAverageItemPrice(), Duration.ofMinutes(5L));
    }

    public String getTrendingItem() {
        return this.get("trending_item", () -> this.placeholderData.getTrendingItem(), Duration.ofMinutes(10L));
    }

    public double getTotalTaxCollected() {
        return this.placeholderData.getTotalTaxCollected();
    }

    public List<UUID> getTopSellers() {
        return this.get("top_sellers", () -> this.placeholderData.getTopSellers(), Duration.ofMinutes(15L));
    }

    public List<UUID> getTopBuyers() {
        return this.get("top_buyers", () -> this.placeholderData.getTopBuyers(), Duration.ofMinutes(15L));
    }

    public List<UUID> getTopEarners() {
        return this.get("top_earners", () -> this.placeholderData.getTopEarners(), Duration.ofMinutes(15L));
    }

    public List<UUID> getTopAuctioneers() {
        return this.get("top_auctioneers", () -> this.placeholderData.getTopAuctioneers(), Duration.ofMinutes(15L));
    }

    public List<UUID> getTopTraders() {
        return this.get("top_traders", () -> this.placeholderData.getTopTraders(), Duration.ofMinutes(15L));
    }

    public double getPlayerTaxSaved(UUID playerId) {
        return this.placeholderData.getPlayerTaxSaved(playerId);
    }

    private static class CacheEntry<T> {
        private final T value;
        private final Instant expiry;
        private final String key;
        private long hits;
        private long misses;

        public CacheEntry(String key, T value, Duration ttl) {
            this.key = key;
            this.value = value;
            this.expiry = Instant.now().plus(ttl);
            this.hits = 0L;
            this.misses = 0L;
        }

        public boolean isExpired() {
            return Instant.now().isAfter(this.expiry);
        }

        public T getValue() {
            ++this.hits;
            return this.value;
        }

        public void recordMiss() {
            ++this.misses;
        }

        public double getHitRate() {
            long total = this.hits + this.misses;
            return total == 0L ? 0.0 : (double)this.hits / (double)total;
        }
    }

    public static class CacheStatistics {
        public final long hits;
        public final long misses;
        public final int entries;
        public final double hitRate;

        public CacheStatistics(long hits, long misses, int entries) {
            this.hits = hits;
            this.misses = misses;
            this.entries = entries;
            long total = hits + misses;
            this.hitRate = total == 0L ? 0.0 : (double)hits / (double)total;
        }

        public String toString() {
            return String.format("Cache Stats - Entries: %d, Hits: %d, Misses: %d, Hit Rate: %.2f%%", this.entries, this.hits, this.misses, this.hitRate * 100.0);
        }
    }
}

