/*
 * Decompiled with CFR 0.152.
 */
package MPP.marketPlacePlus.managers;

import MPP.marketPlacePlus.MarketPlacePlus;
import MPP.marketPlacePlus.gui.PlayerShopGUI;
import MPP.marketPlacePlus.models.CollectableItem;
import MPP.marketPlacePlus.models.PlayerShop;
import MPP.marketPlacePlus.utils.AntiScamUtils;
import MPP.marketPlacePlus.utils.PlayerUtils;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.bukkit.Location;
import org.bukkit.World;
import org.bukkit.entity.Player;
import org.bukkit.inventory.ItemStack;
import org.bukkit.plugin.Plugin;

public class ShopManager {
    private final MarketPlacePlus plugin;
    private final Map<String, PlayerShop> shops;
    private final Map<UUID, List<String>> playerShops;

    public ShopManager(MarketPlacePlus plugin) {
        this.plugin = plugin;
        this.shops = new ConcurrentHashMap<String, PlayerShop>();
        this.playerShops = new ConcurrentHashMap<UUID, List<String>>();
    }

    public boolean createShop(Player owner, String shopName, String description) {
        if (PlayerUtils.checkBanned(owner, this.plugin)) {
            return false;
        }
        UUID ownerId = owner.getUniqueId();
        if (!this.isValidShopName(shopName)) {
            owner.sendMessage("\u00a7cInvalid shop name! Use only letters, numbers, spaces, and basic punctuation.");
            return false;
        }
        if (!this.isValidDescription(description)) {
            owner.sendMessage("\u00a7cInvalid description! Use only letters, numbers, spaces, and basic punctuation.");
            return false;
        }
        if (this.isDuplicateShopName(ownerId, shopName)) {
            owner.sendMessage("\u00a7cYou already have a shop with that name!");
            return false;
        }
        int maxShops = this.plugin.getConfig().getInt("player-shops.max-shops-per-player", 1);
        if (this.getPlayerShopCount(ownerId) >= maxShops) {
            owner.sendMessage("\u00a7cYou have reached your maximum number of shops (" + maxShops + ")!");
            return false;
        }
        double creationCost = this.plugin.getConfig().getDouble("player-shops.shop-creation-cost", 1000.0);
        if (!this.plugin.getEconomyManager().hasBalance(owner, creationCost)) {
            owner.sendMessage("\u00a7cYou need " + this.plugin.getEconomyManager().formatMoney(creationCost) + " to create a shop!");
            return false;
        }
        this.plugin.getEconomyManager().withdrawPlayer(owner, creationCost);
        String id = UUID.randomUUID().toString();
        PlayerShop shop = new PlayerShop(id, ownerId, owner.getName(), shopName, description, LocalDateTime.now());
        shop.setLocation(owner.getLocation().clone());
        this.shops.put(id, shop);
        this.playerShops.computeIfAbsent(ownerId, k -> new ArrayList()).add(id);
        this.plugin.getDatabaseManager().saveShop(shop);
        owner.sendMessage("\u00a7aShop created at your current location!");
        owner.sendMessage("\u00a77Location: \u00a7f" + shop.getLocationString());
        return true;
    }

    public PlayerShop createShop(UUID ownerId, String ownerName, String shopName, String description) {
        String id = UUID.randomUUID().toString();
        PlayerShop shop = new PlayerShop(id, ownerId, ownerName, shopName, description, LocalDateTime.now());
        this.shops.put(id, shop);
        this.playerShops.computeIfAbsent(ownerId, k -> new ArrayList()).add(id);
        this.plugin.getDatabaseManager().saveShop(shop);
        return shop;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean deleteShop(Player owner, String shopId) {
        PlayerShop shop = this.shops.get(shopId);
        if (shop == null || !shop.getOwnerId().equals(owner.getUniqueId())) {
            return false;
        }
        PlayerShop playerShop = shop;
        synchronized (playerShop) {
            int itemsReturned = 0;
            int itemsInCollection = 0;
            ArrayList<ItemStack> overflowItems = new ArrayList<ItemStack>();
            for (PlayerShop.ShopItem shopItem : shop.getItems().values()) {
                ItemStack item = shopItem.getItem().clone();
                item.setAmount(shopItem.getStock());
                AntiScamUtils.clearItemUUID(item, (Plugin)this.plugin);
                HashMap leftover = owner.getInventory().addItem(new ItemStack[]{item});
                if (!leftover.isEmpty()) {
                    for (ItemStack overflow : leftover.values()) {
                        overflowItems.add(overflow);
                        itemsInCollection += overflow.getAmount();
                    }
                    continue;
                }
                itemsReturned += item.getAmount();
            }
            if (!overflowItems.isEmpty()) {
                this.plugin.getCollectionManager().addItems(owner.getUniqueId(), overflowItems, CollectableItem.CollectionType.SHOP_REMOVED, 0.0);
            }
            if (shop.getTotalEarnings() > 0.0) {
                this.plugin.getEconomyManager().depositPlayer(owner, shop.getTotalEarnings());
                owner.sendMessage("\u00a7aCollected \u00a76" + this.plugin.getEconomyManager().formatMoney(shop.getTotalEarnings()) + " \u00a7afrom shop earnings.");
            }
            if (itemsReturned > 0) {
                owner.sendMessage("\u00a7a" + itemsReturned + " item(s) returned to inventory.");
            }
            this.plugin.getDatabaseManager().deleteShop(shopId);
            this.shops.remove(shopId);
            List<String> playerList = this.playerShops.get(owner.getUniqueId());
            if (playerList != null) {
                playerList.remove(shopId);
            }
            return true;
        }
    }

    public PlayerShop getShop(String shopId) {
        return this.shops.get(shopId);
    }

    public void saveShop(PlayerShop shop) {
        this.plugin.getDatabaseManager().updateShop(shop);
    }

    public List<PlayerShop> getPlayerShops(UUID playerId) {
        List<String> ids = this.playerShops.get(playerId);
        if (ids == null) {
            return new ArrayList<PlayerShop>();
        }
        return ids.stream().map(this.shops::get).filter(Objects::nonNull).collect(Collectors.toList());
    }

    public List<PlayerShop> getActiveShops() {
        return this.shops.values().stream().filter(PlayerShop::isActive).collect(Collectors.toList());
    }

    public void loadShops() {
        this.shops.clear();
        this.playerShops.clear();
        List<PlayerShop> savedShops = this.plugin.getDatabaseManager().loadAllShops();
        for (PlayerShop shop : savedShops) {
            this.shops.put(shop.getId(), shop);
            this.playerShops.computeIfAbsent(shop.getOwnerId(), k -> new ArrayList()).add(shop.getId());
        }
    }

    public void loadPlayerShops(UUID playerId) {
        List<PlayerShop> playerShopList = this.getPlayerShops(playerId);
    }

    public void saveShops() {
        for (PlayerShop shop : this.shops.values()) {
            this.plugin.getDatabaseManager().updateShop(shop);
        }
    }

    public boolean updateShopName(Player owner, String shopId, String newName) {
        PlayerShop shop = this.shops.get(shopId);
        if (shop == null || !shop.getOwnerId().equals(owner.getUniqueId())) {
            return false;
        }
        shop.setShopName(newName);
        this.plugin.getDatabaseManager().updateShop(shop);
        return true;
    }

    public void cleanupAbandonedShops() {
        LocalDateTime cutoff = LocalDateTime.now().minusDays(30L);
        List toRemove = this.shops.values().stream().filter(shop -> shop.getCreatedTime().isBefore(cutoff) && shop.getTotalEarnings() == 0.0).map(PlayerShop::getId).collect(Collectors.toList());
        for (String id : toRemove) {
            PlayerShop shop2 = this.shops.remove(id);
            if (shop2 == null) continue;
            this.playerShops.computeIfPresent(shop2.getOwnerId(), (k, list) -> {
                list.remove(id);
                return list.isEmpty() ? null : list;
            });
            this.plugin.getDatabaseManager().deleteShop(id);
        }
    }

    public List<PlayerShop> getAllActiveShops() {
        return this.shops.values().stream().filter(PlayerShop::isActive).sorted((a, b) -> b.getCreatedTime().compareTo(a.getCreatedTime())).collect(Collectors.toList());
    }

    public List<PlayerShop> searchShops(String query) {
        String lowerQuery = query.toLowerCase();
        return this.shops.values().stream().filter(PlayerShop::isActive).filter(shop -> shop.getShopName().toLowerCase().contains(lowerQuery) || shop.getOwnerName().toLowerCase().contains(lowerQuery) || shop.getDescription().toLowerCase().contains(lowerQuery)).collect(Collectors.toList());
    }

    public void incrementVisitCount(String shopId) {
        PlayerShop shop = this.shops.get(shopId);
        if (shop != null) {
            shop.incrementVisitCount();
            this.plugin.getDatabaseManager().updateShop(shop);
        }
    }

    public void deleteShop(String shopId) {
        PlayerShop shop = this.shops.remove(shopId);
        if (shop != null) {
            this.playerShops.computeIfPresent(shop.getOwnerId(), (k, list) -> {
                list.remove(shopId);
                return list.isEmpty() ? null : list;
            });
            this.plugin.getDatabaseManager().deleteShop(shopId);
        }
    }

    public List<PlayerShop> getAllShops() {
        return new ArrayList<PlayerShop>(this.shops.values());
    }

    public void openShop(Player player, PlayerShop shop) {
        if (shop == null || !shop.isActive()) {
            player.sendMessage("\u00a7cThis shop is not available!");
            return;
        }
        this.incrementVisitCount(shop.getId());
        new PlayerShopGUI(player, this.plugin, shop).open();
    }

    public PlayerShop getShop(UUID shopId) {
        return this.shops.get(shopId.toString());
    }

    public PlayerShop getShopByPartialId(String partialId) {
        for (Map.Entry<String, PlayerShop> entry : this.shops.entrySet()) {
            if (!entry.getKey().startsWith(partialId)) continue;
            return entry.getValue();
        }
        return null;
    }

    public void deleteShopAdmin(UUID shopId) {
        String id = shopId.toString();
        PlayerShop shop = this.shops.get(id);
        if (shop != null) {
            this.shops.remove(id);
            List<String> playerShopIds = this.playerShops.get(shop.getOwnerId());
            if (playerShopIds != null) {
                playerShopIds.remove(id);
            }
            this.plugin.getDatabaseManager().deleteShop(id);
        }
    }

    public void removeAllPlayerShops(UUID playerId) {
        List<String> shopIds = this.playerShops.get(playerId);
        if (shopIds != null) {
            for (String id : new ArrayList<String>(shopIds)) {
                this.shops.remove(id);
                this.plugin.getDatabaseManager().deleteShop(id);
            }
            this.playerShops.remove(playerId);
        }
    }

    public int getPlayerShopCount(UUID playerId) {
        List<String> shopIds = this.playerShops.get(playerId);
        return shopIds != null ? shopIds.size() : 0;
    }

    public List<PlayerShop> getNearbyShops(Location location, double radius) {
        if (location == null || location.getWorld() == null) {
            return new ArrayList<PlayerShop>();
        }
        return this.shops.values().stream().filter(PlayerShop::isActive).filter(shop -> shop.getLocation() != null).filter(shop -> shop.getLocation().getWorld() != null).filter(shop -> shop.getLocation().getWorld().equals((Object)location.getWorld())).filter(shop -> shop.getLocation().distance(location) <= radius).sorted((a, b) -> Double.compare(a.getLocation().distance(location), b.getLocation().distance(location))).collect(Collectors.toList());
    }

    public PlayerShop getNearestShop(Location location) {
        return this.shops.values().stream().filter(PlayerShop::isActive).filter(shop -> shop.getLocation() != null).filter(shop -> shop.getLocation().getWorld() != null).filter(shop -> location.getWorld() != null).filter(shop -> shop.getLocation().getWorld().equals((Object)location.getWorld())).min((a, b) -> Double.compare(a.getLocation().distance(location), b.getLocation().distance(location))).orElse(null);
    }

    public boolean teleportToShop(Player player, String shopId) {
        PlayerShop shop = this.shops.get(shopId);
        if (shop == null || !shop.isActive()) {
            player.sendMessage("\u00a7cShop not found or inactive!");
            return false;
        }
        if (shop.getLocation() == null) {
            player.sendMessage("\u00a7cThis shop doesn't have a physical location!");
            return false;
        }
        if (!player.hasPermission("marketplaceplus.shop.teleport")) {
            player.sendMessage("\u00a7cYou don't have permission to teleport to shops!");
            return false;
        }
        double teleportCost = this.plugin.getConfig().getDouble("player-shops.teleport-cost", 100.0);
        if (teleportCost > 0.0 && !player.hasPermission("marketplaceplus.admin.bypass.fee")) {
            if (!this.plugin.getEconomyManager().hasBalance(player, teleportCost)) {
                player.sendMessage("\u00a7cYou need " + this.plugin.getEconomyManager().formatMoney(teleportCost) + " to teleport!");
                return false;
            }
            this.plugin.getEconomyManager().withdrawPlayer(player, teleportCost);
        }
        player.teleport(shop.getLocation());
        player.sendMessage("\u00a7aTeleported to \u00a7f" + shop.getShopName() + "\u00a7a!");
        this.incrementVisitCount(shopId);
        return true;
    }

    public boolean setShopLocation(Player owner, String shopId) {
        PlayerShop shop = this.shops.get(shopId);
        if (shop == null || !shop.getOwnerId().equals(owner.getUniqueId())) {
            return false;
        }
        shop.setLocation(owner.getLocation().clone());
        this.plugin.getDatabaseManager().updateShop(shop);
        owner.sendMessage("\u00a7aShop location updated to your current position!");
        owner.sendMessage("\u00a77New location: \u00a7f" + shop.getLocationString());
        return true;
    }

    public List<PlayerShop> getShopsInWorld(World world) {
        return this.shops.values().stream().filter(PlayerShop::isActive).filter(shop -> shop.getLocation() != null).filter(shop -> shop.getLocation().getWorld() != null).filter(shop -> shop.getLocation().getWorld().equals((Object)world)).collect(Collectors.toList());
    }

    private boolean isValidShopName(String name) {
        if (name == null || name.trim().isEmpty()) {
            return false;
        }
        return name.matches("^[a-zA-Z0-9\\s\\-_.,!']+$");
    }

    private boolean isValidDescription(String description) {
        if (description == null || description.isEmpty()) {
            return true;
        }
        return description.matches("^[a-zA-Z0-9\\s\\-_.,!?':;()]+$");
    }

    private boolean isDuplicateShopName(UUID ownerId, String shopName) {
        List<PlayerShop> playerShopList = this.getPlayerShops(ownerId);
        return playerShopList.stream().anyMatch(shop -> shop.getShopName().equalsIgnoreCase(shopName));
    }

    public PlayerShop getPlayerShop(UUID playerId) {
        List<String> shopIds = this.playerShops.get(playerId);
        if (shopIds != null && !shopIds.isEmpty()) {
            return this.shops.get(shopIds.get(0));
        }
        return null;
    }

    public void createShop(PlayerShop shop) {
        this.shops.put(shop.getId(), shop);
        this.playerShops.computeIfAbsent(shop.getOwnerId(), k -> new ArrayList()).add(shop.getId());
        this.plugin.getDatabaseManager().saveShop(shop);
    }

    public void updateShop(PlayerShop shop) {
        this.shops.put(shop.getId(), shop);
        this.plugin.getDatabaseManager().updateShop(shop);
    }

    public List<PlayerShop> getTopShops() {
        return this.shops.values().stream().filter(PlayerShop::isActive).sorted((a, b) -> Double.compare(b.getTotalEarnings(), a.getTotalEarnings())).limit(10L).collect(Collectors.toList());
    }
}

