/*
 * Decompiled with CFR 0.152.
 */
package com.playtimerewards.managers;

import com.playtimerewards.PlaytimeRewards;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.Bukkit;
import org.bukkit.configuration.file.FileConfiguration;
import org.bukkit.configuration.file.YamlConfiguration;
import org.bukkit.entity.Player;
import org.bukkit.plugin.Plugin;

public class DataManager {
    private final PlaytimeRewards plugin;
    private final File dataFile;
    private FileConfiguration dataConfig;
    private final Map<UUID, PlayerData> playerDataCache;
    private boolean dataModified = false;

    public DataManager(PlaytimeRewards plugin) {
        this.plugin = plugin;
        this.dataFile = new File(plugin.getDataFolder(), "data.yml");
        this.playerDataCache = new ConcurrentHashMap<UUID, PlayerData>();
        this.loadData();
        this.startAutoSaveTask();
    }

    public void loadData() {
        if (!this.dataFile.exists()) {
            try {
                this.dataFile.getParentFile().mkdirs();
                this.dataFile.createNewFile();
            }
            catch (IOException e) {
                this.plugin.getLogger().severe("Failed to create data.yml file!");
                e.printStackTrace();
                return;
            }
        }
        this.dataConfig = YamlConfiguration.loadConfiguration((File)this.dataFile);
        this.playerDataCache.clear();
        if (this.dataConfig.contains("players")) {
            for (String uuidString : this.dataConfig.getConfigurationSection("players").getKeys(false)) {
                try {
                    UUID uuid = UUID.fromString(uuidString);
                    PlayerData data = this.loadPlayerData(uuid);
                    if (data == null) continue;
                    this.playerDataCache.put(uuid, data);
                }
                catch (IllegalArgumentException e) {
                    this.plugin.getLogger().warning("Invalid UUID in data.yml: " + uuidString);
                }
            }
        }
        this.plugin.getLogger().info("Loaded data for " + this.playerDataCache.size() + " players");
    }

    private PlayerData loadPlayerData(UUID uuid) {
        String path = "players." + uuid.toString();
        if (!this.dataConfig.contains(path)) {
            return null;
        }
        String name = this.dataConfig.getString(path + ".name", "Unknown");
        int playtimeMinutes = this.dataConfig.getInt(path + ".playtime_minutes", 0);
        String language = this.dataConfig.getString(path + ".language");
        long lastSeen = this.dataConfig.getLong(path + ".last_seen", System.currentTimeMillis());
        List claimedRewards = this.dataConfig.getStringList(path + ".claimed_rewards");
        HashSet<String> claimedRewardsSet = new HashSet<String>(claimedRewards);
        return new PlayerData(name, playtimeMinutes, language, lastSeen, claimedRewardsSet);
    }

    public void saveAllData() {
        if (!this.dataModified) {
            return;
        }
        Bukkit.getScheduler().runTaskAsynchronously((Plugin)this.plugin, () -> {
            try {
                for (Map.Entry<UUID, PlayerData> entry : this.playerDataCache.entrySet()) {
                    this.savePlayerDataToConfig(entry.getKey(), entry.getValue());
                }
                this.dataConfig.save(this.dataFile);
                this.dataModified = false;
                if (this.plugin.getConfigManager().isDebugEnabled()) {
                    this.plugin.getLogger().info("Saved data for " + this.playerDataCache.size() + " players");
                }
            }
            catch (IOException e) {
                this.plugin.getLogger().severe("Failed to save data.yml!");
                e.printStackTrace();
            }
        });
    }

    public void savePlayerData(UUID uuid) {
        PlayerData data = this.playerDataCache.get(uuid);
        if (data == null) {
            return;
        }
        try {
            this.savePlayerDataToConfig(uuid, data);
            this.dataConfig.save(this.dataFile);
            if (this.plugin.getConfigManager().isDebugEnabled()) {
                this.plugin.getLogger().info("Saved data for player: " + data.getName());
            }
        }
        catch (IOException e) {
            this.plugin.getLogger().severe("Failed to save data for player " + data.getName() + "!");
            e.printStackTrace();
        }
    }

    public void saveAllDataSync() {
        try {
            for (Map.Entry<UUID, PlayerData> entry : this.playerDataCache.entrySet()) {
                this.savePlayerDataToConfig(entry.getKey(), entry.getValue());
            }
            this.dataConfig.save(this.dataFile);
            this.dataModified = false;
            this.plugin.getLogger().info("Synchronously saved data for " + this.playerDataCache.size() + " players");
        }
        catch (IOException e) {
            this.plugin.getLogger().severe("Failed to save data.yml during shutdown!");
            e.printStackTrace();
        }
    }

    private void savePlayerDataToConfig(UUID uuid, PlayerData data) {
        String path = "players." + uuid.toString();
        this.dataConfig.set(path + ".name", (Object)data.getName());
        this.dataConfig.set(path + ".playtime_minutes", (Object)data.getPlaytimeMinutes());
        this.dataConfig.set(path + ".language", (Object)data.getLanguage());
        this.dataConfig.set(path + ".last_seen", (Object)data.getLastSeen());
        this.dataConfig.set(path + ".claimed_rewards", new ArrayList<String>(data.getClaimedRewards()));
    }

    private void startAutoSaveTask() {
        int saveInterval = this.plugin.getConfigManager().getSaveInterval();
        Bukkit.getScheduler().runTaskTimerAsynchronously((Plugin)this.plugin, () -> this.saveAllData(), (long)saveInterval * 1200L, (long)saveInterval * 1200L);
    }

    public void reloadData() {
        this.saveAllData();
        this.loadData();
    }

    public PlayerData getPlayerData(UUID uuid) {
        return this.playerDataCache.computeIfAbsent(uuid, k -> {
            PlayerData data = this.loadPlayerData(uuid);
            return data != null ? data : new PlayerData();
        });
    }

    public void updatePlayerInfo(Player player) {
        PlayerData data = this.getPlayerData(player.getUniqueId());
        data.setName(player.getName());
        data.setLastSeen(System.currentTimeMillis());
        this.dataModified = true;
    }

    public int getPlayerPlaytime(UUID uuid) {
        return this.getPlayerData(uuid).getPlaytimeMinutes();
    }

    public void addPlaytime(UUID uuid, int minutes) {
        PlayerData data = this.getPlayerData(uuid);
        data.addPlaytime(minutes);
        this.dataModified = true;
        if (this.plugin.getConfigManager().shouldLogPlaytimeUpdates()) {
            this.plugin.getLogger().info("Added " + minutes + " minutes playtime to " + data.getName() + " (Total: " + data.getPlaytimeMinutes() + " minutes)");
        }
    }

    public boolean hasClaimedReward(UUID uuid, String rewardId) {
        return this.getPlayerData(uuid).getClaimedRewards().contains(rewardId);
    }

    public void claimReward(UUID uuid, String rewardId) {
        PlayerData data = this.getPlayerData(uuid);
        data.getClaimedRewards().add(rewardId);
        this.dataModified = true;
    }

    public String getPlayerLanguage(UUID uuid) {
        return this.getPlayerData(uuid).getLanguage();
    }

    public void setPlayerLanguage(UUID uuid, String language) {
        PlayerData data = this.getPlayerData(uuid);
        data.setLanguage(language);
        this.dataModified = true;
    }

    public Set<String> getClaimedRewards(UUID uuid) {
        return new HashSet<String>(this.getPlayerData(uuid).getClaimedRewards());
    }

    public static class PlayerData {
        private String name;
        private int playtimeMinutes;
        private String language;
        private long lastSeen;
        private final Set<String> claimedRewards;

        public PlayerData() {
            this.name = "Unknown";
            this.playtimeMinutes = 0;
            this.language = null;
            this.lastSeen = System.currentTimeMillis();
            this.claimedRewards = new HashSet<String>();
        }

        public PlayerData(String name, int playtimeMinutes, String language, long lastSeen, Set<String> claimedRewards) {
            this.name = name;
            this.playtimeMinutes = playtimeMinutes;
            this.language = language;
            this.lastSeen = lastSeen;
            this.claimedRewards = claimedRewards != null ? claimedRewards : new HashSet();
        }

        public String getName() {
            return this.name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public int getPlaytimeMinutes() {
            return this.playtimeMinutes;
        }

        public void addPlaytime(int minutes) {
            this.playtimeMinutes += minutes;
        }

        public String getLanguage() {
            return this.language;
        }

        public void setLanguage(String language) {
            this.language = language;
        }

        public long getLastSeen() {
            return this.lastSeen;
        }

        public void setLastSeen(long lastSeen) {
            this.lastSeen = lastSeen;
        }

        public Set<String> getClaimedRewards() {
            return this.claimedRewards;
        }
    }
}

