/*
 * Decompiled with CFR 0.152.
 */
package com.showcase.config;

import com.showcase.ShowcaseMod;
import com.showcase.command.ShowcaseManager;
import com.showcase.config.ModConfig;
import de.exlll.configlib.YamlConfigurations;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import net.fabricmc.loader.api.FabricLoader;

public class ModConfigManager {
    private static volatile ModConfig CONFIG;
    private static final Path CONFIG_DIR;
    private static final Path CONFIG_PATH;
    private static final Path BACKUP_DIR;
    private static final DateTimeFormatter BACKUP_TIMESTAMP_FORMAT;

    public static void loadConfig() {
        try {
            Files.createDirectories(CONFIG_DIR, new FileAttribute[0]);
            Files.createDirectories(BACKUP_DIR, new FileAttribute[0]);
            if (!Files.exists(CONFIG_PATH, new LinkOption[0])) {
                ShowcaseMod.LOGGER.info("Creating default configuration file...");
                CONFIG = new ModConfig();
                ModConfigManager.saveConfig();
            } else {
                CONFIG = ModConfigManager.loadConfigWithValidation();
            }
        }
        catch (Exception e) {
            ShowcaseMod.LOGGER.error("Failed to load configuration", (Throwable)e);
            throw new RuntimeException("Failed to load or create config file", e);
        }
    }

    private static ModConfig loadConfigWithValidation() throws IOException {
        try {
            ModConfig config = (ModConfig)YamlConfigurations.load((Path)CONFIG_PATH, ModConfig.class);
            if (ModConfigManager.isConfigValid(config)) {
                ShowcaseMod.LOGGER.info("Configuration loaded successfully");
                return config;
            }
            ShowcaseMod.LOGGER.warn("Configuration validation failed, attempting recovery...");
            return ModConfigManager.handleInvalidConfig(config);
        }
        catch (Exception e) {
            ShowcaseMod.LOGGER.error("Failed to parse configuration file: {}", (Object)e.getMessage());
            return ModConfigManager.handleCorruptedConfig(e);
        }
    }

    private static boolean isConfigValid(ModConfig config) {
        try {
            if (config == null) {
                ShowcaseMod.LOGGER.warn("Config is null");
                return false;
            }
            if (config.shareSettings == null || config.shareSettings.isEmpty()) {
                ShowcaseMod.LOGGER.warn("Share settings are missing or empty");
                return false;
            }
            for (ShowcaseManager.ShareType shareType : ShowcaseManager.ShareType.values()) {
                ModConfig.ShareSettings settings = config.shareSettings.get((Object)shareType);
                if (settings == null) {
                    ShowcaseMod.LOGGER.warn("Missing settings for share type: {}", (Object)shareType);
                    return false;
                }
                if (settings.cooldown < 0) {
                    ShowcaseMod.LOGGER.warn("Invalid cooldown value for {}: {}", (Object)shareType, (Object)settings.cooldown);
                    return false;
                }
                if (settings.defaultPermission >= 0 && settings.defaultPermission <= 4) continue;
                ShowcaseMod.LOGGER.warn("Invalid permission level for {}: {}", (Object)shareType, (Object)settings.defaultPermission);
                return false;
            }
            if (config.shareLink == null) {
                ShowcaseMod.LOGGER.warn("Share link settings are missing");
                return false;
            }
            if (config.shareLink.minExpiryTime <= 0 || config.shareLink.defaultExpiryTime <= 0) {
                ShowcaseMod.LOGGER.warn("Invalid expiry time settings");
                return false;
            }
            if (config.placeholders == null) {
                ShowcaseMod.LOGGER.warn("Placeholder settings are missing");
                return false;
            }
            if (config.placeholders.maxSharesPerPlayer <= 0) {
                ShowcaseMod.LOGGER.warn("Invalid max shares per player: {}", (Object)config.placeholders.maxSharesPerPlayer);
                return false;
            }
            return true;
        }
        catch (Exception e) {
            ShowcaseMod.LOGGER.error("Error during config validation", (Throwable)e);
            return false;
        }
    }

    private static ModConfig handleInvalidConfig(ModConfig invalidConfig) throws IOException {
        String timestamp = LocalDateTime.now().format(BACKUP_TIMESTAMP_FORMAT);
        Path backupPath = BACKUP_DIR.resolve("config_invalid_" + timestamp + ".yml");
        try {
            Files.copy(CONFIG_PATH, backupPath, StandardCopyOption.REPLACE_EXISTING);
            ShowcaseMod.LOGGER.info("Backed up invalid configuration to: {}", (Object)backupPath);
        }
        catch (IOException e) {
            ShowcaseMod.LOGGER.error("Failed to backup invalid configuration", (Throwable)e);
        }
        ModConfig defaultConfig = new ModConfig();
        ModConfig mergedConfig = ModConfigManager.mergeConfigs(invalidConfig, defaultConfig);
        if (ModConfigManager.isConfigValid(mergedConfig)) {
            ShowcaseMod.LOGGER.info("Successfully merged configuration with defaults");
            YamlConfigurations.save((Path)CONFIG_PATH, ModConfig.class, (Object)mergedConfig);
            return mergedConfig;
        }
        return ModConfigManager.resetToDefault("Failed to merge configuration");
    }

    private static ModConfig handleCorruptedConfig(Exception originalException) throws IOException {
        String timestamp = LocalDateTime.now().format(BACKUP_TIMESTAMP_FORMAT);
        Path backupPath = BACKUP_DIR.resolve("config_corrupted_" + timestamp + ".yml");
        try {
            Files.copy(CONFIG_PATH, backupPath, StandardCopyOption.REPLACE_EXISTING);
            ShowcaseMod.LOGGER.info("Backed up corrupted configuration to: {}", (Object)backupPath);
        }
        catch (IOException e) {
            ShowcaseMod.LOGGER.error("Failed to backup corrupted configuration", (Throwable)e);
        }
        return ModConfigManager.resetToDefault("Configuration file is corrupted: " + originalException.getMessage());
    }

    private static ModConfig resetToDefault(String reason) throws IOException {
        ShowcaseMod.LOGGER.warn("Resetting configuration to defaults. Reason: {}", (Object)reason);
        ModConfig defaultConfig = new ModConfig();
        YamlConfigurations.save((Path)CONFIG_PATH, ModConfig.class, (Object)defaultConfig);
        ShowcaseMod.LOGGER.info("Configuration has been reset to defaults and saved");
        return defaultConfig;
    }

    private static ModConfig mergeConfigs(ModConfig source, ModConfig defaults) {
        ModConfig merged = new ModConfig();
        try {
            merged.mapViewDuration = source.mapViewDuration >= -1 ? source.mapViewDuration : defaults.mapViewDuration;
            merged.itemViewDuration = source.itemViewDuration >= -1 ? source.itemViewDuration : defaults.itemViewDuration;
            int n = merged.maxPlaceholdersPerMessage = source.maxPlaceholdersPerMessage > 0 ? source.maxPlaceholdersPerMessage : defaults.maxPlaceholdersPerMessage;
            if (source.shareSettings != null && !source.shareSettings.isEmpty()) {
                for (ShowcaseManager.ShareType shareType : ShowcaseManager.ShareType.values()) {
                    ModConfig.ShareSettings sourceSettings = source.shareSettings.get((Object)shareType);
                    ModConfig.ShareSettings defaultSettings = defaults.shareSettings.get((Object)shareType);
                    if (sourceSettings != null && ModConfigManager.isShareSettingsValid(sourceSettings)) {
                        merged.shareSettings.put(shareType, sourceSettings);
                        continue;
                    }
                    merged.shareSettings.put(shareType, defaultSettings);
                    ShowcaseMod.LOGGER.info("Using default settings for share type: {}", (Object)shareType);
                }
            } else {
                merged.shareSettings = defaults.shareSettings;
            }
            merged.shareLink = source.shareLink != null && ModConfigManager.isShareLinkSettingsValid(source.shareLink) ? source.shareLink : defaults.shareLink;
            merged.statsDisplay = source.statsDisplay != null ? source.statsDisplay : defaults.statsDisplay;
            merged.itemIcons = source.itemIcons != null ? source.itemIcons : defaults.itemIcons;
            merged.placeholders = source.placeholders != null && ModConfigManager.isPlaceholderSettingsValid(source.placeholders) ? source.placeholders : defaults.placeholders;
        }
        catch (Exception e) {
            ShowcaseMod.LOGGER.error("Error during config merge, using defaults", (Throwable)e);
            return defaults;
        }
        return merged;
    }

    private static boolean isShareSettingsValid(ModConfig.ShareSettings settings) {
        return settings.cooldown >= 0 && settings.defaultPermission >= 0 && settings.defaultPermission <= 4 && settings.keywords != null;
    }

    private static boolean isShareLinkSettingsValid(ModConfig.ShareLinkSettings settings) {
        return settings.minExpiryTime > 0 && settings.defaultExpiryTime > 0;
    }

    private static boolean isPlaceholderSettingsValid(ModConfig.PlaceholderSettings settings) {
        return settings.maxSharesPerPlayer > 0 && settings.cacheDuration >= 0 && settings.statisticsUpdateInterval > 0;
    }

    public static void reloadConfig() {
        try {
            ModConfigManager.createConfigBackup("reload");
            ModConfigManager.loadConfig();
            ShowcaseMod.LOGGER.info("Configuration reloaded successfully");
        }
        catch (Exception e) {
            ShowcaseMod.LOGGER.error("Failed to reload configuration", (Throwable)e);
            throw new RuntimeException("Configuration reload failed", e);
        }
    }

    public static void saveConfig() {
        if (CONFIG != null) {
            try {
                if (!ModConfigManager.isConfigValid(CONFIG)) {
                    ShowcaseMod.LOGGER.error("Cannot save invalid configuration");
                    throw new IllegalStateException("Configuration validation failed during save");
                }
                YamlConfigurations.save((Path)CONFIG_PATH, ModConfig.class, (Object)CONFIG);
                ShowcaseMod.LOGGER.debug("Configuration saved successfully");
            }
            catch (Exception e) {
                ShowcaseMod.LOGGER.error("Failed to save configuration", (Throwable)e);
                throw new RuntimeException("Failed to save configuration", e);
            }
        }
    }

    public static boolean createConfigBackup(String reason) {
        if (!Files.exists(CONFIG_PATH, new LinkOption[0])) {
            return false;
        }
        try {
            String timestamp = LocalDateTime.now().format(BACKUP_TIMESTAMP_FORMAT);
            String filename = String.format("config_%s_%s.yml", reason, timestamp);
            Path backupPath = BACKUP_DIR.resolve(filename);
            Files.copy(CONFIG_PATH, backupPath, StandardCopyOption.REPLACE_EXISTING);
            ShowcaseMod.LOGGER.info("Configuration backed up to: {}", (Object)backupPath);
            ModConfigManager.cleanupOldBackups();
            return true;
        }
        catch (IOException e) {
            ShowcaseMod.LOGGER.error("Failed to create configuration backup", (Throwable)e);
            return false;
        }
    }

    private static void cleanupOldBackups() {
        try {
            Files.list(BACKUP_DIR).filter(path -> path.toString().endsWith(".yml")).filter(path -> path.getFileName().toString().startsWith("config_")).sorted((p1, p2) -> {
                try {
                    return Files.getLastModifiedTime(p2, new LinkOption[0]).compareTo(Files.getLastModifiedTime(p1, new LinkOption[0]));
                }
                catch (IOException e) {
                    return 0;
                }
            }).skip(10L).forEach(path -> {
                try {
                    Files.delete(path);
                    ShowcaseMod.LOGGER.debug("Deleted old backup: {}", (Object)path.getFileName());
                }
                catch (IOException e) {
                    ShowcaseMod.LOGGER.warn("Failed to delete old backup: {}", (Object)path.getFileName());
                }
            });
        }
        catch (IOException e) {
            ShowcaseMod.LOGGER.warn("Failed to clean up old backups", (Throwable)e);
        }
    }

    public static boolean resetConfigToDefaults() {
        try {
            ModConfigManager.createConfigBackup("manual_reset");
            ModConfig defaultConfig = new ModConfig();
            YamlConfigurations.save((Path)CONFIG_PATH, ModConfig.class, (Object)defaultConfig);
            CONFIG = defaultConfig;
            ShowcaseMod.LOGGER.info("Configuration has been reset to defaults");
            return true;
        }
        catch (Exception e) {
            ShowcaseMod.LOGGER.error("Failed to reset configuration to defaults", (Throwable)e);
            return false;
        }
    }

    public static boolean isCurrentConfigValid() {
        return CONFIG != null && ModConfigManager.isConfigValid(CONFIG);
    }

    public static Path getBackupDirectory() {
        return BACKUP_DIR;
    }

    public static Path getConfigDirectory() {
        return CONFIG_DIR;
    }

    public static Path getConfigPath() {
        return CONFIG_PATH;
    }

    public static ModConfig getConfig() {
        if (CONFIG == null) {
            ModConfigManager.loadConfig();
        }
        return CONFIG;
    }

    public static ModConfig.ShareSettings getShareSettings(ShowcaseManager.ShareType type) {
        return ModConfigManager.getConfig().shareSettings.get((Object)type);
    }

    public static int getShareLinkMinExpiry() {
        return ModConfigManager.getConfig().shareLink.minExpiryTime;
    }

    public static int getShareLinkDefaultExpiry() {
        return ModConfigManager.getConfig().shareLink.defaultExpiryTime;
    }

    public static int getMaxSharesPerPlayer() {
        return ModConfigManager.getConfig().placeholders.maxSharesPerPlayer;
    }

    public static boolean isPlaceholderExtensionsEnabled() {
        return ModConfigManager.getConfig().placeholders.enabled;
    }

    public static int getPlaceholderCacheDuration() {
        return ModConfigManager.getConfig().placeholders.cacheDuration;
    }

    public static boolean isStatisticsTrackingEnabled() {
        return ModConfigManager.getConfig().placeholders.enableStatisticsTracking;
    }

    public static boolean isServerStatisticsEnabled() {
        return ModConfigManager.getConfig().placeholders.enableServerStatistics;
    }

    public static boolean isPerformanceMetricsEnabled() {
        return ModConfigManager.getConfig().placeholders.enablePerformanceMetrics;
    }

    public static int getStatisticsUpdateInterval() {
        return ModConfigManager.getConfig().placeholders.statisticsUpdateInterval;
    }

    public static boolean isConditionalPlaceholdersEnabled() {
        return ModConfigManager.getConfig().placeholders.enableConditionalPlaceholders;
    }

    static {
        CONFIG_DIR = FabricLoader.getInstance().getConfigDir().resolve("showcase");
        CONFIG_PATH = CONFIG_DIR.resolve("config.yml");
        BACKUP_DIR = CONFIG_DIR.resolve("backups");
        BACKUP_TIMESTAMP_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd_HH-mm-ss");
    }
}

