/*
 * Decompiled with CFR 0.152.
 */
package dev.isxander.controlify.controllermanager;

import com.google.common.io.ByteStreams;
import com.sun.jna.Memory;
import com.sun.jna.Pointer;
import dev.isxander.controlify.Controlify;
import dev.isxander.controlify.controller.ControllerEntity;
import dev.isxander.controlify.controller.id.ControllerType;
import dev.isxander.controlify.controller.info.ControllerInfo;
import dev.isxander.controlify.controllermanager.AbstractControllerManager;
import dev.isxander.controlify.controllermanager.UniqueControllerID;
import dev.isxander.controlify.debug.DebugProperties;
import dev.isxander.controlify.driver.CompoundDriver;
import dev.isxander.controlify.driver.Driver;
import dev.isxander.controlify.driver.sdl.SDL3GamepadDriver;
import dev.isxander.controlify.driver.sdl.SDL3JoystickDriver;
import dev.isxander.controlify.driver.sdl.SDL3NativesManager;
import dev.isxander.controlify.driver.sdl.SDLUtil;
import dev.isxander.controlify.driver.steamdeck.SteamDeckDriver;
import dev.isxander.controlify.driver.steamdeck.SteamDeckUtil;
import dev.isxander.controlify.hid.ControllerHIDService;
import dev.isxander.controlify.hid.HIDDevice;
import dev.isxander.controlify.hid.HIDIdentifier;
import dev.isxander.controlify.utils.CUtil;
import dev.isxander.controlify.utils.ControllerUtils;
import dev.isxander.controlify.utils.log.ControlifyLogger;
import dev.isxander.sdl3java.api.error.SdlError;
import dev.isxander.sdl3java.api.events.SDL_EventFilter;
import dev.isxander.sdl3java.api.events.SdlEvents;
import dev.isxander.sdl3java.api.events.events.SDL_Event;
import dev.isxander.sdl3java.api.gamepad.SDL_Gamepad;
import dev.isxander.sdl3java.api.gamepad.SdlGamepad;
import dev.isxander.sdl3java.api.iostream.SDL_IOStream;
import dev.isxander.sdl3java.api.iostream.SdlIOStream;
import dev.isxander.sdl3java.api.joystick.SDL_Joystick;
import dev.isxander.sdl3java.api.joystick.SDL_JoystickGUID;
import dev.isxander.sdl3java.api.joystick.SDL_JoystickID;
import dev.isxander.sdl3java.api.joystick.SdlJoystick;
import dev.isxander.sdl3java.jna.size_t;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import net.minecraft.server.packs.resources.Resource;
import net.minecraft.server.packs.resources.ResourceProvider;
import org.jetbrains.annotations.NotNull;

public class SDLControllerManager
extends AbstractControllerManager {
    private SDL_Event event = new SDL_Event();
    private final EventFilter eventFilter;
    private boolean steamDeckConsumed = false;

    public SDLControllerManager(ControlifyLogger logger) {
        super(logger);
        logger.debugLog("Controller manager using SDL3");
        logger.validateIsTrue(SDL3NativesManager.isLoaded(), "SDL3 natives must be loaded before creating SDLControllerManager");
        this.eventFilter = new EventFilter();
        SdlEvents.SDL_SetEventFilter((SDL_EventFilter)this.eventFilter, (Pointer)Pointer.NULL);
    }

    @Override
    public void tick(boolean outOfFocus) {
        super.tick(outOfFocus);
        SdlEvents.SDL_PumpEvents();
        if (this.event == null) {
            this.logger.error("SDL_Event has somehow been set to null. Recreating...");
            this.event = new SDL_Event();
        }
        while (SdlEvents.SDL_PollEvent((SDL_Event)this.event)) {
            switch (this.event.type) {
                case 1541: {
                    SDL_JoystickID jid = this.event.jdevice.which;
                    this.logger.validateIsTrue(jid != null, "event.jdevice.which was null during SDL_EVENT_JOYSTICK_ADDED event");
                    this.logger.debugLog("SDL event: Joystick added: {}", jid.intValue());
                    SDLUniqueControllerID ucid = new SDLUniqueControllerID(jid);
                    Optional<ControllerEntity> controllerOpt = this.tryCreate(ucid, SDLControllerManager.fetchTypeFromSDL(jid).orElse(new ControllerHIDService.ControllerHIDInfo(ControllerType.DEFAULT, Optional.empty())));
                    controllerOpt.ifPresent(controller -> ControllerUtils.wrapControllerError(() -> this.onControllerConnected((ControllerEntity)controller, true), "Connecting controller", controller));
                    break;
                }
                case 1542: {
                    SDL_JoystickID jid = this.event.jdevice.which;
                    this.logger.validateIsTrue(jid != null, "event.jdevice.which was null during SDL_EVENT_JOYSTICK_REMOVED event");
                    this.logger.debugLog("SDL event: Joystick removed: {}", jid.intValue());
                    this.getController(new SDLUniqueControllerID(jid)).ifPresentOrElse(this::onControllerRemoved, () -> CUtil.LOGGER.warn("Controller removed but not found: {}", jid.intValue()));
                }
            }
        }
        SdlGamepad.SDL_UpdateGamepads();
        SdlJoystick.SDL_UpdateJoysticks();
    }

    @Override
    public void discoverControllers() {
        SDL_JoystickID[] joysticks;
        this.logger.debugLog("Discovering controllers...");
        for (SDL_JoystickID jid : joysticks = SdlJoystick.SDL_GetJoysticks()) {
            Optional<ControllerEntity> controllerOpt = this.tryCreate(new SDLUniqueControllerID(jid), SDLControllerManager.fetchTypeFromSDL(jid).orElse(new ControllerHIDService.ControllerHIDInfo(ControllerType.DEFAULT, Optional.empty())));
            controllerOpt.ifPresent(controller -> this.onControllerConnected((ControllerEntity)controller, false));
        }
    }

    @Override
    protected Optional<ControllerEntity> createController(UniqueControllerID ucid, ControllerHIDService.ControllerHIDInfo hidInfo, ControlifyLogger controllerLogger) {
        SDL_JoystickID jid = ((SDLUniqueControllerID)ucid).jid();
        controllerLogger.debugLog("Creating controller: {}", jid.intValue());
        boolean isGamepad = this.isControllerGamepad(ucid) && !DebugProperties.FORCE_JOYSTICK;
        controllerLogger.debugLog("Controller is gamepad: {}", isGamepad);
        ArrayList<Driver> drivers = new ArrayList<Driver>();
        if (SteamDeckUtil.DECK_MODE.isGamingMode() && !this.steamDeckConsumed && hidInfo.type().namespace().equals((Object)SteamDeckUtil.STEAM_DECK_NAMESPACE)) {
            controllerLogger.debugLog("Controller is steam deck candidate");
            Optional<SteamDeckDriver> steamDeckDriver = SteamDeckDriver.create(controllerLogger);
            if (steamDeckDriver.isPresent()) {
                drivers.add(steamDeckDriver.get());
                this.steamDeckConsumed = true;
                controllerLogger.debugLog("Adding SteamDeckDriver - this controller has been reserved for Steam Deck");
            }
        }
        if (isGamepad) {
            SDL_Gamepad ptrGamepad = SDLUtil.openGamepad(jid);
            drivers.add(new SDL3GamepadDriver(ptrGamepad, jid, hidInfo.type(), controllerLogger));
        } else {
            SDL_Joystick ptrJoystick = SDLUtil.openJoystick(jid);
            drivers.add(new SDL3JoystickDriver(ptrJoystick, jid, hidInfo.type(), controllerLogger));
        }
        controllerLogger.debugLog("Drivers: {}", drivers.stream().map(driver -> driver.getClass().getSimpleName()).collect(Collectors.joining(", ")));
        CompoundDriver compoundDriver = new CompoundDriver(drivers);
        ControllerInfo info = new ControllerInfo(ucid, hidInfo.type(), hidInfo.hidDevice());
        ControllerEntity controller = new ControllerEntity(info, compoundDriver, controllerLogger);
        controllerLogger.debugLog("Unique Controller ID: {}", info.ucid());
        this.addController(ucid, controller);
        return Optional.of(controller);
    }

    @Override
    public boolean probeConnectedControllers() {
        return SdlJoystick.SDL_HasJoystick() || SdlGamepad.SDL_HasGamepad();
    }

    @Override
    public boolean isControllerGamepad(UniqueControllerID ucid) {
        SDL_JoystickID jid = ((SDLUniqueControllerID)ucid).jid;
        return SdlGamepad.SDL_IsGamepad((SDL_JoystickID)jid);
    }

    @Override
    protected String getControllerSystemName(UniqueControllerID ucid) {
        SDL_JoystickID jid = ((SDLUniqueControllerID)ucid).jid;
        return this.isControllerGamepad(ucid) ? SdlGamepad.SDL_GetGamepadNameForID((SDL_JoystickID)jid) : SdlJoystick.SDL_GetJoystickNameForID((SDL_JoystickID)jid);
    }

    private Optional<ControllerEntity> getController(UniqueControllerID ucid) {
        return Optional.ofNullable(this.controllersByJid.getOrDefault(ucid, null));
    }

    @Override
    protected void loadGamepadMappings(ResourceProvider resourceProvider) {
        CUtil.LOGGER.debugLog("Loading gamepad mappings...");
        Optional resourceOpt = resourceProvider.getResource(CUtil.rl("controllers/gamecontrollerdb-sdl3.txt"));
        if (resourceOpt.isEmpty()) {
            CUtil.LOGGER.error("Failed to find game controller database.");
            return;
        }
        try (InputStream is = ((Resource)resourceOpt.get()).open();){
            byte[] bytes = ByteStreams.toByteArray((InputStream)is);
            try (Memory memory = new Memory((long)bytes.length);){
                memory.write(0L, bytes, 0, bytes.length);
                SDL_IOStream stream = SdlIOStream.SDL_IOFromConstMem((Pointer)memory, (size_t)new size_t((long)bytes.length));
                if (stream == null) {
                    throw new IllegalStateException("Failed to open stream");
                }
                int count = SdlGamepad.SDL_AddGamepadMappingsFromIO((SDL_IOStream)stream, (boolean)true);
                if (count < 0) {
                    CUtil.LOGGER.error("Failed to load gamepad mappings: {}", SdlError.SDL_GetError());
                } else if (count == 0) {
                    CUtil.LOGGER.warn("Successfully applied gamepad mappings but none were found for this OS. Unsupported OS?");
                } else {
                    CUtil.LOGGER.log("Successfully loaded {} gamepad mapping entries!", count);
                }
            }
        }
        catch (Throwable e) {
            CUtil.LOGGER.error("Failed to load gamepad mappings", e);
        }
    }

    private static Optional<ControllerHIDService.ControllerHIDInfo> fetchTypeFromSDL(SDL_JoystickID jid) {
        short vid = SdlJoystick.SDL_GetJoystickVendorForID((SDL_JoystickID)jid);
        short pid = SdlJoystick.SDL_GetJoystickProductForID((SDL_JoystickID)jid);
        SDL_JoystickGUID guid = SdlJoystick.SDL_GetJoystickGUIDForID((SDL_JoystickID)jid);
        String guidStr = guid.toString();
        if (vid != 0 && pid != 0) {
            CUtil.LOGGER.log("Using SDL to identify controller type.");
            return Optional.of(new ControllerHIDService.ControllerHIDInfo(Controlify.instance().controllerTypeManager().getControllerType(new HIDIdentifier(vid, pid)), Optional.of(new HIDDevice.SDLHidApi(vid, pid, guidStr))));
        }
        return Optional.empty();
    }

    private static class EventFilter
    implements SDL_EventFilter {
        private EventFilter() {
        }

        public boolean filterEvent(Pointer userdata, SDL_Event event) {
            switch (event.type) {
                case 1541: 
                case 1542: {
                    return true;
                }
            }
            return false;
        }
    }

    public record SDLUniqueControllerID(@NotNull SDL_JoystickID jid) implements UniqueControllerID
    {
        @Override
        public boolean equals(Object obj) {
            return obj instanceof SDLUniqueControllerID && ((SDLUniqueControllerID)obj).jid.equals((Object)this.jid);
        }

        @Override
        public String toString() {
            return "SDL-" + this.jid.longValue();
        }

        @Override
        public int hashCode() {
            return Objects.hash(this.jid.longValue());
        }
    }
}

