/*
 * Decompiled with CFR 0.152.
 */
package com.cff1028.schematicsfix;

import com.cff1028.schematicsfix.Config;
import com.cff1028.schematicsfix.SchematicNBTDetector;
import java.io.IOException;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.FileAttribute;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import net.neoforged.fml.util.thread.SidedThreadGroups;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class FileWatcher {
    private static final Logger LOGGER = LogManager.getLogger();
    private final Path watchDir;
    private WatchService watchService;
    private final Map<Path, FileTracker> trackedFiles = new ConcurrentHashMap<Path, FileTracker>();
    private final Map<WatchKey, Path> watchKeys = new HashMap<WatchKey, Path>();
    private ScheduledExecutorService scheduler;
    private volatile boolean running = false;
    private final SchematicNBTDetector nbtDetector;

    public FileWatcher(SchematicNBTDetector nbtDetector) {
        this.watchDir = Paths.get("schematics/uploaded", new String[0]).toAbsolutePath();
        this.nbtDetector = nbtDetector;
    }

    public void start() throws IOException {
        if (this.running || !((Boolean)Config.INSTANCE.enableFileWatcher.get()).booleanValue()) {
            return;
        }
        this.running = true;
        this.watchService = FileSystems.getDefault().newWatchService();
        this.scheduler = Executors.newSingleThreadScheduledExecutor(r -> new Thread((ThreadGroup)SidedThreadGroups.SERVER, r, "FileWatcher-Scheduler"));
        this.registerAllDirectories(this.watchDir);
        Thread monitorThread = new Thread((ThreadGroup)SidedThreadGroups.SERVER, this::monitorFiles, "FileWatcher-Monitor");
        monitorThread.setDaemon(true);
        monitorThread.start();
        long interval = (Long)Config.INSTANCE.stableCheckInterval.get();
        this.scheduler.scheduleAtFixedRate(this::checkFileStability, interval, interval, TimeUnit.MILLISECONDS);
        LOGGER.info("Started monitoring directory: {}", (Object)this.watchDir);
    }

    public void stop() throws IOException {
        this.running = false;
        if (this.scheduler != null) {
            this.scheduler.shutdown();
            try {
                if (!this.scheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.scheduler.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.scheduler.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
        if (this.watchService != null) {
            this.watchService.close();
        }
        this.trackedFiles.clear();
        this.watchKeys.clear();
    }

    private void registerAllDirectories(Path dir) throws IOException {
        if (!Files.exists(dir, new LinkOption[0])) {
            Files.createDirectories(dir, new FileAttribute[0]);
        }
        Files.walk(dir, new FileVisitOption[0]).filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).forEach(subDir -> {
            try {
                WatchKey key = subDir.register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.ENTRY_DELETE);
                this.watchKeys.put(key, (Path)subDir);
                LOGGER.debug("Registered directory for monitoring: {}", subDir);
            }
            catch (IOException e) {
                LOGGER.warn("Failed to register directory: {}", subDir, (Object)e);
            }
        });
    }

    private void monitorFiles() {
        while (this.running && !Thread.currentThread().isInterrupted()) {
            try {
                WatchKey key = this.watchService.take();
                Path dir = this.watchKeys.get(key);
                if (dir == null) {
                    key.reset();
                    continue;
                }
                for (WatchEvent<?> event : key.pollEvents()) {
                    WatchEvent.Kind<?> kind = event.kind();
                    if (kind == StandardWatchEventKinds.OVERFLOW) continue;
                    WatchEvent<?> ev = event;
                    Path fileName = (Path)ev.context();
                    Path filePath = dir.resolve(fileName);
                    if (Files.isDirectory(filePath, new LinkOption[0])) {
                        this.handleDirectoryEvent(kind, filePath);
                        continue;
                    }
                    if (!this.isNbtFile(filePath)) continue;
                    this.handleFileEvent(kind, filePath, dir);
                }
                if (key.reset()) continue;
                this.watchKeys.remove(key);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                break;
            }
            catch (ClosedWatchServiceException e) {
                break;
            }
            catch (Exception e) {
                LOGGER.error("Error in file monitor", (Throwable)e);
            }
        }
    }

    private void handleDirectoryEvent(WatchEvent.Kind<?> kind, Path dirPath) {
        if (kind == StandardWatchEventKinds.ENTRY_CREATE) {
            try {
                this.registerAllDirectories(dirPath);
                LOGGER.debug("Registered new directory and its subdirectories: {}", (Object)dirPath);
                this.checkExistingFilesInDirectory(dirPath);
            }
            catch (IOException e) {
                LOGGER.warn("Failed to register new directory: {}", (Object)dirPath, (Object)e);
            }
        }
    }

    private void checkExistingFilesInDirectory(Path directory) {
        try {
            Files.walk(directory, new FileVisitOption[0]).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(this::isNbtFile).forEach(filePath -> {
                this.handleFileCreate((Path)filePath, this.getFolderName((Path)filePath), filePath.getFileName().toString());
                LOGGER.debug("Found existing NBT file during directory registration: {}", filePath);
            });
        }
        catch (IOException e) {
            LOGGER.warn("Failed to check existing files in directory: {}", (Object)directory, (Object)e);
        }
    }

    private String getFolderName(Path filePath) {
        Path relativePath = this.watchDir.relativize(filePath.getParent());
        return relativePath.toString();
    }

    private void handleFileEvent(WatchEvent.Kind<?> kind, Path filePath, Path parentDir) {
        String folderName = this.getFolderName(filePath);
        String fileName = filePath.getFileName().toString();
        try {
            if (kind == StandardWatchEventKinds.ENTRY_CREATE) {
                this.handleFileCreate(filePath, folderName, fileName);
            } else if (kind == StandardWatchEventKinds.ENTRY_MODIFY) {
                this.handleFileModify(filePath, folderName, fileName);
            } else if (kind == StandardWatchEventKinds.ENTRY_DELETE) {
                this.trackedFiles.remove(filePath);
                LOGGER.debug("File {} in folder {} has been deleted", (Object)fileName, (Object)folderName);
            }
        }
        catch (Exception e) {
            LOGGER.warn("Error handling file event for: {}", (Object)filePath, (Object)e);
        }
    }

    private void handleFileCreate(Path filePath, String folderName, String fileName) {
        FileTracker tracker = new FileTracker(filePath);
        this.trackedFiles.put(filePath, tracker);
        LOGGER.info("\u001b[94mDetected new file {} in folder {}\u001b[0m", (Object)fileName, (Object)folderName);
    }

    private void handleFileModify(Path filePath, String folderName, String fileName) {
        FileTracker tracker = this.trackedFiles.get(filePath);
        if (tracker == null) {
            tracker = new FileTracker(filePath);
            this.trackedFiles.put(filePath, tracker);
            LOGGER.info("\u001b[94mDetected existing file {} in folder {}\u001b[0m", (Object)fileName, (Object)folderName);
        } else {
            tracker.update();
            LOGGER.debug("File {} in folder {} has been modified", (Object)fileName, (Object)folderName);
        }
    }

    private void checkFileStability() {
        long currentTime = System.currentTimeMillis();
        long maxStableTime = (Long)Config.INSTANCE.maxStableTime.get();
        this.trackedFiles.entrySet().removeIf(entry -> {
            Path filePath = (Path)entry.getKey();
            FileTracker tracker = (FileTracker)entry.getValue();
            try {
                boolean shouldRemove;
                if (!Files.exists(filePath, new LinkOption[0])) {
                    LOGGER.debug("File {} has been deleted, removing from tracking", (Object)filePath.getFileName());
                    return true;
                }
                long currentSize = Files.size(filePath);
                long timeSinceLastChange = currentTime - tracker.getLastModifiedTime();
                if (currentSize != tracker.getLastSize()) {
                    tracker.updateSize(currentSize);
                    tracker.setNotifiedStable(false);
                    LOGGER.debug("File {} size changed to {} bytes", (Object)filePath.getFileName(), (Object)currentSize);
                    return false;
                }
                if (timeSinceLastChange >= maxStableTime && !tracker.hasNotifiedStable()) {
                    String folderName = this.getFolderName(filePath);
                    String fileName = filePath.getFileName().toString().replace(".nbt", "");
                    LOGGER.info("\u001b[94mFile {} in folder {} upload completed (stable for {}ms), starting anomaly detection\u001b[0m", (Object)fileName, (Object)folderName, (Object)timeSinceLastChange);
                    if (((Boolean)Config.INSTANCE.autoCleanAnomalies.get()).booleanValue()) {
                        SchematicNBTDetector.DetectionResult result = this.nbtDetector.detectAnomalies(folderName, fileName);
                        if (result.hasAnomalies) {
                            LOGGER.warn("\u001b[91mAnomalies detected and processed: {}\u001b[0m", (Object)result.message);
                        } else {
                            LOGGER.info("\u001b[92mSchematic validation passed: {}\u001b[0m", (Object)result.message);
                        }
                    }
                    tracker.setNotifiedStable(true);
                }
                boolean bl = shouldRemove = tracker.hasNotifiedStable() && timeSinceLastChange > maxStableTime * 2L;
                if (shouldRemove) {
                    LOGGER.debug("File {} has been stable for long enough, removing from tracking", (Object)filePath.getFileName());
                }
                return shouldRemove;
            }
            catch (IOException e) {
                LOGGER.warn("Error checking file stability: {}", (Object)filePath, (Object)e);
                return true;
            }
        });
    }

    private boolean isNbtFile(Path filePath) {
        String fileName = filePath.getFileName().toString();
        return fileName.toLowerCase().endsWith(".nbt");
    }

    private static class FileTracker {
        private final Path filePath;
        private long lastModifiedTime;
        private long lastSize;
        private boolean notifiedStable = false;

        public FileTracker(Path filePath) {
            this.filePath = filePath;
            this.update();
        }

        public void update() {
            try {
                this.lastModifiedTime = System.currentTimeMillis();
                this.lastSize = Files.size(this.filePath);
            }
            catch (IOException e) {
                this.lastSize = 0L;
            }
        }

        public void updateSize(long newSize) {
            this.lastSize = newSize;
            this.lastModifiedTime = System.currentTimeMillis();
        }

        public long getLastModifiedTime() {
            return this.lastModifiedTime;
        }

        public long getLastSize() {
            return this.lastSize;
        }

        public boolean hasNotifiedStable() {
            return this.notifiedStable;
        }

        public void setNotifiedStable(boolean notifiedStable) {
            this.notifiedStable = notifiedStable;
        }
    }
}

