/*
 * Decompiled with CFR 0.152.
 */
package com.cff1028.schematicsfix;

import com.cff1028.schematicsfix.Config;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Writer;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import net.minecraft.ChatFormatting;
import net.minecraft.nbt.ByteArrayTag;
import net.minecraft.nbt.ByteTag;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.DoubleTag;
import net.minecraft.nbt.FloatTag;
import net.minecraft.nbt.IntTag;
import net.minecraft.nbt.ListTag;
import net.minecraft.nbt.LongTag;
import net.minecraft.nbt.NbtIo;
import net.minecraft.nbt.ShortTag;
import net.minecraft.nbt.StringTag;
import net.minecraft.nbt.Tag;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import net.neoforged.neoforge.server.ServerLifecycleHooks;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.yaml.snakeyaml.DumperOptions;
import org.yaml.snakeyaml.Yaml;

public class SchematicNBTDetector {
    private static final Logger LOGGER = LogManager.getLogger();
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final Yaml YAML = SchematicNBTDetector.createYaml();
    private final Path configPath = Paths.get("config/schematicsfix.yaml", new String[0]).toAbsolutePath();
    private final List<FilterRule> filterRules = new ArrayList<FilterRule>();

    public SchematicNBTDetector() {
        this.loadConfig();
    }

    private static Yaml createYaml() {
        DumperOptions options = new DumperOptions();
        options.setDefaultFlowStyle(DumperOptions.FlowStyle.BLOCK);
        options.setPrettyFlow(true);
        return new Yaml(options);
    }

    private void loadConfig() {
        if (!Files.exists(this.configPath, new LinkOption[0])) {
            try {
                Files.createDirectories(this.configPath.getParent(), new FileAttribute[0]);
                this.copyDefaultConfig();
                LOGGER.info("\u001b[94mCreated default config file from assets: {}\u001b[0m", (Object)this.configPath);
            }
            catch (IOException e) {
                LOGGER.error("\u001b[91mFailed to create config file: {}\u001b[0m", (Object)this.configPath, (Object)e);
            }
            return;
        }
        this.reloadConfigInternal();
    }

    public boolean reloadConfig() {
        LOGGER.info("\u001b[94mReloading schematic filter configuration...\u001b[0m");
        return this.reloadConfigInternal();
    }

    public List<String> getRuleNames() {
        return this.filterRules.stream().map(rule -> rule.name).collect(Collectors.toList());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean reloadConfigInternal() {
        this.filterRules.clear();
        try (InputStream input = Files.newInputStream(this.configPath, new OpenOption[0]);){
            Map config = (Map)YAML.load(input);
            if (config == null || !config.containsKey("rules")) {
                LOGGER.warn("\u001b[91mNo rules found in config file\u001b[0m");
                boolean bl2 = false;
                return bl2;
            }
            List rules = (List)config.get("rules");
            int loadedRules = 0;
            for (Map ruleData : rules) {
                try {
                    FilterRule rule = this.parseFilterRule(ruleData);
                    if (rule == null) continue;
                    this.filterRules.add(rule);
                    ++loadedRules;
                    LOGGER.debug("Loaded filter rule: {}", (Object)rule.name);
                }
                catch (Exception e) {
                    LOGGER.warn("\u001b[91mFailed to parse rule: {}\u001b[0m", (Object)ruleData, (Object)e);
                }
            }
            LOGGER.info("\u001b[94mLoaded {} filter rules from config\u001b[0m", (Object)loadedRules);
            boolean bl = true;
            return bl;
        }
        catch (IOException e) {
            LOGGER.error("\u001b[91mFailed to load config file: {}\u001b[0m", (Object)this.configPath, (Object)e);
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void copyDefaultConfig() {
        InputStream inputStream = null;
        try {
            inputStream = this.getClass().getResourceAsStream("/assets/schematicsfix/schematicsfix.yaml");
            if (inputStream == null) {
                inputStream = this.getClass().getClassLoader().getResourceAsStream("assets/schematicsfix/schematicsfix.yaml");
            }
            if (inputStream != null) {
                Files.copy(inputStream, this.configPath, StandardCopyOption.REPLACE_EXISTING);
                LOGGER.info("\u001b[94mSuccessfully copied default config from assets\u001b[0m");
            } else {
                LOGGER.warn("\u001b[91mDefault config not found in assets, creating example config file\u001b[0m");
                this.createExampleConfig();
            }
        }
        catch (Exception e) {
            LOGGER.error("\u001b[91mFailed to copy default config from assets\u001b[0m", (Throwable)e);
            try {
                this.createExampleConfig();
            }
            catch (IOException ex) {
                LOGGER.error("\u001b[91mFailed to create example config file\u001b[0m", (Throwable)ex);
            }
        }
        finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                }
                catch (IOException e) {
                    LOGGER.error("\u001b[91mError closing input stream\u001b[0m", (Throwable)e);
                }
            }
        }
    }

    private void createExampleConfig() throws IOException {
        LinkedHashMap<String, Object> config = new LinkedHashMap<String, Object>();
        config.put("version", "1.0");
        ArrayList rules = new ArrayList();
        LinkedHashMap<String, Object> valveRule = new LinkedHashMap<String, Object>();
        valveRule.put("name", "valve_handle_capacity_cleanup");
        valveRule.put("target_path", "blocks[*].nbt");
        ArrayList valveConditions = new ArrayList();
        LinkedHashMap<String, String> valveCondition = new LinkedHashMap<String, String>();
        valveCondition.put("type", "field_equals");
        valveCondition.put("path", "id");
        valveCondition.put("value", "create:valve_handle");
        valveConditions.add(valveCondition);
        valveRule.put("conditions", valveConditions);
        ArrayList valveActions = new ArrayList();
        LinkedHashMap<String, Object> capacityAction = new LinkedHashMap<String, Object>();
        capacityAction.put("type", "conditional_remove");
        capacityAction.put("target_path", "Network.Capacity");
        LinkedHashMap<String, Object> capacityCondition = new LinkedHashMap<String, Object>();
        capacityCondition.put("type", "field_not_equals");
        capacityCondition.put("path", "Network.Capacity");
        capacityCondition.put("value", 256.0);
        capacityAction.put("condition", capacityCondition);
        capacityAction.put("remove_strategy", "field_only");
        valveActions.add(capacityAction);
        LinkedHashMap<String, Object> speedAction = new LinkedHashMap<String, Object>();
        speedAction.put("type", "conditional_remove");
        speedAction.put("target_path", "Speed");
        LinkedHashMap<String, Object> speedCondition = new LinkedHashMap<String, Object>();
        speedCondition.put("type", "field_not_equals");
        speedCondition.put("path", "Speed");
        speedCondition.put("value", 32.0);
        speedAction.put("condition", speedCondition);
        speedAction.put("remove_strategy", "field_only");
        valveActions.add(speedAction);
        valveRule.put("cleanup_actions", valveActions);
        rules.add(valveRule);
        config.put("rules", rules);
        try (FileWriter writer = new FileWriter(this.configPath.toFile());){
            YAML.dump(config, (Writer)writer);
        }
        LOGGER.info("Created example configuration");
    }

    private FilterRule parseFilterRule(Map<String, Object> ruleData) {
        if (!ruleData.containsKey("name") || !ruleData.containsKey("target_path")) {
            LOGGER.warn("\u001b[91mRule missing required fields: name and target_path\u001b[0m");
            return null;
        }
        String name = (String)ruleData.get("name");
        String targetPath = (String)ruleData.get("target_path");
        FilterRule rule = new FilterRule(name, targetPath);
        if (ruleData.containsKey("conditions")) {
            List conditions = (List)ruleData.get("conditions");
            for (Map conditionData : conditions) {
                Condition condition = this.parseCondition(conditionData);
                if (condition == null) continue;
                rule.conditions.add(condition);
            }
        }
        if (ruleData.containsKey("cleanup_actions")) {
            List actions = (List)ruleData.get("cleanup_actions");
            for (Map actionData : actions) {
                CleanupAction action = this.parseCleanupAction(actionData);
                if (action == null) continue;
                rule.cleanupActions.add(action);
            }
        }
        return rule;
    }

    private Condition parseCondition(Map<String, Object> conditionData) {
        if (!conditionData.containsKey("type")) {
            return null;
        }
        String type = (String)conditionData.get("type");
        Condition condition = new Condition(type);
        condition.path = (String)conditionData.get("path");
        condition.value = conditionData.get("value");
        return condition;
    }

    private CleanupAction parseCleanupAction(Map<String, Object> actionData) {
        if (!actionData.containsKey("type") || !actionData.containsKey("target_path")) {
            return null;
        }
        String type = (String)actionData.get("type");
        String targetPath = (String)actionData.get("target_path");
        CleanupAction action = new CleanupAction(type, targetPath);
        action.removeStrategy = (String)actionData.get("remove_strategy");
        if (actionData.containsKey("condition")) {
            Map conditionData = (Map)actionData.get("condition");
            action.condition = this.parseCondition(conditionData);
        }
        if (actionData.containsKey("remove_pattern")) {
            action.removePattern = Pattern.compile((String)actionData.get("remove_pattern"));
        }
        return action;
    }

    public int getRuleCount() {
        return this.filterRules.size();
    }

    public DetectionResult detectAnomalies(String playerName, String fileName) {
        Path schematicPath = Paths.get("schematics/uploaded", playerName, fileName + ".nbt").toAbsolutePath();
        Path anomalyPath = Paths.get("schematics/anomaly", playerName, fileName + ".nbt").toAbsolutePath();
        if (!Files.exists(schematicPath, new LinkOption[0])) {
            return new DetectionResult(false, "Schematic file not found: " + String.valueOf(schematicPath));
        }
        try {
            CompoundTag nbt = this.readNbtFile(schematicPath);
            if (nbt == null) {
                return new DetectionResult(false, "Failed to read NBT file - unsupported compression or corrupted file");
            }
            boolean modified = this.processWithRules(nbt, playerName, fileName);
            if (modified) {
                this.writeNbtFile(nbt, schematicPath);
                if (((Boolean)Config.INSTANCE.backupAnomalousFiles.get()).booleanValue()) {
                    Files.createDirectories(anomalyPath.getParent(), new FileAttribute[0]);
                    Files.copy(schematicPath, anomalyPath, StandardCopyOption.REPLACE_EXISTING);
                    LOGGER.warn("\u001b[91mRule-based anomalies detected and cleaned in {}.nbt from player {}. Original backed up.\u001b[0m", (Object)fileName, (Object)playerName);
                    if (modified && ((Boolean)Config.INSTANCE.notifyPlayerOnAnomaly.get()).booleanValue()) {
                        this.notifyPlayer(playerName, fileName);
                    }
                    return new DetectionResult(true, "Rule-based anomalies detected and cleaned. Original backed up to anomaly directory.");
                }
                LOGGER.warn("\u001b[91mRule-based anomalies detected and cleaned in {}.nbt from player {}.\u001b[0m", (Object)fileName, (Object)playerName);
                if (modified && ((Boolean)Config.INSTANCE.notifyPlayerOnAnomaly.get()).booleanValue()) {
                    this.notifyPlayer(playerName, fileName);
                }
                return new DetectionResult(true, "Rule-based anomalies detected and cleaned.");
            }
            return new DetectionResult(false, "No anomalies detected in schematic.");
        }
        catch (IOException e) {
            LOGGER.error("\u001b[91mError processing schematic file: {}\u001b[0m", (Object)schematicPath, (Object)e);
            return new DetectionResult(false, "Error processing schematic: " + e.getMessage());
        }
        catch (Exception e) {
            LOGGER.error("\u001b[91mUnexpected error processing schematic file: {}\u001b[0m", (Object)schematicPath, (Object)e);
            return new DetectionResult(false, "Unexpected error: " + e.getMessage());
        }
    }

    private boolean processWithRules(CompoundTag rootNbt, String playerName, String fileName) {
        boolean modified = false;
        for (FilterRule rule : this.filterRules) {
            modified |= this.applyRule(rootNbt, rule, playerName, fileName);
        }
        return modified;
    }

    private boolean applyRule(CompoundTag root, FilterRule rule, String playerName, String fileName) {
        boolean modified = false;
        String[] pathParts = rule.targetPath.split("\\.");
        List<CompoundTag> targets = this.findTargets((Tag)root, pathParts, 0);
        for (CompoundTag target : targets) {
            if (!this.checkConditions(target, rule.conditions)) continue;
            for (CleanupAction action : rule.cleanupActions) {
                modified |= this.applyCleanupAction(target, action, playerName, fileName, rule.name);
            }
        }
        return modified;
    }

    private List<CompoundTag> findTargets(Tag current, String[] pathParts, int depth) {
        CompoundTag compound;
        ArrayList<CompoundTag> results = new ArrayList<CompoundTag>();
        if (depth >= pathParts.length) {
            if (current instanceof CompoundTag) {
                results.add((CompoundTag)current);
            }
            return results;
        }
        String part = pathParts[depth];
        if (part.equals("*")) {
            if (current instanceof CompoundTag) {
                CompoundTag compound2 = (CompoundTag)current;
                for (String key : compound2.getAllKeys()) {
                    Tag child = compound2.get(key);
                    results.addAll(this.findTargets(child, pathParts, depth + 1));
                }
            } else if (current instanceof ListTag) {
                ListTag list = (ListTag)current;
                for (int i = 0; i < list.size(); ++i) {
                    Tag child = list.get(i);
                    results.addAll(this.findTargets(child, pathParts, depth + 1));
                }
            }
        } else if (part.endsWith("[*]")) {
            Tag array;
            CompoundTag compound3;
            String arrayName = part.substring(0, part.length() - 3);
            if (current instanceof CompoundTag && (compound3 = (CompoundTag)current).contains(arrayName) && (array = compound3.get(arrayName)) instanceof ListTag) {
                ListTag list = (ListTag)array;
                for (int i = 0; i < list.size(); ++i) {
                    Tag child = list.get(i);
                    results.addAll(this.findTargets(child, pathParts, depth + 1));
                }
            }
        } else if (current instanceof CompoundTag && (compound = (CompoundTag)current).contains(part)) {
            Tag child = compound.get(part);
            results.addAll(this.findTargets(child, pathParts, depth + 1));
        }
        return results;
    }

    private boolean checkConditions(CompoundTag target, List<Condition> conditions) {
        for (Condition condition : conditions) {
            if (this.checkCondition(target, condition)) continue;
            return false;
        }
        return true;
    }

    private boolean checkCondition(CompoundTag target, Condition condition) {
        try {
            switch (condition.type) {
                case "field_equals": {
                    return this.checkFieldEquals(target, condition);
                }
                case "field_not_equals": {
                    return !this.checkFieldEquals(target, condition);
                }
                case "field_exists": {
                    return this.checkFieldExists(target, condition);
                }
                case "string_contains": {
                    return this.checkStringContains(target, condition);
                }
            }
            LOGGER.warn("\u001b[95mUnknown condition type: {}\u001b[0m", (Object)condition.type);
            return false;
        }
        catch (Exception e) {
            LOGGER.warn("\u001b[95mError checking condition: {}\u001b[0m", (Object)condition.type, (Object)e);
            return false;
        }
    }

    private boolean checkFieldEquals(CompoundTag target, Condition condition) {
        Tag field = this.getTagByPath(target, condition.path);
        if (field == null) {
            return false;
        }
        Object actualValue = this.getTagValue(field);
        if (condition.value instanceof Double && actualValue instanceof Float) {
            return (double)Math.abs(((Float)actualValue).floatValue() - ((Double)condition.value).floatValue()) < 1.0E-4;
        }
        return Objects.equals(actualValue, condition.value);
    }

    private boolean checkFieldExists(CompoundTag target, Condition condition) {
        return this.getTagByPath(target, condition.path) != null;
    }

    private boolean checkStringContains(CompoundTag target, Condition condition) {
        List<TagWithPath> foundTags = this.findTagsByPath((Tag)target, condition.path);
        for (TagWithPath tagWithPath : foundTags) {
            StringTag stringTag;
            String value;
            Tag field = tagWithPath.tag;
            if (!(field instanceof StringTag) || !(value = (stringTag = (StringTag)field).getAsString()).contains((String)condition.value)) continue;
            return true;
        }
        return false;
    }

    private Tag getTagByPath(CompoundTag root, String path) {
        String[] parts = path.split("\\.");
        CompoundTag current = root;
        for (String part : parts) {
            CompoundTag compound;
            if (current instanceof CompoundTag) {
                compound = current;
                if (!compound.contains(part)) {
                    return null;
                }
            } else {
                return null;
            }
            current = compound.get(part);
        }
        return current;
    }

    private Object getTagValue(Tag tag) {
        if (tag instanceof StringTag) {
            return ((StringTag)tag).getAsString();
        }
        if (tag instanceof ByteTag) {
            return ((ByteTag)tag).getAsByte();
        }
        if (tag instanceof ShortTag) {
            return ((ShortTag)tag).getAsShort();
        }
        if (tag instanceof IntTag) {
            return ((IntTag)tag).getAsInt();
        }
        if (tag instanceof LongTag) {
            return ((LongTag)tag).getAsLong();
        }
        if (tag instanceof FloatTag) {
            return Float.valueOf(((FloatTag)tag).getAsFloat());
        }
        if (tag instanceof DoubleTag) {
            return ((DoubleTag)tag).getAsDouble();
        }
        if (tag instanceof ByteArrayTag) {
            return ((ByteArrayTag)tag).getAsByteArray();
        }
        if (tag instanceof StringTag) {
            return ((StringTag)tag).getAsString();
        }
        if (tag instanceof ListTag) {
            return tag;
        }
        if (tag instanceof CompoundTag) {
            return tag;
        }
        return null;
    }

    private boolean applyCleanupAction(CompoundTag target, CleanupAction action, String playerName, String fileName, String ruleName) {
        try {
            switch (action.type) {
                case "conditional_remove": {
                    return this.applyConditionalRemove(target, action, playerName, fileName, ruleName);
                }
                case "path_remove": {
                    return this.applyPathRemove(target, action, playerName, fileName, ruleName);
                }
                case "nested_string_remove": {
                    return this.applyNestedStringRemove(target, action, playerName, fileName, ruleName);
                }
            }
            LOGGER.warn("\u001b[95mUnknown cleanup action type: {}\u001b[0m", (Object)action.type);
            return false;
        }
        catch (Exception e) {
            LOGGER.warn("\u001b[95mError applying cleanup action: {}\u001b[0m", (Object)action.type, (Object)e);
            return false;
        }
    }

    private boolean applyConditionalRemove(CompoundTag target, CleanupAction action, String playerName, String fileName, String ruleName) {
        if (action.condition != null && !this.checkCondition(target, action.condition)) {
            return false;
        }
        return this.removeTagAtPath(target, action.targetPath, playerName, fileName, ruleName);
    }

    private boolean applyPathRemove(CompoundTag target, CleanupAction action, String playerName, String fileName, String ruleName) {
        return this.removeTagAtPath(target, action.targetPath, playerName, fileName, ruleName);
    }

    private boolean applyNestedStringRemove(CompoundTag target, CleanupAction action, String playerName, String fileName, String ruleName) {
        boolean modified = false;
        try {
            List<TagWithPath> foundTags = this.findTagsByPath((Tag)target, action.targetPath);
            for (TagWithPath tagWithPath : foundTags) {
                String cleaned;
                Tag tag = tagWithPath.tag;
                String fullPath = tagWithPath.path;
                if (!(tag instanceof StringTag)) continue;
                StringTag stringTag = (StringTag)tag;
                String original = stringTag.getAsString();
                if (action.condition != null && !original.contains((String)action.condition.value) || (cleaned = action.removePattern.matcher(original).replaceAll("")).equals(original) || !this.updateTagAtPath(target, fullPath, (Tag)StringTag.valueOf((String)cleaned))) continue;
                modified = true;
                LOGGER.info("\u001b[93mRule '{}' cleaned string at path '{}' in file {}.nbt\u001b[0m", (Object)ruleName, (Object)fullPath, (Object)fileName);
            }
        }
        catch (Exception e) {
            LOGGER.warn("\u001b[95mError applying nested string remove for rule '{}': {}\u001b[0m", (Object)ruleName, (Object)e.getMessage());
        }
        return modified;
    }

    private List<TagWithPath> findTagsByPath(Tag current, String path) {
        return this.findTagsByPath(current, path.split("\\."), 0, "");
    }

    private List<TagWithPath> findTagsByPath(Tag current, String[] pathParts, int depth, String currentPath) {
        CompoundTag compound;
        String newPath;
        ArrayList<TagWithPath> results = new ArrayList<TagWithPath>();
        if (depth >= pathParts.length) {
            results.add(new TagWithPath(current, currentPath));
            return results;
        }
        String part = pathParts[depth];
        String string = newPath = currentPath.isEmpty() ? part : currentPath + "." + part;
        if (part.equals("*")) {
            if (current instanceof CompoundTag) {
                CompoundTag compound2 = (CompoundTag)current;
                for (String key : compound2.getAllKeys()) {
                    Tag child = compound2.get(key);
                    String childPath = currentPath.isEmpty() ? key : currentPath + "." + key;
                    results.addAll(this.findTagsByPath(child, pathParts, depth + 1, childPath));
                }
            } else if (current instanceof ListTag) {
                ListTag list = (ListTag)current;
                for (int i = 0; i < list.size(); ++i) {
                    Tag child = list.get(i);
                    String childPath = currentPath + "[" + i + "]";
                    results.addAll(this.findTagsByPath(child, pathParts, depth + 1, childPath));
                }
            }
        } else if (part.endsWith("[*]")) {
            Tag array;
            CompoundTag compound3;
            String arrayName = part.substring(0, part.length() - 3);
            if (current instanceof CompoundTag && (compound3 = (CompoundTag)current).contains(arrayName) && (array = compound3.get(arrayName)) instanceof ListTag) {
                ListTag list = (ListTag)array;
                for (int i = 0; i < list.size(); ++i) {
                    Tag child = list.get(i);
                    String childPath = currentPath.isEmpty() ? arrayName + "[" + i + "]" : currentPath + "." + arrayName + "[" + i + "]";
                    results.addAll(this.findTagsByPath(child, pathParts, depth + 1, childPath));
                }
            }
        } else if (part.contains("[")) {
            Tag listTag;
            CompoundTag compound4;
            String listName = part.substring(0, part.indexOf("["));
            String indexStr = part.substring(part.indexOf("[") + 1, part.indexOf("]"));
            if (current instanceof CompoundTag && (compound4 = (CompoundTag)current).contains(listName) && (listTag = compound4.get(listName)) instanceof ListTag) {
                ListTag list = (ListTag)listTag;
                if (indexStr.equals("*")) {
                    for (int i = 0; i < list.size(); ++i) {
                        Tag child = list.get(i);
                        String childPath = currentPath.isEmpty() ? listName + "[" + i + "]" : currentPath + "." + listName + "[" + i + "]";
                        results.addAll(this.findTagsByPath(child, pathParts, depth + 1, childPath));
                    }
                } else {
                    try {
                        int index = Integer.parseInt(indexStr);
                        if (index < list.size()) {
                            Tag child = list.get(index);
                            String childPath = currentPath.isEmpty() ? listName + "[" + index + "]" : currentPath + "." + listName + "[" + index + "]";
                            results.addAll(this.findTagsByPath(child, pathParts, depth + 1, childPath));
                        }
                    }
                    catch (NumberFormatException e) {
                        LOGGER.warn("\u001b[95mInvalid array index: {}\u001b[0m", (Object)indexStr);
                    }
                }
            }
        } else if (current instanceof CompoundTag && (compound = (CompoundTag)current).contains(part)) {
            Tag child = compound.get(part);
            results.addAll(this.findTagsByPath(child, pathParts, depth + 1, newPath));
        }
        return results;
    }

    private boolean updateTagAtPath(CompoundTag root, String path, Tag newTag) {
        if (path.isEmpty()) {
            return false;
        }
        String[] pathParts = path.split("\\.");
        CompoundTag current = root;
        for (int i = 0; i < pathParts.length - 1; ++i) {
            Tag index2;
            String part = pathParts[i];
            if (part.contains("[")) {
                Tag tag;
                String listName = part.substring(0, part.indexOf("["));
                int index2 = Integer.parseInt(part.substring(part.indexOf("[") + 1, part.indexOf("]")));
                if (current.contains(listName) && (tag = current.get(listName)) instanceof ListTag) {
                    Tag tag2;
                    ListTag list = (ListTag)tag;
                    if (index2 < list.size() && (tag2 = list.get(index2)) instanceof CompoundTag) {
                        CompoundTag compound;
                        current = compound = (CompoundTag)tag2;
                        continue;
                    }
                    return false;
                }
                return false;
            }
            if (current.contains(part) && (index2 = current.get(part)) instanceof CompoundTag) {
                CompoundTag compound;
                current = compound = (CompoundTag)index2;
                continue;
            }
            return false;
        }
        String targetKey = pathParts[pathParts.length - 1];
        if (targetKey.contains("[")) {
            ListTag list;
            Tag tag;
            String listName = targetKey.substring(0, targetKey.indexOf("["));
            int index = Integer.parseInt(targetKey.substring(targetKey.indexOf("[") + 1, targetKey.indexOf("]")));
            if (current.contains(listName) && (tag = current.get(listName)) instanceof ListTag && index < (list = (ListTag)tag).size()) {
                list.set(index, newTag);
                return true;
            }
        } else if (current.contains(targetKey)) {
            current.put(targetKey, newTag);
            return true;
        }
        return false;
    }

    private boolean removeTagAtPath(CompoundTag root, String path, String playerName, String fileName, String ruleName) {
        String[] parts = path.split("\\.");
        CompoundTag current = root;
        for (int i = 0; i < parts.length - 1; ++i) {
            String part = parts[i];
            if (!current.contains(part) || !(current.get(part) instanceof CompoundTag)) {
                return false;
            }
            current = (CompoundTag)current.get(part);
        }
        String targetKey = parts[parts.length - 1];
        if (current.contains(targetKey)) {
            current.remove(targetKey);
            LOGGER.info("\u001b[93mRule '{}' removed path '{}' in file {}.nbt\u001b[0m", (Object)ruleName, (Object)path, (Object)fileName);
            return true;
        }
        return false;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private CompoundTag readNbtFile(Path filePath) throws IOException {
        int maxRetries = 6;
        int retryDelayMs = 500;
        int attempt = 1;
        while (attempt <= maxRetries) {
            try (InputStream is = Files.newInputStream(filePath, new OpenOption[0]);
                 GZIPInputStream gzipIs = new GZIPInputStream(is);
                 DataInputStream dataIs = new DataInputStream(gzipIs);){
                CompoundTag nbt = NbtIo.read((DataInput)dataIs);
                if (nbt != null) {
                    if (attempt > 1) {
                        LOGGER.info("\u001b[92mSuccessfully read NBT file after {} attempts: {}\u001b[0m", (Object)attempt, (Object)filePath);
                    }
                    CompoundTag compoundTag = nbt;
                    return compoundTag;
                }
            }
            catch (Exception e) {
                if (attempt < maxRetries) {
                    LOGGER.warn("\u001b[91mFailed to read NBT file (attempt {}/{}): {}. Retrying in {}ms...\u001b[0m", (Object)attempt, (Object)maxRetries, (Object)filePath, (Object)retryDelayMs, (Object)e);
                    try {
                        Thread.sleep(retryDelayMs);
                    }
                    catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                        throw new IOException("Interrupted during retry delay", ie);
                    }
                }
                LOGGER.error("\u001b[91mFailed to read NBT file after {} attempts: {}\u001b[0m", (Object)maxRetries, (Object)filePath, (Object)e);
            }
            ++attempt;
        }
        return null;
    }

    private void writeNbtFile(CompoundTag nbt, Path filePath) throws IOException {
        try (OutputStream os = Files.newOutputStream(filePath, new OpenOption[0]);
             GZIPOutputStream gzipOs = new GZIPOutputStream(os);
             DataOutputStream dataOs = new DataOutputStream(gzipOs);){
            NbtIo.write((CompoundTag)nbt, (DataOutput)dataOs);
        }
    }

    private void notifyPlayer(String playerName, String fileName) {
        try {
            MinecraftServer server = ServerLifecycleHooks.getCurrentServer();
            if (server != null) {
                ServerPlayer player = server.getPlayerList().getPlayerByName(playerName);
                if (player != null) {
                    MutableComponent message = Component.literal((String)"The schematic diagram you uploaded appears to be abnormal.").withStyle(ChatFormatting.GOLD);
                    player.sendSystemMessage((Component)message);
                    LOGGER.debug("Notified player {} about anomalous schematic: {}.nbt", (Object)playerName, (Object)fileName);
                } else {
                    LOGGER.debug("Player {} not online, cannot send anomaly notification", (Object)playerName);
                }
            }
        }
        catch (Exception e) {
            LOGGER.warn("\u001b[95mFailed to notify player {} about anomalous schematic\u001b[0m", (Object)playerName, (Object)e);
        }
    }

    private static class FilterRule {
        final String name;
        final String targetPath;
        final List<Condition> conditions = new ArrayList<Condition>();
        final List<CleanupAction> cleanupActions = new ArrayList<CleanupAction>();

        FilterRule(String name, String targetPath) {
            this.name = name;
            this.targetPath = targetPath;
        }
    }

    private static class Condition {
        final String type;
        String path;
        Object value;

        Condition(String type) {
            this.type = type;
        }
    }

    private static class CleanupAction {
        final String type;
        final String targetPath;
        String removeStrategy;
        Condition condition;
        Pattern removePattern;

        CleanupAction(String type, String targetPath) {
            this.type = type;
            this.targetPath = targetPath;
        }
    }

    public static class DetectionResult {
        public final boolean hasAnomalies;
        public final String message;

        public DetectionResult(boolean hasAnomalies, String message) {
            this.hasAnomalies = hasAnomalies;
            this.message = message;
        }
    }

    private static class TagWithPath {
        public final Tag tag;
        public final String path;

        public TagWithPath(Tag tag, String path) {
            this.tag = tag;
            this.path = path;
        }
    }
}

