#version 330 compatibility

#include "distort.glsl"

#define FRESNEL_STRENGTH 1.0 // [0.0 0.1 0.2 0.3 0.4 0.5 0.6 0.7 0.8 0.9 1.0 1.1 1.2 1.3 1.4 1.5 1.6 1.7 1.8 1.9 2.0]
#define SHADOW_RESOLUTION 1024 // [128 192 256 384 512 768 1024 1536 2048 3072 4096 6144 8192]
#define SHADOW_SAMPLES 3 // [1 3 5 7 9 11 13 15 17 19 21 23 25]
#define SHADOW_SOFTNESS 1.0 // [0.0 0.1 0.2 0.3 0.4 0.5 0.6 0.7 0.8 0.9 1.0 1.1 1.2 1.3 1.4 1.5 1.6 1.7 1.8 1.9 2.0 2.1 2.2 2.3 2.4 2.5 2.6 2.7 2.8 2.9 3.0 3.1 3.2 3.3 3.4 3.5 3.6 3.7 3.8 3.9 4.0 4.1 4.2 4.3 4.4 4.5 4.6 4.7 4.8 4.9 5.0]

uniform sampler2D colortex0;
uniform sampler2D colortex1;
uniform sampler2D colortex2;
uniform sampler2D colortex3;
uniform sampler2D depthtex0;
uniform sampler2D shadowtex0;

uniform mat4 gbufferProjectionInverse;
uniform mat4 gbufferModelViewInverse;
uniform mat4 shadowModelView;
uniform mat4 shadowProjection;

uniform vec3 shadowLightPosition;
uniform int worldTime;
uniform float rainStrength;

const vec3 blocklightColor = vec3(3.0, 1.0, 0.5);
const vec3 skylightColor = vec3(0.57, 0.6, 0.9);
const vec3 sunlightColor = vec3(4.0, 1.5, 0.6);
const vec3 ambientLight = vec3(0.01, 0.011, 0.013) * 3.0;
const float fresnelIntensity = 0.1 * FRESNEL_STRENGTH;

const int times[6] = int[](0, 6000, 12785, 18000, 23215, 24000);
const float values[6] = float[](0.7, 0.8, 0.0, 0.03, 0.0, 0.7);

in vec2 texcoord;

layout(location = 0) out vec4 color;

vec3 projectAndDivide(mat4 projectionMatrix, vec3 position) {
    vec4 homPos = projectionMatrix * vec4(position, 1.0);
    return homPos.xyz / homPos.w;
}

float sampleShadow(vec2 uv, float compare) {
    return step(compare, texture(shadowtex0, uv).r);
}

float getSoftShadow(vec3 shadowPos) {
    float result = 0.0;
    int half = SHADOW_SAMPLES / 2;
    float size = (1024.0 / float(SHADOW_RESOLUTION) * (0.001 * SHADOW_SOFTNESS)) * (5.0 / float(SHADOW_SAMPLES));
    for (int x = -half; x <= half; x++)
        for (int y = -half; y <= half; y++)
            result += sampleShadow(shadowPos.xy + vec2(x, y) * size, shadowPos.z);
    return result / float(SHADOW_SAMPLES * SHADOW_SAMPLES);
}

void main() {
    color = texture(colortex0, texcoord);

    float depth = texture(depthtex0, texcoord).r;
    if (depth == 1.0) return;

    vec3 main = texture(colortex0, texcoord).rgb;
    vec2 lightmap = texture(colortex1, texcoord).rg;
    vec3 normalData = texture(colortex2, texcoord).rgb;

    float dayCycle = 0.0;
    for (int i = 0; i < 5; i++) {
        if (worldTime >= times[i] && worldTime < times[i + 1]) {
            float t = float(worldTime - times[i]) / float(times[i + 1] - times[i]);
            dayCycle = mix(values[i], values[i + 1], t);
            break;
        }
    }

    vec3 normal = normalData * 2.0 - 1.0;
    
    vec3 NDCPos = vec3(texcoord, depth) * 2.0 - 1.0;
    vec3 viewPos = projectAndDivide(gbufferProjectionInverse, NDCPos);
    vec3 feetPlayerPos = (gbufferModelViewInverse * vec4(viewPos, 1.0)).xyz;
    
    vec3 shadowViewPos = (shadowModelView * vec4(feetPlayerPos, 1.0)).xyz;
    vec4 shadowClipPos = shadowProjection * vec4(shadowViewPos, 1.0);
    shadowClipPos.z -= 0.001;
    shadowClipPos.xyz = distortShadowClipPos(shadowClipPos.xyz);
    vec3 shadowScreenPos = (shadowClipPos.xyz / shadowClipPos.w) * 0.5 + 0.5;
    
    float shadow = getSoftShadow(shadowScreenPos);
    
    vec3 worldLightVector = mat3(gbufferModelViewInverse) * normalize(shadowLightPosition);
    float NdotL = max(dot(worldLightVector, normal), 0.0);
    
    lightmap = pow(lightmap, vec2(3.0));
    
    float fresnel = 0.0;
	if (FRESNEL_STRENGTH != 0.0) {
 	   vec3 psNormal = inverse(mat3(gbufferModelViewInverse)) * normal;
	    vec3 viewDir = normalize(feetPlayerPos - viewPos);
 	   fresnel = pow(1.0 - dot(viewDir, psNormal), 3.0) * 0.1;
	}
    
    float mask = texture(colortex3, texcoord).r;
    vec3 blocklight = lightmap.r * blocklightColor;
    vec3 skylight = lightmap.g * skylightColor * dayCycle;
    vec3 sunlight = sunlightColor * mix(NdotL, 1.0, step(1.0, mask)) * shadow * lightmap.g * (1.2 - rainStrength) * dayCycle;
    vec3 ambient = ambientLight;
    
    main = pow(main, vec3(2.2));
    main += vec3(fresnel) * fresnelIntensity;
    color.rgb = main * (blocklight + skylight + sunlight + ambient) * 0.6;
}