/*
 * Decompiled with CFR 0.152.
 */
package fr.tylwen.satyria.dynashop.system;

import fr.tylwen.satyria.dynashop.DynaShopPlugin;
import fr.tylwen.satyria.dynashop.system.chart.PriceHistory;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class MarketTrendAnalyzer {
    private final DynaShopPlugin plugin;

    public MarketTrendAnalyzer(DynaShopPlugin plugin) {
        this.plugin = plugin;
    }

    public MarketTrend analyzeTrend(String shopId, String itemId, int days) {
        PriceHistory history = this.plugin.getStorageManager().getPriceHistory(shopId, itemId);
        List<PriceHistory.PriceDataPoint> dataPoints = history.getDataPoints();
        LocalDateTime startDate = LocalDateTime.now().minusDays(days);
        List<PriceHistory.PriceDataPoint> filteredPoints = dataPoints.stream().filter(p -> p.getTimestamp().isAfter(startDate)).collect(Collectors.toList());
        if (filteredPoints.size() < 5) {
            return new MarketTrend(TrendType.UNKNOWN, 0.0, 0.0, 0.0, 0.0, 0.0, null, null, null, null, List.of(), List.of(), List.of(), List.of());
        }
        TechnicalAnalysis buyAnalysis = this.analyzeBuyPrices(filteredPoints);
        TechnicalAnalysis sellAnalysis = this.analyzeSellPrices(filteredPoints);
        double buyPriceChange = this.calculatePriceChange(filteredPoints, true);
        double sellPriceChange = this.calculatePriceChange(filteredPoints, false);
        double volumeChange = this.calculateVolumeChange(filteredPoints);
        List<Map<String, Object>> buyForecastData = this.generateForecastData(filteredPoints, 7, true);
        List<Map<String, Object>> sellForecastData = this.generateForecastData(filteredPoints, 7, false);
        List<Double> buySupportLevels = this.calculateSupportLevels(filteredPoints, true);
        List<Double> buyResistanceLevels = this.calculateResistanceLevels(filteredPoints, true);
        List<Double> sellSupportLevels = this.calculateSupportLevels(filteredPoints, false);
        List<Double> sellResistanceLevels = this.calculateResistanceLevels(filteredPoints, false);
        return new MarketTrend(this.determineCombinedTrend(buyAnalysis.getTrendType(), sellAnalysis.getTrendType()), Math.max(buyAnalysis.getStrength(), sellAnalysis.getStrength()), Math.max(buyAnalysis.getVolatility(), sellAnalysis.getVolatility()), buyPriceChange, sellPriceChange, volumeChange, buyAnalysis, sellAnalysis, buyForecastData, sellForecastData, buySupportLevels, buyResistanceLevels, sellSupportLevels, sellResistanceLevels);
    }

    private TrendType determineCombinedTrend(TrendType buyTrend, TrendType sellTrend) {
        if (buyTrend == sellTrend) {
            return buyTrend;
        }
        if (buyTrend == TrendType.UNKNOWN) {
            return sellTrend;
        }
        if (sellTrend == TrendType.UNKNOWN) {
            return buyTrend;
        }
        if (buyTrend == TrendType.VOLATILE || sellTrend == TrendType.VOLATILE) {
            return TrendType.VOLATILE;
        }
        if (buyTrend == TrendType.RISING && sellTrend == TrendType.FALLING || buyTrend == TrendType.FALLING && sellTrend == TrendType.RISING) {
            return TrendType.SPREAD_CHANGING;
        }
        if (buyTrend == TrendType.STABLE) {
            return sellTrend;
        }
        if (sellTrend == TrendType.STABLE) {
            return buyTrend;
        }
        return TrendType.STABLE;
    }

    private TechnicalAnalysis analyzeBuyPrices(List<PriceHistory.PriceDataPoint> points) {
        double[] closePrices = points.stream().mapToDouble(PriceHistory.PriceDataPoint::getCloseBuyPrice).filter(price -> price > 0.0).toArray();
        if (closePrices.length < 5) {
            return new TechnicalAnalysis(TrendType.UNKNOWN, 0.0, 0.0);
        }
        TrendType trendType = this.calculateBasicTrend(points, true);
        double strength = this.calculateTrendStrength(points, true);
        double volatility = this.calculateVolatility(points, true);
        double[] sma5 = this.calculateSMA(closePrices, 5);
        double[] sma20 = this.calculateSMA(closePrices, Math.min(20, closePrices.length / 2));
        double[] rsi = this.calculateRSI(closePrices, 14);
        MacdResult macd = this.calculateMACD(closePrices);
        BollingerBands bands = this.calculateBollingerBands(closePrices, 20, 2.0);
        boolean priceRsiDivergence = this.detectPriceRsiDivergence(closePrices, rsi);
        List<TradingSignal> signals = this.generateTradingSignals(closePrices, sma5, sma20, rsi, macd, bands, points);
        return new TechnicalAnalysis(trendType, strength, volatility, sma5, sma20, rsi, macd, bands, signals, priceRsiDivergence);
    }

    private TechnicalAnalysis analyzeSellPrices(List<PriceHistory.PriceDataPoint> points) {
        double[] closePrices = points.stream().mapToDouble(PriceHistory.PriceDataPoint::getCloseSellPrice).filter(price -> price > 0.0).toArray();
        if (closePrices.length < 5) {
            return new TechnicalAnalysis(TrendType.UNKNOWN, 0.0, 0.0);
        }
        TrendType trendType = this.calculateBasicTrend(points, false);
        double strength = this.calculateTrendStrength(points, false);
        double volatility = this.calculateVolatility(points, false);
        double[] sma5 = this.calculateSMA(closePrices, 5);
        double[] sma20 = this.calculateSMA(closePrices, Math.min(20, closePrices.length / 2));
        double[] rsi = this.calculateRSI(closePrices, 14);
        MacdResult macd = this.calculateMACD(closePrices);
        BollingerBands bands = this.calculateBollingerBands(closePrices, 20, 2.0);
        boolean priceRsiDivergence = this.detectPriceRsiDivergence(closePrices, rsi);
        List<TradingSignal> signals = this.generateTradingSignals(closePrices, sma5, sma20, rsi, macd, bands, points);
        return new TechnicalAnalysis(trendType, strength, volatility, sma5, sma20, rsi, macd, bands, signals, priceRsiDivergence);
    }

    private TrendType calculateBasicTrend(List<PriceHistory.PriceDataPoint> points, boolean isBuy) {
        double[] prices;
        double[] dArray = prices = isBuy ? points.stream().mapToDouble(PriceHistory.PriceDataPoint::getCloseBuyPrice).filter(price -> price > 0.0).toArray() : points.stream().mapToDouble(PriceHistory.PriceDataPoint::getCloseSellPrice).filter(price -> price > 0.0).toArray();
        if (prices.length < 3) {
            return TrendType.UNKNOWN;
        }
        double[] trend = this.calculateLinearTrend(prices);
        double slope = trend[0];
        double volatility = this.calculateVolatility(points, isBuy);
        if (Math.abs(slope) < 0.001) {
            return TrendType.STABLE;
        }
        if (volatility > 0.15) {
            return TrendType.VOLATILE;
        }
        if (slope > 0.0) {
            return TrendType.RISING;
        }
        return TrendType.FALLING;
    }

    private double calculateTrendStrength(List<PriceHistory.PriceDataPoint> points, boolean isBuy) {
        double[] prices;
        double[] dArray = prices = isBuy ? points.stream().mapToDouble(PriceHistory.PriceDataPoint::getCloseBuyPrice).filter(price -> price > 0.0).toArray() : points.stream().mapToDouble(PriceHistory.PriceDataPoint::getCloseSellPrice).filter(price -> price > 0.0).toArray();
        if (prices.length < 3) {
            return 0.0;
        }
        double[] trend = this.calculateLinearTrend(prices);
        double slope = trend[0];
        double intercept = trend[1];
        double rSquared = this.calculateRSquared(prices, slope, intercept);
        return Math.min(1.0, Math.abs(rSquared));
    }

    private double calculateVolatility(List<PriceHistory.PriceDataPoint> points, boolean isBuy) {
        double[] prices;
        double[] dArray = prices = isBuy ? points.stream().mapToDouble(PriceHistory.PriceDataPoint::getCloseBuyPrice).filter(price -> price > 0.0).toArray() : points.stream().mapToDouble(PriceHistory.PriceDataPoint::getCloseSellPrice).filter(price -> price > 0.0).toArray();
        if (prices.length < 3) {
            return 0.0;
        }
        double mean = Arrays.stream(prices).average().orElse(0.0);
        double variance = Arrays.stream(prices).map(p -> Math.pow(p - mean, 2.0)).sum() / (double)prices.length;
        double stdDev = Math.sqrt(variance);
        return mean > 0.0 ? stdDev / mean : 0.0;
    }

    private double calculatePriceChange(List<PriceHistory.PriceDataPoint> points, boolean isBuy) {
        double lastPrice;
        if (points.isEmpty()) {
            return 0.0;
        }
        double firstPrice = isBuy ? points.get(0).getCloseBuyPrice() : points.get(0).getCloseSellPrice();
        double d = lastPrice = isBuy ? points.get(points.size() - 1).getCloseBuyPrice() : points.get(points.size() - 1).getCloseSellPrice();
        if (firstPrice <= 0.0) {
            return 0.0;
        }
        return (lastPrice - firstPrice) / firstPrice * 100.0;
    }

    private double calculateVolumeChange(List<PriceHistory.PriceDataPoint> points) {
        if (points.size() < 2) {
            return 0.0;
        }
        int midPoint = points.size() / 2;
        double volumeFirstHalf = points.subList(0, midPoint).stream().mapToDouble(PriceHistory.PriceDataPoint::getVolume).sum();
        double volumeSecondHalf = points.subList(midPoint, points.size()).stream().mapToDouble(PriceHistory.PriceDataPoint::getVolume).sum();
        if (volumeFirstHalf <= 0.0) {
            return 0.0;
        }
        return (volumeSecondHalf - volumeFirstHalf) / volumeFirstHalf * 100.0;
    }

    private double[] calculateSMA(double[] prices, int period) {
        if (prices.length < period) {
            return new double[0];
        }
        double[] sma = new double[prices.length - period + 1];
        for (int i = 0; i < sma.length; ++i) {
            double sum = 0.0;
            for (int j = 0; j < period; ++j) {
                sum += prices[i + j];
            }
            sma[i] = sum / (double)period;
        }
        return sma;
    }

    private double[] calculateRSI(double[] prices, int period) {
        int i;
        if (prices.length <= period) {
            return new double[0];
        }
        double[] rsi = new double[prices.length - period];
        double[] gains = new double[prices.length - 1];
        double[] losses = new double[prices.length - 1];
        for (i = 1; i < prices.length; ++i) {
            double change = prices[i] - prices[i - 1];
            gains[i - 1] = Math.max(0.0, change);
            losses[i - 1] = Math.max(0.0, -change);
        }
        for (i = 0; i < rsi.length; ++i) {
            double avgGain = 0.0;
            double avgLoss = 0.0;
            if (i == 0) {
                for (int j = 0; j < period; ++j) {
                    avgGain += gains[j];
                    avgLoss += losses[j];
                }
                avgGain /= (double)period;
                avgLoss /= (double)period;
            } else {
                avgGain = (rsi[i - 1] * (double)(period - 1) + gains[i + period - 1]) / (double)period;
                avgLoss = (rsi[i - 1] * (double)(period - 1) + losses[i + period - 1]) / (double)period;
            }
            double rs = avgGain / (avgLoss > 0.0 ? avgLoss : 0.001);
            rsi[i] = 100.0 - 100.0 / (1.0 + rs);
        }
        return rsi;
    }

    private MacdResult calculateMACD(double[] prices) {
        if (prices.length < 26) {
            return new MacdResult(new double[0], new double[0], new double[0]);
        }
        int fastPeriod = 12;
        int slowPeriod = 26;
        int signalPeriod = 9;
        double[] emaFast = this.calculateEMA(prices, fastPeriod);
        double[] emaSlow = this.calculateEMA(prices, slowPeriod);
        double[] macdLine = new double[emaSlow.length];
        for (int i = 0; i < macdLine.length; ++i) {
            int fastIndex = i + (emaFast.length - emaSlow.length);
            macdLine[i] = emaFast[fastIndex] - emaSlow[i];
        }
        double[] signalLine = this.calculateEMA(macdLine, signalPeriod);
        double[] histogram = new double[signalLine.length];
        for (int i = 0; i < histogram.length; ++i) {
            int macdIndex = i + (macdLine.length - signalLine.length);
            histogram[i] = macdLine[macdIndex] - signalLine[i];
        }
        return new MacdResult(macdLine, signalLine, histogram);
    }

    private double[] calculateEMA(double[] prices, int period) {
        int i;
        if (prices.length < period) {
            return new double[0];
        }
        double[] ema = new double[prices.length - period + 1];
        double k = 2.0 / (double)(period + 1);
        double sum = 0.0;
        for (i = 0; i < period; ++i) {
            sum += prices[i];
        }
        ema[0] = sum / (double)period;
        for (i = 1; i < ema.length; ++i) {
            ema[i] = prices[i + period - 1] * k + ema[i - 1] * (1.0 - k);
        }
        return ema;
    }

    private BollingerBands calculateBollingerBands(double[] prices, int period, double multiplier) {
        if (prices.length < period) {
            return new BollingerBands(new double[0], new double[0], new double[0]);
        }
        double[] sma = this.calculateSMA(prices, period);
        double[] upper = new double[sma.length];
        double[] lower = new double[sma.length];
        for (int i = 0; i < sma.length; ++i) {
            double sum = 0.0;
            for (int j = 0; j < period; ++j) {
                sum += Math.pow(prices[i + j] - sma[i], 2.0);
            }
            double stdDev = Math.sqrt(sum / (double)period);
            upper[i] = sma[i] + multiplier * stdDev;
            lower[i] = sma[i] - multiplier * stdDev;
        }
        return new BollingerBands(sma, upper, lower);
    }

    private boolean detectPriceRsiDivergence(double[] prices, double[] rsi) {
        boolean rsiDowntrend;
        if (prices.length < 10 || rsi.length < 5) {
            return false;
        }
        int startPrice = prices.length - 10;
        int startRsi = rsi.length - 5;
        boolean priceUptrend = prices[prices.length - 1] > prices[startPrice];
        boolean bl = rsiDowntrend = rsi[rsi.length - 1] < rsi[startRsi];
        if (priceUptrend && rsiDowntrend) {
            return true;
        }
        boolean priceDowntrend = prices[prices.length - 1] < prices[startPrice];
        boolean rsiUptrend = rsi[rsi.length - 1] > rsi[startRsi];
        return priceDowntrend && rsiUptrend;
    }

    private List<TradingSignal> generateTradingSignals(double[] prices, double[] sma5, double[] sma20, double[] rsi, MacdResult macd, BollingerBands bands, List<PriceHistory.PriceDataPoint> points) {
        ArrayList<TradingSignal> signals = new ArrayList<TradingSignal>();
        if (prices.length < 30 || sma5.length < 2 || sma20.length < 2 || rsi.length < 2 || macd.macdLine.length < 2 || bands.middle.length < 2) {
            return signals;
        }
        int lastPrice = prices.length - 1;
        int lastSma5 = sma5.length - 1;
        int lastSma20 = sma20.length - 1;
        int lastRsi = rsi.length - 1;
        int lastMacd = macd.macdLine.length - 1;
        int lastSignal = macd.signalLine.length - 1;
        int lastBands = bands.middle.length - 1;
        if (sma5[lastSma5 - 1] < sma20[lastSma20 - 1] && sma5[lastSma5] > sma20[lastSma20]) {
            signals.add(new TradingSignal(TradingSignalType.GOLDEN_CROSS, "Croisement haussier SMA5 > SMA20", points.get(points.size() - 1).getTimestamp()));
        } else if (sma5[lastSma5 - 1] > sma20[lastSma20 - 1] && sma5[lastSma5] < sma20[lastSma20]) {
            signals.add(new TradingSignal(TradingSignalType.DEATH_CROSS, "Croisement baissier SMA5 < SMA20", points.get(points.size() - 1).getTimestamp()));
        }
        if (rsi[lastRsi] > 70.0) {
            signals.add(new TradingSignal(TradingSignalType.OVERBOUGHT, "RSI en zone de surachat (>" + Math.round(rsi[lastRsi]) + ")", points.get(points.size() - 1).getTimestamp()));
        } else if (rsi[lastRsi] < 30.0) {
            signals.add(new TradingSignal(TradingSignalType.OVERSOLD, "RSI en zone de survente (<" + Math.round(rsi[lastRsi]) + ")", points.get(points.size() - 1).getTimestamp()));
        }
        if (lastMacd > 0 && lastSignal > 0) {
            if (macd.macdLine[lastMacd - 1] < macd.signalLine[lastSignal - 1] && macd.macdLine[lastMacd] > macd.signalLine[lastSignal]) {
                signals.add(new TradingSignal(TradingSignalType.MACD_BULLISH_CROSS, "Croisement haussier du MACD", points.get(points.size() - 1).getTimestamp()));
            } else if (macd.macdLine[lastMacd - 1] > macd.signalLine[lastSignal - 1] && macd.macdLine[lastMacd] < macd.signalLine[lastSignal]) {
                signals.add(new TradingSignal(TradingSignalType.MACD_BEARISH_CROSS, "Croisement baissier du MACD", points.get(points.size() - 1).getTimestamp()));
            }
        }
        if (prices[lastPrice] > bands.upper[lastBands]) {
            signals.add(new TradingSignal(TradingSignalType.PRICE_ABOVE_UPPER_BAND, "Prix au-dessus de la bande sup\u00e9rieure de Bollinger", points.get(points.size() - 1).getTimestamp()));
        } else if (prices[lastPrice] < bands.lower[lastBands]) {
            signals.add(new TradingSignal(TradingSignalType.PRICE_BELOW_LOWER_BAND, "Prix en-dessous de la bande inf\u00e9rieure de Bollinger", points.get(points.size() - 1).getTimestamp()));
        }
        return signals;
    }

    private List<Map<String, Object>> generateForecastData(List<PriceHistory.PriceDataPoint> points, int daysToForecast, boolean isBuy) {
        double[] prices;
        ArrayList<Map<String, Object>> forecast = new ArrayList<Map<String, Object>>();
        double[] dArray = prices = isBuy ? points.stream().mapToDouble(PriceHistory.PriceDataPoint::getCloseBuyPrice).filter(price -> price > 0.0).toArray() : points.stream().mapToDouble(PriceHistory.PriceDataPoint::getCloseSellPrice).filter(price -> price > 0.0).toArray();
        if (prices.length < 5) {
            return forecast;
        }
        double[] trend = this.calculateLinearTrend(prices);
        double slope = trend[0];
        double intercept = trend[1];
        double rSquared = this.calculateRSquared(prices, slope, intercept);
        double errorMultiplier = 0.1 + (1.0 - rSquared) * 0.2;
        LocalDateTime lastDate = points.get(points.size() - 1).getTimestamp();
        for (int i = 1; i <= daysToForecast; ++i) {
            LocalDateTime forecastDate = lastDate.plusDays(i);
            double forecastPrice = intercept + slope * (double)(prices.length + i);
            double errorMargin = forecastPrice * errorMultiplier;
            HashMap<String, Object> forecastPoint = new HashMap<String, Object>();
            forecastPoint.put("date", forecastDate.toString());
            forecastPoint.put("price", Math.max(0.01, forecastPrice));
            forecastPoint.put("lowEstimate", Math.max(0.01, forecastPrice - errorMargin));
            forecastPoint.put("highEstimate", forecastPrice + errorMargin);
            forecastPoint.put("confidence", rSquared);
            forecast.add(forecastPoint);
        }
        return forecast;
    }

    private List<Double> calculateSupportLevels(List<PriceHistory.PriceDataPoint> points, boolean isBuy) {
        List<Double> minimaPrices = this.findLocalMinima(points, isBuy);
        return this.clusterPriceLevels(minimaPrices);
    }

    private List<Double> calculateResistanceLevels(List<PriceHistory.PriceDataPoint> points, boolean isBuy) {
        List<Double> maximaPrices = this.findLocalMaxima(points, isBuy);
        return this.clusterPriceLevels(maximaPrices);
    }

    private List<Double> findLocalMinima(List<PriceHistory.PriceDataPoint> points, boolean isBuy) {
        ArrayList<Double> minima = new ArrayList<Double>();
        if (points.size() < 3) {
            return minima;
        }
        for (int i = 1; i < points.size() - 1; ++i) {
            double next;
            double prev = isBuy ? points.get(i - 1).getLowBuyPrice() : points.get(i - 1).getLowSellPrice();
            double current = isBuy ? points.get(i).getLowBuyPrice() : points.get(i).getLowSellPrice();
            double d = next = isBuy ? points.get(i + 1).getLowBuyPrice() : points.get(i + 1).getLowSellPrice();
            if (!(current > 0.0) || !(current < prev) || !(current < next)) continue;
            minima.add(current);
        }
        return minima;
    }

    private List<Double> findLocalMaxima(List<PriceHistory.PriceDataPoint> points, boolean isBuy) {
        ArrayList<Double> maxima = new ArrayList<Double>();
        if (points.size() < 3) {
            return maxima;
        }
        for (int i = 1; i < points.size() - 1; ++i) {
            double next;
            double prev = isBuy ? points.get(i - 1).getHighBuyPrice() : points.get(i - 1).getHighSellPrice();
            double current = isBuy ? points.get(i).getHighBuyPrice() : points.get(i).getHighSellPrice();
            double d = next = isBuy ? points.get(i + 1).getHighBuyPrice() : points.get(i + 1).getHighSellPrice();
            if (!(current > prev) || !(current > next)) continue;
            maxima.add(current);
        }
        return maxima;
    }

    private List<Double> clusterPriceLevels(List<Double> prices) {
        if (prices.isEmpty()) {
            return List.of();
        }
        ArrayList<Double> sortedPrices = new ArrayList<Double>(prices);
        Collections.sort(sortedPrices);
        double avgPrice = sortedPrices.stream().mapToDouble(Double::doubleValue).average().orElse(0.0);
        double clusterTolerance = avgPrice * 0.05;
        ArrayList<Double> clusters = new ArrayList<Double>();
        ArrayList<Double> currentCluster = new ArrayList<Double>();
        Iterator iterator = sortedPrices.iterator();
        while (iterator.hasNext()) {
            double price = (Double)iterator.next();
            if (currentCluster.isEmpty()) {
                currentCluster.add(price);
                continue;
            }
            double clusterAvg = currentCluster.stream().mapToDouble(Double::doubleValue).average().orElse(0.0);
            if (Math.abs(price - clusterAvg) <= clusterTolerance) {
                currentCluster.add(price);
                continue;
            }
            clusters.add(currentCluster.stream().mapToDouble(Double::doubleValue).average().orElse(0.0));
            currentCluster = new ArrayList();
            currentCluster.add(price);
        }
        if (!currentCluster.isEmpty()) {
            clusters.add(currentCluster.stream().mapToDouble(Double::doubleValue).average().orElse(0.0));
        }
        return clusters;
    }

    private double[] calculateLinearTrend(double[] prices) {
        int n = prices.length;
        double sumX = 0.0;
        double sumY = 0.0;
        double sumXY = 0.0;
        double sumX2 = 0.0;
        for (int i = 0; i < n; ++i) {
            sumX += (double)i;
            sumY += prices[i];
            sumXY += (double)i * prices[i];
            sumX2 += (double)(i * i);
        }
        double denominator = (double)n * sumX2 - sumX * sumX;
        if (Math.abs(denominator) < 1.0E-10) {
            return new double[]{0.0, prices.length > 0 ? prices[0] : 0.0};
        }
        double a = ((double)n * sumXY - sumX * sumY) / denominator;
        double b = (sumY - a * sumX) / (double)n;
        return new double[]{a, b};
    }

    private double calculateRSquared(double[] prices, double slope, double intercept) {
        int n = prices.length;
        double mean = Arrays.stream(prices).average().orElse(0.0);
        double sst = 0.0;
        for (double price : prices) {
            sst += Math.pow(price - mean, 2.0);
        }
        double ssr = 0.0;
        for (int i = 0; i < n; ++i) {
            double fitted = intercept + slope * (double)i;
            ssr += Math.pow(fitted - mean, 2.0);
        }
        return sst > 0.0 ? ssr / sst : 0.0;
    }

    public static class MarketTrend {
        private final TrendType trendType;
        private final double strength;
        private final double volatility;
        private final double buyPriceChangePercent;
        private final double sellPriceChangePercent;
        private final double volumeChangePercent;
        public final TechnicalAnalysis buyAnalysis;
        public final TechnicalAnalysis sellAnalysis;
        private final List<Map<String, Object>> buyForecastData;
        private final List<Map<String, Object>> sellForecastData;
        private final List<Double> buySupportLevels;
        private final List<Double> buyResistanceLevels;
        private final List<Double> sellSupportLevels;
        private final List<Double> sellResistanceLevels;

        public MarketTrend(TrendType trendType, double strength, double volatility, double buyPriceChangePercent, double sellPriceChangePercent, double volumeChangePercent, TechnicalAnalysis buyAnalysis, TechnicalAnalysis sellAnalysis, List<Map<String, Object>> buyForecastData, List<Map<String, Object>> sellForecastData, List<Double> buySupportLevels, List<Double> buyResistanceLevels, List<Double> sellSupportLevels, List<Double> sellResistanceLevels) {
            this.trendType = trendType;
            this.strength = strength;
            this.volatility = volatility;
            this.buyPriceChangePercent = buyPriceChangePercent;
            this.sellPriceChangePercent = sellPriceChangePercent;
            this.volumeChangePercent = volumeChangePercent;
            this.buyAnalysis = buyAnalysis;
            this.sellAnalysis = sellAnalysis;
            this.buyForecastData = buyForecastData;
            this.sellForecastData = sellForecastData;
            this.buySupportLevels = buySupportLevels;
            this.buyResistanceLevels = buyResistanceLevels;
            this.sellSupportLevels = sellSupportLevels;
            this.sellResistanceLevels = sellResistanceLevels;
        }

        public TrendType getTrendType() {
            return this.trendType;
        }

        public double getStrength() {
            return this.strength;
        }

        public double getVolatility() {
            return this.volatility;
        }

        public double getBuyPriceChangePercent() {
            return this.buyPriceChangePercent;
        }

        public double getSellPriceChangePercent() {
            return this.sellPriceChangePercent;
        }

        public double getPriceChangePercent() {
            if (Math.abs(this.buyPriceChangePercent) > Math.abs(this.sellPriceChangePercent)) {
                return this.buyPriceChangePercent;
            }
            return this.sellPriceChangePercent;
        }

        public double getVolumeChangePercent() {
            return this.volumeChangePercent;
        }

        public List<Map<String, Object>> getBuyForecastData() {
            return this.buyForecastData;
        }

        public List<Map<String, Object>> getSellForecastData() {
            return this.sellForecastData;
        }

        public List<Double> getBuySupportLevels() {
            return this.buySupportLevels;
        }

        public List<Double> getBuyResistanceLevels() {
            return this.buyResistanceLevels;
        }

        public List<Double> getSellSupportLevels() {
            return this.sellSupportLevels;
        }

        public List<Double> getSellResistanceLevels() {
            return this.sellResistanceLevels;
        }

        public Map<String, Object> toMap() {
            HashMap<String, Object> map = new HashMap<String, Object>();
            map.put("trend", this.trendType.name());
            map.put("strength", this.strength);
            map.put("volatility", this.volatility);
            map.put("buyPriceChange", this.buyPriceChangePercent);
            map.put("sellPriceChange", this.sellPriceChangePercent);
            map.put("buyAnalysis", this.buyAnalysis != null ? this.buyAnalysis.toMap() : null);
            map.put("sellAnalysis", this.sellAnalysis != null ? this.sellAnalysis.toMap() : null);
            map.put("buyForecast", this.buyForecastData);
            map.put("sellForecast", this.sellForecastData);
            map.put("buySupportLevels", this.buySupportLevels);
            map.put("buyResistanceLevels", this.buyResistanceLevels);
            map.put("sellSupportLevels", this.sellSupportLevels);
            map.put("sellResistanceLevels", this.sellResistanceLevels);
            return map;
        }
    }

    public static enum TrendType {
        RISING,
        FALLING,
        STABLE,
        VOLATILE,
        SPREAD_CHANGING,
        UNKNOWN;

    }

    public static class TechnicalAnalysis {
        private final TrendType trendType;
        private final double strength;
        private final double volatility;
        private final double[] sma5;
        private final double[] sma20;
        private final double[] rsi;
        private final MacdResult macd;
        private final BollingerBands bollingerBands;
        private final List<TradingSignal> signals;
        private final boolean priceRsiDivergence;

        public TechnicalAnalysis(TrendType trendType, double strength, double volatility) {
            this(trendType, strength, volatility, new double[0], new double[0], new double[0], new MacdResult(new double[0], new double[0], new double[0]), new BollingerBands(new double[0], new double[0], new double[0]), List.of(), false);
        }

        public TechnicalAnalysis(TrendType trendType, double strength, double volatility, double[] sma5, double[] sma20, double[] rsi, MacdResult macd, BollingerBands bollingerBands, List<TradingSignal> signals, boolean priceRsiDivergence) {
            this.trendType = trendType;
            this.strength = strength;
            this.volatility = volatility;
            this.sma5 = sma5;
            this.sma20 = sma20;
            this.rsi = rsi;
            this.macd = macd;
            this.bollingerBands = bollingerBands;
            this.signals = signals;
            this.priceRsiDivergence = priceRsiDivergence;
        }

        public TrendType getTrendType() {
            return this.trendType;
        }

        public double getStrength() {
            return this.strength;
        }

        public double getVolatility() {
            return this.volatility;
        }

        public Map<String, Object> toMap() {
            HashMap<String, Object> map = new HashMap<String, Object>();
            map.put("trend", this.trendType.name());
            map.put("strength", this.strength);
            map.put("volatility", this.volatility);
            int pointsToTake = 30;
            double[] sma5Subset = this.sma5.length > pointsToTake ? Arrays.copyOfRange(this.sma5, this.sma5.length - pointsToTake, this.sma5.length) : this.sma5;
            double[] sma20Subset = this.sma20.length > pointsToTake ? Arrays.copyOfRange(this.sma20, this.sma20.length - pointsToTake, this.sma20.length) : this.sma20;
            double[] rsiSubset = this.rsi.length > pointsToTake ? Arrays.copyOfRange(this.rsi, this.rsi.length - pointsToTake, this.rsi.length) : this.rsi;
            map.put("sma5", sma5Subset);
            map.put("sma20", sma20Subset);
            map.put("rsi", rsiSubset);
            map.put("macd", this.macd.toMap());
            map.put("bollingerBands", this.bollingerBands.toMap());
            map.put("priceRsiDivergence", this.priceRsiDivergence);
            List signalMaps = this.signals.stream().map(TradingSignal::toMap).collect(Collectors.toList());
            map.put("signals", signalMaps);
            return map;
        }
    }

    public static class MacdResult {
        public final double[] macdLine;
        public final double[] signalLine;
        public final double[] histogram;

        public MacdResult(double[] macdLine, double[] signalLine, double[] histogram) {
            this.macdLine = macdLine;
            this.signalLine = signalLine;
            this.histogram = histogram;
        }

        public Map<String, Object> toMap() {
            HashMap<String, Object> map = new HashMap<String, Object>();
            int pointsToTake = Math.min(30, this.macdLine.length);
            int start = Math.max(0, this.macdLine.length - pointsToTake);
            double[] macdSubset = Arrays.copyOfRange(this.macdLine, start, this.macdLine.length);
            double[] signalSubset = this.signalLine.length >= pointsToTake ? Arrays.copyOfRange(this.signalLine, Math.max(0, this.signalLine.length - pointsToTake), this.signalLine.length) : this.signalLine;
            double[] histogramSubset = this.histogram.length >= pointsToTake ? Arrays.copyOfRange(this.histogram, Math.max(0, this.histogram.length - pointsToTake), this.histogram.length) : this.histogram;
            map.put("macd", macdSubset);
            map.put("signal", signalSubset);
            map.put("histogram", histogramSubset);
            return map;
        }
    }

    public static class BollingerBands {
        public final double[] middle;
        public final double[] upper;
        public final double[] lower;

        public BollingerBands(double[] middle, double[] upper, double[] lower) {
            this.middle = middle;
            this.upper = upper;
            this.lower = lower;
        }

        public Map<String, Object> toMap() {
            HashMap<String, Object> map = new HashMap<String, Object>();
            int pointsToTake = Math.min(30, this.middle.length);
            int start = Math.max(0, this.middle.length - pointsToTake);
            double[] middleSubset = Arrays.copyOfRange(this.middle, start, this.middle.length);
            double[] upperSubset = Arrays.copyOfRange(this.upper, start, this.upper.length);
            double[] lowerSubset = Arrays.copyOfRange(this.lower, start, this.lower.length);
            map.put("middle", middleSubset);
            map.put("upper", upperSubset);
            map.put("lower", lowerSubset);
            return map;
        }
    }

    public static class TradingSignal {
        private final TradingSignalType type;
        private final String description;
        private final LocalDateTime timestamp;

        public TradingSignal(TradingSignalType type, String description, LocalDateTime timestamp) {
            this.type = type;
            this.description = description;
            this.timestamp = timestamp;
        }

        public TradingSignalType getType() {
            return this.type;
        }

        public String getDescription() {
            return this.description;
        }

        public LocalDateTime getTimestamp() {
            return this.timestamp;
        }

        public Map<String, Object> toMap() {
            HashMap<String, Object> map = new HashMap<String, Object>();
            map.put("type", this.type.name());
            map.put("description", this.description);
            map.put("timestamp", this.timestamp.toString());
            return map;
        }
    }

    public static enum TradingSignalType {
        GOLDEN_CROSS,
        DEATH_CROSS,
        OVERBOUGHT,
        OVERSOLD,
        MACD_BULLISH_CROSS,
        MACD_BEARISH_CROSS,
        PRICE_ABOVE_UPPER_BAND,
        PRICE_BELOW_LOWER_BAND;

    }
}

