/*
 * Decompiled with CFR 0.152.
 */
package fr.tylwen.satyria.dynashop.system;

import fr.tylwen.satyria.dynashop.DynaShopPlugin;
import fr.tylwen.satyria.dynashop.data.cache.LimitCacheEntry;
import fr.tylwen.satyria.dynashop.data.model.TransactionRecord;
import fr.tylwen.satyria.dynashop.data.storage.StorageManager;
import java.time.DayOfWeek;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAdjusters;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.bukkit.entity.Player;

public class TransactionLimiter {
    private final DynaShopPlugin plugin;
    private final StorageManager storageManager;
    private final ConcurrentLinkedQueue<TransactionRecord> transactionQueue;
    private final Map<String, Integer> metricsCounter;
    private final ScheduledExecutorService scheduler;
    private final AtomicBoolean running = new AtomicBoolean(true);

    public TransactionLimiter(DynaShopPlugin plugin) {
        this.plugin = plugin;
        this.transactionQueue = new ConcurrentLinkedQueue();
        this.metricsCounter = new ConcurrentHashMap<String, Integer>();
        this.storageManager = plugin.getStorageManager();
        this.scheduler = Executors.newScheduledThreadPool(1);
        this.scheduler.scheduleWithFixedDelay(this::processQueue, 100L, 100L, TimeUnit.MILLISECONDS);
        this.scheduler.scheduleAtFixedRate(this::cleanupOldData, 12L, 24L, TimeUnit.HOURS);
    }

    private void processQueue() {
        try {
            TransactionRecord record;
            ArrayList<TransactionRecord> batch = new ArrayList<TransactionRecord>();
            for (int batchSize = 0; (record = this.transactionQueue.poll()) != null && batchSize < 50; ++batchSize) {
                batch.add(record);
            }
            if (!batch.isEmpty()) {
                this.storageManager.saveTransactionsBatch(batch);
                this.incrementMetric("processed_transactions", batch.size());
            }
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Erreur lors du traitement des transactions: " + e.getMessage());
        }
    }

    private void cleanupOldData() {
        try {
            this.storageManager.cleanupExpiredTransactions();
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Erreur lors du nettoyage des donn\u00e9es: " + e.getMessage());
        }
    }

    public void shutdown() {
        this.running.set(false);
        try {
            this.scheduler.shutdown();
            if (!this.scheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.scheduler.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.scheduler.shutdownNow();
            Thread.currentThread().interrupt();
        }
        this.processQueue();
    }

    public void queueTransaction(Player player, String shopId, String itemId, boolean isBuy, int amount) {
        if (amount <= 0 || !this.running.get()) {
            return;
        }
        TransactionRecord record = new TransactionRecord(player.getUniqueId(), shopId, itemId, isBuy, amount, LocalDateTime.now());
        this.transactionQueue.add(record);
        String cacheKey = DynaShopPlugin.getLimitCacheKey(player.getUniqueId(), shopId, itemId, isBuy);
        this.plugin.getLimitCache().invalidate(cacheKey);
    }

    public boolean canPerformTransactionSync(Player player, String shopId, String itemId, boolean isBuy, int amount) {
        LimitCacheEntry entry = this.getTransactionLimit(player, shopId, itemId, isBuy);
        if (entry == null || entry.baseLimit <= 0) {
            return true;
        }
        return entry.remaining >= amount;
    }

    public LimitCacheEntry getTransactionLimit(Player player, String shopId, String itemId, boolean isBuy) {
        String cacheKey = DynaShopPlugin.getLimitCacheKey(player.getUniqueId(), shopId, itemId, isBuy);
        return this.plugin.getLimitCache().get(cacheKey, () -> {
            try {
                String limitPath = isBuy ? "limit.buy" : "limit.sell";
                int limitAmount = this.plugin.getShopConfigManager().getItemValue(shopId, itemId, limitPath, Integer.class).orElse(0);
                if (limitAmount <= 0) {
                    return null;
                }
                int cooldownSeconds = 0;
                LimitPeriod period = LimitPeriod.NONE;
                Optional<Integer> cooldownInt = this.plugin.getShopConfigManager().getItemValue(shopId, itemId, "limit.cooldown", Integer.class);
                if (cooldownInt.isPresent()) {
                    cooldownSeconds = cooldownInt.get();
                } else {
                    Optional<String> cooldownStr = this.plugin.getShopConfigManager().getItemValue(shopId, itemId, "limit.cooldown", String.class);
                    if (cooldownStr.isPresent()) {
                        String periodStr = cooldownStr.get().toUpperCase();
                        try {
                            period = LimitPeriod.valueOf(periodStr);
                            cooldownSeconds = period.getSeconds();
                        }
                        catch (IllegalArgumentException e) {
                            try {
                                cooldownSeconds = Integer.parseInt(periodStr);
                            }
                            catch (NumberFormatException ex) {
                                cooldownSeconds = 0;
                            }
                        }
                    }
                }
                if (cooldownSeconds <= 0) {
                    return null;
                }
                if (period == LimitPeriod.NONE) {
                    period = this.getPeriodForCooldown(cooldownSeconds);
                }
                int remaining = this.calculateRemainingAmount(player.getUniqueId(), shopId, itemId, isBuy, limitAmount, cooldownSeconds, period);
                long nextAvailable = this.calculateNextAvailableTime(player.getUniqueId(), shopId, itemId, isBuy, cooldownSeconds, period);
                return new LimitCacheEntry(player.getUniqueId(), shopId, itemId, isBuy, limitAmount, cooldownSeconds, remaining, nextAvailable);
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Erreur lors du calcul des limites: " + e.getMessage());
                return null;
            }
        });
    }

    private int calculateRemainingAmount(UUID playerUuid, String shopId, String itemId, boolean isBuy, int limitAmount, int cooldownSeconds, LimitPeriod period) {
        LocalDateTime startDate = period != LimitPeriod.NONE ? period.getStartDate() : LocalDateTime.now().minusSeconds(cooldownSeconds);
        int used = this.storageManager.getUsedAmount(playerUuid, shopId, itemId, isBuy, startDate);
        return Math.max(0, limitAmount - used);
    }

    public int getRemainingAmountSync(Player player, String shopId, String itemId, boolean isBuy) {
        String cacheKey = DynaShopPlugin.getLimitCacheKey(player.getUniqueId(), shopId, itemId, isBuy);
        LimitCacheEntry entry = this.plugin.getLimitCache().getIfPresent(cacheKey);
        if (entry != null) {
            return entry.remaining;
        }
        entry = this.getTransactionLimit(player, shopId, itemId, isBuy);
        if (entry == null) {
            return Integer.MAX_VALUE;
        }
        return entry.remaining;
    }

    private long calculateNextAvailableTime(UUID playerUuid, String shopId, String itemId, boolean isBuy, int cooldownSeconds, LimitPeriod period) {
        if (period != LimitPeriod.NONE && period != LimitPeriod.FOREVER) {
            LocalDateTime nextReset = period.getNextReset();
            return ChronoUnit.MILLIS.between(LocalDateTime.now(), nextReset);
        }
        Optional<LocalDateTime> lastTransaction = this.storageManager.getLastTransactionTime(playerUuid, shopId, itemId, isBuy);
        if (lastTransaction.isPresent()) {
            LocalDateTime nextAvailable = lastTransaction.get().plusSeconds(cooldownSeconds);
            long millisUntilAvailable = ChronoUnit.MILLIS.between(LocalDateTime.now(), nextAvailable);
            return Math.max(0L, millisUntilAvailable);
        }
        return 0L;
    }

    public long getNextAvailableTimeSync(Player player, String shopId, String itemId, boolean isBuy) {
        String cacheKey = DynaShopPlugin.getLimitCacheKey(player.getUniqueId(), shopId, itemId, isBuy);
        LimitCacheEntry entry = this.plugin.getLimitCache().getIfPresent(cacheKey);
        if (entry != null) {
            return entry.nextAvailable;
        }
        entry = this.getTransactionLimit(player, shopId, itemId, isBuy);
        if (entry == null) {
            return 0L;
        }
        return entry.nextAvailable;
    }

    private LimitPeriod getPeriodForCooldown(int cooldown) {
        if (cooldown >= 31536000) {
            return LimitPeriod.FOREVER;
        }
        if (cooldown >= 2592000) {
            return LimitPeriod.MONTHLY;
        }
        if (cooldown >= 604800) {
            return LimitPeriod.WEEKLY;
        }
        if (cooldown >= 86400) {
            return LimitPeriod.DAILY;
        }
        return LimitPeriod.NONE;
    }

    public CompletableFuture<Boolean> resetLimits(Player player, String shopId, String itemId) {
        UUID playerUuid = player.getUniqueId();
        return CompletableFuture.supplyAsync(() -> {
            boolean success = this.storageManager.resetLimits(playerUuid, shopId, itemId);
            String buyCacheKey = DynaShopPlugin.getLimitCacheKey(playerUuid, shopId, itemId, true);
            String sellCacheKey = DynaShopPlugin.getLimitCacheKey(playerUuid, shopId, itemId, false);
            this.plugin.getLimitCache().invalidate(buyCacheKey);
            this.plugin.getLimitCache().invalidate(sellCacheKey);
            return success;
        });
    }

    public CompletableFuture<Boolean> resetAllLimits(UUID playerUuid) {
        return CompletableFuture.supplyAsync(() -> this.storageManager.resetAllLimits(playerUuid));
    }

    public CompletableFuture<Boolean> resetAllLimits() {
        return CompletableFuture.supplyAsync(() -> this.storageManager.resetAllLimits());
    }

    public CompletableFuture<Map<String, Object>> getStatistics() {
        return CompletableFuture.supplyAsync(() -> {
            HashMap<String, Object> stats = new HashMap<String, Object>();
            stats.putAll(this.metricsCounter);
            stats.putAll(this.storageManager.getStatistics());
            return stats;
        });
    }

    private void incrementMetric(String metricName, int value) {
        this.metricsCounter.merge(metricName, value, Integer::sum);
    }

    public static enum LimitPeriod {
        DAILY(86400),
        WEEKLY(604800),
        MONTHLY(2592000),
        YEARLY(31536000),
        FOREVER(Integer.MAX_VALUE),
        NONE(0);

        private final int seconds;

        private LimitPeriod(int seconds) {
            this.seconds = seconds;
        }

        public int getSeconds() {
            return this.seconds;
        }

        public static LimitPeriod fromString(String value) {
            try {
                return LimitPeriod.valueOf(value.toUpperCase());
            }
            catch (IllegalArgumentException e) {
                try {
                    int seconds = Integer.parseInt(value);
                    for (LimitPeriod period : LimitPeriod.values()) {
                        if (period.getSeconds() != seconds) continue;
                        return period;
                    }
                    return NONE;
                }
                catch (NumberFormatException ex) {
                    return NONE;
                }
            }
        }

        public LocalDateTime getStartDate() {
            return LocalDateTime.now().with(TemporalAdjusters.firstDayOfMonth()).withHour(0).withMinute(0).withSecond(0);
        }

        public LocalDateTime getNextReset() {
            switch (this.ordinal()) {
                case 0: {
                    return LocalDateTime.now().plusDays(1L).withHour(0).withMinute(0).withSecond(0);
                }
                case 1: {
                    return LocalDateTime.now().plusWeeks(1L).with(TemporalAdjusters.next(DayOfWeek.MONDAY)).withHour(0).withMinute(0).withSecond(0);
                }
                case 2: {
                    return LocalDateTime.now().plusMonths(1L).with(TemporalAdjusters.firstDayOfMonth()).withHour(0).withMinute(0).withSecond(0);
                }
                case 3: {
                    return LocalDateTime.now().plusYears(1L).withMonth(1).withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0);
                }
            }
            return LocalDateTime.MAX;
        }
    }
}

