/*
 * Decompiled with CFR 0.152.
 */
package fr.tylwen.satyria.dynashop.web;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpServer;
import fr.tylwen.satyria.dynashop.DynaShopPlugin;
import fr.tylwen.satyria.dynashop.data.param.DynaShopType;
import fr.tylwen.satyria.dynashop.system.MarketTrendAnalyzer;
import fr.tylwen.satyria.dynashop.system.chart.PriceHistory;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalDouble;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.stream.Collectors;
import net.brcdev.shopgui.ShopGuiPlusApi;
import net.brcdev.shopgui.shop.Shop;
import net.brcdev.shopgui.shop.item.ShopItem;
import net.brcdev.shopgui.shop.item.ShopItemType;
import org.bukkit.ChatColor;
import org.bukkit.configuration.ConfigurationSection;
import org.bukkit.configuration.file.FileConfiguration;

public class MarketWebServer {
    private final DynaShopPlugin plugin;
    private HttpServer server;
    private final int port;
    private final Gson gson;
    private static final String WEB_DIR = "/web";

    public MarketWebServer(DynaShopPlugin plugin, int port) {
        this.plugin = plugin;
        this.port = port;
        this.gson = new GsonBuilder().setPrettyPrinting().create();
    }

    public void start() {
        try {
            this.server = HttpServer.create(new InetSocketAddress(this.port), 0);
            this.server.createContext("/", this::handleRoot);
            this.server.createContext("/assets", this::handleAssets);
            this.server.createContext("/api/shops", this::handleShopsList);
            this.server.createContext("/api/items", this::handleItemsList);
            this.server.createContext("/api/prices", this::handlePricesData);
            this.server.createContext("/api/price-stats", this::handlePriceStats);
            this.server.createContext("/api/shop-type", this::handleShopType);
            this.server.createContext("/api/market-trends", this::handleMarketTrends);
            ThreadPoolExecutor threadPoolExecutor = (ThreadPoolExecutor)Executors.newFixedThreadPool(10);
            this.server.setExecutor(threadPoolExecutor);
            this.server.start();
            this.extractWebFiles();
        }
        catch (IOException e) {
            this.plugin.getLogger().severe("Error starting web server: " + e.getMessage());
            e.printStackTrace();
        }
    }

    public void stop() {
        if (this.server != null) {
            this.server.stop(0);
            this.plugin.getLogger().info("Web server stopped");
        }
    }

    private void extractWebFiles() {
        Path webDir = Paths.get(this.plugin.getDataFolder().getPath(), "web");
        if (!Files.exists(webDir, new LinkOption[0])) {
            try {
                Files.createDirectories(webDir, new FileAttribute[0]);
                this.extractResourceFile("index.html", webDir);
                Path assetsDir = Paths.get(webDir.toString(), "assets");
                Files.createDirectories(assetsDir, new FileAttribute[0]);
                this.extractResourceFile("assets/style.css", webDir);
                this.extractResourceFile("assets/script.js", webDir);
                this.extractResourceFile("assets/locales/en.json", webDir);
                this.extractResourceFile("assets/locales/fr.json", webDir);
                this.extractResourceFile("assets/locales/de.json", webDir);
                this.extractResourceFile("assets/locales/es.json", webDir);
                this.extractResourceFile("assets/locales/it.json", webDir);
                this.extractResourceFile("assets/locales/pt.json", webDir);
                this.extractResourceFile("assets/locales/ar.json", webDir);
                this.extractResourceFile("assets/locales/zh.json", webDir);
                this.extractResourceFile("assets/locales/hi.json", webDir);
                this.plugin.getLogger().info("Web files extracted successfully");
            }
            catch (IOException e) {
                this.plugin.getLogger().severe("Error extracting web files: " + e.getMessage());
            }
        }
    }

    private void extractResourceFile(String resourcePath, Path targetDir) {
        try {
            Path targetPath = Paths.get(targetDir.toString(), resourcePath);
            Files.createDirectories(targetPath.getParent(), new FileAttribute[0]);
            try (OutputStream os = Files.newOutputStream(targetPath, new OpenOption[0]);){
                byte[] buffer = new byte[1024];
                try (InputStream is = this.getClass().getResourceAsStream("/web/" + resourcePath);){
                    int bytesRead;
                    if (is == null) {
                        this.plugin.getLogger().warning("Resource not found: " + resourcePath);
                        return;
                    }
                    while ((bytesRead = is.read(buffer)) != -1) {
                        os.write(buffer, 0, bytesRead);
                    }
                }
            }
        }
        catch (IOException e) {
            this.plugin.getLogger().warning("Error extracting " + resourcePath + ": " + e.getMessage());
        }
    }

    private void handleRoot(HttpExchange exchange) throws IOException {
        Path indexPath = Paths.get(this.plugin.getDataFolder().getPath(), "web", "index.html");
        this.serveFile(exchange, indexPath, "text/html");
    }

    private void handleAssets(HttpExchange exchange) throws IOException {
        String path = exchange.getRequestURI().getPath().substring(1);
        Path filePath = Paths.get(this.plugin.getDataFolder().getPath(), "web", path);
        if (!Files.exists(filePath, new LinkOption[0])) {
            exchange.sendResponseHeaders(404, 0L);
            exchange.getResponseBody().close();
            return;
        }
        String contentType = "application/octet-stream";
        if (path.endsWith(".css")) {
            contentType = "text/css";
        } else if (path.endsWith(".js")) {
            contentType = "application/javascript";
        } else if (path.endsWith(".html")) {
            contentType = "text/html";
        } else if (path.endsWith(".json")) {
            contentType = "application/json";
        }
        this.serveFile(exchange, filePath, contentType);
    }

    private void handleShopsList(HttpExchange exchange) throws IOException {
        if (!exchange.getRequestMethod().equals("GET")) {
            exchange.sendResponseHeaders(405, 0L);
            exchange.getResponseBody().close();
            return;
        }
        Set<String> shopIds = this.plugin.getShopConfigManager().getShops();
        ArrayList<Map> shopsList = new ArrayList<Map>();
        FileConfiguration sgpConfig = ShopGuiPlusApi.getPlugin().getConfigMain().getConfig();
        HashMap<String, String> shopNamesMap = new HashMap<String, String>();
        ConfigurationSection shopMenuItems = sgpConfig.getConfigurationSection("shopMenuItems");
        if (shopMenuItems != null) {
            for (String key : shopMenuItems.getKeys(false)) {
                String shopId = shopMenuItems.getString(key + ".shop");
                String shopName = shopMenuItems.getString(key + ".item.name");
                if (shopId == null || shopName == null) continue;
                shopName = ChatColor.stripColor((String)ChatColor.translateAlternateColorCodes((char)'&', (String)shopName)).trim();
                shopNamesMap.put(shopId, shopName);
            }
        }
        for (String shopId : shopIds) {
            Shop shop = ShopGuiPlusApi.getPlugin().getShopManager().getShopById(shopId);
            if (shop == null) continue;
            HashMap<String, String> shopData = new HashMap<String, String>();
            shopData.put("id", shopId);
            String shopName = shopNamesMap.getOrDefault(shopId, null);
            if (shopName == null || shopName.isEmpty()) {
                shopName = ChatColor.stripColor((String)shop.getName().replace("%page%", "")).trim();
            }
            if (shopName.isEmpty()) {
                shopName = shopId;
            }
            shopData.put("name", shopName);
            shopsList.add(shopData);
        }
        shopsList.sort(Comparator.comparing(map -> (String)map.get("name")));
        String jsonResponse = this.gson.toJson(shopsList);
        this.sendJsonResponse(exchange, jsonResponse);
    }

    private void handleItemsList(HttpExchange exchange) throws IOException {
        String[] langs;
        String shopId;
        if (!exchange.getRequestMethod().equals("GET")) {
            exchange.sendResponseHeaders(405, 0L);
            exchange.getResponseBody().close();
            return;
        }
        Map<String, String[]> queryParams = this.parseQueryParams(exchange.getRequestURI().getQuery());
        String string = shopId = queryParams.containsKey("shop") ? queryParams.get("shop")[0] : null;
        if (shopId == null) {
            exchange.sendResponseHeaders(400, 0L);
            exchange.getResponseBody().close();
            return;
        }
        Shop shop = ShopGuiPlusApi.getPlugin().getShopManager().getShopById(shopId);
        if (shop == null) {
            exchange.sendResponseHeaders(404, 0L);
            exchange.getResponseBody().close();
            return;
        }
        String locale = "en";
        String acceptLanguage = exchange.getRequestHeaders().getFirst("Accept-Language");
        if (acceptLanguage != null && !acceptLanguage.isEmpty() && (langs = acceptLanguage.split(",")).length > 0) {
            String primaryLang;
            locale = primaryLang = langs[0].split(";")[0].split("-")[0];
        }
        String finalLocale = locale;
        HashMap uniqueItemsMap = new HashMap();
        this.plugin.getShopConfigManager().getShopItems(shopId).stream().filter(itemId -> {
            ShopItem shopItem = shop.getShopItem(itemId);
            return shopItem != null && shopItem.getType() != ShopItemType.DUMMY && shopItem.getType() != ShopItemType.SPECIAL;
        }).forEach(itemId -> {
            String itemName = this.plugin.getShopConfigManager().getItemNameWithLocale(shopId, (String)itemId, finalLocale);
            String normalizedName = itemName.toLowerCase().trim();
            HashMap<String, String> itemData = new HashMap<String, String>();
            itemData.put("id", (String)itemId);
            itemData.put("name", itemName);
            if (!uniqueItemsMap.containsKey(normalizedName) || ((String)((Map)uniqueItemsMap.get(normalizedName)).get("id")).length() > itemId.length()) {
                uniqueItemsMap.put(normalizedName, itemData);
            }
        });
        ArrayList items = new ArrayList(uniqueItemsMap.values());
        items.sort(Comparator.comparing(map -> (String)map.get("name")));
        String jsonResponse = this.gson.toJson(items);
        this.sendJsonResponse(exchange, jsonResponse);
    }

    private void handlePricesData(HttpExchange exchange) throws IOException {
        try {
            List<PriceHistory.PriceDataPoint> dataPoints;
            int interval;
            if (!exchange.getRequestMethod().equals("GET")) {
                exchange.sendResponseHeaders(405, 0L);
                exchange.getResponseBody().close();
                return;
            }
            Map<String, String[]> queryParams = this.parseQueryParams(exchange.getRequestURI().getQuery());
            String shopId = queryParams.containsKey("shop") ? queryParams.get("shop")[0] : null;
            String itemId = queryParams.containsKey("item") ? queryParams.get("item")[0] : null;
            String period = queryParams.containsKey("period") ? queryParams.get("period")[0] : "all";
            String granularity = queryParams.containsKey("granularity") ? queryParams.get("granularity")[0] : "auto";
            int maxPoints = queryParams.containsKey("maxPoints") ? Integer.parseInt(queryParams.get("maxPoints")[0]) : 2000;
            LocalDateTime startTime = null;
            if (!period.equals("all")) {
                LocalDateTime now = LocalDateTime.now();
                switch (period) {
                    case "1h": {
                        startTime = now.minusHours(1L);
                        break;
                    }
                    case "6h": {
                        startTime = now.minusHours(6L);
                        break;
                    }
                    case "12h": {
                        startTime = now.minusHours(12L);
                        break;
                    }
                    case "1d": {
                        startTime = now.minusDays(1L);
                        break;
                    }
                    case "1w": {
                        startTime = now.minusWeeks(1L);
                        break;
                    }
                    case "1m": {
                        startTime = now.minusMonths(1L);
                    }
                }
            }
            int INTERVAL_MINUTES = this.plugin.getConfigMain().getInt("history.save-interval", 15);
            if (granularity.equals("auto")) {
                interval = period.equals("1h") ? INTERVAL_MINUTES : (period.equals("6h") ? INTERVAL_MINUTES : (period.equals("12h") ? INTERVAL_MINUTES * 2 : (period.equals("1d") ? INTERVAL_MINUTES * 4 : (period.equals("1w") ? INTERVAL_MINUTES * 16 : (period.equals("1m") ? INTERVAL_MINUTES * 48 : INTERVAL_MINUTES)))));
            } else {
                switch (granularity) {
                    case "minute": {
                        int n = INTERVAL_MINUTES;
                        break;
                    }
                    case "5min": {
                        int n = INTERVAL_MINUTES;
                        break;
                    }
                    case "15min": {
                        int n = INTERVAL_MINUTES;
                        break;
                    }
                    case "30min": {
                        int n = INTERVAL_MINUTES * 2;
                        break;
                    }
                    case "hour": {
                        int n = INTERVAL_MINUTES * 4;
                        break;
                    }
                    case "day": {
                        int n = INTERVAL_MINUTES * 96;
                        break;
                    }
                    default: {
                        int n = interval = INTERVAL_MINUTES;
                    }
                }
            }
            if (shopId == null || itemId == null) {
                exchange.sendResponseHeaders(400, 0L);
                exchange.getResponseBody().close();
                return;
            }
            if (this.plugin.getDataConfig().getDatabaseType().equalsIgnoreCase("sqlite")) {
                PriceHistory history = this.plugin.getStorageManager().getPriceHistory(shopId, itemId);
                dataPoints = history.getDataPoints();
            } else {
                dataPoints = this.plugin.getStorageManager().getAggregatedPriceHistory(shopId, itemId, interval, startTime, maxPoints);
            }
            List<PriceHistory.PriceDataPoint> filteredPoints = this.filterByPeriod(dataPoints, period);
            List<Map<String, Object>> chartData = this.aggregateOrSampleData(filteredPoints, granularity, maxPoints);
            String jsonResponse = this.gson.toJson(chartData);
            this.sendJsonResponse(exchange, jsonResponse);
        }
        catch (Exception ex) {
            this.plugin.getLogger().severe("Erreur dans /api/prices : " + ex.getMessage());
            ex.printStackTrace();
            String errorJson = "{\"error\": \"Internal server error: " + ex.getMessage() + "\"}";
            exchange.getResponseHeaders().set("Content-Type", "application/json");
            exchange.sendResponseHeaders(500, errorJson.length());
            exchange.getResponseBody().write(errorJson.getBytes());
            exchange.getResponseBody().close();
        }
    }

    private void handlePriceStats(HttpExchange exchange) throws IOException {
        int recipeStock;
        String itemId;
        if (!exchange.getRequestMethod().equals("GET")) {
            exchange.sendResponseHeaders(405, 0L);
            exchange.getResponseBody().close();
            return;
        }
        Map<String, String[]> queryParams = this.parseQueryParams(exchange.getRequestURI().getQuery());
        String shopId = queryParams.containsKey("shop") ? queryParams.get("shop")[0] : null;
        String string = itemId = queryParams.containsKey("item") ? queryParams.get("item")[0] : null;
        if (shopId == null || itemId == null) {
            exchange.sendResponseHeaders(400, 0L);
            exchange.getResponseBody().close();
            return;
        }
        PriceHistory history = this.plugin.getStorageManager().getPriceHistory(shopId, itemId);
        List<PriceHistory.PriceDataPoint> dataPoints = history.getDataPoints();
        HashMap<String, Object> stats = new HashMap<String, Object>();
        stats.put("totalPoints", dataPoints.size());
        if (!dataPoints.isEmpty()) {
            PriceHistory.PriceDataPoint first = dataPoints.get(0);
            PriceHistory.PriceDataPoint last = dataPoints.get(dataPoints.size() - 1);
            stats.put("firstTimestamp", first.getTimestamp().toString());
            stats.put("lastTimestamp", last.getTimestamp().toString());
            stats.put("timeSpanHours", ChronoUnit.HOURS.between(first.getTimestamp(), last.getTimestamp()));
            OptionalDouble minBuyOpt = dataPoints.stream().mapToDouble(PriceHistory.PriceDataPoint::getLowBuyPrice).min();
            OptionalDouble maxBuyOpt = dataPoints.stream().mapToDouble(PriceHistory.PriceDataPoint::getHighBuyPrice).max();
            OptionalDouble minSellOpt = dataPoints.stream().mapToDouble(PriceHistory.PriceDataPoint::getLowSellPrice).min();
            OptionalDouble maxSellOpt = dataPoints.stream().mapToDouble(PriceHistory.PriceDataPoint::getHighSellPrice).max();
            double totalVolume = dataPoints.stream().mapToDouble(PriceHistory.PriceDataPoint::getVolume).sum();
            stats.put("minBuyPrice", minBuyOpt.orElse(0.0));
            stats.put("maxBuyPrice", maxBuyOpt.orElse(0.0));
            stats.put("minSellPrice", minSellOpt.orElse(0.0));
            stats.put("maxSellPrice", maxSellOpt.orElse(0.0));
            stats.put("currentBuyPrice", last.getCloseBuyPrice());
            stats.put("currentSellPrice", last.getCloseSellPrice());
            stats.put("totalVolume", totalVolume);
            stats.put("recommendedGranularity", this.recommendGranularity(dataPoints.size()));
        }
        DynaShopType dynaShopType = this.plugin.getShopConfigManager().getTypeDynaShop(shopId, itemId);
        DynaShopType buyType = this.plugin.getShopConfigManager().getTypeDynaShop(shopId, itemId, "buy");
        DynaShopType sellType = this.plugin.getShopConfigManager().getTypeDynaShop(shopId, itemId, "sell");
        if (dynaShopType == DynaShopType.STOCK || dynaShopType == DynaShopType.STATIC_STOCK) {
            Optional<Integer> stockOpt = this.plugin.getStorageManager().getStock(shopId, itemId);
            int stock = stockOpt.orElse(0);
            int maxStock = this.plugin.getShopConfigManager().getItemValue(shopId, itemId, "stock.max", Integer.class).orElse(this.plugin.getDataConfig().getStockMax());
            stats.put("currentStock", stock);
            stats.put("maxStock", maxStock);
            stats.put("isStockItem", true);
        } else if (dynaShopType == DynaShopType.RECIPE && (recipeStock = this.plugin.getPriceRecipe().calculateStock(shopId, itemId, new ArrayList<String>())) > 0) {
            int maxStock = this.plugin.getPriceRecipe().calculateMaxStock(shopId, itemId, new ArrayList<String>());
            stats.put("currentStock", recipeStock);
            stats.put("maxStock", maxStock > 0 ? maxStock : recipeStock * 2);
            stats.put("isRecipeStock", true);
        }
        stats.put("shopType", dynaShopType.toString());
        stats.put("buyType", buyType.toString());
        stats.put("sellType", sellType.toString());
        String jsonResponse = this.gson.toJson(stats);
        this.sendJsonResponse(exchange, jsonResponse);
    }

    private String recommendGranularity(int dataPointCount) {
        if (dataPointCount > 100000) {
            return "day";
        }
        if (dataPointCount > 10000) {
            return "hour";
        }
        if (dataPointCount > 2000) {
            return "minute";
        }
        return "raw";
    }

    private List<PriceHistory.PriceDataPoint> filterByPeriod(List<PriceHistory.PriceDataPoint> dataPoints, String period) {
        LocalDateTime cutoff;
        if (dataPoints.isEmpty() || period.equals("all")) {
            return dataPoints;
        }
        LocalDateTime now = LocalDateTime.now();
        switch (period) {
            case "1h": {
                cutoff = now.minusHours(1L);
                break;
            }
            case "6h": {
                cutoff = now.minusHours(6L);
                break;
            }
            case "12h": {
                cutoff = now.minusHours(12L);
                break;
            }
            case "1d": {
                cutoff = now.minusHours(24L);
                break;
            }
            case "1w": {
                cutoff = now.minusWeeks(1L);
                break;
            }
            case "1m": {
                cutoff = now.minusMonths(1L);
                break;
            }
            case "3m": {
                cutoff = now.minusMonths(3L);
                break;
            }
            default: {
                return dataPoints;
            }
        }
        return dataPoints.stream().filter(p -> p.getTimestamp().isAfter(cutoff)).collect(Collectors.toList());
    }

    private List<Map<String, Object>> aggregateOrSampleData(List<PriceHistory.PriceDataPoint> dataPoints, String granularity, int maxPoints) {
        if (dataPoints.isEmpty()) {
            return List.of();
        }
        if (dataPoints.size() <= maxPoints && granularity.equals("raw")) {
            return dataPoints.stream().map(this::convertDataPointToMap).collect(Collectors.toList());
        }
        if (granularity.equals("auto")) {
            int size = dataPoints.size();
            granularity = size > maxPoints * 10 ? "day" : (size > maxPoints * 3 ? "hour" : (size > maxPoints ? "minute" : "raw"));
        }
        switch (granularity) {
            case "minute": {
                return this.aggregateByTimeUnit(dataPoints, ChronoUnit.MINUTES);
            }
            case "hour": {
                return this.aggregateByTimeUnit(dataPoints, ChronoUnit.HOURS);
            }
            case "day": {
                return this.aggregateByTimeUnit(dataPoints, ChronoUnit.DAYS);
            }
        }
        return this.sampleDataPoints(dataPoints, maxPoints);
    }

    private Map<String, Object> convertDataPointToMap(PriceHistory.PriceDataPoint point) {
        HashMap<String, Object> pointData = new HashMap<String, Object>();
        pointData.put("timestamp", point.getTimestamp().toString());
        pointData.put("openBuy", point.getOpenBuyPrice());
        pointData.put("closeBuy", point.getCloseBuyPrice());
        pointData.put("highBuy", point.getHighBuyPrice());
        pointData.put("lowBuy", point.getLowBuyPrice());
        pointData.put("openSell", point.getOpenSellPrice());
        pointData.put("closeSell", point.getCloseSellPrice());
        pointData.put("highSell", point.getHighSellPrice());
        pointData.put("lowSell", point.getLowSellPrice());
        pointData.put("volume", point.getVolume());
        return pointData;
    }

    private List<Map<String, Object>> sampleDataPoints(List<PriceHistory.PriceDataPoint> points, int maxPoints) {
        double step;
        if (points.size() <= maxPoints) {
            return points.stream().map(this::convertDataPointToMap).collect(Collectors.toList());
        }
        ArrayList<Map<String, Object>> sampled = new ArrayList<Map<String, Object>>(maxPoints);
        sampled.add(this.convertDataPointToMap(points.get(0)));
        for (double i = step = (double)(points.size() - 2) / (double)(maxPoints - 2); i < (double)(points.size() - 1); i += step) {
            int index = (int)Math.floor(i);
            sampled.add(this.convertDataPointToMap(points.get(index)));
        }
        if (points.size() > 1) {
            sampled.add(this.convertDataPointToMap(points.get(points.size() - 1)));
        }
        return sampled;
    }

    private List<Map<String, Object>> aggregateByTimeUnit(List<PriceHistory.PriceDataPoint> points, ChronoUnit unit) {
        HashMap<String, List> groups = new HashMap<String, List>();
        for (PriceHistory.PriceDataPoint point : points) {
            LocalDateTime truncated = point.getTimestamp().truncatedTo(unit);
            String key = truncated.toString();
            groups.computeIfAbsent(key, k -> new ArrayList()).add(point);
        }
        ArrayList<Map<String, Object>> result = new ArrayList<Map<String, Object>>();
        for (Map.Entry entry : groups.entrySet()) {
            List group = (List)entry.getValue();
            PriceHistory.PriceDataPoint first = (PriceHistory.PriceDataPoint)group.get(0);
            PriceHistory.PriceDataPoint last = (PriceHistory.PriceDataPoint)group.get(group.size() - 1);
            double highBuy = group.stream().mapToDouble(PriceHistory.PriceDataPoint::getHighBuyPrice).max().orElse(0.0);
            double lowBuy = group.stream().mapToDouble(PriceHistory.PriceDataPoint::getLowBuyPrice).min().orElse(Double.MAX_VALUE);
            if (lowBuy == Double.MAX_VALUE) {
                lowBuy = 0.0;
            }
            double highSell = group.stream().mapToDouble(PriceHistory.PriceDataPoint::getHighSellPrice).max().orElse(0.0);
            double lowSell = group.stream().mapToDouble(PriceHistory.PriceDataPoint::getLowSellPrice).min().orElse(Double.MAX_VALUE);
            if (lowSell == Double.MAX_VALUE) {
                lowSell = 0.0;
            }
            double totalVolume = group.stream().mapToDouble(PriceHistory.PriceDataPoint::getVolume).sum();
            HashMap<String, Object> aggregated = new HashMap<String, Object>();
            aggregated.put("timestamp", entry.getKey());
            aggregated.put("openBuy", first.getOpenBuyPrice());
            aggregated.put("closeBuy", last.getCloseBuyPrice());
            aggregated.put("highBuy", highBuy);
            aggregated.put("lowBuy", lowBuy);
            aggregated.put("openSell", first.getOpenSellPrice());
            aggregated.put("closeSell", last.getCloseSellPrice());
            aggregated.put("highSell", highSell);
            aggregated.put("lowSell", lowSell);
            aggregated.put("volume", totalVolume);
            result.add(aggregated);
        }
        result.sort(Comparator.comparing(m -> m.get("timestamp").toString()));
        return result;
    }

    private void handleShopType(HttpExchange exchange) throws IOException {
        String itemId;
        if (!exchange.getRequestMethod().equals("GET")) {
            exchange.sendResponseHeaders(405, 0L);
            exchange.getResponseBody().close();
            return;
        }
        Map<String, String[]> queryParams = this.parseQueryParams(exchange.getRequestURI().getQuery());
        String shopId = queryParams.containsKey("shop") ? queryParams.get("shop")[0] : null;
        String string = itemId = queryParams.containsKey("item") ? queryParams.get("item")[0] : null;
        if (shopId == null || itemId == null) {
            exchange.sendResponseHeaders(400, 0L);
            exchange.getResponseBody().close();
            return;
        }
        DynaShopType generalType = this.plugin.getShopConfigManager().getTypeDynaShop(shopId, itemId);
        DynaShopType buyType = this.plugin.getShopConfigManager().getTypeDynaShop(shopId, itemId, "buy");
        DynaShopType sellType = this.plugin.getShopConfigManager().getTypeDynaShop(shopId, itemId, "sell");
        DynaShopType realBuyType = this.plugin.getShopConfigManager().getRealTypeDynaShop(shopId, itemId, "buy");
        DynaShopType realSellType = this.plugin.getShopConfigManager().getRealTypeDynaShop(shopId, itemId, "sell");
        HashMap<String, String> types = new HashMap<String, String>();
        types.put("general", generalType.name());
        types.put("buy", buyType.name());
        types.put("sell", sellType.name());
        types.put("realBuy", realBuyType.name());
        types.put("realSell", realSellType.name());
        String jsonResponse = this.gson.toJson(types);
        this.sendJsonResponse(exchange, jsonResponse);
    }

    private void serveFile(HttpExchange exchange, Path filePath, String contentType) throws IOException {
        if (!Files.exists(filePath, new LinkOption[0])) {
            exchange.sendResponseHeaders(404, 0L);
            exchange.getResponseBody().close();
            return;
        }
        byte[] fileContent = Files.readAllBytes(filePath);
        exchange.getResponseHeaders().set("Content-Type", contentType);
        exchange.sendResponseHeaders(200, fileContent.length);
        try (OutputStream os = exchange.getResponseBody();){
            os.write(fileContent);
        }
    }

    private void sendJsonResponse(HttpExchange exchange, String jsonResponse) throws IOException {
        exchange.getResponseHeaders().set("Content-Type", "application/json");
        exchange.getResponseHeaders().set("Access-Control-Allow-Origin", "*");
        byte[] responseBytes = jsonResponse.getBytes();
        exchange.sendResponseHeaders(200, responseBytes.length);
        try (OutputStream os = exchange.getResponseBody();){
            os.write(responseBytes);
        }
    }

    private Map<String, String[]> parseQueryParams(String query) {
        String[] pairs;
        HashMap<String, String[]> result = new HashMap<String, String[]>();
        if (query == null) {
            return result;
        }
        for (String pair : pairs = query.split("&")) {
            String[] keyValue = pair.split("=");
            if (keyValue.length != 2) continue;
            String key = keyValue[0];
            String value = keyValue[1];
            if (result.containsKey(key)) {
                String[] values = (String[])result.get(key);
                String[] newValues = new String[values.length + 1];
                System.arraycopy(values, 0, newValues, 0, values.length);
                newValues[values.length] = value;
                result.put(key, newValues);
                continue;
            }
            result.put(key, new String[]{value});
        }
        return result;
    }

    private void handleMarketTrends(HttpExchange exchange) throws IOException {
        int days;
        if (!exchange.getRequestMethod().equals("GET")) {
            exchange.sendResponseHeaders(405, 0L);
            exchange.getResponseBody().close();
            return;
        }
        Map<String, String[]> queryParams = this.parseQueryParams(exchange.getRequestURI().getQuery());
        String shopId = queryParams.containsKey("shop") ? queryParams.get("shop")[0] : null;
        String itemId = queryParams.containsKey("item") ? queryParams.get("item")[0] : null;
        int n = days = queryParams.containsKey("days") ? Integer.parseInt(queryParams.get("days")[0]) : 7;
        if (shopId == null || itemId == null) {
            exchange.sendResponseHeaders(400, 0L);
            exchange.getResponseBody().close();
            return;
        }
        try {
            MarketTrendAnalyzer analyzer = new MarketTrendAnalyzer(this.plugin);
            MarketTrendAnalyzer.MarketTrend trend = analyzer.analyzeTrend(shopId, itemId, days);
            HashMap<String, Object> response = new HashMap<String, Object>();
            response.put("shopId", shopId);
            response.put("itemId", itemId);
            response.put("itemName", this.plugin.getShopConfigManager().getItemName(null, shopId, itemId));
            response.put("period", days + "d");
            response.put("trend", trend.getTrendType().name());
            response.put("strength", trend.getStrength());
            response.put("volatility", trend.getVolatility());
            response.put("buyPriceChange", trend.getBuyPriceChangePercent());
            response.put("sellPriceChange", trend.getSellPriceChangePercent());
            response.put("priceChange", trend.getPriceChangePercent());
            response.put("volumeChange", trend.getVolumeChangePercent());
            response.put("buyAnalysis", trend.buyAnalysis != null ? trend.buyAnalysis.toMap() : null);
            response.put("sellAnalysis", trend.sellAnalysis != null ? trend.sellAnalysis.toMap() : null);
            response.put("buyForecast", trend.getBuyForecastData());
            response.put("sellForecast", trend.getSellForecastData());
            response.put("buySupportLevels", trend.getBuySupportLevels());
            response.put("buyResistanceLevels", trend.getBuyResistanceLevels());
            response.put("sellSupportLevels", trend.getSellSupportLevels());
            response.put("sellResistanceLevels", trend.getSellResistanceLevels());
            String jsonResponse = this.gson.toJson(response);
            this.sendJsonResponse(exchange, jsonResponse);
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Erreur lors de l'analyse des tendances: " + e.getMessage());
            e.printStackTrace();
            String errorJson = "{\"error\": \"Failed to analyze market trends: " + e.getMessage() + "\"}";
            exchange.getResponseHeaders().set("Content-Type", "application/json");
            exchange.sendResponseHeaders(500, errorJson.length());
            exchange.getResponseBody().write(errorJson.getBytes());
            exchange.getResponseBody().close();
        }
    }
}

