#ifdef FXAA_TAA_INTERACTION
    ivec2 neighbourhoodOffsets[8] = ivec2[8](
        ivec2( 1, 1), ivec2( 1,-1), ivec2(-1, 1), ivec2(-1,-1),
        ivec2( 1, 0), ivec2( 0, 1), ivec2(-1, 0), ivec2( 0,-1)
    );
#endif

// FXAA 3.11 constants from http://blog.simonrodriguez.fr/articles/30-07-2016_implementing_fxaa.html
const float quality[12] = float[12](
    1.0, 1.0, 1.0, 1.0, 1.0, 1.5, 2.0, 2.0, 2.0, 2.0, 4.0, 8.0
);

void FXAA311(inout vec3 color) {
    const float edgeThresholdMin = 0.03125;
    const float edgeThresholdMax = 0.0625;
    const float subpixelQuality = 0.75;
    const int maxIterations = 12;

    vec2 invResolution = 1.0 / vec2(viewWidth, viewHeight);

    // Fetch luminance at center and neighbors
    float lumaCenter = GetLuminance(color);
    float lumaDown  = GetLuminance(texelFetch(colortex3, texelCoord + ivec2(0, -1), 0).rgb);
    float lumaUp    = GetLuminance(texelFetch(colortex3, texelCoord + ivec2(0,  1), 0).rgb);
    float lumaLeft  = GetLuminance(texelFetch(colortex3, texelCoord + ivec2(-1, 0), 0).rgb);
    float lumaRight = GetLuminance(texelFetch(colortex3, texelCoord + ivec2(1,  0), 0).rgb);

    float lumaMin = min(lumaCenter, min(min(lumaDown, lumaUp), min(lumaLeft, lumaRight)));
    float lumaMax = max(lumaCenter, max(max(lumaDown, lumaUp), max(lumaLeft, lumaRight)));
    float lumaRange = lumaMax - lumaMin;

    if (lumaRange > max(edgeThresholdMin, lumaMax * edgeThresholdMax)) {
        // Diagonal neighbors luminance
        float lumaDownLeft  = GetLuminance(texelFetch(colortex3, texelCoord + ivec2(-1, -1), 0).rgb);
        float lumaUpRight   = GetLuminance(texelFetch(colortex3, texelCoord + ivec2( 1,  1), 0).rgb);
        float lumaUpLeft    = GetLuminance(texelFetch(colortex3, texelCoord + ivec2(-1,  1), 0).rgb);
        float lumaDownRight = GetLuminance(texelFetch(colortex3, texelCoord + ivec2( 1, -1), 0).rgb);

        float lumaDownUp    = lumaDown + lumaUp;
        float lumaLeftRight = lumaLeft + lumaRight;

        float lumaLeftCorners  = lumaDownLeft + lumaUpLeft;
        float lumaDownCorners  = lumaDownLeft + lumaDownRight;
        float lumaRightCorners = lumaDownRight + lumaUpRight;
        float lumaUpCorners    = lumaUpRight + lumaUpLeft;

        // Edge detection on horizontal and vertical axes
        float edgeHorizontal = abs(-2.0 * lumaLeft + lumaLeftCorners) +
                               abs(-2.0 * lumaCenter + lumaDownUp) * 2.0 +
                               abs(-2.0 * lumaRight + lumaRightCorners);
                               
        float edgeVertical = abs(-2.0 * lumaUp + lumaUpCorners) +
                             abs(-2.0 * lumaCenter + lumaLeftRight) * 2.0 +
                             abs(-2.0 * lumaDown + lumaDownCorners);

        bool isHorizontal = edgeHorizontal >= edgeVertical;

        // Choose neighbors along detected edge direction
        float luma1 = isHorizontal ? lumaDown : lumaLeft;
        float luma2 = isHorizontal ? lumaUp   : lumaRight;
        float grad1 = luma1 - lumaCenter;
        float grad2 = luma2 - lumaCenter;

        bool grad1IsSteeper = abs(grad1) >= abs(grad2);
        float gradMaxAbs = 0.25 * max(abs(grad1), abs(grad2));
        float stepLength = isHorizontal ? invResolution.y : invResolution.x;

        float lumaLocalAvg = 0.0;
        if (grad1IsSteeper) {
            stepLength = -stepLength;
            lumaLocalAvg = 0.5 * (luma1 + lumaCenter);
        } else {
            lumaLocalAvg = 0.5 * (luma2 + lumaCenter);
        }

        vec2 currentUV = texCoord + (isHorizontal ? vec2(0.0, stepLength * 0.5) : vec2(stepLength * 0.5, 0.0));
        vec2 offset = isHorizontal ? vec2(invResolution.x, 0.0) : vec2(0.0, invResolution.y);

        vec2 uv1 = currentUV - offset;
        vec2 uv2 = currentUV + offset;

        float lumaEnd1 = GetLuminance(texture2D(colortex3, uv1).rgb) - lumaLocalAvg;
        float lumaEnd2 = GetLuminance(texture2D(colortex3, uv2).rgb) - lumaLocalAvg;

        bool reached1 = abs(lumaEnd1) >= gradMaxAbs;
        bool reached2 = abs(lumaEnd2) >= gradMaxAbs;

        if (!reached1) uv1 -= offset;
        if (!reached2) uv2 += offset;

        bool reachedBoth = reached1 && reached2;

        // Iterate to find precise edge crossing
        for (int i = 2; i < maxIterations && !reachedBoth; i++) {
            if (!reached1) {
                lumaEnd1 = GetLuminance(texture2D(colortex3, uv1).rgb) - lumaLocalAvg;
                reached1 = abs(lumaEnd1) >= gradMaxAbs;
            }
            if (!reached2) {
                lumaEnd2 = GetLuminance(texture2D(colortex3, uv2).rgb) - lumaLocalAvg;
                reached2 = abs(lumaEnd2) >= gradMaxAbs;
            }
            reachedBoth = reached1 && reached2;

            if (!reached1) uv1 -= offset * quality[i];
            if (!reached2) uv2 += offset * quality[i];
        }

        float dist1 = isHorizontal ? (texCoord.x - uv1.x) : (texCoord.y - uv1.y);
        float dist2 = isHorizontal ? (uv2.x - texCoord.x) : (uv2.y - texCoord.y);

        bool useDist1 = dist1 < dist2;
        float minDist = min(dist1, dist2);
        float totalDist = dist1 + dist2;

        float pixelOffset = -minDist / totalDist + 0.5;

        bool centerLumaSmaller = lumaCenter < lumaLocalAvg;
        bool correctVariation = ((useDist1 ? lumaEnd1 : lumaEnd2) < 0.0) != centerLumaSmaller;

        float finalOffset = correctVariation ? pixelOffset : 0.0;

        float lumaAverage = (1.0 / 12.0) * (2.0 * (lumaDownUp + lumaLeftRight) + lumaLeftCorners + lumaRightCorners);
        float subPixelOffset1 = clamp(abs(lumaAverage - lumaCenter) / lumaRange, 0.0, 1.0);
        float subPixelOffset2 = (-2.0 * subPixelOffset1 + 3.0) * subPixelOffset1 * subPixelOffset1;
        float subPixelOffsetFinal = subPixelOffset2 * subPixelOffset2 * subpixelQuality;

        finalOffset = max(finalOffset, subPixelOffsetFinal);

        vec2 finalUV = texCoord;
        if (isHorizontal) {
            finalUV.y += finalOffset * stepLength;
        } else {
            finalUV.x += finalOffset * stepLength;
        }

        #if defined TAA && defined FXAA_TAA_INTERACTION && TAA_MODE == 1
            // Reduce FXAA when moving camera
            vec3 newColor = texture2D(colortex3, finalUV).rgb;
            float movementFactor = min1(
                20.0 * length(cameraPosition - previousCameraPosition)
                #ifdef TAA_MOVEMENT_IMPROVEMENT_FILTER
                    + 0.25
                #endif
            );

            float z0 = texelFetch(depthtex0, texelCoord, 0).r;
            float z1 = texelFetch(depthtex1, texelCoord, 0).r;
            bool edgeDetected = false;
            for (int i = 0; i < 8; i++) {
                ivec2 neighborCoord = texelCoord + neighbourhoodOffsets[i];
                float z0Neighbor = texelFetch(depthtex0, neighborCoord, 0).r;
                float z1Neighbor = texelFetch(depthtex1, neighborCoord, 0).r;
                if (max(abs(GetLinearDepth(z0Neighbor) - GetLinearDepth(z0)), abs(GetLinearDepth(z1Neighbor) - GetLinearDepth(z1))) > 0.09) {
                    edgeDetected = true;
                    break;
                }
            }
            if (edgeDetected) movementFactor = 0.0;

            if (dot(texelFetch(colortex2, texelCoord, 0).rgb, vec3(1.0)) < 0.01) movementFactor = 0.0;

            color = mix(newColor, color, movementFactor);
        #else
            color = texture2D(colortex3, finalUV).rgb;
        #endif
    }
}
