/*
 * Decompiled with CFR 0.152.
 */
package xaeroplus.feature.highlights;

import it.unimi.dsi.fastutil.longs.Long2LongMap;
import it.unimi.dsi.fastutil.longs.Long2LongMaps;
import it.unimi.dsi.fastutil.objects.ObjectIterator;
import java.io.Closeable;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import net.minecraft.resources.ResourceKey;
import net.minecraft.world.level.Level;
import org.rfresh.sqlite.JDBC;
import org.rfresh.sqlite.SQLiteConnection;
import org.rfresh.sqlite.SQLiteErrorCode;
import xaero.map.WorldMap;
import xaeroplus.XaeroPlus;
import xaeroplus.feature.db.DatabaseMigrator;
import xaeroplus.feature.highlights.db.V0ToV1Migration;
import xaeroplus.util.ChunkUtils;
import xaeroplus.util.Wait;

public class ChunkHighlightDatabase
implements Closeable {
    public static final int MAX_HIGHLIGHTS_LIST = 25000;
    private Connection connection;
    protected final String databaseName;
    protected final Path dbPath;
    private static final DatabaseMigrator MIGRATOR = new DatabaseMigrator(List.of(new V0ToV1Migration()));
    boolean recoveryAttempted = false;
    private static final int MAX_RETRIES = 3;

    public ChunkHighlightDatabase(String worldId, String databaseName) {
        this.databaseName = databaseName;
        try {
            Class<JDBC> jdbcClass = JDBC.class;
            this.dbPath = WorldMap.saveFolder.toPath().resolve(worldId).resolve(databaseName + ".db");
            boolean init = !this.dbPath.toFile().exists();
            this.connection = DriverManager.getConnection("jdbc:rfresh_sqlite:" + String.valueOf(this.dbPath));
            ((SQLiteConnection)this.connection).setBusyTimeout(5000);
            MIGRATOR.migrate(this.dbPath, databaseName, this.connection, init);
            this.createMetadataTable();
            this.setPragmas();
        }
        catch (Exception e) {
            XaeroPlus.LOGGER.error("Error while creating chunk highlight database: {} for worldId: {}", new Object[]{databaseName, worldId, e});
            throw new RuntimeException(e);
        }
    }

    public void initializeDimension(ResourceKey<Level> dimension) {
        this.createHighlightsTableIfNotExists(dimension);
    }

    private String getTableName(ResourceKey<Level> dimension) {
        return dimension.location().toString();
    }

    private void setPragmas() {
        try (Statement statement = this.connection.createStatement();){
            statement.executeUpdate("pragma journal_mode = WAL;");
            statement.executeUpdate("pragma synchronous = NORMAL;");
        }
        catch (SQLException e) {
            throw new RuntimeException("Failed to set sqlite pragmas", e);
        }
    }

    private void createMetadataTable() {
        int tryCount = 0;
        while (tryCount++ < 3) {
            if (this.createMetadataTable0()) {
                return;
            }
            XaeroPlus.LOGGER.info("Retrying creation of metadata table in {} database (attempt {}/{})", new Object[]{this.databaseName, tryCount, 3});
            Wait.waitMs(50);
        }
        throw new RuntimeException("Failed to create metadata table");
    }

    private boolean createMetadataTable0() {
        try (Statement statement = this.connection.createStatement();){
            statement.executeUpdate("CREATE TABLE IF NOT EXISTS metadata (id INTEGER PRIMARY KEY, version INTEGER)");
            statement.executeUpdate("INSERT OR REPLACE INTO metadata (id, version) VALUES (0, 1)");
        }
        catch (SQLException e) {
            XaeroPlus.LOGGER.error("Error creating metadata table for db: {}", (Object)this.databaseName, (Object)e);
            if (e.getErrorCode() == SQLiteErrorCode.SQLITE_CORRUPT.code) {
                XaeroPlus.LOGGER.error("Corruption detected in {} database", (Object)this.databaseName, (Object)e);
                this.recoverCorruptDatabase();
            }
            return false;
        }
        return true;
    }

    private void recoverCorruptDatabase() {
        if (this.recoveryAttempted) {
            return;
        }
        this.recoveryAttempted = true;
        XaeroPlus.LOGGER.info("Attempting to recover corrupt database: {}", (Object)this.databaseName);
        Path recoveredDbPath = this.dbPath.getParent().resolve("recovered_" + this.databaseName + "-" + System.currentTimeMillis() + ".db");
        try (Statement statement = this.connection.createStatement();){
            statement.executeUpdate("recover to \"" + String.valueOf(recoveredDbPath.toAbsolutePath()) + "\"");
            XaeroPlus.LOGGER.info("Wrote recovered database to: {}", (Object)recoveredDbPath);
        }
        catch (Exception e) {
            XaeroPlus.LOGGER.error("Error recovering corrupt database: {}", (Object)this.databaseName, (Object)e);
            return;
        }
        try {
            this.connection.close();
            XaeroPlus.LOGGER.info("Closed DB connection to corrupt database: {}", (Object)this.databaseName);
        }
        catch (Exception e) {
            XaeroPlus.LOGGER.error("Error closing connection to corrupt database: {}", (Object)this.databaseName, (Object)e);
            throw new RuntimeException(e);
        }
        Path corruptedBackDbPath = this.dbPath.getParent().resolve("corrupted_" + this.databaseName + "-" + System.currentTimeMillis() + ".db");
        try {
            Files.move(this.dbPath, corruptedBackDbPath, new CopyOption[0]);
            Files.move(recoveredDbPath, this.dbPath, new CopyOption[0]);
            XaeroPlus.LOGGER.info("Replaced corrupt database with recovered: {}", (Object)this.databaseName);
            this.connection = DriverManager.getConnection("jdbc:rfresh_sqlite:" + String.valueOf(this.dbPath));
            XaeroPlus.LOGGER.info("Opened DB connection to recovered database: {}", (Object)this.databaseName);
        }
        catch (Exception e) {
            XaeroPlus.LOGGER.error("Error reopening connection to recovered database: {}", (Object)this.databaseName, (Object)e);
            throw new RuntimeException(e);
        }
        try {
            Files.delete(corruptedBackDbPath);
            XaeroPlus.LOGGER.info("Deleted corrupted database backup: {}", (Object)corruptedBackDbPath);
        }
        catch (Exception e) {
            XaeroPlus.LOGGER.error("Error deleting corrupted backup database: {}", (Object)this.databaseName, (Object)e);
        }
        XaeroPlus.LOGGER.info("Completed recovering corrupt database: {}", (Object)this.databaseName);
    }

    private void createHighlightsTableIfNotExists(ResourceKey<Level> dimension) {
        int tryCount = 0;
        while (tryCount++ < 3) {
            if (this.createHighlightsTableIfNotExists0(dimension)) {
                return;
            }
            XaeroPlus.LOGGER.info("Retrying creation of highlights table in {} database in dimension: {} (attempt {}/{})", new Object[]{this.databaseName, dimension.location(), tryCount, 3});
            Wait.waitMs(50);
        }
    }

    private boolean createHighlightsTableIfNotExists0(ResourceKey<Level> dimension) {
        try (Statement statement = this.connection.createStatement();){
            statement.executeUpdate("CREATE TABLE IF NOT EXISTS \"" + this.getTableName(dimension) + "\" (x INTEGER, z INTEGER, foundTime INTEGER)");
            statement.executeUpdate("CREATE UNIQUE INDEX IF NOT EXISTS \"unique_xz_" + this.getTableName(dimension) + "\" ON \"" + this.getTableName(dimension) + "\" (x, z)");
        }
        catch (SQLException e) {
            XaeroPlus.LOGGER.error("Error creating highlights table for db: {} in dimension: {}", new Object[]{this.databaseName, dimension.location(), e});
            if (e.getErrorCode() == SQLiteErrorCode.SQLITE_CORRUPT.code) {
                XaeroPlus.LOGGER.error("Corruption detected in {} database", (Object)this.databaseName, (Object)e);
                this.recoverCorruptDatabase();
            }
            return false;
        }
        return true;
    }

    public void insertHighlightList(Long2LongMap chunks, ResourceKey<Level> dimension) {
        int tryCount = 0;
        while (tryCount++ < 3) {
            if (this.insertHighlightList0(chunks, dimension)) {
                return;
            }
            XaeroPlus.LOGGER.info("Retrying insert of {} chunks into {} database in dimension: {} (attempt {}/{})", new Object[]{chunks.size(), this.databaseName, dimension.location(), tryCount, 3});
            Wait.waitMs(50);
        }
    }

    private boolean insertHighlightList0(Long2LongMap chunks, ResourceKey<Level> dimension) {
        if (chunks.isEmpty()) {
            return true;
        }
        try {
            int batchSize = 25000;
            ObjectIterator it = Long2LongMaps.fastIterator((Long2LongMap)chunks);
            StringBuilder sb = new StringBuilder(50 * Math.min(batchSize, chunks.size()) + 75);
            while (it.hasNext()) {
                sb.setLength(0);
                sb.append("INSERT OR IGNORE INTO \"").append(this.getTableName(dimension)).append("\" VALUES ");
                boolean trailingComma = false;
                for (int i = 0; i < batchSize && it.hasNext(); ++i) {
                    Long2LongMap.Entry entry = (Long2LongMap.Entry)it.next();
                    long chunk = entry.getLongKey();
                    int chunkX = ChunkUtils.longToChunkX(chunk);
                    int chunkZ = ChunkUtils.longToChunkZ(chunk);
                    long foundTime = entry.getLongValue();
                    sb.append("(").append(chunkX).append(", ").append(chunkZ).append(", ").append(foundTime).append(")");
                    sb.append(", ");
                    trailingComma = true;
                }
                if (trailingComma) {
                    sb.replace(sb.length() - 2, sb.length(), "");
                }
                Statement stmt = this.connection.createStatement();
                try {
                    stmt.executeUpdate(sb.toString());
                }
                finally {
                    if (stmt == null) continue;
                    stmt.close();
                }
            }
        }
        catch (SQLException e) {
            XaeroPlus.LOGGER.error("Error inserting {} chunks into {} database in dimension: {}", new Object[]{chunks.size(), this.databaseName, dimension.location(), e});
            if (e.getErrorCode() == SQLiteErrorCode.SQLITE_CORRUPT.code) {
                XaeroPlus.LOGGER.error("Corruption detected in {} database", (Object)this.databaseName, (Object)e);
                this.recoverCorruptDatabase();
            }
            return false;
        }
        return true;
    }

    public void getHighlightsInWindow(ResourceKey<Level> dimension, int regionXMin, int regionXMax, int regionZMin, int regionZMax, HighlightConsumer consumer) {
        int tryCount = 0;
        while (tryCount++ < 3) {
            if (this.getHighlightsInWindow0(dimension, regionXMin, regionXMax, regionZMin, regionZMax, consumer)) {
                return;
            }
            XaeroPlus.LOGGER.info("Retrying get highlights from {} database in dimension: {}, (attempt {}/{})", new Object[]{this.databaseName, dimension.location(), tryCount, 3});
            Wait.waitMs(50);
        }
    }

    private boolean getHighlightsInWindow0(ResourceKey<Level> dimension, int regionXMin, int regionXMax, int regionZMin, int regionZMax, HighlightConsumer consumer) {
        try (Statement statement = this.connection.createStatement();
             ResultSet resultSet = statement.executeQuery("SELECT * FROM \"" + this.getTableName(dimension) + "\" WHERE x >= " + ChunkUtils.regionCoordToChunkCoord(regionXMin) + " AND x <= " + ChunkUtils.regionCoordToChunkCoord(regionXMax) + " AND z >= " + ChunkUtils.regionCoordToChunkCoord(regionZMin) + " AND z <= " + ChunkUtils.regionCoordToChunkCoord(regionZMax));){
            while (resultSet.next()) {
                consumer.accept(resultSet.getInt("x"), resultSet.getInt("z"), resultSet.getLong("foundTime"));
            }
        }
        catch (SQLException e) {
            XaeroPlus.LOGGER.error("Error getting chunks from {} database in dimension: {}, window: {}-{}, {}-{}", new Object[]{this.databaseName, dimension.location(), regionXMin, regionXMax, regionZMin, regionZMax, e});
            if (e.getErrorCode() == SQLiteErrorCode.SQLITE_CORRUPT.code) {
                XaeroPlus.LOGGER.error("Corruption detected in {} database", (Object)this.databaseName, (Object)e);
                this.recoverCorruptDatabase();
            }
            return false;
        }
        return true;
    }

    public void getHighlightsInWindowAndOutsidePrevWindow(ResourceKey<Level> dimension, int regionXMin, int regionXMax, int regionZMin, int regionZMax, int prevRegionXMin, int prevRegionXMax, int prevRegionZMin, int prevRegionZMax, HighlightConsumer consumer) {
        int tryCount = 0;
        while (tryCount++ < 3) {
            if (this.getHighlightsInWindowAndOutsidePrevWindow0(dimension, regionXMin, regionXMax, regionZMin, regionZMax, prevRegionXMin, prevRegionXMax, prevRegionZMin, prevRegionZMax, consumer)) {
                return;
            }
            XaeroPlus.LOGGER.info("Retrying get of highlights from {} database in dimension: {}, (attempt {}/{})", new Object[]{this.databaseName, dimension.location(), tryCount, 3});
            Wait.waitMs(50);
        }
    }

    private boolean getHighlightsInWindowAndOutsidePrevWindow0(ResourceKey<Level> dimension, int regionXMin, int regionXMax, int regionZMin, int regionZMax, int prevRegionXMin, int prevRegionXMax, int prevRegionZMin, int prevRegionZMax, HighlightConsumer consumer) {
        int xMin = ChunkUtils.regionCoordToChunkCoord(regionXMin);
        int xMax = ChunkUtils.regionCoordToChunkCoord(regionXMax);
        int zMin = ChunkUtils.regionCoordToChunkCoord(regionZMin);
        int zMax = ChunkUtils.regionCoordToChunkCoord(regionZMax);
        int prevXMin = ChunkUtils.regionCoordToChunkCoord(prevRegionXMin);
        int prevXMax = ChunkUtils.regionCoordToChunkCoord(prevRegionXMax);
        int prevZMin = ChunkUtils.regionCoordToChunkCoord(prevRegionZMin);
        int prevZMax = ChunkUtils.regionCoordToChunkCoord(prevRegionZMax);
        try (Statement statement = this.connection.createStatement();
             ResultSet resultSet = statement.executeQuery("SELECT * FROM \"" + this.getTableName(dimension) + "\" WHERE x BETWEEN " + xMin + " AND " + xMax + " AND z BETWEEN " + zMin + " AND " + zMax + " AND NOT (x BETWEEN " + prevXMin + " AND " + prevXMax + " AND z BETWEEN " + prevZMin + " AND " + prevZMax + ")");){
            while (resultSet.next()) {
                consumer.accept(resultSet.getInt("x"), resultSet.getInt("z"), resultSet.getLong("foundTime"));
            }
        }
        catch (SQLException e) {
            XaeroPlus.LOGGER.error("Error getting chunks from {} database in dimension: {}, window: {}-{}, {}-{}", new Object[]{this.databaseName, dimension.location(), regionXMin, regionXMax, regionZMin, regionZMax, e});
            if (e.getErrorCode() == SQLiteErrorCode.SQLITE_CORRUPT.code) {
                XaeroPlus.LOGGER.error("Corruption detected in {} database", (Object)this.databaseName, (Object)e);
                this.recoverCorruptDatabase();
            }
            return false;
        }
        return true;
    }

    public void removeHighlight(int x, int z, ResourceKey<Level> dimension) {
        int tryCount = 0;
        while (tryCount++ < 3) {
            if (this.removeHighlight0(x, z, dimension)) {
                return;
            }
            XaeroPlus.LOGGER.info("Retrying removal of highlight from {} database in dimension: {} (attempt {}/{})", new Object[]{this.databaseName, dimension.location(), tryCount, 3});
            Wait.waitMs(50);
        }
    }

    private boolean removeHighlight0(int x, int z, ResourceKey<Level> dimension) {
        try (Statement statement = this.connection.createStatement();){
            statement.executeUpdate("DELETE FROM \"" + this.getTableName(dimension) + "\" WHERE x = " + x + " AND z = " + z);
        }
        catch (SQLException e) {
            XaeroPlus.LOGGER.error("Error while removing highlight from {} database in dimension: {}, at {}, {}", new Object[]{this.databaseName, dimension.location(), x, z, e});
            if (e.getErrorCode() == SQLiteErrorCode.SQLITE_CORRUPT.code) {
                XaeroPlus.LOGGER.error("Corruption detected in {} database", (Object)this.databaseName, (Object)e);
                this.recoverCorruptDatabase();
            }
            return false;
        }
        return true;
    }

    @Override
    public void close() {
        try {
            this.connection.close();
        }
        catch (Exception e) {
            XaeroPlus.LOGGER.warn("Failed closing {} database connection", (Object)this.databaseName, (Object)e);
        }
    }

    @FunctionalInterface
    public static interface HighlightConsumer {
        public void accept(int var1, int var2, long var3);
    }
}

