/*
 * Decompiled with CFR 0.152.
 */
package xaeroplus.feature.drawing.db;

import it.unimi.dsi.fastutil.longs.Long2LongMap;
import it.unimi.dsi.fastutil.longs.Long2LongMaps;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectMaps;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntMaps;
import it.unimi.dsi.fastutil.objects.ObjectIterator;
import java.io.Closeable;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import java.util.function.Consumer;
import net.minecraft.resources.ResourceKey;
import net.minecraft.world.level.Level;
import org.rfresh.sqlite.JDBC;
import org.rfresh.sqlite.NativeLibraryNotFoundException;
import org.rfresh.sqlite.SQLiteErrorCode;
import xaero.map.WorldMap;
import xaeroplus.XaeroPlus;
import xaeroplus.feature.db.DatabaseMigrator;
import xaeroplus.feature.drawing.db.V0Migration;
import xaeroplus.feature.render.line.Line;
import xaeroplus.feature.render.text.Text;
import xaeroplus.module.impl.TickTaskExecutor;
import xaeroplus.util.ChunkUtils;
import xaeroplus.util.NotificationUtil;

public class DrawingDatabase
implements Closeable {
    public static final int MAX_HIGHLIGHTS_LIST = 25000;
    public static final String HIGHLIGHTS_TABLE = "highlights";
    public static final String LINES_TABLE = "lines";
    public static final String TEXTS_TABLE = "texts";
    private Connection connection;
    public final String databaseName;
    protected final Path dbPath;
    private static final DatabaseMigrator MIGRATOR = new DatabaseMigrator(List.of(new V0Migration()));
    boolean recoveryAttempted = false;
    static boolean nativeLibraryErrorSent = false;

    public DrawingDatabase(String worldId, String databaseName) {
        this.databaseName = databaseName;
        try {
            Class<JDBC> jdbcClass = JDBC.class;
            this.dbPath = WorldMap.saveFolder.toPath().resolve(worldId).resolve(databaseName + ".db");
            boolean init = !this.dbPath.toFile().exists();
            this.connection = DriverManager.getConnection("jdbc:rfresh_sqlite:" + String.valueOf(this.dbPath));
            MIGRATOR.migrate(this.dbPath, databaseName, this.connection, init);
        }
        catch (Exception e) {
            Throwable throwable;
            if (!nativeLibraryErrorSent && (throwable = e.getCause()) instanceof NativeLibraryNotFoundException) {
                NativeLibraryNotFoundException nativeException = (NativeLibraryNotFoundException)throwable;
                nativeLibraryErrorSent = true;
                TickTaskExecutor.INSTANCE.execute(() -> NotificationUtil.errorNotification("Error initializing Drawing database, Drawing features will not work.\n" + nativeException.getMessage()));
            }
            XaeroPlus.LOGGER.error("Error while creating drawing database: {} for worldId: {}", new Object[]{databaseName, worldId, e});
            throw new RuntimeException(e);
        }
    }

    private void createHighlightsTable(String databaseName, Connection connection, ResourceKey<Level> dimension) {
        try (Statement statement = connection.createStatement();){
            statement.executeUpdate("CREATE TABLE IF NOT EXISTS \"" + this.getTableName(dimension, HIGHLIGHTS_TABLE) + "\" (x INTEGER, z INTEGER, color INTEGER, PRIMARY KEY (x, z))");
        }
        catch (SQLException e) {
            XaeroPlus.LOGGER.error("Error creating highlights table for db: {}", (Object)databaseName, (Object)e);
            throw new RuntimeException(e);
        }
    }

    private void createLinesTable(String databaseName, Connection connection, ResourceKey<Level> dimension) {
        try (Statement statement = connection.createStatement();){
            statement.executeUpdate("CREATE TABLE IF NOT EXISTS \"" + this.getTableName(dimension, LINES_TABLE) + "\" (x1 INTEGER, z1 INTEGER, x2 INTEGER, z2 INTEGER, color INTEGER, PRIMARY KEY (x1, z1, x2, z2))");
        }
        catch (SQLException e) {
            XaeroPlus.LOGGER.error("Error creating lines table for db: {}", (Object)databaseName, (Object)e);
            throw new RuntimeException(e);
        }
    }

    private void createTextsTable(String databaseName, Connection connection, ResourceKey<Level> dimension) {
        try (Statement statement = connection.createStatement();){
            statement.executeUpdate("CREATE TABLE IF NOT EXISTS \"" + this.getTableName(dimension, TEXTS_TABLE) + "\" (value TEXT, x INTEGER, z INTEGER, color INTEGER, scale REAL, PRIMARY KEY (x, z))");
        }
        catch (SQLException e) {
            XaeroPlus.LOGGER.error("Error creating texts table for db: {}", (Object)databaseName, (Object)e);
            throw new RuntimeException(e);
        }
    }

    private String getTableName(ResourceKey<Level> dimension, String type) {
        return dimension.location().toString() + "-" + type;
    }

    private void recoverCorruptDatabase() {
        if (this.recoveryAttempted) {
            return;
        }
        this.recoveryAttempted = true;
        XaeroPlus.LOGGER.info("Attempting to recover corrupt database: {}", (Object)this.databaseName);
        Path recoveredDbPath = this.dbPath.getParent().resolve("recovered_" + this.databaseName + "-" + System.currentTimeMillis() + ".db");
        try (Statement statement = this.connection.createStatement();){
            statement.executeUpdate("recover to \"" + String.valueOf(recoveredDbPath.toAbsolutePath()) + "\"");
            XaeroPlus.LOGGER.info("Wrote recovered database to: {}", (Object)recoveredDbPath);
        }
        catch (Exception e) {
            XaeroPlus.LOGGER.error("Error recovering corrupt database: {}", (Object)this.databaseName, (Object)e);
            return;
        }
        try {
            this.connection.close();
            XaeroPlus.LOGGER.info("Closed DB connection to corrupt database: {}", (Object)this.databaseName);
        }
        catch (Exception e) {
            XaeroPlus.LOGGER.error("Error closing connection to corrupt database: {}", (Object)this.databaseName, (Object)e);
            throw new RuntimeException(e);
        }
        Path corruptedBackDbPath = this.dbPath.getParent().resolve("corrupted_" + this.databaseName + "-" + System.currentTimeMillis() + ".db");
        try {
            Files.move(this.dbPath, corruptedBackDbPath, new CopyOption[0]);
            Files.move(recoveredDbPath, this.dbPath, new CopyOption[0]);
            XaeroPlus.LOGGER.info("Replaced corrupt database with recovered: {}", (Object)this.databaseName);
            this.connection = DriverManager.getConnection("jdbc:rfresh_sqlite:" + String.valueOf(this.dbPath));
            XaeroPlus.LOGGER.info("Opened DB connection to recovered database: {}", (Object)this.databaseName);
        }
        catch (Exception e) {
            XaeroPlus.LOGGER.error("Error reopening connection to recovered database: {}", (Object)this.databaseName, (Object)e);
            throw new RuntimeException(e);
        }
        try {
            Files.delete(corruptedBackDbPath);
            XaeroPlus.LOGGER.info("Deleted corrupted database backup: {}", (Object)corruptedBackDbPath);
        }
        catch (Exception e) {
            XaeroPlus.LOGGER.error("Error deleting corrupted backup database: {}", (Object)this.databaseName, (Object)e);
        }
        XaeroPlus.LOGGER.info("Completed recovering corrupt database: {}", (Object)this.databaseName);
    }

    public void initializeDimension(ResourceKey<Level> dimension) {
        this.createHighlightsTable(this.databaseName, this.connection, dimension);
        this.createLinesTable(this.databaseName, this.connection, dimension);
        this.createTextsTable(this.databaseName, this.connection, dimension);
    }

    public void getLinesInDimension(ResourceKey<Level> dimension, LineConsumer consumer) {
        block15: {
            try (Statement statement = this.connection.createStatement();
                 ResultSet resultSet = statement.executeQuery("SELECT * FROM \"" + this.getTableName(dimension, LINES_TABLE) + "\"");){
                while (resultSet.next()) {
                    consumer.accept(resultSet.getInt("x1"), resultSet.getInt("z1"), resultSet.getInt("x2"), resultSet.getInt("z2"), resultSet.getInt("color"));
                }
            }
            catch (SQLException e) {
                XaeroPlus.LOGGER.error("Error getting lines from {} database in dimension: {}", new Object[]{this.databaseName, dimension.location(), e});
                if (e.getErrorCode() != SQLiteErrorCode.SQLITE_CORRUPT.code) break block15;
                XaeroPlus.LOGGER.error("Corruption detected in {} database", (Object)this.databaseName, (Object)e);
                this.recoverCorruptDatabase();
            }
        }
    }

    public void getTextsInWindow(ResourceKey<Level> dimension, int regionXMin, int regionXMax, int regionZMin, int regionZMax, Consumer<Text> consumer) {
        block15: {
            try (Statement statement = this.connection.createStatement();
                 ResultSet resultSet = statement.executeQuery("SELECT * FROM \"" + this.getTableName(dimension, TEXTS_TABLE) + "\" WHERE x >= " + ChunkUtils.regionCoordToCoord(regionXMin) + " AND x <= " + ChunkUtils.regionCoordToCoord(regionXMax) + " AND z >= " + ChunkUtils.regionCoordToCoord(regionZMin) + " AND z <= " + ChunkUtils.regionCoordToCoord(regionZMax));){
                while (resultSet.next()) {
                    Text text = new Text(resultSet.getString("value"), resultSet.getInt("x"), resultSet.getInt("z"), resultSet.getInt("color"), resultSet.getFloat("scale"));
                    consumer.accept(text);
                }
            }
            catch (SQLException e) {
                XaeroPlus.LOGGER.error("Error getting texts from {} database in dimension: {}", new Object[]{this.databaseName, dimension.location(), e});
                if (e.getErrorCode() != SQLiteErrorCode.SQLITE_CORRUPT.code) break block15;
                XaeroPlus.LOGGER.error("Corruption detected in {} database", (Object)this.databaseName, (Object)e);
                this.recoverCorruptDatabase();
            }
        }
    }

    public void getHighlightsInWindow(ResourceKey<Level> dimension, int regionXMin, int regionXMax, int regionZMin, int regionZMax, HighlightConsumer consumer) {
        block15: {
            try (Statement statement = this.connection.createStatement();
                 ResultSet resultSet = statement.executeQuery("SELECT * FROM \"" + this.getTableName(dimension, HIGHLIGHTS_TABLE) + "\" WHERE x >= " + ChunkUtils.regionCoordToChunkCoord(regionXMin) + " AND x <= " + ChunkUtils.regionCoordToChunkCoord(regionXMax) + " AND z >= " + ChunkUtils.regionCoordToChunkCoord(regionZMin) + " AND z <= " + ChunkUtils.regionCoordToChunkCoord(regionZMax));){
                while (resultSet.next()) {
                    consumer.accept(resultSet.getInt("x"), resultSet.getInt("z"), resultSet.getInt("color"));
                }
            }
            catch (SQLException e) {
                XaeroPlus.LOGGER.error("Error getting chunks from {} database in dimension: {}, window: {}-{}, {}-{}", new Object[]{this.databaseName, dimension.location(), regionXMin, regionXMax, regionZMin, regionZMax, e});
                if (e.getErrorCode() != SQLiteErrorCode.SQLITE_CORRUPT.code) break block15;
                XaeroPlus.LOGGER.error("Corruption detected in {} database", (Object)this.databaseName, (Object)e);
                this.recoverCorruptDatabase();
            }
        }
    }

    public void insertLinesList(Object2IntMap<Line> lines, ResourceKey<Level> dimension) {
        block11: {
            if (lines.isEmpty()) {
                return;
            }
            try {
                this.createLinesTable(this.databaseName, this.connection, dimension);
                int batchSize = 25000;
                StringBuilder sb = new StringBuilder(50 * Math.min(batchSize, lines.size()) + 75);
                ObjectIterator it = Object2IntMaps.fastIterator(lines);
                while (it.hasNext()) {
                    sb.setLength(0);
                    sb.append("INSERT OR REPLACE INTO \"").append(this.getTableName(dimension, LINES_TABLE)).append("\" VALUES ");
                    boolean trailingComma = false;
                    for (int i = 0; i < batchSize && it.hasNext(); ++i) {
                        Object2IntMap.Entry entry = (Object2IntMap.Entry)it.next();
                        Line line = (Line)entry.getKey();
                        sb.append("(").append(line.x1()).append(", ").append(line.z1()).append(", ").append(line.x2()).append(", ").append(line.z2()).append(", ").append(entry.getIntValue()).append(")");
                        sb.append(", ");
                        trailingComma = true;
                    }
                    if (trailingComma) {
                        sb.replace(sb.length() - 2, sb.length(), "");
                    }
                    Statement stmt = this.connection.createStatement();
                    try {
                        stmt.executeUpdate(sb.toString());
                    }
                    finally {
                        if (stmt == null) continue;
                        stmt.close();
                    }
                }
            }
            catch (SQLException e) {
                XaeroPlus.LOGGER.error("Error inserting {} lines into {} database in dimension: {}", new Object[]{lines.size(), this.databaseName, dimension.location(), e});
                if (e.getErrorCode() != SQLiteErrorCode.SQLITE_CORRUPT.code) break block11;
                XaeroPlus.LOGGER.error("Corruption detected in {} database", (Object)this.databaseName, (Object)e);
                this.recoverCorruptDatabase();
            }
        }
    }

    public void insertHighlightList(Long2LongMap chunks, ResourceKey<Level> dimension) {
        block11: {
            if (chunks.isEmpty()) {
                return;
            }
            try {
                int batchSize = 25000;
                ObjectIterator it = Long2LongMaps.fastIterator((Long2LongMap)chunks);
                StringBuilder sb = new StringBuilder(50 * Math.min(batchSize, chunks.size()) + 75);
                while (it.hasNext()) {
                    sb.setLength(0);
                    sb.append("INSERT OR REPLACE INTO \"").append(this.getTableName(dimension, HIGHLIGHTS_TABLE)).append("\" VALUES ");
                    boolean trailingComma = false;
                    for (int i = 0; i < batchSize && it.hasNext(); ++i) {
                        Long2LongMap.Entry entry = (Long2LongMap.Entry)it.next();
                        long chunk = entry.getLongKey();
                        int chunkX = ChunkUtils.longToChunkX(chunk);
                        int chunkZ = ChunkUtils.longToChunkZ(chunk);
                        long color = entry.getLongValue();
                        sb.append("(").append(chunkX).append(", ").append(chunkZ).append(", ").append(color).append(")");
                        sb.append(", ");
                        trailingComma = true;
                    }
                    if (trailingComma) {
                        sb.replace(sb.length() - 2, sb.length(), "");
                    }
                    Statement stmt = this.connection.createStatement();
                    try {
                        stmt.executeUpdate(sb.toString());
                    }
                    finally {
                        if (stmt == null) continue;
                        stmt.close();
                    }
                }
            }
            catch (SQLException e) {
                XaeroPlus.LOGGER.error("Error inserting {} chunks into {} database in dimension: {}", new Object[]{chunks.size(), this.databaseName, dimension.location(), e});
                if (e.getErrorCode() != SQLiteErrorCode.SQLITE_CORRUPT.code) break block11;
                XaeroPlus.LOGGER.error("Corruption detected in {} database", (Object)this.databaseName, (Object)e);
                this.recoverCorruptDatabase();
            }
        }
    }

    public void insertTextsList(Long2ObjectMap<Text> texts, ResourceKey<Level> dimension) {
        block11: {
            if (texts.isEmpty()) {
                return;
            }
            try {
                int batchSize = 25000;
                ObjectIterator it = Long2ObjectMaps.fastIterator(texts);
                StringBuilder sb = new StringBuilder(50 * Math.min(batchSize, texts.size()) + 75);
                while (it.hasNext()) {
                    sb.setLength(0);
                    sb.append("INSERT OR REPLACE INTO \"").append(this.getTableName(dimension, TEXTS_TABLE)).append("\" VALUES ");
                    boolean trailingComma = false;
                    for (int i = 0; i < batchSize && it.hasNext(); ++i) {
                        Text entry = (Text)((Long2ObjectMap.Entry)it.next()).getValue();
                        sb.append("(").append("'").append(entry.value()).append("', ").append(entry.x()).append(", ").append(entry.z()).append(", ").append(entry.color()).append(", ").append(entry.scale()).append(")");
                        sb.append(", ");
                        trailingComma = true;
                    }
                    if (trailingComma) {
                        sb.replace(sb.length() - 2, sb.length(), "");
                    }
                    Statement stmt = this.connection.createStatement();
                    try {
                        stmt.executeUpdate(sb.toString());
                    }
                    finally {
                        if (stmt == null) continue;
                        stmt.close();
                    }
                }
            }
            catch (SQLException e) {
                XaeroPlus.LOGGER.error("Error inserting {} texts into {} database in dimension: {}", new Object[]{texts.size(), this.databaseName, dimension.location(), e});
                if (e.getErrorCode() != SQLiteErrorCode.SQLITE_CORRUPT.code) break block11;
                XaeroPlus.LOGGER.error("Corruption detected in {} database", (Object)this.databaseName, (Object)e);
                this.recoverCorruptDatabase();
            }
        }
    }

    public void removeLine(int x1, int z1, int x2, int z2, ResourceKey<Level> dimension) {
        block8: {
            try (Statement statement = this.connection.createStatement();){
                statement.executeUpdate("DELETE FROM \"" + this.getTableName(dimension, LINES_TABLE) + "\" WHERE x1 = " + x1 + " AND z1 = " + z1 + " AND x2 = " + x2 + " AND z2 = " + z2);
            }
            catch (SQLException e) {
                XaeroPlus.LOGGER.error("Error while removing line from {} database in dimension: {}, from ({}, {}) to ({}, {})", new Object[]{this.databaseName, dimension.location(), x1, z1, x2, z2, e});
                if (e.getErrorCode() != SQLiteErrorCode.SQLITE_CORRUPT.code) break block8;
                XaeroPlus.LOGGER.error("Corruption detected in {} database", (Object)this.databaseName, (Object)e);
                this.recoverCorruptDatabase();
            }
        }
    }

    public void removeHighlight(int x, int z, ResourceKey<Level> dimension) {
        block8: {
            try (Statement statement = this.connection.createStatement();){
                statement.executeUpdate("DELETE FROM \"" + this.getTableName(dimension, HIGHLIGHTS_TABLE) + "\" WHERE x = " + x + " AND z = " + z);
            }
            catch (SQLException e) {
                XaeroPlus.LOGGER.error("Error while removing highlight from {} database in dimension: {}, at {}, {}", new Object[]{this.databaseName, dimension.location(), x, z, e});
                if (e.getErrorCode() != SQLiteErrorCode.SQLITE_CORRUPT.code) break block8;
                XaeroPlus.LOGGER.error("Corruption detected in {} database", (Object)this.databaseName, (Object)e);
                this.recoverCorruptDatabase();
            }
        }
    }

    public void removeText(int x, int z, ResourceKey<Level> dimension) {
        block8: {
            try (Statement statement = this.connection.createStatement();){
                statement.executeUpdate("DELETE FROM \"" + this.getTableName(dimension, TEXTS_TABLE) + "\" WHERE x = " + x + " AND z = " + z);
            }
            catch (SQLException e) {
                XaeroPlus.LOGGER.error("Error while removing text from {} database in dimension: {}, at {}, {}", new Object[]{this.databaseName, dimension.location(), x, z, e});
                if (e.getErrorCode() != SQLiteErrorCode.SQLITE_CORRUPT.code) break block8;
                XaeroPlus.LOGGER.error("Corruption detected in {} database", (Object)this.databaseName, (Object)e);
                this.recoverCorruptDatabase();
            }
        }
    }

    @Override
    public void close() {
        try {
            this.connection.close();
        }
        catch (Exception e) {
            XaeroPlus.LOGGER.warn("Failed closing {} database connection", (Object)this.databaseName, (Object)e);
        }
    }

    @FunctionalInterface
    public static interface LineConsumer {
        public void accept(int var1, int var2, int var3, int var4, int var5);
    }

    @FunctionalInterface
    public static interface HighlightConsumer {
        public void accept(int var1, int var2, int var3);
    }
}

