/*
 * Decompiled with CFR 0.152.
 */
package de.markusbordihn.easynpc.client.texture;

import com.mojang.blaze3d.platform.NativeImage;
import de.markusbordihn.easynpc.client.texture.TextureModelKey;
import de.markusbordihn.easynpc.data.skin.SkinModel;
import de.markusbordihn.easynpc.validator.ImageValidator;
import de.markusbordihn.easynpc.validator.UrlValidator;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;
import javax.imageio.ImageIO;
import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.texture.DynamicTexture;
import net.minecraft.resources.ResourceLocation;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class TextureManager {
    protected static final Logger log = LogManager.getLogger((String)"Easy NPC: Core");
    private static final String TEXTURE_PREFIX = "easy_npc_client_texture_";
    private static final String LOG_PREFIX = "[Texture Manager]";
    private static final Map<TextureModelKey, String> errorMessageMap = new HashMap<TextureModelKey, String>();
    private static String lastErrorMessage;

    private TextureManager() {
    }

    public static ResourceLocation addCustomTexture(TextureModelKey textureModelKey, File file) {
        BufferedImage image;
        if (textureModelKey == null) {
            log.error("{} Texture model key for {} is invalid!", (Object)LOG_PREFIX, (Object)file);
            return null;
        }
        if (file == null || !file.exists() || !file.canRead() || file.isDirectory()) {
            log.error("{} Texture file {} is invalid!", (Object)LOG_PREFIX, (Object)file);
            return null;
        }
        try {
            image = ImageIO.read(file);
        }
        catch (IOException | IllegalArgumentException exception) {
            log.error("{} Unable to load Texture file {} for {} because of:", (Object)LOG_PREFIX, (Object)file, (Object)textureModelKey, (Object)exception);
            return null;
        }
        if (!ImageValidator.isValidImage(image)) {
            log.error("{} Unable to get any valid texture from file {} for {}!", (Object)LOG_PREFIX, (Object)file, (Object)textureModelKey);
            return null;
        }
        return TextureManager.registerTexture(textureModelKey, file);
    }

    private static ResourceLocation registerTexture(TextureModelKey textureModelKey, File file) {
        DynamicTexture dynamicTexture;
        NativeImage nativeImage;
        Minecraft client = Minecraft.getInstance();
        net.minecraft.client.renderer.texture.TextureManager textureManager = client.getTextureManager();
        NativeImage nativeImage2 = nativeImage = textureModelKey.getSkinModel() == SkinModel.HUMANOID || textureModelKey.getSkinModel() == SkinModel.HUMANOID_SLIM ? TextureManager.getNativePlayerImage(file) : TextureManager.getNativeImage(file);
        if (nativeImage == null) {
            log.error("{} Unable to create native image for file {} for {}.", (Object)LOG_PREFIX, (Object)file, (Object)textureModelKey);
            return null;
        }
        try {
            dynamicTexture = new DynamicTexture(nativeImage);
        }
        catch (Exception exception) {
            log.error("{} Unable to create dynamic texture for file {} for {}:", (Object)LOG_PREFIX, (Object)file, (Object)textureModelKey, (Object)exception);
            return null;
        }
        String resourceName = TextureManager.getResourceName(textureModelKey);
        ResourceLocation resourceLocation = textureManager.register(resourceName, dynamicTexture);
        log.info("{} Registered file {} with image {} for texture {} with {}.", (Object)LOG_PREFIX, (Object)file, (Object)nativeImage, (Object)textureModelKey, (Object)resourceLocation);
        return resourceLocation;
    }

    public static ResourceLocation addRemoteTexture(TextureModelKey textureModelKey, String remoteUrl, Path targetDirectory) {
        BufferedImage image;
        if (!UrlValidator.isValidUrl(remoteUrl)) {
            TextureManager.urlLoadErrorMessage(textureModelKey, remoteUrl, "Invalid URL");
            return null;
        }
        ResourceLocation cachedTexture = TextureManager.getCachedTexture(textureModelKey, targetDirectory);
        if (cachedTexture != null) {
            log.info("{} Found downloaded file in cache, will re-used {} for {}", (Object)LOG_PREFIX, (Object)cachedTexture, (Object)remoteUrl);
            return cachedTexture;
        }
        log.warn("{} Starting download of remote texture from {} for {}", (Object)LOG_PREFIX, (Object)remoteUrl, (Object)textureModelKey);
        try {
            URL remoteImageURL = new URL(remoteUrl);
            HttpURLConnection connection = (HttpURLConnection)remoteImageURL.openConnection();
            if (connection.getResponseCode() == 301 || connection.getResponseCode() == 302) {
                String redirectUrl = connection.getHeaderField("Location");
                log.info("{} Following redirect from {} > {}", (Object)LOG_PREFIX, (Object)remoteUrl, (Object)redirectUrl);
                remoteUrl = redirectUrl;
            } else if (connection.getResponseCode() != 200) {
                TextureManager.urlLoadErrorMessage(textureModelKey, remoteUrl, connection.getResponseMessage());
                return null;
            }
        }
        catch (IOException | IllegalArgumentException exception) {
            TextureManager.urlLoadErrorMessage(textureModelKey, remoteUrl, exception.getMessage());
            return null;
        }
        try {
            image = ImageIO.read(new URL(remoteUrl));
        }
        catch (IOException | IllegalArgumentException exception) {
            TextureManager.processingErrorMessage(textureModelKey, remoteUrl, exception.getMessage());
            return null;
        }
        if (!ImageValidator.isValidImage(image)) {
            TextureManager.processingErrorMessage(textureModelKey, remoteUrl, "Unable to get any valid texture");
            return null;
        }
        UUID uuid = textureModelKey.getUUID();
        File file = targetDirectory.resolve(TextureManager.getFileName(uuid)).toFile();
        try {
            ImageIO.write((RenderedImage)image, "png", file);
        }
        catch (IOException | IllegalArgumentException exception) {
            TextureManager.processingErrorMessage(textureModelKey, remoteUrl, exception.getMessage());
            return null;
        }
        log.info("{} Downloaded texture from {} and stored it as {} for {}", (Object)LOG_PREFIX, (Object)remoteUrl, (Object)file, (Object)textureModelKey);
        return TextureManager.registerTexture(textureModelKey, file);
    }

    public static String getResourceName(TextureModelKey textureModelKey) {
        return TextureManager.getResourceName(textureModelKey.getUUID().toString(), textureModelKey.getSubType());
    }

    public static String getResourceName(String name, String type) {
        return (TEXTURE_PREFIX + type + "_" + name.replaceAll("[^a-zA-Z0-9_.-]", "")).toLowerCase(Locale.ROOT);
    }

    public static String getFileName(UUID uuid) {
        return TextureManager.getFileName(uuid.toString());
    }

    public static String getFileName(String name) {
        return name.replaceAll("[^a-zA-Z0-9_.-]", "") + ".png";
    }

    public static ResourceLocation getCachedTexture(TextureModelKey textureModelKey, Path targetDirectory) {
        String fileName = String.format("%s.png", textureModelKey.getUUID());
        File file = targetDirectory.resolve(fileName).toFile();
        if (file.exists()) {
            log.info("{} Found texture file in cache, will re-used file {} for {}", (Object)LOG_PREFIX, (Object)file, (Object)textureModelKey);
            return TextureManager.registerTexture(textureModelKey, file);
        }
        return null;
    }

    public static ResourceLocation searchCachedTexture(TextureModelKey textureModelKey, Path targetDirectory) {
        ResourceLocation resourceLocation = TextureManager.getCachedTexture(textureModelKey, targetDirectory);
        if (resourceLocation != null) {
            return resourceLocation;
        }
        UUID textureUUID = textureModelKey.getUUID();
        for (File file : targetDirectory.toFile().listFiles()) {
            String filename;
            UUID uuid;
            if (file == null || !textureUUID.equals(uuid = TextureManager.getUUIDFromFilename(filename = file.getName()))) continue;
            ResourceLocation textureResourceLocation = TextureManager.registerTexture(textureModelKey, file);
            if (textureResourceLocation != null) {
                log.info("{} Registered cached texture file {} for {} with {}", (Object)LOG_PREFIX, (Object)file, (Object)textureModelKey, (Object)textureResourceLocation);
            } else {
                log.error("{} Unable to register cached texture file {} for {}", (Object)LOG_PREFIX, (Object)file, (Object)textureModelKey);
            }
            return textureResourceLocation;
        }
        log.warn("{} Unable to find any cached texture file for {} in {}", (Object)LOG_PREFIX, (Object)textureModelKey, (Object)targetDirectory);
        return null;
    }

    public static NativeImage getNativeImage(File file) {
        return TextureManager.getNativeImage(file, false);
    }

    public static NativeImage getNativePlayerImage(File file) {
        return TextureManager.getNativeImage(file, true);
    }

    public static NativeImage getNativeImage(File file, boolean legacySupport) {
        NativeImage nativeImage;
        try (FileInputStream inputStream = new FileInputStream(file);){
            nativeImage = NativeImage.read((InputStream)inputStream);
        }
        catch (Exception exception) {
            log.error("{} Unable to get native image for file {} because of:", (Object)LOG_PREFIX, (Object)file, (Object)exception);
            return null;
        }
        if (legacySupport && nativeImage.getWidth() == 64 && nativeImage.getHeight() == 32) {
            log.info("{} Processing legacy image {} from 64x32 to 64x64 ...", (Object)LOG_PREFIX, (Object)nativeImage);
            nativeImage = TextureManager.getNativeImageFromLegacyImage(nativeImage);
        }
        return nativeImage;
    }

    public static NativeImage getNativeImageFromLegacyImage(NativeImage legacyNativeImage) {
        NativeImage nativeImage = new NativeImage(64, 64, true);
        nativeImage.copyFrom(legacyNativeImage);
        legacyNativeImage.close();
        nativeImage.fillRect(0, 32, 64, 32, 0);
        nativeImage.copyRect(4, 16, 16, 32, 4, 4, true, false);
        nativeImage.copyRect(8, 16, 16, 32, 4, 4, true, false);
        nativeImage.copyRect(0, 20, 24, 32, 4, 12, true, false);
        nativeImage.copyRect(4, 20, 16, 32, 4, 12, true, false);
        nativeImage.copyRect(8, 20, 8, 32, 4, 12, true, false);
        nativeImage.copyRect(12, 20, 16, 32, 4, 12, true, false);
        nativeImage.copyRect(44, 16, -8, 32, 4, 4, true, false);
        nativeImage.copyRect(48, 16, -8, 32, 4, 4, true, false);
        nativeImage.copyRect(40, 20, 0, 32, 4, 12, true, false);
        nativeImage.copyRect(44, 20, -8, 32, 4, 12, true, false);
        nativeImage.copyRect(48, 20, -16, 32, 4, 12, true, false);
        nativeImage.copyRect(52, 20, -8, 32, 4, 12, true, false);
        return nativeImage;
    }

    public static TextureModelKey getTextureModelKey(SkinModel skinModel, File textureFile) {
        String filename = textureFile.getName();
        UUID uuid = TextureManager.getUUIDFromFilename(filename);
        if (uuid == null) {
            log.error("{} Unable to get UUID for {} and texture file {}!", (Object)LOG_PREFIX, (Object)skinModel, (Object)filename);
            return null;
        }
        return new TextureModelKey(uuid, skinModel, filename);
    }

    public static UUID getUUIDFromFilename(String fileName) {
        if (fileName == null || fileName.isEmpty()) {
            return null;
        }
        if (!fileName.endsWith(".png")) {
            log.error("{} Unable to get UUID from invalid file name {}!", (Object)LOG_PREFIX, (Object)fileName);
            return null;
        }
        try {
            return UUID.fromString(fileName.substring(0, fileName.indexOf(46)));
        }
        catch (IllegalArgumentException e) {
            return UUID.nameUUIDFromBytes(fileName.getBytes());
        }
    }

    private static void processingErrorMessage(TextureModelKey textureModelKey, String remoteUrl, String reason) {
        String errorMessage = String.format("Unable to process texture from %s: %s", remoteUrl, reason);
        log.error("{} {}", (Object)LOG_PREFIX, (Object)errorMessage);
        TextureManager.addErrorMessage(textureModelKey, errorMessage);
    }

    private static void urlLoadErrorMessage(TextureModelKey textureModelKey, String remoteUrl, String reason) {
        String errorMessage = String.format("Unable to load texture from %s: %s", remoteUrl, reason);
        log.error("{} {}", (Object)LOG_PREFIX, (Object)errorMessage);
        TextureManager.addErrorMessage(textureModelKey, errorMessage);
    }

    private static void addErrorMessage(TextureModelKey textureModelKey, String errorMessage) {
        errorMessageMap.put(textureModelKey, errorMessage);
        lastErrorMessage = errorMessage;
    }

    public static boolean hasLastErrorMessage() {
        return lastErrorMessage != null && !lastErrorMessage.isEmpty();
    }

    public static String getLastErrorMessage() {
        return lastErrorMessage;
    }

    public static void clearLastErrorMessage() {
        lastErrorMessage = null;
    }
}

