/*
 * Decompiled with CFR 0.152.
 */
package com.kneaf.core.performance.monitoring;

import com.kneaf.core.data.block.BlockEntityData;
import com.kneaf.core.data.entity.EntityData;
import com.kneaf.core.data.entity.MobData;
import com.kneaf.core.data.entity.PlayerData;
import com.kneaf.core.data.item.ItemEntityData;
import com.kneaf.core.performance.RustPerformance;
import com.kneaf.core.performance.core.ItemProcessResult;
import com.kneaf.core.performance.core.MobProcessResult;
import com.kneaf.core.performance.core.PerformanceOptimizer;
import com.kneaf.core.performance.core.PerformanceProcessor;
import com.kneaf.core.performance.core.RustPerformanceFacade;
import com.kneaf.core.performance.monitoring.PerformanceConfig;
import com.kneaf.core.performance.monitoring.PerformanceManager;
import com.kneaf.core.performance.monitoring.PerformanceMetricsLogger;
import com.kneaf.core.performance.spatial.SpatialGrid;
import com.mojang.logging.LogUtils;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import net.minecraft.network.chat.Component;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.Mob;
import net.minecraft.world.entity.item.ItemEntity;
import net.minecraft.world.entity.monster.Monster;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.ChunkPos;
import net.minecraft.world.phys.AABB;
import org.slf4j.Logger;

public class EntityProcessor {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static final PerformanceConfig CONFIG = PerformanceConfig.load();
    private static final Map<ServerLevel, SpatialGrid> LEVEL_SPATIAL_GRIDS = new HashMap<ServerLevel, SpatialGrid>();
    private static final Object SPATIAL_GRID_LOCK = new Object();
    private static final Map<ServerLevel, Integer> ORIGINAL_VIEW_DISTANCE = new ConcurrentHashMap<ServerLevel, Integer>();
    private static final Map<ServerLevel, Integer> TARGET_DISTANCE = new ConcurrentHashMap<ServerLevel, Integer>();
    private static final Map<ServerLevel, Integer> TRANSITION_REMAINING = new ConcurrentHashMap<ServerLevel, Integer>();
    private static final Map<ServerLevel, Integer> ORIGINAL_SIMULATION_DISTANCE = new ConcurrentHashMap<ServerLevel, Integer>();
    private static final Map<ServerLevel, Integer> TARGET_SIMULATION_DISTANCE = new ConcurrentHashMap<ServerLevel, Integer>();
    private static final Map<ServerLevel, Integer> TRANSITION_REMAINING_SIM = new ConcurrentHashMap<ServerLevel, Integer>();
    private static int TICK_COUNTER = 0;
    private static double currentTpsThreshold;
    private static final Map<Class<?>, Method> VIEW_GET_CACHE;
    private static final Map<Class<?>, Method> VIEW_SET_CACHE;
    private static final Map<Class<?>, Method> SIM_GET_CACHE;
    private static final Map<Class<?>, Method> SIM_SET_CACHE;
    private static final Map<Class<?>, Boolean> METHOD_RESOLVED;
    private static final int DEFAULT_TRANSITION_TICKS = 20;
    private static final int TRANSITION_TICKS = 20;

    public EntityDataCollection collectAndConsolidate(MinecraftServer server, boolean shouldProfile) {
        long entityCollectionStart = shouldProfile ? System.nanoTime() : 0L;
        EntityDataCollection rawData = this.collectEntityData(server);
        if (shouldProfile) {
            long durationMs = (System.nanoTime() - entityCollectionStart) / 1000000L;
            PerformanceMetricsLogger.logLine(String.format("PERF: entity_collection duration=%dms", durationMs));
        }
        long consolidationStart = shouldProfile ? System.nanoTime() : 0L;
        List<ItemEntityData> consolidatedItems = this.consolidateItemEntities(rawData.items());
        if (shouldProfile) {
            long durationMs = (System.nanoTime() - consolidationStart) / 1000000L;
            PerformanceMetricsLogger.logLine(String.format("PERF: item_consolidation duration=%dms", durationMs));
        }
        return new EntityDataCollection(rawData.entities(), consolidatedItems, rawData.mobs(), rawData.blockEntities(), rawData.players());
    }

    private EntityDataCollection collectEntityData(MinecraftServer server) {
        int estimatedEntities = Math.min(CONFIG.getMaxEntitiesToCollect(), 5000);
        ArrayList<EntityData> entities = new ArrayList<EntityData>(estimatedEntities);
        ArrayList<ItemEntityData> items = new ArrayList<ItemEntityData>(estimatedEntities / 4);
        ArrayList<MobData> mobs = new ArrayList<MobData>(estimatedEntities / 8);
        ArrayList<BlockEntityData> blockEntities = new ArrayList<BlockEntityData>(128);
        ArrayList<PlayerData> players = new ArrayList<PlayerData>(32);
        int maxEntities = CONFIG.getMaxEntitiesToCollect();
        double distanceCutoff = CONFIG.getEntityDistanceCutoff();
        String[] excludedTypes = CONFIG.getExcludedEntityTypes();
        double cutoffSq = distanceCutoff * distanceCutoff;
        for (ServerLevel level : server.getAllLevels()) {
            List serverPlayers = level.players();
            ArrayList<PlayerData> levelPlayers = new ArrayList<PlayerData>(serverPlayers.size());
            for (ServerPlayer p : serverPlayers) {
                levelPlayers.add(new PlayerData(p.getId(), p.getX(), p.getY(), p.getZ()));
            }
            this.collectEntitiesFromLevel(level, new EntityCollectionContext(entities, items, mobs, maxEntities, distanceCutoff, levelPlayers, excludedTypes, cutoffSq));
            players.addAll(levelPlayers);
            if (entities.size() < maxEntities) continue;
            break;
        }
        return new EntityDataCollection(entities, items, mobs, blockEntities, players);
    }

    private void collectEntitiesFromLevel(ServerLevel level, EntityCollectionContext context) {
        long spatialStart = System.nanoTime();
        SpatialGrid spatialGrid = this.getOrCreateSpatialGrid(level, context.players());
        long durationMs = (System.nanoTime() - spatialStart) / 1000000L;
        PerformanceMetricsLogger.logLine(String.format("PERF: spatial_grid_init duration=%dms", durationMs));
        AABB searchBounds = this.createSearchBounds(context.players(), context.distanceCutoff());
        if (this.shouldPerformAdaptiveDistanceCalculation()) {
            this.processEntitiesWithFullDistanceCheck(level, searchBounds, context, spatialGrid);
        } else {
            this.processEntitiesWithReducedDistanceCheck(level, searchBounds, context, spatialGrid);
        }
    }

    private boolean shouldPerformAdaptiveDistanceCalculation() {
        return PerformanceManager.getAverageTPS() < CONFIG.getTpsThresholdForAsync() * 0.9;
    }

    private void processEntitiesWithFullDistanceCheck(ServerLevel level, AABB searchBounds, EntityCollectionContext context, SpatialGrid spatialGrid) {
        List entityList = level.getEntities(null, searchBounds);
        int entityCount = entityList.size();
        for (int i = 0; i < entityCount && context.entities().size() < context.maxEntities(); ++i) {
            Entity entity = (Entity)entityList.get(i);
            double minSq = this.computeMinSquaredDistanceToPlayersOptimized(entity, spatialGrid, context.distanceCutoff());
            if (!(minSq <= context.cutoffSq())) continue;
            this.processEntityWithinCutoff(entity, minSq, context.excluded(), context.entities(), context.items(), context.mobs());
        }
    }

    private void processEntitiesWithReducedDistanceCheck(ServerLevel level, AABB searchBounds, EntityCollectionContext context, SpatialGrid spatialGrid) {
        double approximateCutoff = context.distanceCutoff() * 1.2;
        double approximateCutoffSq = approximateCutoff * approximateCutoff;
        ArrayList<Entity> candidateEntities = new ArrayList<Entity>();
        for (Entity entity : level.getEntities(null, searchBounds)) {
            if (context.entities().size() >= context.maxEntities()) break;
            double approxMinSq = spatialGrid.findMinSquaredDistance(entity.getX(), entity.getY(), entity.getZ(), approximateCutoff);
            if (!(approxMinSq <= approximateCutoffSq)) continue;
            candidateEntities.add(entity);
        }
        for (Entity entity : candidateEntities) {
            double minSq;
            if (context.entities().size() >= context.maxEntities()) break;
            String typeStr = entity.getType().toString();
            if (this.isExcludedType(typeStr, context.excluded()) || !((minSq = this.computeMinSquaredDistanceToPlayersOptimized(entity, spatialGrid, context.distanceCutoff())) <= context.cutoffSq())) continue;
            this.processEntityWithinCutoff(entity, minSq, context.excluded(), context.entities(), context.items(), context.mobs());
        }
    }

    private double computeMinSquaredDistanceToPlayersOptimized(Entity entity, SpatialGrid spatialGrid, double maxSearchRadius) {
        return spatialGrid.findMinSquaredDistance(entity.getX(), entity.getY(), entity.getZ(), maxSearchRadius);
    }

    private AABB createSearchBounds(List<PlayerData> players, double distanceCutoff) {
        if (players.isEmpty()) {
            return new AABB(0.0, 0.0, 0.0, 0.0, 0.0, 0.0);
        }
        double minX = Double.MAX_VALUE;
        double minY = Double.MAX_VALUE;
        double minZ = Double.MAX_VALUE;
        double maxX = Double.MIN_VALUE;
        double maxY = Double.MIN_VALUE;
        double maxZ = Double.MIN_VALUE;
        for (PlayerData player : players) {
            minX = Math.min(minX, player.getX());
            minY = Math.min(minY, player.getY());
            minZ = Math.min(minZ, player.getZ());
            maxX = Math.max(maxX, player.getX());
            maxY = Math.max(maxY, player.getY());
            maxZ = Math.max(maxZ, player.getZ());
        }
        return new AABB(minX -= distanceCutoff, minY -= distanceCutoff, minZ -= distanceCutoff, maxX += distanceCutoff, maxY += distanceCutoff, maxZ += distanceCutoff);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private SpatialGrid getOrCreateSpatialGrid(ServerLevel level, List<PlayerData> players) {
        Object object = SPATIAL_GRID_LOCK;
        synchronized (object) {
            SpatialGrid grid = LEVEL_SPATIAL_GRIDS.computeIfAbsent(level, k -> new SpatialGrid(Math.max(CONFIG.getEntityDistanceCutoff() / 4.0, 16.0)));
            grid.clear();
            for (PlayerData player : players) {
                grid.updatePlayer(player);
            }
            return grid;
        }
    }

    private void processEntityWithinCutoff(Entity entity, double minSq, String[] excluded, List<EntityData> entities, List<ItemEntityData> items, List<MobData> mobs) {
        String typeStr = entity.getType().toString();
        if (this.isExcludedType(typeStr, excluded)) {
            return;
        }
        double distance = Math.sqrt(minSq);
        entities.add(new EntityData(entity.getId(), entity.getX(), entity.getY(), entity.getZ(), distance, false, typeStr));
        if (entity instanceof ItemEntity) {
            ItemEntity itemEntity = (ItemEntity)entity;
            this.collectItemEntity(entity, itemEntity, items);
        } else if (entity instanceof Mob) {
            Mob mob = (Mob)entity;
            this.collectMobEntity(entity, mob, mobs, distance, typeStr);
        }
    }

    private boolean isExcludedType(String typeStr, String[] excluded) {
        if (excluded == null || excluded.length == 0) {
            return false;
        }
        for (String ex : excluded) {
            if (ex == null || ex.isEmpty() || !typeStr.contains(ex)) continue;
            return true;
        }
        return false;
    }

    private void collectItemEntity(Entity entity, ItemEntity itemEntity, List<ItemEntityData> items) {
        ChunkPos chunkPos = entity.chunkPosition();
        ItemStack itemStack = itemEntity.getItem();
        String itemType = itemStack.getItem().getDescriptionId();
        int count = itemStack.getCount();
        int ageSeconds = itemEntity.getAge() / 20;
        items.add(new ItemEntityData(entity.getId(), chunkPos.x, chunkPos.z, itemType, count, ageSeconds));
    }

    private void collectMobEntity(Entity entity, Mob mob, List<MobData> mobs, double distance, String typeStr) {
        boolean isPassive = !(mob instanceof Monster);
        mobs.add(new MobData(entity.getId(), distance, isPassive, typeStr));
    }

    public List<ItemEntityData> consolidateItemEntities(List<ItemEntityData> items) {
        if (items == null || items.isEmpty()) {
            return items;
        }
        int estimatedSize = Math.min(items.size(), items.size() / 2 + 1);
        HashMap<Long, ItemEntityData> agg = new HashMap<Long, ItemEntityData>(estimatedSize);
        for (ItemEntityData it : items) {
            long key = this.packItemKey(it.getChunkX(), it.getChunkZ(), it.getItemType());
            ItemEntityData cur = (ItemEntityData)agg.get(key);
            if (cur == null) {
                agg.put(key, it);
                continue;
            }
            int newCount = cur.getCount() + it.getCount();
            int newAge = Math.min(cur.getAgeSeconds(), it.getAgeSeconds());
            long preservedId = cur.getId();
            agg.put(key, new ItemEntityData(preservedId, it.getChunkX(), it.getChunkZ(), it.getItemType(), newCount, newAge));
        }
        return new ArrayList<ItemEntityData>(agg.values());
    }

    private long packItemKey(int chunkX, int chunkZ, String itemType) {
        long packedChunkX = (long)chunkX & 0x1FFFFFL;
        long packedChunkZ = (long)chunkZ & 0x1FFFFFL;
        long itemHash = itemType == null ? 0L : (long)itemType.hashCode() & 0x3FFFFFL;
        return packedChunkX << 43 | packedChunkZ << 22 | itemHash;
    }

    public OptimizationResults processOptimizations(EntityDataCollection data) {
        List<Long> toTick = RustPerformance.getEntitiesToTick(data.entities(), data.players());
        List<Long> blockResult = RustPerformance.getBlockEntitiesToTick(data.blockEntities());
        ItemProcessResult itemResult = RustPerformance.processItemEntities(data.items());
        MobProcessResult mobResult = RustPerformance.processMobAI(data.mobs());
        return new OptimizationResults(toTick, blockResult, itemResult, mobResult);
    }

    public void applyOptimizations(MinecraftServer server, OptimizationResults results) {
        this.applyItemUpdates(server, results.itemResult());
        this.applyMobOptimizations(server, results.mobResult());
        this.enforceServerDistanceBounds(server);
    }

    private void applyItemUpdates(MinecraftServer server, ItemProcessResult itemResult) {
        if (itemResult == null || itemResult.getItemUpdates() == null) {
            return;
        }
        HashMap<Integer, PerformanceProcessor.ItemUpdate> updateMap = new HashMap<Integer, PerformanceProcessor.ItemUpdate>();
        for (PerformanceProcessor.ItemUpdate update : itemResult.getItemUpdates()) {
            updateMap.put((int)update.getId(), update);
        }
        for (ServerLevel level : server.getAllLevels()) {
            for (Map.Entry entry : updateMap.entrySet()) {
                Entity entity = level.getEntity(((Integer)entry.getKey()).intValue());
                if (!(entity instanceof ItemEntity)) continue;
                ItemEntity itemEntity = (ItemEntity)entity;
                itemEntity.getItem().setCount(((PerformanceProcessor.ItemUpdate)entry.getValue()).getNewCount());
            }
        }
    }

    private void applyMobOptimizations(MinecraftServer server, MobProcessResult mobResult) {
        if (mobResult == null) {
            return;
        }
        HashSet<Integer> disableAiIds = new HashSet<Integer>();
        for (Long id : mobResult.getDisableList()) {
            disableAiIds.add(id.intValue());
        }
        HashSet<Integer> simplifyAiIds = new HashSet<Integer>();
        for (Long id : mobResult.getSimplifyList()) {
            simplifyAiIds.add(id.intValue());
        }
        for (ServerLevel level : server.getAllLevels()) {
            Entity entity;
            for (Integer id : disableAiIds) {
                entity = level.getEntity(id.intValue());
                if (!(entity instanceof Mob)) continue;
                Mob mob = (Mob)entity;
                mob.setNoAi(true);
            }
            for (Integer id : simplifyAiIds) {
                entity = level.getEntity(id.intValue());
                if (!(entity instanceof Mob)) continue;
            }
        }
    }

    private void enforceServerDistanceBounds(MinecraftServer server) {
        try {
            PerformanceOptimizer.OptimizationLevel level;
            RustPerformanceFacade facade = RustPerformanceFacade.getInstance();
            PerformanceOptimizer.OptimizationLevel optimizationLevel = level = facade.isNativeAvailable() ? facade.getCurrentOptimizationLevel() : null;
            if (level != null) {
                int target = this.mapOptimizationLevelToDistance(level);
                this.setTargetDistance(server, target);
            }
        }
        catch (Throwable t) {
            LOGGER.debug("Failed to enforce server distance bounds: {}", (Object)t.getMessage());
        }
    }

    private int mapOptimizationLevelToDistance(PerformanceOptimizer.OptimizationLevel level) {
        return switch (level) {
            case PerformanceOptimizer.OptimizationLevel.AGGRESSIVE -> 8;
            case PerformanceOptimizer.OptimizationLevel.HIGH -> 12;
            case PerformanceOptimizer.OptimizationLevel.MEDIUM -> 16;
            case PerformanceOptimizer.OptimizationLevel.NORMAL -> 32;
            default -> 16;
        };
    }

    public void setTargetDistance(MinecraftServer server, int targetChunks) {
        if (server == null) {
            return;
        }
        for (ServerLevel level : server.getAllLevels()) {
            try {
                this.recordOriginalDistances(level);
                TARGET_DISTANCE.put(level, targetChunks);
                TRANSITION_REMAINING.put(level, 20);
                TARGET_SIMULATION_DISTANCE.put(level, targetChunks);
                TRANSITION_REMAINING_SIM.put(level, 20);
            }
            catch (Throwable t) {
                LOGGER.debug("Failed to set target distance for level {}: {}", (Object)level, (Object)t.getMessage());
            }
        }
    }

    private void recordOriginalDistances(ServerLevel level) {
        if (!ORIGINAL_VIEW_DISTANCE.containsKey(level)) {
            try {
                Method m = this.resolveViewGetter(level.getClass());
                if (m != null) {
                    int current = (Integer)m.invoke((Object)level, new Object[0]);
                    ORIGINAL_VIEW_DISTANCE.put(level, current);
                }
            }
            catch (Throwable m) {
                // empty catch block
            }
        }
        if (!ORIGINAL_SIMULATION_DISTANCE.containsKey(level)) {
            try {
                Method sm = this.resolveSimGetter(level.getClass());
                if (sm != null) {
                    int curSim = (Integer)sm.invoke((Object)level, new Object[0]);
                    ORIGINAL_SIMULATION_DISTANCE.put(level, curSim);
                }
            }
            catch (Throwable throwable) {
                // empty catch block
            }
        }
    }

    public void applyDistanceTransitions(MinecraftServer server) {
        if (server == null) {
            return;
        }
        this.applyViewDistanceTransitions(server);
        this.applySimulationDistanceTransitions(server);
        this.restoreOriginalDistancesWhenAppropriate(server);
    }

    private void applyViewDistanceTransitions(MinecraftServer server) {
        for (ServerLevel level : server.getAllLevels()) {
            Integer remaining = TRANSITION_REMAINING.get(level);
            Integer target = TARGET_DISTANCE.get(level);
            if (remaining == null || target == null) continue;
            try {
                Method vg = this.resolveViewGetter(level.getClass());
                if (vg == null) {
                    this.cleanupTransitionState(level, TRANSITION_REMAINING, TARGET_DISTANCE, ORIGINAL_VIEW_DISTANCE);
                    continue;
                }
                int current = (Integer)vg.invoke((Object)level, new Object[0]);
                if (remaining <= 1) {
                    this.applyFinalDistanceChange(level, target, this.resolveViewSetter(level.getClass()));
                    this.cleanupTransitionState(level, TRANSITION_REMAINING, TARGET_DISTANCE, ORIGINAL_VIEW_DISTANCE);
                    continue;
                }
                this.applyGradualDistanceChange(level, current, target, remaining, this.resolveViewSetter(level.getClass()));
                this.updateTransitionState(level, remaining, TRANSITION_REMAINING);
            }
            catch (Throwable t) {
                this.handleTransitionError(level, "view distance", t);
            }
        }
    }

    private void applySimulationDistanceTransitions(MinecraftServer server) {
        for (ServerLevel level : server.getAllLevels()) {
            Integer remaining = TRANSITION_REMAINING_SIM.get(level);
            Integer target = TARGET_SIMULATION_DISTANCE.get(level);
            if (remaining == null || target == null) continue;
            try {
                Method sg = this.resolveSimGetter(level.getClass());
                if (sg == null) {
                    this.cleanupTransitionState(level, TRANSITION_REMAINING_SIM, TARGET_SIMULATION_DISTANCE, ORIGINAL_SIMULATION_DISTANCE);
                    continue;
                }
                int current = (Integer)sg.invoke((Object)level, new Object[0]);
                if (remaining <= 1) {
                    this.applyFinalDistanceChange(level, target, this.resolveSimSetter(level.getClass()));
                    this.cleanupTransitionState(level, TRANSITION_REMAINING_SIM, TARGET_SIMULATION_DISTANCE, ORIGINAL_SIMULATION_DISTANCE);
                    continue;
                }
                this.applyGradualDistanceChange(level, current, target, remaining, this.resolveSimSetter(level.getClass()));
                this.updateTransitionState(level, remaining, TRANSITION_REMAINING_SIM);
            }
            catch (Throwable t) {
                this.handleTransitionError(level, "simulation distance", t);
            }
        }
    }

    private void applyFinalDistanceChange(ServerLevel level, int target, Method setter) {
        if (setter != null) {
            try {
                setter.invoke((Object)level, target);
            }
            catch (Throwable throwable) {
                // empty catch block
            }
        }
    }

    private void applyGradualDistanceChange(ServerLevel level, int current, int target, int remaining, Method setter) {
        int step = (int)Math.ceil((double)(target - current) / (double)remaining);
        int next = current + step;
        if (setter != null) {
            try {
                setter.invoke((Object)level, next);
            }
            catch (Throwable throwable) {
                // empty catch block
            }
        }
    }

    private void updateTransitionState(ServerLevel level, int remaining, Map<ServerLevel, Integer> stateMap) {
        stateMap.put(level, remaining - 1);
    }

    private void cleanupTransitionState(ServerLevel level, Map<ServerLevel, Integer> remainingMap, Map<ServerLevel, Integer> targetMap, Map<ServerLevel, Integer> originalMap) {
        remainingMap.remove(level);
        targetMap.remove(level);
        Integer orig = originalMap.get(level);
        if (orig != null) {
            originalMap.remove(level);
        }
    }

    private void handleTransitionError(ServerLevel level, String distanceType, Throwable t) {
        LOGGER.debug("Error transitioning {} for level {}: {}", new Object[]{distanceType, level, t.getMessage()});
        this.cleanupTransitionState(level, TRANSITION_REMAINING, TARGET_DISTANCE, ORIGINAL_VIEW_DISTANCE);
        this.cleanupTransitionState(level, TRANSITION_REMAINING_SIM, TARGET_SIMULATION_DISTANCE, ORIGINAL_SIMULATION_DISTANCE);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void restoreOriginalDistancesWhenAppropriate(MinecraftServer server) {
        for (ServerLevel level : server.getAllLevels()) {
            if (TARGET_DISTANCE.containsKey(level) || !ORIGINAL_VIEW_DISTANCE.containsKey(level)) continue;
            try {
                this.restoreOriginalDistance(level, this.resolveViewGetter(level.getClass()), this.resolveViewSetter(level.getClass()), ORIGINAL_VIEW_DISTANCE);
            }
            catch (Throwable t) {
                this.disableReflectionForClass(level.getClass(), "restoreView", t);
            }
            finally {
                ORIGINAL_VIEW_DISTANCE.remove(level);
            }
        }
    }

    private void restoreOriginalDistance(ServerLevel level, Method getter, Method setter, Map<ServerLevel, Integer> originalMap) throws Throwable {
        int orig;
        int current;
        if (getter != null && setter != null && (current = ((Number)getter.invoke((Object)level, new Object[0])).intValue()) != (orig = originalMap.get(level).intValue())) {
            setter.invoke((Object)level, orig);
        }
    }

    private Method resolveViewGetter(Class<?> cls) {
        Method cached = VIEW_GET_CACHE.get(cls);
        if (cached != null) {
            return cached;
        }
        if (METHOD_RESOLVED.containsKey(cls) && !VIEW_GET_CACHE.containsKey(cls)) {
            return null;
        }
        try {
            Method method = null;
            try {
                method = cls.getMethod("getViewDistance", new Class[0]);
            }
            catch (NoSuchMethodException e) {
                method = cls.getMethod("viewDistance", new Class[0]);
            }
            VIEW_GET_CACHE.put(cls, method);
            METHOD_RESOLVED.put(cls, true);
            return method;
        }
        catch (Throwable t) {
            METHOD_RESOLVED.put(cls, false);
            return null;
        }
    }

    private Method resolveViewSetter(Class<?> cls) {
        Method cached = VIEW_SET_CACHE.get(cls);
        if (cached != null) {
            return cached;
        }
        try {
            Method method = cls.getMethod("setViewDistance", Integer.TYPE);
            VIEW_SET_CACHE.put(cls, method);
            return method;
        }
        catch (Throwable t) {
            return null;
        }
    }

    private Method resolveSimGetter(Class<?> cls) {
        Method cached = SIM_GET_CACHE.get(cls);
        if (cached != null) {
            return cached;
        }
        try {
            Method method = cls.getMethod("getSimulationDistance", new Class[0]);
            SIM_GET_CACHE.put(cls, method);
            return method;
        }
        catch (Throwable t) {
            return null;
        }
    }

    private Method resolveSimSetter(Class<?> cls) {
        Method cached = SIM_SET_CACHE.get(cls);
        if (cached != null) {
            return cached;
        }
        try {
            Method method = cls.getMethod("setSimulationDistance", Integer.TYPE);
            SIM_SET_CACHE.put(cls, method);
            return method;
        }
        catch (Throwable t) {
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void disableReflectionForClass(Class<?> cls, String why, Throwable t) {
        try {
            VIEW_GET_CACHE.remove(cls);
            VIEW_SET_CACHE.remove(cls);
            SIM_GET_CACHE.remove(cls);
            SIM_SET_CACHE.remove(cls);
            METHOD_RESOLVED.put(cls, false);
        }
        finally {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Disabling reflection distance adjustments for {}: {}", (Object)cls.getName(), (Object)why);
            }
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Reflection disable cause: ", t);
            }
        }
    }

    public void removeItems(MinecraftServer server, ItemProcessResult itemResult) {
        if (itemResult == null || itemResult.getItemsToRemove() == null || itemResult.getItemsToRemove().isEmpty()) {
            return;
        }
        for (ServerLevel level : server.getAllLevels()) {
            for (Long id : itemResult.getItemsToRemove()) {
                try {
                    Entity entity = level.getEntity(id.intValue());
                    if (entity == null) continue;
                    entity.remove(Entity.RemovalReason.DISCARDED);
                }
                catch (Exception e) {
                    LOGGER.debug("Error removing item entity {} on level {}", new Object[]{id, level.dimension(), e});
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void logSpatialGridStats(MinecraftServer server) {
        Object object = SPATIAL_GRID_LOCK;
        synchronized (object) {
            int totalLevels = LEVEL_SPATIAL_GRIDS.size();
            int totalPlayers = 0;
            int totalCells = 0;
            for (SpatialGrid grid : LEVEL_SPATIAL_GRIDS.values()) {
                SpatialGrid.GridStats stats = grid.getStats();
                totalPlayers += stats.totalPlayers();
                totalCells += stats.totalCells();
            }
            String summary = String.format("SpatialGrid Stats: %d levels, %d players, %d cells, avg %.2f players/cell", totalLevels, totalPlayers, totalCells, totalCells > 0 ? (double)totalPlayers / (double)totalCells : 0.0);
            PerformanceMetricsLogger.logLine(summary);
            if (CONFIG.isBroadcastToClient()) {
                this.broadcastToPlayers(server, summary);
            }
        }
    }

    private void broadcastToPlayers(MinecraftServer server, String message) {
        try {
            for (ServerLevel level : server.getAllLevels()) {
                for (ServerPlayer player : level.players()) {
                    player.displayClientMessage((Component)Component.literal((String)message), false);
                }
            }
        }
        catch (Exception e) {
            LOGGER.debug("Failed to broadcast performance message to players", (Throwable)e);
        }
    }

    public void onServerTick(MinecraftServer server) {
        this.updateTPS();
        if (++TICK_COUNTER % 2 == 0) {
            this.adjustDynamicThreshold();
        }
        if (!this.shouldPerformScan()) {
            return;
        }
        EntityDataCollection data = this.collectAndConsolidate(server, false);
        try {
            this.applyDistanceTransitions(server);
        }
        catch (Throwable t) {
            LOGGER.debug("Error applying distance transitions: {}", (Object)t.getMessage());
        }
        double avgTps = this.getRollingAvgTPS();
        if (avgTps >= currentTpsThreshold) {
            this.runSynchronousOptimizations(server, data, false);
        } else {
            this.runSynchronousOptimizations(server, data, false);
        }
    }

    private void updateTPS() {
    }

    private double getRollingAvgTPS() {
        return 20.0;
    }

    private boolean shouldPerformScan() {
        return true;
    }

    private void adjustDynamicThreshold() {
    }

    private void runSynchronousOptimizations(MinecraftServer server, EntityDataCollection data, boolean shouldProfile) {
        try {
            OptimizationResults results = this.processOptimizations(data);
            this.applyOptimizations(server, results);
        }
        catch (Exception ex) {
            LOGGER.warn("Error processing optimizations synchronously", (Throwable)ex);
        }
    }

    static {
        VIEW_GET_CACHE = new ConcurrentHashMap();
        VIEW_SET_CACHE = new ConcurrentHashMap();
        SIM_GET_CACHE = new ConcurrentHashMap();
        SIM_SET_CACHE = new ConcurrentHashMap();
        METHOD_RESOLVED = new ConcurrentHashMap();
    }

    public record EntityDataCollection(List<EntityData> entities, List<ItemEntityData> items, List<MobData> mobs, List<BlockEntityData> blockEntities, List<PlayerData> players) {
    }

    public record EntityCollectionContext(List<EntityData> entities, List<ItemEntityData> items, List<MobData> mobs, int maxEntities, double distanceCutoff, List<PlayerData> players, String[] excluded, double cutoffSq) {
    }

    public record OptimizationResults(List<Long> toTick, List<Long> blockResult, ItemProcessResult itemResult, MobProcessResult mobResult) {
    }
}

