/*
 * Decompiled with CFR 0.152.
 */
package com.kneaf.core.protocol.core;

import com.kneaf.core.protocol.core.ProtocolException;
import com.kneaf.core.protocol.core.ProtocolHandler;
import com.kneaf.core.protocol.core.ProtocolLogger;
import com.kneaf.core.protocol.core.ProtocolUtils;
import com.kneaf.core.protocol.core.ProtocolValidator;
import com.kneaf.core.protocol.utils.ProtocolContext;
import com.kneaf.core.protocol.utils.ProtocolLoggerImpl;
import com.kneaf.core.protocol.utils.ProtocolValidatorImpl;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;

public class EnhancedProtocolProcessor
implements ProtocolHandler<Object, Object> {
    private final ProtocolLogger logger;
    private final String protocolVersion;
    private final long defaultTimeoutMs;
    private final boolean batchingEnabled;
    private final int batchSizeThreshold;
    private final long batchTimeThresholdMs;
    private final List<Object> packetBuffer;
    private final ScheduledExecutorService batchScheduler;
    private volatile ScheduledFuture<?> batchFlushTask;
    private final Map<Object, CompletableFuture<Object>> pendingResults;

    public EnhancedProtocolProcessor(ProtocolLogger logger, String protocolVersion, long defaultTimeoutMs) {
        this(logger, protocolVersion, defaultTimeoutMs, false, 10, 1000L);
    }

    public EnhancedProtocolProcessor(ProtocolLogger logger, String protocolVersion, long defaultTimeoutMs, boolean batchingEnabled, int batchSizeThreshold, long batchTimeThresholdMs) {
        this.logger = logger;
        this.protocolVersion = protocolVersion;
        this.defaultTimeoutMs = defaultTimeoutMs;
        this.batchingEnabled = batchingEnabled;
        this.batchSizeThreshold = batchSizeThreshold;
        this.batchTimeThresholdMs = batchTimeThresholdMs;
        this.packetBuffer = new ArrayList<Object>();
        this.batchScheduler = batchingEnabled ? Executors.newSingleThreadScheduledExecutor() : null;
        this.pendingResults = new ConcurrentHashMap<Object, CompletableFuture<Object>>();
    }

    public static EnhancedProtocolProcessor createDefault() {
        return new EnhancedProtocolProcessor(new ProtocolLoggerImpl("EnhancedProtocolProcessor"), "2.0", 30000L);
    }

    @Override
    public Object processBinary(Object input) throws Exception {
        ProtocolContext context = ProtocolContext.builder().operation("process_binary").protocolFormat("binary").protocolVersion(this.protocolVersion).logger(this.logger).validator(ProtocolValidatorImpl.createBinaryValidator()).addMetadata("input_type", input.getClass().getSimpleName()).build();
        return this.processWithContext(input, context, inputData -> {
            try {
                return this.executeBinaryProcessing(inputData);
            }
            catch (Exception e) {
                throw new RuntimeException("Binary processing failed", e);
            }
        });
    }

    @Override
    public Object processJson(Object input) throws Exception {
        ProtocolContext context = ProtocolContext.builder().operation("process_json").protocolFormat("json").protocolVersion(this.protocolVersion).logger(this.logger).validator(ProtocolValidatorImpl.createJsonValidator()).addMetadata("input_type", input.getClass().getSimpleName()).build();
        return this.processWithContext(input, context, inputData -> {
            try {
                return this.executeJsonProcessing(inputData);
            }
            catch (Exception e) {
                throw new RuntimeException("JSON processing failed", e);
            }
        });
    }

    @Override
    public Object processWithFallback(Object input) throws Exception {
        if (this.batchingEnabled) {
            return this.addToBatch(input).get();
        }
        String traceId = this.logger.generateTraceId();
        String detectedFormat = ProtocolUtils.detectFormat(null, input);
        ProtocolContext context = ProtocolContext.builder().operation("process_with_fallback").protocolFormat(detectedFormat).protocolVersion(this.protocolVersion).logger(this.logger).addMetadata("detected_format", detectedFormat).addMetadata("input_type", input.getClass().getSimpleName()).build();
        this.logger.logOperationStart("process_with_fallback", detectedFormat, traceId, Map.of("input_type", input.getClass().getSimpleName()));
        try {
            Object result = switch (detectedFormat) {
                case "binary" -> this.processBinary(input);
                case "json" -> this.processJson(input);
                default -> {
                    this.logger.logWarning("process_with_fallback", "Unknown format '" + detectedFormat + "', falling back to JSON", traceId, Map.of("detected_format", detectedFormat));
                    yield this.processJson(input);
                }
            };
            this.logger.logOperationComplete("process_with_fallback", detectedFormat, traceId, context.getElapsedTime(), true, Map.of("result_type", result.getClass().getSimpleName()));
            return result;
        }
        catch (Exception e) {
            this.logger.logError("process_with_fallback", e, traceId, Map.of("detected_format", detectedFormat));
            throw e;
        }
    }

    @Override
    public CompletableFuture<Object> processAsync(Object input) {
        if (this.batchingEnabled) {
            return this.addToBatch(input);
        }
        return CompletableFuture.supplyAsync(() -> {
            try {
                return this.processWithFallback(input);
            }
            catch (Exception e) {
                throw new CompletionException(e);
            }
        });
    }

    @Override
    public String getProtocolVersion() {
        return this.protocolVersion;
    }

    @Override
    public boolean supportsFormat(String format) {
        return "binary".equals(format) || "json".equals(format);
    }

    private Object processWithContext(Object input, ProtocolContext context, Function<Object, Object> processor) throws Exception {
        String traceId = context.getTraceId();
        String operation = context.getOperation();
        String format = context.getProtocolFormat();
        this.logger.logOperationStart(operation, format, traceId, context.getMetadata());
        try {
            if (context.getValidator() != null) {
                ProtocolValidator<?> typedValidator = context.getValidator();
                ProtocolValidator.ValidationResult validationResult = typedValidator.validate(input);
                this.logger.logValidation("input_validation", validationResult, traceId, context.getMetadata());
                if (!validationResult.isValid()) {
                    throw ProtocolException.validationFailed("Input validation failed: " + String.join((CharSequence)", ", validationResult.getErrors()), format, traceId);
                }
            }
            Object result = this.executeWithTimeout(() -> processor.apply(input), this.defaultTimeoutMs);
            this.logger.logOperationComplete(operation, format, traceId, context.getElapsedTime(), true, Map.of("result_type", result != null ? result.getClass().getSimpleName() : "null"));
            return result;
        }
        catch (TimeoutException e) {
            ProtocolException timeoutError = ProtocolException.timeout(operation, this.defaultTimeoutMs, traceId);
            this.logger.logError(operation, timeoutError, traceId, context.getMetadata());
            throw timeoutError;
        }
        catch (Exception e) {
            if (e instanceof ProtocolException) {
                this.logger.logError(operation, e, traceId, context.getMetadata());
                throw e;
            }
            ProtocolException wrappedError = ProtocolException.serializationFailed("Processing failed: " + e.getMessage(), e, format, traceId);
            this.logger.logError(operation, wrappedError, traceId, context.getMetadata());
            throw wrappedError;
        }
    }

    private Object executeBinaryProcessing(Object input) throws Exception {
        ByteBuffer buffer;
        if (input instanceof byte[]) {
            buffer = ByteBuffer.wrap((byte[])input);
        } else if (input instanceof ByteBuffer) {
            buffer = (ByteBuffer)input;
        } else if (input instanceof String) {
            byte[] decoded = Base64.getDecoder().decode((String)input);
            buffer = ByteBuffer.wrap(decoded);
        } else {
            throw new IllegalArgumentException("Unsupported binary input type: " + input.getClass().getSimpleName());
        }
        return buffer;
    }

    private Object executeJsonProcessing(Object input) throws Exception {
        String jsonString = input instanceof String ? (String)input : ProtocolUtils.toJson(input);
        if (!ProtocolUtils.isValidJson(jsonString)) {
            throw ProtocolException.validationFailed("Invalid JSON format", "json", this.logger.generateTraceId());
        }
        return ProtocolUtils.parseJson(jsonString);
    }

    private Object executeWithTimeout(Callable<Object> function, long timeoutMs) throws TimeoutException, Exception {
        CompletableFuture<Object> future = CompletableFuture.supplyAsync(() -> {
            try {
                return function.call();
            }
            catch (Exception e) {
                throw new CompletionException(e);
            }
        });
        try {
            return future.get(timeoutMs, TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            future.cancel(true);
            throw new TimeoutException("Operation timed out after " + timeoutMs + " ms");
        }
        catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof CompletionException) {
                cause = cause.getCause();
            }
            if (cause instanceof Exception) {
                throw (Exception)cause;
            }
            throw new Exception(cause);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new Exception("Operation was interrupted", e);
        }
    }

    private synchronized CompletableFuture<Object> addToBatch(Object packet) {
        if (!this.batchingEnabled) {
            throw new IllegalStateException("Batching not enabled");
        }
        CompletableFuture<Object> future = new CompletableFuture<Object>();
        this.pendingResults.put(packet, future);
        this.packetBuffer.add(packet);
        if (this.packetBuffer.size() >= this.batchSizeThreshold) {
            this.flushBatch();
        } else {
            this.scheduleFlush();
        }
        return future;
    }

    private void scheduleFlush() {
        if (this.batchFlushTask != null && !this.batchFlushTask.isDone()) {
            this.batchFlushTask.cancel(false);
        }
        this.batchFlushTask = this.batchScheduler.schedule(this::flushBatch, this.batchTimeThresholdMs, TimeUnit.MILLISECONDS);
    }

    private synchronized void flushBatch() {
        if (this.packetBuffer.isEmpty()) {
            return;
        }
        ArrayList<Object> batch = new ArrayList<Object>(this.packetBuffer);
        this.packetBuffer.clear();
        if (this.batchFlushTask != null) {
            this.batchFlushTask.cancel(false);
            this.batchFlushTask = null;
        }
        CompletableFuture.runAsync(() -> {
            for (Object packet : batch) {
                CompletableFuture<Object> future;
                try {
                    Object result = this.processWithFallback(packet);
                    future = this.pendingResults.remove(packet);
                    if (future == null) continue;
                    future.complete(result);
                }
                catch (Exception e) {
                    future = this.pendingResults.remove(packet);
                    if (future != null) {
                        future.completeExceptionally(e);
                    }
                    this.logger.logError("batch_processing", e, this.logger.generateTraceId(), Map.of("packet_type", packet.getClass().getSimpleName()));
                }
            }
        });
    }

    public void shutdown() {
        if (this.batchScheduler != null) {
            this.batchScheduler.shutdown();
            try {
                if (!this.batchScheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.batchScheduler.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.batchScheduler.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
    }

    public <T, R> ProtocolResult<R> processWithFallback(T input, String operationName, BinarySerializer<T> binarySerializer, BinaryNativeCaller<byte[]> binaryCaller, BinaryDeserializer<R> binaryDeserializer, JsonInputPreparer<T> jsonInputPreparer, JsonNativeCaller<String> jsonCaller, JsonResultParser<R> jsonResultParser, R fallbackResult) {
        String traceId = this.logger.generateTraceId();
        try {
            ByteBuffer binaryInput = binarySerializer.serialize(input);
            byte[] binaryResult = binaryCaller.callNative(binaryInput);
            if (binaryResult != null && binaryResult.length > 0) {
                R result = binaryDeserializer.deserialize(binaryResult);
                return new ProtocolResult<R>(result, null, true);
            }
        }
        catch (Exception e) {
            this.logger.logWarning(operationName, "Binary processing failed, trying JSON fallback", traceId, Map.of("error", e.getMessage()));
        }
        try {
            Map<String, Object> jsonInput = jsonInputPreparer.prepareInput(input);
            String jsonInputStr = ProtocolUtils.toJson(jsonInput);
            String jsonResult = jsonCaller.callNative(jsonInputStr);
            if (jsonResult != null && !jsonResult.isEmpty()) {
                R result = jsonResultParser.parseResult(jsonResult);
                return new ProtocolResult<R>(result, null, true);
            }
        }
        catch (Exception e) {
            this.logger.logWarning(operationName, "JSON processing also failed, returning fallback", traceId, Map.of("error", e.getMessage()));
        }
        return new ProtocolResult<R>(fallbackResult, null, true);
    }

    public static interface BinarySerializer<T> {
        public ByteBuffer serialize(T var1) throws Exception;
    }

    public static interface BinaryNativeCaller<R> {
        public R callNative(ByteBuffer var1) throws Exception;
    }

    public static interface BinaryDeserializer<T> {
        public T deserialize(byte[] var1) throws Exception;
    }

    public static class ProtocolResult<T> {
        private final T data;
        private final Exception error;
        private final boolean success;

        public ProtocolResult(T data, Exception error, boolean success) {
            this.data = data;
            this.error = error;
            this.success = success;
        }

        public T getDataOrThrow() throws Exception {
            if (!this.success && this.error != null) {
                throw this.error;
            }
            return this.data;
        }

        public T getData() {
            return this.data;
        }

        public Exception getError() {
            return this.error;
        }

        public boolean isSuccess() {
            return this.success;
        }
    }

    public static interface JsonInputPreparer<T> {
        public Map<String, Object> prepareInput(T var1);
    }

    public static interface JsonNativeCaller<R> {
        public R callNative(String var1) throws Exception;
    }

    public static interface JsonResultParser<T> {
        public T parseResult(String var1) throws Exception;
    }
}

