/*
 * Decompiled with CFR 0.152.
 */
package com.kneaf.core.chunkstorage.cache;

import com.kneaf.core.chunkstorage.common.StorageStatisticsProvider;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChunkCache
implements StorageStatisticsProvider {
    private static final Logger LOGGER = LoggerFactory.getLogger(ChunkCache.class);
    private final Map<String, CachedChunk> cache = new ConcurrentHashMap<String, CachedChunk>();
    private final AtomicInteger maxCapacity;
    private volatile EvictionPolicy evictionPolicy;
    private final ReadWriteLock cacheLock = new ReentrantReadWriteLock();
    private final AtomicInteger hitCount = new AtomicInteger(0);
    private final AtomicInteger missCount = new AtomicInteger(0);
    private final AtomicInteger evictionCount = new AtomicInteger(0);
    private final AtomicLong totalHits = new AtomicLong(0L);
    private final AtomicLong totalMisses = new AtomicLong(0L);
    private final AtomicInteger swapOutCount = new AtomicInteger(0);
    private final AtomicInteger swapInCount = new AtomicInteger(0);
    private final AtomicInteger swapOutFailureCount = new AtomicInteger(0);
    private final AtomicInteger swapInFailureCount = new AtomicInteger(0);
    private final AtomicLong totalSwapOutTime = new AtomicLong(0L);
    private final AtomicLong totalSwapInTime = new AtomicLong(0L);
    private volatile MemoryPressureLevel memoryPressureLevel = MemoryPressureLevel.NORMAL;

    public ChunkCache(int maxCapacity, EvictionPolicy evictionPolicy) {
        if (maxCapacity <= 0) {
            throw new IllegalArgumentException("Max capacity must be positive");
        }
        if (evictionPolicy == null) {
            throw new IllegalArgumentException("Eviction policy cannot be null");
        }
        this.maxCapacity = new AtomicInteger(maxCapacity);
        this.evictionPolicy = evictionPolicy;
        LOGGER.info("Initialized ChunkCache with capacity { } and { } eviction policy", (Object)this.maxCapacity.get(), (Object)evictionPolicy.getPolicyName());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Optional<CachedChunk> getChunk(String key) {
        if (key == null || key.isEmpty()) {
            return Optional.empty();
        }
        this.cacheLock.readLock().lock();
        try {
            CachedChunk cached = this.cache.get(key);
            if (cached != null) {
                cached.markAccessed();
                this.hitCount.incrementAndGet();
                this.totalHits.incrementAndGet();
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace("Cache hit for chunk { }", (Object)key);
                }
                Optional<CachedChunk> optional = Optional.of(cached);
                return optional;
            }
            this.missCount.incrementAndGet();
            this.totalMisses.incrementAndGet();
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Cache miss for chunk { }", (Object)key);
            }
            Optional<CachedChunk> optional = Optional.empty();
            return optional;
        }
        finally {
            this.cacheLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CachedChunk putChunk(String key, Object chunk) {
        if (key == null || key.isEmpty() || chunk == null) {
            throw new IllegalArgumentException("Key and chunk cannot be null or empty");
        }
        this.cacheLock.writeLock().lock();
        try {
            String keyToEvict;
            if (this.cache.size() >= this.maxCapacity.get() && !this.cache.containsKey(key) && (keyToEvict = this.evictionPolicy.selectChunkToEvict(this.cache)) != null) {
                CachedChunk evicted = this.cache.remove(keyToEvict);
                this.evictionCount.incrementAndGet();
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Evicted chunk { } from cache (policy: { })", (Object)keyToEvict, (Object)this.evictionPolicy.getPolicyName());
                }
                CachedChunk cachedChunk = evicted;
                return cachedChunk;
            }
            CachedChunk cachedChunk = new CachedChunk(chunk);
            this.cache.put(key, cachedChunk);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Cached chunk { } (total cached: { })", (Object)key, (Object)this.cache.size());
            }
            CachedChunk cachedChunk2 = null;
            return cachedChunk2;
        }
        finally {
            this.cacheLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CachedChunk removeChunk(String key) {
        if (key == null || key.isEmpty()) {
            return null;
        }
        this.cacheLock.writeLock().lock();
        try {
            CachedChunk removed = this.cache.remove(key);
            if (removed != null && LOGGER.isDebugEnabled()) {
                LOGGER.debug("Removed chunk { } from cache", (Object)key);
            }
            CachedChunk cachedChunk = removed;
            return cachedChunk;
        }
        finally {
            this.cacheLock.writeLock().unlock();
        }
    }

    public boolean hasChunk(String key) {
        if (key == null || key.isEmpty()) {
            return false;
        }
        this.cacheLock.readLock().lock();
        try {
            boolean bl = this.cache.containsKey(key);
            return bl;
        }
        finally {
            this.cacheLock.readLock().unlock();
        }
    }

    public int getCacheSize() {
        this.cacheLock.readLock().lock();
        try {
            int n = this.cache.size();
            return n;
        }
        finally {
            this.cacheLock.readLock().unlock();
        }
    }

    public int getMaxCapacity() {
        return this.maxCapacity.get();
    }

    public void clear() {
        this.cacheLock.writeLock().lock();
        try {
            int sizeBefore = this.cache.size();
            this.cache.clear();
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("Cleared { } chunks from cache", (Object)sizeBefore);
            }
        }
        finally {
            this.cacheLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public CacheStats getStats() {
        this.cacheLock.readLock().lock();
        try {
            int currentHits = this.hitCount.get();
            int currentMisses = this.missCount.get();
            int currentEvictions = this.evictionCount.get();
            long totalHitCount = this.totalHits.get();
            long totalMissCount = this.totalMisses.get();
            int cacheSize = this.cache.size();
            double hitRate = totalHitCount + totalMissCount > 0L ? (double)totalHitCount / (double)(totalHitCount + totalMissCount) : 0.0;
            int currentSwapOuts = this.swapOutCount.get();
            int currentSwapIns = this.swapInCount.get();
            int currentSwapOutFailures = this.swapOutFailureCount.get();
            int currentSwapInFailures = this.swapInFailureCount.get();
            long totalSwapOutDuration = this.totalSwapOutTime.get();
            long totalSwapInDuration = this.totalSwapInTime.get();
            CacheStats cacheStats = new CacheStats(currentHits, currentMisses, currentEvictions, cacheSize, this.maxCapacity.get(), hitRate, this.evictionPolicy.getPolicyName(), currentSwapOuts, currentSwapIns, currentSwapOutFailures, currentSwapInFailures, totalSwapOutDuration, totalSwapInDuration);
            return cacheStats;
        }
        finally {
            this.cacheLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean setMaxCapacity(int newCapacity) {
        if (newCapacity <= 0) {
            return false;
        }
        this.cacheLock.writeLock().lock();
        try {
            String keyToEvict;
            int old = this.maxCapacity.getAndSet(newCapacity);
            LOGGER.info("Cache capacity changed from { } to { }", (Object)old, (Object)newCapacity);
            while (this.cache.size() > newCapacity && (keyToEvict = this.evictionPolicy.selectChunkToEvict(this.cache)) != null) {
                CachedChunk evicted = this.cache.remove(keyToEvict);
                if (evicted == null) continue;
                this.evictionCount.incrementAndGet();
                if (!LOGGER.isDebugEnabled()) continue;
                LOGGER.debug("Evicted chunk { } due to capacity change (policy: { })", (Object)keyToEvict, (Object)this.evictionPolicy.getPolicyName());
            }
            boolean bl = true;
            return bl;
        }
        finally {
            this.cacheLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean setEvictionPolicy(String policyName) {
        if (policyName == null || policyName.isEmpty()) {
            return false;
        }
        this.cacheLock.writeLock().lock();
        try {
            EvictionPolicy newPolicy;
            switch (policyName.toLowerCase()) {
                case "lru": {
                    newPolicy = new LRUEvictionPolicy();
                    break;
                }
                case "distance": {
                    newPolicy = new DistanceEvictionPolicy(0, 0);
                    break;
                }
                case "hybrid": {
                    newPolicy = new HybridEvictionPolicy();
                    break;
                }
                case "swapaware": 
                case "swap_aware": 
                case "swap-aware": {
                    newPolicy = new SwapAwareEvictionPolicy(this.memoryPressureLevel);
                    break;
                }
                default: {
                    boolean bl = false;
                    return bl;
                }
            }
            this.evictionPolicy = newPolicy;
            LOGGER.info("Eviction policy switched to { }", (Object)newPolicy.getPolicyName());
            boolean bl = true;
            return bl;
        }
        finally {
            this.cacheLock.writeLock().unlock();
        }
    }

    public void resetStats() {
        this.hitCount.set(0);
        this.missCount.set(0);
        this.evictionCount.set(0);
        this.swapOutCount.set(0);
        this.swapInCount.set(0);
        this.swapOutFailureCount.set(0);
        this.swapInFailureCount.set(0);
        this.totalSwapOutTime.set(0L);
        this.totalSwapInTime.set(0L);
    }

    public void setMemoryPressureLevel(MemoryPressureLevel pressureLevel) {
        if (pressureLevel != null) {
            this.memoryPressureLevel = pressureLevel;
            LOGGER.debug("Memory pressure level set to { }", (Object)pressureLevel);
        }
    }

    public MemoryPressureLevel getMemoryPressureLevel() {
        return this.memoryPressureLevel;
    }

    public CompletableFuture<Boolean> initiateSwapOut(String key, Consumer<Boolean> swapCallback) {
        if (key == null || key.isEmpty()) {
            return CompletableFuture.completedFuture(false);
        }
        return CompletableFuture.supplyAsync(() -> {
            this.cacheLock.writeLock().lock();
            try {
                CachedChunk cached = this.cache.get(key);
                if (cached == null) {
                    LOGGER.warn("Cannot swap out non-existent chunk: { }", (Object)key);
                    Boolean bl = false;
                    return bl;
                }
                if (cached.isSwapping() || cached.isSwapped()) {
                    LOGGER.debug("Chunk { } is already swapping or swapped", (Object)key);
                    Boolean bl = false;
                    return bl;
                }
                long startTime = System.currentTimeMillis();
                cached.setState(ChunkState.SWAPPING_OUT);
                cached.setSwapStartTime(startTime);
                LOGGER.debug("Initiated swap-out for chunk { }", (Object)key);
                try {
                    Thread.sleep(100L);
                    cached.setState(ChunkState.SWAPPED);
                    cached.setSwapCompleteTime(System.currentTimeMillis());
                    long duration = cached.getSwapDuration();
                    this.totalSwapOutTime.addAndGet(duration);
                    this.swapOutCount.incrementAndGet();
                    LOGGER.debug("Completed swap-out for chunk { } in { }ms", (Object)key, (Object)duration);
                    if (swapCallback != null) {
                        swapCallback.accept(true);
                    }
                    Boolean bl = true;
                    return bl;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    cached.setState(ChunkState.COLD);
                    cached.setSwapError("Swap interrupted");
                    this.swapOutFailureCount.incrementAndGet();
                    LOGGER.error("Swap-out interrupted for chunk { }", (Object)key, (Object)e);
                    if (swapCallback != null) {
                        swapCallback.accept(false);
                    }
                    Boolean bl = false;
                    this.cacheLock.writeLock().unlock();
                    return bl;
                }
            }
            finally {
                this.cacheLock.writeLock().unlock();
            }
        });
    }

    public CompletableFuture<Boolean> initiateSwapIn(String key, Consumer<Boolean> swapCallback) {
        if (key == null || key.isEmpty()) {
            return CompletableFuture.completedFuture(false);
        }
        return CompletableFuture.supplyAsync(() -> {
            this.cacheLock.writeLock().lock();
            try {
                CachedChunk cached = this.cache.get(key);
                if (cached == null) {
                    LOGGER.warn("Cannot swap in non-existent chunk: { }", (Object)key);
                    Boolean bl = false;
                    return bl;
                }
                if (!cached.isSwapped()) {
                    LOGGER.debug("Chunk { } is not swapped out", (Object)key);
                    Boolean bl = false;
                    return bl;
                }
                long startTime = System.currentTimeMillis();
                cached.setState(ChunkState.SWAPPING_IN);
                cached.setSwapStartTime(startTime);
                LOGGER.debug("Initiated swap-in for chunk { }", (Object)key);
                try {
                    Thread.sleep(150L);
                    cached.setState(ChunkState.HOT);
                    cached.setSwapCompleteTime(System.currentTimeMillis());
                    long duration = cached.getSwapDuration();
                    this.totalSwapInTime.addAndGet(duration);
                    this.swapInCount.incrementAndGet();
                    LOGGER.debug("Completed swap-in for chunk { } in { }ms", (Object)key, (Object)duration);
                    if (swapCallback != null) {
                        swapCallback.accept(true);
                    }
                    Boolean bl = true;
                    return bl;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    cached.setState(ChunkState.SWAPPED);
                    cached.setSwapError("Swap-in interrupted");
                    this.swapInFailureCount.incrementAndGet();
                    LOGGER.error("Swap-in interrupted for chunk { }", (Object)key, (Object)e);
                    if (swapCallback != null) {
                        swapCallback.accept(false);
                    }
                    Boolean bl = false;
                    this.cacheLock.writeLock().unlock();
                    return bl;
                }
            }
            finally {
                this.cacheLock.writeLock().unlock();
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int performSwapAwareEviction(int targetChunks) {
        if (targetChunks <= 0) {
            return 0;
        }
        int swappedCount = 0;
        this.cacheLock.writeLock().lock();
        try {
            SwapAwareEvictionPolicy swapPolicy = new SwapAwareEvictionPolicy(this.memoryPressureLevel);
            for (int i = 0; i < targetChunks; ++i) {
                CachedChunk cached;
                String keyToSwap = swapPolicy.selectChunkToEvict(this.cache);
                if (keyToSwap == null || (cached = this.cache.get(keyToSwap)) == null || cached.isSwapping() || cached.isSwapped()) continue;
                this.initiateSwapOut(keyToSwap, success -> {
                    if (success.booleanValue()) {
                        LOGGER.debug("Lazy swap-out completed for chunk { }", (Object)keyToSwap);
                    }
                });
                ++swappedCount;
            }
            LOGGER.info("Initiated swap-aware eviction for { } chunks (pressure: { })", (Object)swappedCount, (Object)this.memoryPressureLevel);
        }
        finally {
            this.cacheLock.writeLock().unlock();
        }
        return swappedCount;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean canEvict(String key) {
        if (key == null || key.isEmpty()) {
            return false;
        }
        this.cacheLock.readLock().lock();
        try {
            CachedChunk cached = this.cache.get(key);
            boolean bl = cached != null && !cached.isSwapping();
            return bl;
        }
        finally {
            this.cacheLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public SwapStats getSwapStats() {
        this.cacheLock.readLock().lock();
        try {
            int outs = this.swapOutCount.get();
            int ins = this.swapInCount.get();
            int outFailures = this.swapOutFailureCount.get();
            int inFailures = this.swapInFailureCount.get();
            long totalOutTime = this.totalSwapOutTime.get();
            long totalInTime = this.totalSwapInTime.get();
            double avgOutTime = outs > 0 ? (double)totalOutTime / (double)outs : 0.0;
            double avgInTime = ins > 0 ? (double)totalInTime / (double)ins : 0.0;
            int totalSwaps = outs + ins;
            int totalFailures = outFailures + inFailures;
            double failureRate = totalSwaps > 0 ? (double)totalFailures / (double)totalSwaps : 0.0;
            SwapStats swapStats = new SwapStats(outs, ins, outFailures, inFailures, avgOutTime, avgInTime, failureRate);
            return swapStats;
        }
        finally {
            this.cacheLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void updateChunkState(String key, ChunkState state) {
        if (key == null || key.isEmpty() || state == null) {
            return;
        }
        this.cacheLock.writeLock().lock();
        try {
            CachedChunk cached = this.cache.get(key);
            if (cached != null) {
                cached.setState(state);
            }
        }
        finally {
            this.cacheLock.writeLock().unlock();
        }
    }

    public void markChunkDirty(String key) {
        if (key == null || key.isEmpty()) {
            return;
        }
        this.cacheLock.writeLock().lock();
        try {
            CachedChunk cached = this.cache.get(key);
            if (cached != null) {
                cached.markDirty();
                cached.setState(ChunkState.DIRTY);
            }
        }
        finally {
            this.cacheLock.writeLock().unlock();
        }
    }

    public static enum MemoryPressureLevel {
        NORMAL,
        ELEVATED,
        HIGH,
        CRITICAL;

    }

    public static interface EvictionPolicy {
        public String selectChunkToEvict(Map<String, CachedChunk> var1);

        public String getPolicyName();
    }

    public static class CachedChunk {
        private final Object chunk;
        private final long lastAccessTime;
        private final long creationTime;
        private final AtomicInteger accessCount;
        private volatile ChunkState state;
        private volatile boolean dirty;
        private volatile long swapStartTime;
        private volatile long swapCompleteTime;
        private volatile String swapError;

        public CachedChunk(Object chunk) {
            this.chunk = chunk;
            this.lastAccessTime = this.creationTime = System.currentTimeMillis();
            this.accessCount = new AtomicInteger(1);
            this.state = ChunkState.HOT;
            this.dirty = false;
            this.swapStartTime = 0L;
            this.swapCompleteTime = 0L;
            this.swapError = null;
        }

        public Object getChunk() {
            return this.chunk;
        }

        public Object getChunkAsLevelChunk() {
            try {
                if (this.chunk != null && Class.forName("net.minecraft.world.level.chunk.LevelChunk").isInstance(this.chunk)) {
                    return this.chunk;
                }
            }
            catch (ClassNotFoundException classNotFoundException) {
                // empty catch block
            }
            return null;
        }

        public boolean isLevelChunk() {
            try {
                return this.chunk != null && Class.forName("net.minecraft.world.level.chunk.LevelChunk").isInstance(this.chunk);
            }
            catch (ClassNotFoundException e) {
                return false;
            }
        }

        public long getLastAccessTime() {
            return this.lastAccessTime;
        }

        public long getCreationTime() {
            return this.creationTime;
        }

        public int getAccessCount() {
            return this.accessCount.get();
        }

        public ChunkState getState() {
            return this.state;
        }

        public boolean isDirty() {
            return this.dirty;
        }

        public long getSwapStartTime() {
            return this.swapStartTime;
        }

        public long getSwapCompleteTime() {
            return this.swapCompleteTime;
        }

        public String getSwapError() {
            return this.swapError;
        }

        public void markAccessed() {
            this.accessCount.incrementAndGet();
        }

        public void markDirty() {
            this.dirty = true;
        }

        public void markClean() {
            this.dirty = false;
        }

        public void setState(ChunkState state) {
            this.state = state;
        }

        public void setSwapStartTime(long time) {
            this.swapStartTime = time;
        }

        public void setSwapCompleteTime(long time) {
            this.swapCompleteTime = time;
        }

        public void setSwapError(String error) {
            this.swapError = error;
        }

        public boolean isSwapping() {
            return this.state == ChunkState.SWAPPING_OUT || this.state == ChunkState.SWAPPING_IN;
        }

        public boolean isSwapped() {
            return this.state == ChunkState.SWAPPED;
        }

        public long getSwapDuration() {
            if (this.swapStartTime > 0L && this.swapCompleteTime > this.swapStartTime) {
                return this.swapCompleteTime - this.swapStartTime;
            }
            return 0L;
        }
    }

    public static class CacheStats {
        private final int hits;
        private final int misses;
        private final int evictions;
        private final int cacheSize;
        private final int maxCapacity;
        private final double hitRate;
        private final String evictionPolicy;
        private final int swapOuts;
        private final int swapIns;
        private final int swapOutFailures;
        private final int swapInFailures;
        private final long totalSwapOutTime;
        private final long totalSwapInTime;

        public CacheStats(int hits, int misses, int evictions, int cacheSize, int maxCapacity, double hitRate, String evictionPolicy, int swapOuts, int swapIns, int swapOutFailures, int swapInFailures, long totalSwapOutTime, long totalSwapInTime) {
            this.hits = hits;
            this.misses = misses;
            this.evictions = evictions;
            this.cacheSize = cacheSize;
            this.maxCapacity = maxCapacity;
            this.hitRate = hitRate;
            this.evictionPolicy = evictionPolicy;
            this.swapOuts = swapOuts;
            this.swapIns = swapIns;
            this.swapOutFailures = swapOutFailures;
            this.swapInFailures = swapInFailures;
            this.totalSwapOutTime = totalSwapOutTime;
            this.totalSwapInTime = totalSwapInTime;
        }

        public int getHits() {
            return this.hits;
        }

        public int getMisses() {
            return this.misses;
        }

        public int getEvictions() {
            return this.evictions;
        }

        public int getCacheSize() {
            return this.cacheSize;
        }

        public int getMaxCapacity() {
            return this.maxCapacity;
        }

        public double getHitRate() {
            return this.hitRate;
        }

        public String getEvictionPolicy() {
            return this.evictionPolicy;
        }

        public int getSwapOuts() {
            return this.swapOuts;
        }

        public int getSwapIns() {
            return this.swapIns;
        }

        public int getSwapOutFailures() {
            return this.swapOutFailures;
        }

        public int getSwapInFailures() {
            return this.swapInFailures;
        }

        public long getTotalSwapOutTime() {
            return this.totalSwapOutTime;
        }

        public long getTotalSwapInTime() {
            return this.totalSwapInTime;
        }

        public double getAverageSwapOutTime() {
            return this.swapOuts > 0 ? (double)this.totalSwapOutTime / (double)this.swapOuts : 0.0;
        }

        public double getAverageSwapInTime() {
            return this.swapIns > 0 ? (double)this.totalSwapInTime / (double)this.swapIns : 0.0;
        }

        public String toString() {
            return String.format("CacheStats{hits=%d, misses=%d, evictions=%d, size=%d/%d, hitRate=%.2f%%, policy=%s, swapOuts=%d, swapIns=%d, swapFailures=%d/%d, avgSwapOutTime=%.2fms, avgSwapInTime=%.2fms}", this.hits, this.misses, this.evictions, this.cacheSize, this.maxCapacity, this.hitRate * 100.0, this.evictionPolicy, this.swapOuts, this.swapIns, this.swapOutFailures, this.swapInFailures, this.getAverageSwapOutTime(), this.getAverageSwapInTime());
        }
    }

    public static class LRUEvictionPolicy
    implements EvictionPolicy {
        @Override
        public String selectChunkToEvict(Map<String, CachedChunk> cache) {
            String oldestKey = null;
            long oldestTime = Long.MAX_VALUE;
            for (Map.Entry<String, CachedChunk> entry : cache.entrySet()) {
                CachedChunk cached = entry.getValue();
                if (cached.isSwapping() || cached.getLastAccessTime() >= oldestTime) continue;
                oldestTime = cached.getLastAccessTime();
                oldestKey = entry.getKey();
            }
            return oldestKey;
        }

        @Override
        public String getPolicyName() {
            return "LRU";
        }
    }

    public static class DistanceEvictionPolicy
    implements EvictionPolicy {
        private final int centerX;
        private final int centerZ;

        public DistanceEvictionPolicy(int centerX, int centerZ) {
            this.centerX = centerX;
            this.centerZ = centerZ;
        }

        @Override
        public String selectChunkToEvict(Map<String, CachedChunk> cache) {
            String farthestKey = null;
            double maxDistance = -1.0;
            for (Map.Entry<String, CachedChunk> entry : cache.entrySet()) {
                String[] parts = entry.getKey().split(":");
                if (parts.length < 3) continue;
                try {
                    int chunkX = Integer.parseInt(parts[1]);
                    int chunkZ = Integer.parseInt(parts[2]);
                    double distance = Math.sqrt(Math.pow((double)chunkX - (double)this.centerX, 2.0) + Math.pow((double)chunkZ - (double)this.centerZ, 2.0));
                    if (!(distance > maxDistance)) continue;
                    maxDistance = distance;
                    farthestKey = entry.getKey();
                }
                catch (NumberFormatException e) {
                    LOGGER.debug("Invalid chunk key format: { }", (Object)entry.getKey());
                }
            }
            return farthestKey;
        }

        @Override
        public String getPolicyName() {
            return "Distance";
        }
    }

    public static class HybridEvictionPolicy
    implements EvictionPolicy {
        @Override
        public String selectChunkToEvict(Map<String, CachedChunk> cache) {
            String bestKey = null;
            double bestScore = Double.MAX_VALUE;
            long currentTime = System.currentTimeMillis();
            for (Map.Entry<String, CachedChunk> entry : cache.entrySet()) {
                double accessScore;
                double timeScore;
                double score;
                CachedChunk cached = entry.getValue();
                if (cached.isSwapping() || !((score = (timeScore = (double)(currentTime - cached.getLastAccessTime()) / 1000.0) * (accessScore = 1.0 / (double)Math.max(1, cached.getAccessCount()))) < bestScore)) continue;
                bestScore = score;
                bestKey = entry.getKey();
            }
            return bestKey;
        }

        @Override
        public String getPolicyName() {
            return "Hybrid";
        }
    }

    public static class SwapAwareEvictionPolicy
    implements EvictionPolicy {
        private final MemoryPressureLevel memoryPressure;

        public SwapAwareEvictionPolicy(MemoryPressureLevel memoryPressure) {
            this.memoryPressure = memoryPressure;
        }

        @Override
        public String selectChunkToEvict(Map<String, CachedChunk> cache) {
            String bestKey = null;
            double bestScore = Double.MAX_VALUE;
            long currentTime = System.currentTimeMillis();
            for (Map.Entry<String, CachedChunk> entry : cache.entrySet()) {
                double score;
                CachedChunk cached = entry.getValue();
                if (cached.isSwapping() || !((score = this.calculateSwapPriorityScore(cached, currentTime)) < bestScore)) continue;
                bestScore = score;
                bestKey = entry.getKey();
            }
            return bestKey;
        }

        private double calculateSwapPriorityScore(CachedChunk cached, long currentTime) {
            double baseScore = 0.0;
            switch (cached.getState().ordinal()) {
                case 1: {
                    baseScore += 10.0;
                    break;
                }
                case 3: {
                    baseScore += 5.0;
                    break;
                }
                case 0: {
                    baseScore += 50.0;
                    break;
                }
                case 2: {
                    baseScore += 100.0;
                    break;
                }
                case 5: {
                    baseScore += 1000.0;
                    break;
                }
                default: {
                    baseScore += 25.0;
                }
            }
            double timeScore = (double)(currentTime - cached.getLastAccessTime()) / 1000.0;
            baseScore += timeScore * 0.1;
            double accessScore = 1.0 / (double)Math.max(1, cached.getAccessCount());
            baseScore *= accessScore;
            double pressureMultiplier = this.getMemoryPressureMultiplier();
            baseScore *= pressureMultiplier;
            if (cached.isDirty()) {
                baseScore *= 2.0;
            }
            return baseScore;
        }

        private double getMemoryPressureMultiplier() {
            switch (this.memoryPressure.ordinal()) {
                case 3: {
                    return 0.1;
                }
                case 2: {
                    return 0.3;
                }
                case 1: {
                    return 0.7;
                }
            }
            return 1.0;
        }

        @Override
        public String getPolicyName() {
            return "SwapAware";
        }
    }

    public static class SwapStats {
        private final int swapOuts;
        private final int swapIns;
        private final int swapOutFailures;
        private final int swapInFailures;
        private final double avgSwapOutTime;
        private final double avgSwapInTime;
        private final double failureRate;

        public SwapStats(int swapOuts, int swapIns, int swapOutFailures, int swapInFailures, double avgSwapOutTime, double avgSwapInTime, double failureRate) {
            this.swapOuts = swapOuts;
            this.swapIns = swapIns;
            this.swapOutFailures = swapOutFailures;
            this.swapInFailures = swapInFailures;
            this.avgSwapOutTime = avgSwapOutTime;
            this.avgSwapInTime = avgSwapInTime;
            this.failureRate = failureRate;
        }

        public int getSwapOuts() {
            return this.swapOuts;
        }

        public int getSwapIns() {
            return this.swapIns;
        }

        public int getSwapOutFailures() {
            return this.swapOutFailures;
        }

        public int getSwapInFailures() {
            return this.swapInFailures;
        }

        public double getAvgSwapOutTime() {
            return this.avgSwapOutTime;
        }

        public double getAvgSwapInTime() {
            return this.avgSwapInTime;
        }

        public double getFailureRate() {
            return this.failureRate;
        }

        public int getTotalSwaps() {
            return this.swapOuts + this.swapIns;
        }

        public int getTotalFailures() {
            return this.swapOutFailures + this.swapInFailures;
        }

        public String toString() {
            return String.format("SwapStats{swapOuts=%d, swapIns=%d, failures=%d/%d, failureRate=%.2f%%, avgSwapOutTime=%.2fms, avgSwapInTime=%.2fms}", this.swapOuts, this.swapIns, this.getTotalFailures(), this.getTotalSwaps(), this.failureRate * 100.0, this.avgSwapOutTime, this.avgSwapInTime);
        }
    }

    public static enum ChunkState {
        HOT,
        COLD,
        DIRTY,
        SERIALIZED,
        SWAPPING_OUT,
        SWAPPED,
        SWAPPING_IN;

    }
}

