/*
 * Decompiled with CFR 0.152.
 */
package com.kneaf.core.performance;

import com.kneaf.core.performance.core.PerformanceConstants;
import com.kneaf.core.performance.monitoring.PerformanceManager;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

public final class EnhancedNativeBridge {
    public static final byte OPERATION_TYPE_ENTITY = 1;
    public static final byte OPERATION_TYPE_ITEM = 2;
    public static final byte OPERATION_TYPE_MOB = 3;
    public static final byte OPERATION_TYPE_BLOCK = 4;
    public static final byte OPERATION_TYPE_SPATIAL = 5;
    public static final byte OPERATION_TYPE_SIMD = 6;
    private static final AtomicLong TOTAL_BATCHES_PROCESSED;
    private static final AtomicLong TOTAL_OPERATIONS_BATCHED;
    private static final AtomicLong AVERAGE_BATCH_SIZE;
    private static final AtomicBoolean BATCH_PROCESSOR_RUNNING;
    private static final ConcurrentLinkedQueue<BatchOperation>[] OPERATION_QUEUES;
    private static final Thread[] BATCH_PROCESSOR_THREADS;

    private EnhancedNativeBridge() {
    }

    private static int adaptiveBatchTimeoutMs() {
        double tps = PerformanceManager.getAverageTPS();
        double tickDelay = PerformanceManager.getLastTickDurationMs();
        return (int)PerformanceConstants.getAdaptiveBatchTimeoutMs(tps, tickDelay);
    }

    private static int minBatchSize() {
        return PerformanceConstants.getAdaptiveBatchSize(PerformanceManager.getAverageTPS(), PerformanceManager.getLastTickDurationMs());
    }

    private static int maxBatchSize() {
        return Math.max(50, PerformanceConstants.getAdaptiveBatchSize(PerformanceManager.getAverageTPS(), PerformanceManager.getLastTickDurationMs()) * 2);
    }

    private static int maxPendingBatches() {
        return Math.max(10, PerformanceConstants.getAdaptiveQueueCapacity(PerformanceManager.getAverageTPS()) / 20);
    }

    public static CompletableFuture<byte[]> submitOperation(byte operationType, ByteBuffer inputData) {
        CompletableFuture<byte[]> future = new CompletableFuture<byte[]>();
        BatchOperation operation = new BatchOperation(operationType, inputData, future);
        if (operationType >= 0 && operationType < OPERATION_QUEUES.length) {
            OPERATION_QUEUES[operationType].offer(operation);
        } else {
            future.completeExceptionally(new IllegalArgumentException("Invalid operation type: " + operationType));
        }
        return future;
    }

    private static void processBatch(byte operationType, List<BatchOperation> batch) {
        if (batch.isEmpty()) {
            return;
        }
        try {
            int totalSize = 0;
            for (BatchOperation batchOperation : batch) {
                totalSize += batchOperation.estimatedSize;
            }
            ByteBuffer batchBuffer = ByteBuffer.allocateDirect(totalSize + batch.size() * 8);
            batchBuffer.order(ByteOrder.LITTLE_ENDIAN);
            batchBuffer.putInt(batch.size());
            batchBuffer.putInt(totalSize);
            for (BatchOperation op : batch) {
                batchBuffer.putInt(op.estimatedSize);
                if (op.inputData == null) continue;
                batchBuffer.put(op.inputData);
            }
            batchBuffer.flip();
            byte[] byArray = EnhancedNativeBridge.nativeProcessBatch(operationType, batchBuffer);
            if (byArray != null) {
                int i;
                ByteBuffer resultBuffer = ByteBuffer.wrap(byArray).order(ByteOrder.LITTLE_ENDIAN);
                int resultCount = resultBuffer.getInt();
                ArrayList<byte[]> results = new ArrayList<byte[]>(resultCount);
                for (i = 0; i < resultCount; ++i) {
                    int resultSize = resultBuffer.getInt();
                    if (resultSize > 0 && resultBuffer.remaining() >= resultSize) {
                        byte[] resultData = new byte[resultSize];
                        resultBuffer.get(resultData);
                        results.add(resultData);
                        continue;
                    }
                    results.add(new byte[0]);
                }
                for (i = 0; i < Math.min(batch.size(), results.size()); ++i) {
                    BatchOperation op = batch.get(i);
                    if (op.resultFuture.isDone()) continue;
                    op.resultFuture.complete((byte[])results.get(i));
                }
            } else {
                for (BatchOperation op : batch) {
                    if (op.resultFuture.isDone()) continue;
                    op.resultFuture.complete(new byte[0]);
                }
            }
        }
        catch (Exception e) {
            for (BatchOperation batchOperation : batch) {
                if (batchOperation.resultFuture.isDone()) continue;
                batchOperation.resultFuture.completeExceptionally(e);
            }
        }
        TOTAL_BATCHES_PROCESSED.incrementAndGet();
        TOTAL_OPERATIONS_BATCHED.addAndGet(batch.size());
        long currentAvg = AVERAGE_BATCH_SIZE.get();
        long l = (currentAvg * 9L + (long)batch.size()) / 10L;
        AVERAGE_BATCH_SIZE.set(l);
    }

    private static void startBatchProcessors() {
        if (BATCH_PROCESSOR_RUNNING.compareAndSet(false, true)) {
            for (int i = 0; i < BATCH_PROCESSOR_THREADS.length; ++i) {
                int processorId = i;
                EnhancedNativeBridge.BATCH_PROCESSOR_THREADS[i] = new Thread(() -> {
                    for (byte opType = 0; opType < OPERATION_QUEUES.length; opType = (byte)(opType + 1)) {
                        if (opType % BATCH_PROCESSOR_THREADS.length != processorId) continue;
                        new BatchProcessor(opType, processorId).run();
                    }
                });
                BATCH_PROCESSOR_THREADS[i].setDaemon(true);
                BATCH_PROCESSOR_THREADS[i].setName("EnhancedNativeBridge-Processor-" + i);
                BATCH_PROCESSOR_THREADS[i].start();
            }
        }
    }

    public static void shutdown() {
        BATCH_PROCESSOR_RUNNING.set(false);
        for (Thread thread : BATCH_PROCESSOR_THREADS) {
            if (thread == null) continue;
            thread.interrupt();
        }
    }

    public static BatchMetrics getMetrics() {
        return new BatchMetrics(TOTAL_BATCHES_PROCESSED.get(), TOTAL_OPERATIONS_BATCHED.get(), AVERAGE_BATCH_SIZE.get(), EnhancedNativeBridge.getCurrentQueueDepth());
    }

    private static int getCurrentQueueDepth() {
        int totalDepth = 0;
        for (ConcurrentLinkedQueue<BatchOperation> queue : OPERATION_QUEUES) {
            totalDepth += queue.size();
        }
        return totalDepth;
    }

    public static native void initRustAllocator();

    public static native long nativeCreateWorker(int var0);

    public static native void nativePushTask(long var0, byte[] var2);

    public static native byte[] nativePollResult(long var0);

    public static native void nativeDestroyWorker(long var0);

    private static native byte[] nativeProcessBatch(byte var0, ByteBuffer var1);

    public static native String getSimdFeatures();

    public static native int initSimd(boolean var0, boolean var1, boolean var2, boolean var3, boolean var4);

    public static native String getBatchProcessorMetrics();

    public static native int initBatchProcessor(int var0, long var1, int var3, int var4);

    static {
        try {
            System.loadLibrary("rustperf");
        }
        catch (UnsatisfiedLinkError unsatisfiedLinkError) {
            // empty catch block
        }
        TOTAL_BATCHES_PROCESSED = new AtomicLong(0L);
        TOTAL_OPERATIONS_BATCHED = new AtomicLong(0L);
        AVERAGE_BATCH_SIZE = new AtomicLong(0L);
        BATCH_PROCESSOR_RUNNING = new AtomicBoolean(false);
        ConcurrentLinkedQueue[] queues = new ConcurrentLinkedQueue[7];
        for (int i = 0; i < queues.length; ++i) {
            queues[i] = new ConcurrentLinkedQueue();
        }
        OPERATION_QUEUES = queues;
        BATCH_PROCESSOR_THREADS = new Thread[4];
        EnhancedNativeBridge.startBatchProcessors();
    }

    public static class BatchOperation {
        public final byte operationType;
        public final ByteBuffer inputData;
        public final CompletableFuture<byte[]> resultFuture;
        public final long timestamp;
        public final int estimatedSize;

        public BatchOperation(byte operationType, ByteBuffer inputData, CompletableFuture<byte[]> resultFuture) {
            this.operationType = operationType;
            this.inputData = inputData;
            this.resultFuture = resultFuture;
            this.timestamp = System.nanoTime();
            this.estimatedSize = inputData != null ? inputData.remaining() : 0;
        }
    }

    public static class BatchMetrics {
        public final long TOTAL_BATCHES_PROCESSED;
        public final long TOTAL_OPERATIONS_BATCHED;
        public final long AVERAGE_BATCH_SIZE;
        public final int currentQueueDepth;

        public BatchMetrics(long TOTAL_BATCHES_PROCESSED, long TOTAL_OPERATIONS_BATCHED, long AVERAGE_BATCH_SIZE, int currentQueueDepth) {
            this.TOTAL_BATCHES_PROCESSED = TOTAL_BATCHES_PROCESSED;
            this.TOTAL_OPERATIONS_BATCHED = TOTAL_OPERATIONS_BATCHED;
            this.AVERAGE_BATCH_SIZE = AVERAGE_BATCH_SIZE;
            this.currentQueueDepth = currentQueueDepth;
        }
    }

    private static class BatchProcessor
    implements Runnable {
        private final byte operationType;
        private final int processorId;

        public BatchProcessor(byte operationType, int processorId) {
            this.operationType = operationType;
            this.processorId = processorId;
        }

        @Override
        public void run() {
            Thread.currentThread().setName("EnhancedNativeBridge-BatchProcessor-" + this.operationType + "-" + this.processorId);
            while (BATCH_PROCESSOR_RUNNING.get()) {
                try {
                    List<BatchOperation> batch = this.collectBatch();
                    if (!batch.isEmpty()) {
                        EnhancedNativeBridge.processBatch(this.operationType, batch);
                    }
                    if (batch.size() >= EnhancedNativeBridge.minBatchSize()) continue;
                    Thread.sleep(EnhancedNativeBridge.adaptiveBatchTimeoutMs());
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    break;
                }
                catch (Exception e) {
                    System.err.println("Error in batch processor: " + e.getMessage());
                }
            }
        }

        private List<BatchOperation> collectBatch() {
            ArrayList<BatchOperation> batch = new ArrayList<BatchOperation>();
            ConcurrentLinkedQueue<BatchOperation> queue = OPERATION_QUEUES[this.operationType];
            long startTime = System.currentTimeMillis();
            int targetSize = this.calculateOptimalBatchSize();
            while (batch.size() < targetSize && System.currentTimeMillis() - startTime < (long)EnhancedNativeBridge.adaptiveBatchTimeoutMs()) {
                BatchOperation op = queue.poll();
                if (op != null) {
                    batch.add(op);
                    continue;
                }
                if (!batch.isEmpty()) break;
                try {
                    Thread.sleep(1L);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    break;
                }
            }
            return batch;
        }

        private int calculateOptimalBatchSize() {
            long avgSize = AVERAGE_BATCH_SIZE.get();
            long totalOps = TOTAL_OPERATIONS_BATCHED.get();
            long totalBatches = TOTAL_BATCHES_PROCESSED.get();
            if (totalBatches > 0L && totalOps > 0L) {
                double actualAvg = (double)totalOps / (double)totalBatches;
                if (actualAvg > (double)avgSize * 1.2) {
                    return Math.min(EnhancedNativeBridge.maxBatchSize(), (int)((double)avgSize * 1.1));
                }
                if (actualAvg < (double)avgSize * 0.8) {
                    return Math.max(EnhancedNativeBridge.minBatchSize(), (int)((double)avgSize * 0.9));
                }
            }
            return (int)Math.max((long)EnhancedNativeBridge.minBatchSize(), Math.min((long)EnhancedNativeBridge.maxBatchSize(), avgSize));
        }
    }

    public static class BatchResult {
        public final byte operationType;
        public final List<byte[]> results;
        public final long processingTimeNs;

        public BatchResult(byte operationType, List<byte[]> results, long processingTimeNs) {
            this.operationType = operationType;
            this.results = results;
            this.processingTimeNs = processingTimeNs;
        }
    }
}

