/*
 * Decompiled with CFR 0.152.
 */
package com.kneaf.core.chunkstorage.core;

import com.kneaf.core.chunkstorage.cache.ChunkCache;
import com.kneaf.core.chunkstorage.common.ChunkStorageConfig;
import com.kneaf.core.chunkstorage.common.StorageStatisticsProvider;
import com.kneaf.core.chunkstorage.core.ChunkStorageConfigManager;
import com.kneaf.core.chunkstorage.core.ChunkStorageCoordinator;
import com.kneaf.core.chunkstorage.core.ChunkStorageCore;
import com.kneaf.core.chunkstorage.database.DatabaseAdapter;
import com.kneaf.core.chunkstorage.database.InMemoryDatabaseAdapter;
import com.kneaf.core.chunkstorage.database.RustDatabaseAdapter;
import com.kneaf.core.chunkstorage.serialization.ChunkSerializer;
import com.kneaf.core.chunkstorage.serialization.FastNbtSerializer;
import com.kneaf.core.chunkstorage.serialization.NbtChunkSerializer;
import com.kneaf.core.chunkstorage.swap.SwapManager;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChunkStorageManager
implements StorageStatisticsProvider {
    private static final Logger LOGGER = LoggerFactory.getLogger(ChunkStorageManager.class);
    private final String worldName;
    private final ChunkStorageCore core;
    private final ChunkStorageCoordinator coordinator;
    private final ChunkStorageConfigManager configManager;
    private final boolean enabled;
    private volatile boolean shutdown = false;

    public ChunkStorageManager(String worldName, ChunkStorageConfig config) {
        if (worldName == null || worldName.isEmpty()) {
            throw new IllegalArgumentException("World name cannot be null or empty");
        }
        if (config == null) {
            throw new IllegalArgumentException("Config cannot be null");
        }
        this.worldName = worldName;
        this.enabled = config.isEnabled();
        if (this.enabled) {
            this.configManager = new ChunkStorageConfigManager(config);
            this.configManager.validateConfiguration();
            ChunkSerializer serializer = this.createSerializer(config);
            DatabaseAdapter database = this.createDatabaseAdapter(worldName, config);
            ChunkCache cache = this.createCache(config);
            SwapManager swapManager = this.createSwapManager(config);
            this.core = new ChunkStorageCore(worldName, serializer, database, cache, config.isEnabled());
            if (swapManager != null) {
                try {
                    this.core.setSwapManager(swapManager);
                }
                catch (Exception e) {
                    LOGGER.warn("Failed to set SwapManager on ChunkStorageCore: { }", (Object)e.getMessage());
                }
            }
            this.coordinator = swapManager != null ? new ChunkStorageCoordinator(worldName, this.core, swapManager, config.isEnabled()) : null;
            LOGGER.info("Initialized ChunkStorageManager FACADE for world '{ }' with config: { }", (Object)worldName, (Object)this.configManager.getConfigurationSummary());
        } else {
            this.configManager = null;
            this.core = null;
            this.coordinator = null;
            LOGGER.info("ChunkStorageManager FACADE disabled for world '{ }'", (Object)worldName);
        }
    }

    public void clearCache() {
        if (!this.enabled || this.shutdown || this.core == null) {
            return;
        }
        try {
            this.core.clearCache();
        }
        catch (Exception e) {
            LOGGER.error("ChunkStorageManager: Failed to clear cache for world '" + this.worldName + "'", (Throwable)e);
        }
    }

    public void setCacheCapacity(int capacity) {
        if (!this.enabled || this.shutdown || this.core == null) {
            return;
        }
    }

    public void setEvictionPolicy(String policy) {
        if (!this.enabled || this.shutdown || this.core == null) {
            return;
        }
    }

    public CompletableFuture<Void> saveChunk(Object chunk) {
        if (!this.enabled || this.shutdown || this.core == null) {
            return CompletableFuture.completedFuture(null);
        }
        try {
            return this.core.saveChunk(chunk);
        }
        catch (Exception e) {
            LOGGER.error("ChunkStorageManager: Failed to save chunk", (Throwable)e);
            return CompletableFuture.failedFuture(e);
        }
    }

    public Optional<Object> loadChunk(Object level, int chunkX, int chunkZ) {
        if (!this.enabled || this.shutdown || this.core == null) {
            return Optional.empty();
        }
        try {
            return this.core.loadChunk(level, chunkX, chunkZ);
        }
        catch (Exception e) {
            LOGGER.error("ChunkStorageManager: Failed to load chunk at (" + chunkX + ", " + chunkZ + ")", (Throwable)e);
            return Optional.empty();
        }
    }

    public CompletableFuture<Optional<Object>> loadChunkAsync(Object level, int chunkX, int chunkZ) {
        if (!this.enabled || this.shutdown || this.core == null) {
            return CompletableFuture.completedFuture(Optional.empty());
        }
        try {
            return CompletableFuture.supplyAsync(() -> this.core.loadChunk(level, chunkX, chunkZ), Executors.newSingleThreadExecutor());
        }
        catch (Exception e) {
            LOGGER.error("ChunkStorageManager: Failed to load chunk async at (" + chunkX + ", " + chunkZ + ")", (Throwable)e);
            return CompletableFuture.completedFuture(Optional.empty());
        }
    }

    public boolean deleteChunk(Object level, int chunkX, int chunkZ) {
        if (!this.enabled || this.shutdown || this.core == null) {
            return false;
        }
        try {
            return this.core.deleteChunk(level, chunkX, chunkZ);
        }
        catch (Exception e) {
            LOGGER.error("ChunkStorageManager: Failed to delete chunk at (" + chunkX + ", " + chunkZ + ")", (Throwable)e);
            return false;
        }
    }

    @Override
    public Object getStats() {
        if (!this.enabled || this.shutdown || this.core == null) {
            return this.createDisabledStats();
        }
        try {
            return this.coordinator.getStats();
        }
        catch (Exception e) {
            LOGGER.error("ChunkStorageManager: Failed to get storage Stats", (Throwable)e);
            return this.createErrorStats();
        }
    }

    public void performMaintenance() {
        if (!this.enabled || this.shutdown || this.coordinator == null) {
            return;
        }
        try {
            LOGGER.info("Performing storage maintenance for world '{ }' via FACADE", (Object)this.worldName);
            this.core.performMaintenance();
            LOGGER.info("Storage maintenance completed for world '{ }' via FACADE", (Object)this.worldName);
        }
        catch (Exception e) {
            LOGGER.error("ChunkStorageManager: Failed to perform storage maintenance for world '" + this.worldName + "'", (Throwable)e);
        }
    }

    public void shutdown() {
        if (!this.enabled || this.shutdown) {
            return;
        }
        this.shutdown = true;
        try {
            LOGGER.info("Shutting down ChunkStorageManager FACADE for world '{ }'", (Object)this.worldName);
            if (this.core != null) {
                this.core.close();
            }
            LOGGER.info("ChunkStorageManager FACADE shutdown completed for world '{ }'", (Object)this.worldName);
        }
        catch (Exception e) {
            LOGGER.error("ChunkStorageManager: Error during shutdown of ChunkStorageManager FACADE for world '" + this.worldName + "'", (Throwable)e);
        }
    }

    public boolean isHealthy() {
        if (!this.enabled || this.shutdown || this.core == null) {
            return false;
        }
        try {
            return this.core.isHealthy();
        }
        catch (Exception e) {
            LOGGER.error("ChunkStorageManager: Health check failed for world '" + this.worldName + "'", (Throwable)e);
            return false;
        }
    }

    public void createBackup(String backupPath) {
        if (!this.enabled || this.shutdown || this.coordinator == null) {
            return;
        }
        try {
            LOGGER.info("Creating backup for world '{ }' at '{ }' via FACADE", (Object)this.worldName, (Object)backupPath);
            this.core.createBackup(backupPath);
            LOGGER.info("Backup completed for world '{ }' via FACADE", (Object)this.worldName);
        }
        catch (Exception e) {
            LOGGER.error("ChunkStorageManager: Failed to create backup for world '" + this.worldName + "' at '" + backupPath + "'", (Throwable)e);
        }
    }

    public CompletableFuture<Boolean> swapOutChunk(String chunkKey) {
        if (!this.enabled || this.shutdown || this.coordinator == null) {
            return CompletableFuture.completedFuture(false);
        }
        try {
            return this.coordinator != null ? this.coordinator.swapOutChunk(chunkKey) : CompletableFuture.completedFuture(false);
        }
        catch (Exception e) {
            LOGGER.error("ChunkStorageManager: Failed to swap out chunk '" + chunkKey + "'", (Throwable)e);
            return CompletableFuture.completedFuture(false);
        }
    }

    public CompletableFuture<Boolean> swapInChunk(String chunkKey) {
        if (!this.enabled || this.shutdown || this.coordinator == null) {
            return CompletableFuture.completedFuture(false);
        }
        try {
            return this.coordinator != null ? this.coordinator.swapInChunk(chunkKey) : CompletableFuture.completedFuture(false);
        }
        catch (Exception e) {
            LOGGER.error("ChunkStorageManager: Failed to swap in chunk '" + chunkKey + "'", (Throwable)e);
            return CompletableFuture.completedFuture(false);
        }
    }

    public Object getSwapStats() {
        if (!this.enabled || this.shutdown || this.coordinator == null) {
            return this.createEmptySwapStats();
        }
        try {
            Object swapStats = this.coordinator != null ? this.coordinator.getSwapStats() : this.createEmptySwapStats();
            return swapStats;
        }
        catch (Exception e) {
            LOGGER.error("ChunkStorageManager: Failed to get swap Stats", (Throwable)e);
            return this.createEmptySwapStats();
        }
    }

    public Object getMemoryPressureLevel() {
        if (!this.enabled || this.shutdown || this.coordinator == null) {
            return SwapManager.MemoryPressureLevel.NORMAL;
        }
        try {
            return this.coordinator != null ? this.coordinator.getMemoryPressureLevel() : SwapManager.MemoryPressureLevel.NORMAL;
        }
        catch (Exception e) {
            LOGGER.error("ChunkStorageManager: Failed to get memory pressure level", (Throwable)e);
            return SwapManager.MemoryPressureLevel.NORMAL;
        }
    }

    public Object getMemoryUsage() {
        if (!this.enabled || this.shutdown || this.coordinator == null) {
            return this.createEmptyMemoryUsage();
        }
        try {
            return this.coordinator != null ? this.coordinator.getMemoryUsage() : this.createEmptyMemoryUsage();
        }
        catch (Exception e) {
            LOGGER.error("ChunkStorageManager: Failed to get memory usage", (Throwable)e);
            return this.createEmptyMemoryUsage();
        }
    }

    private ChunkSerializer createSerializer(ChunkStorageConfig config) {
        try {
            if (config.isEnableFastNbt() && FastNbtSerializer.isFastNbtAvailable()) {
                LOGGER.info("Using FastNbtSerializer for chunk serialization");
                return new FastNbtSerializer();
            }
            if (NbtChunkSerializer.isMinecraftAvailable()) {
                LOGGER.info("Using NbtChunkSerializer for chunk serialization (FastNBT not available or disabled)");
                return new NbtChunkSerializer();
            }
            LOGGER.warn("Neither FastNBT nor standard NBT classes are available - serializer will be null");
            return null;
        }
        catch (Exception e) {
            LOGGER.warn("Failed to initialize serializer, will use fallback: { }", (Object)e.getMessage());
            if (NbtChunkSerializer.isMinecraftAvailable()) {
                LOGGER.info("Falling back to NbtChunkSerializer after FastNBT initialization failed");
                return new NbtChunkSerializer();
            }
            LOGGER.error("All serialization options failed - serializer will be null");
            return null;
        }
    }

    private DatabaseAdapter createDatabaseAdapter(String worldName, ChunkStorageConfig config) {
        if (config.isUseRustDatabase()) {
            try {
                LOGGER.info("Attempting to create RustDatabaseAdapter with type: { }, checksums: { }", (Object)config.getDatabaseType(), (Object)config.isEnableChecksums());
                return new RustDatabaseAdapter(config.getDatabaseType(), config.isEnableChecksums());
            }
            catch (Exception e) {
                LOGGER.error("Failed to initialize Rust database adapter, falling back to in-memory database", (Throwable)e);
                return new InMemoryDatabaseAdapter("in-memory-" + worldName);
            }
        }
        return new InMemoryDatabaseAdapter("in-memory-" + worldName);
    }

    private ChunkCache createCache(ChunkStorageConfig config) {
        ChunkCache.EvictionPolicy evictionPolicy;
        switch (config.getEvictionPolicy().toLowerCase()) {
            case "lru": {
                evictionPolicy = new ChunkCache.LRUEvictionPolicy();
                break;
            }
            case "distance": {
                evictionPolicy = new ChunkCache.DistanceEvictionPolicy(0, 0);
                break;
            }
            case "hybrid": {
                evictionPolicy = new ChunkCache.HybridEvictionPolicy();
                break;
            }
            default: {
                evictionPolicy = new ChunkCache.LRUEvictionPolicy();
                LOGGER.warn("Unknown eviction policy '{ }', defaulting to LRU", (Object)config.getEvictionPolicy());
            }
        }
        return new ChunkCache(config.getCacheCapacity(), evictionPolicy);
    }

    private SwapManager createSwapManager(ChunkStorageConfig config) {
        if (config.isEnableSwapManager() && config.isUseRustDatabase()) {
            try {
                SwapManager.SwapConfig swapConfig = this.createSwapConfig(config);
                SwapManager swapManager = new SwapManager(swapConfig);
                LOGGER.info("Initialized SwapManager");
                return swapManager;
            }
            catch (Exception e) {
                LOGGER.warn("Failed to initialize SwapManager, disabling swap functionality: { }", (Object)e.getMessage());
                return null;
            }
        }
        LOGGER.info("SwapManager disabled");
        return null;
    }

    private SwapManager.SwapConfig createSwapConfig(ChunkStorageConfig config) {
        SwapManager.SwapConfig swapConfig = new SwapManager.SwapConfig();
        swapConfig.setEnabled(config.isEnableSwapManager());
        swapConfig.setMemoryCheckIntervalMs(config.getSwapMemoryCheckIntervalMs());
        swapConfig.setMaxConcurrentSwaps(config.getMaxConcurrentSwaps());
        swapConfig.setSwapBatchSize(config.getSwapBatchSize());
        swapConfig.setSwapTimeoutMs(config.getSwapTimeoutMs());
        swapConfig.setEnableAutomaticSwapping(config.isEnableAutomaticSwapping());
        swapConfig.setCriticalMemoryThreshold(config.getCriticalMemoryThreshold());
        swapConfig.setHighMemoryThreshold(config.getHighMemoryThreshold());
        swapConfig.setElevatedMemoryThreshold(config.getElevatedMemoryThreshold());
        swapConfig.setMinSwapChunkAgeMs(config.getMinSwapChunkAgeMs());
        swapConfig.setEnableSwapStatistics(config.isEnableSwapStatistics());
        swapConfig.setEnablePerformanceMonitoring(true);
        return swapConfig;
    }

    private StorageStats createDisabledStats() {
        return StorageStats.builder().enabled(false).totalChunksInDb(0L).cachedChunks(0).avgReadLatencyMs(0L).avgWriteLatencyMs(0L).cacheHitRate(0.0).overallHitRate(0.0).status("disabled").swapEnabled(false).memoryPressureLevel("NORMAL").swapOperationsTotal(0L).swapOperationsFailed(0L).build();
    }

    private StorageStats createErrorStats() {
        return StorageStats.builder().enabled(false).totalChunksInDb(0L).cachedChunks(0).avgReadLatencyMs(0L).avgWriteLatencyMs(0L).cacheHitRate(0.0).overallHitRate(0.0).status("error").swapEnabled(false).memoryPressureLevel("NORMAL").swapOperationsTotal(0L).swapOperationsFailed(0L).build();
    }

    private Object createEmptySwapStats() {
        SwapManager.MemoryUsageInfo memInfo = new SwapManager.MemoryUsageInfo(0L, 0L, 0L, 0L, 0.0);
        SwapManager.SwapStatistics swapStatistics = new SwapManager.SwapStatistics();
        return new SwapManager.SwapManagerStats(false, SwapManager.MemoryPressureLevel.NORMAL, 0L, 0L, 0, 0, memInfo, swapStatistics);
    }

    private Object createEmptyMemoryUsage() {
        return new SwapManager.MemoryUsageInfo(0L, 0L, 0L, 0L, 0.0);
    }

    public static class StorageStats {
        private final boolean enabled;
        private final long totalChunksInDb;
        private final int cachedChunks;
        private final long avgReadLatencyMs;
        private final long avgWriteLatencyMs;
        private final double cacheHitRate;
        private final double overallHitRate;
        private final String status;
        private final boolean swapEnabled;
        private final String memoryPressureLevel;
        private final long swapOperationsTotal;
        private final long swapOperationsFailed;

        private StorageStats(Builder builder) {
            this.enabled = builder.enabled;
            this.totalChunksInDb = builder.totalChunksInDb;
            this.cachedChunks = builder.cachedChunks;
            this.avgReadLatencyMs = builder.avgReadLatencyMs;
            this.avgWriteLatencyMs = builder.avgWriteLatencyMs;
            this.cacheHitRate = builder.cacheHitRate;
            this.overallHitRate = builder.overallHitRate;
            this.status = builder.status;
            this.swapEnabled = builder.swapEnabled;
            this.memoryPressureLevel = builder.memoryPressureLevel;
            this.swapOperationsTotal = builder.swapOperationsTotal;
            this.swapOperationsFailed = builder.swapOperationsFailed;
        }

        public static Builder builder() {
            return new Builder();
        }

        public boolean isEnabled() {
            return this.enabled;
        }

        public long getTotalChunksInDb() {
            return this.totalChunksInDb;
        }

        public int getCachedChunks() {
            return this.cachedChunks;
        }

        public long getAvgReadLatencyMs() {
            return this.avgReadLatencyMs;
        }

        public long getAvgWriteLatencyMs() {
            return this.avgWriteLatencyMs;
        }

        public double getCacheHitRate() {
            return this.cacheHitRate;
        }

        public double getOverallHitRate() {
            return this.overallHitRate;
        }

        public String getStatus() {
            return this.status;
        }

        public boolean isSwapEnabled() {
            return this.swapEnabled;
        }

        public String getMemoryPressureLevel() {
            return this.memoryPressureLevel;
        }

        public long getSwapOperationsTotal() {
            return this.swapOperationsTotal;
        }

        public long getSwapOperationsFailed() {
            return this.swapOperationsFailed;
        }

        public String toString() {
            return String.format("StorageStats{enabled=%s, dbChunks=%d, cached=%d, readLatency=%d ms, writeLatency=%d ms, cacheHitRate=%.2f%%, overallHitRate=%.2f%%, status=%s, swapEnabled=%s, pressure=%s, swapOps=%d, swapFailed=%d}", this.enabled, this.totalChunksInDb, this.cachedChunks, this.avgReadLatencyMs, this.avgWriteLatencyMs, this.cacheHitRate * 100.0, this.overallHitRate * 100.0, this.status, this.swapEnabled, this.memoryPressureLevel, this.swapOperationsTotal, this.swapOperationsFailed);
        }

        public static class Builder {
            private boolean enabled;
            private long totalChunksInDb;
            private int cachedChunks;
            private long avgReadLatencyMs;
            private long avgWriteLatencyMs;
            private double cacheHitRate;
            private double overallHitRate;
            private String status;
            private boolean swapEnabled;
            private String memoryPressureLevel;
            private long swapOperationsTotal;
            private long swapOperationsFailed;

            public Builder enabled(boolean enabled) {
                this.enabled = enabled;
                return this;
            }

            public Builder totalChunksInDb(long totalChunksInDb) {
                this.totalChunksInDb = totalChunksInDb;
                return this;
            }

            public Builder cachedChunks(int cachedChunks) {
                this.cachedChunks = cachedChunks;
                return this;
            }

            public Builder avgReadLatencyMs(long avgReadLatencyMs) {
                this.avgReadLatencyMs = avgReadLatencyMs;
                return this;
            }

            public Builder avgWriteLatencyMs(long avgWriteLatencyMs) {
                this.avgWriteLatencyMs = avgWriteLatencyMs;
                return this;
            }

            public Builder cacheHitRate(double cacheHitRate) {
                this.cacheHitRate = cacheHitRate;
                return this;
            }

            public Builder overallHitRate(double overallHitRate) {
                this.overallHitRate = overallHitRate;
                return this;
            }

            public Builder status(String status) {
                this.status = status;
                return this;
            }

            public Builder swapEnabled(boolean swapEnabled) {
                this.swapEnabled = swapEnabled;
                return this;
            }

            public Builder memoryPressureLevel(String memoryPressureLevel) {
                this.memoryPressureLevel = memoryPressureLevel;
                return this;
            }

            public Builder swapOperationsTotal(long swapOperationsTotal) {
                this.swapOperationsTotal = swapOperationsTotal;
                return this;
            }

            public Builder swapOperationsFailed(long swapOperationsFailed) {
                this.swapOperationsFailed = swapOperationsFailed;
                return this;
            }

            public StorageStats build() {
                return new StorageStats(this);
            }
        }
    }
}

