/*
 * Decompiled with CFR 0.152.
 */
package com.kneaf.core.performance.integration;

import com.kneaf.core.data.entity.VillagerData;
import com.kneaf.core.performance.RustPerformance;
import com.kneaf.core.performance.core.PerformanceConstants;
import com.kneaf.core.performance.monitoring.EntityProcessor;
import com.kneaf.core.performance.monitoring.PerformanceConfig;
import com.kneaf.core.performance.monitoring.PerformanceManager;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.entity.npc.Villager;
import net.minecraft.world.entity.npc.VillagerProfession;
import net.minecraft.world.level.ChunkPos;
import net.minecraft.world.level.chunk.LevelChunk;
import net.neoforged.bus.api.SubscribeEvent;
import net.neoforged.fml.common.EventBusSubscriber;
import net.neoforged.neoforge.event.level.ChunkEvent;
import net.neoforged.neoforge.event.server.ServerAboutToStartEvent;
import net.neoforged.neoforge.event.server.ServerStartedEvent;
import net.neoforged.neoforge.event.server.ServerStoppingEvent;
import net.neoforged.neoforge.event.tick.ServerTickEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@EventBusSubscriber(modid="kneafcore")
public class NeoForgeEventIntegration {
    private static final Logger LOGGER = LoggerFactory.getLogger(NeoForgeEventIntegration.class);
    private static final AtomicLong TICK_COUNTER = new AtomicLong(0L);
    private static final AtomicLong TOTAL_TICK_TIME = new AtomicLong(0L);
    private static final AtomicLong MAX_TICK_TIME = new AtomicLong(0L);
    private static final AtomicLong MIN_TICK_TIME = new AtomicLong(Long.MAX_VALUE);
    private static final Map<UUID, PlayerMovementData> PLAYER_MOVEMENT_TRACKER = new ConcurrentHashMap<UUID, PlayerMovementData>();
    private static final Set<ChunkPos> PREDICTED_CHUNKS = ConcurrentHashMap.newKeySet();
    private static final Map<Integer, VillagerProcessingData> VILLAGER_PROCESSING_TRACKER = new ConcurrentHashMap<Integer, VillagerProcessingData>();
    private static boolean optimizationsEnabled = false;
    private static boolean benchmarkMode = false;
    private static long benchmarkStartTime;
    private static long lastMemoryCheck;
    private static EntityProcessor entityProcessor;

    private static long getTargetTickTimeNanos() {
        double avgTps = PerformanceManager.getAverageTPS();
        double msPerTick = 1000.0 / Math.max(0.1, avgTps);
        return (long)(msPerTick * 1000000.0);
    }

    private static long getCriticalTickTimeNanos() {
        return NeoForgeEventIntegration.getTargetTickTimeNanos() * 2L;
    }

    private static int getPredictiveChunkRadius() {
        double tps = PerformanceManager.getAverageTPS();
        return Math.max(1, (int)Math.round(Math.min(6.0, Math.max(1.0, tps / 6.0))));
    }

    private static int getMaxVillagersPerTick() {
        PerformanceConfig cfg = PerformanceConfig.load();
        int base = Math.max(10, cfg.getMaxEntitiesToCollect() / 200);
        double tpsFactor = Math.max(0.5, Math.min(1.5, PerformanceManager.getAverageTPS() / 20.0));
        return Math.max(5, (int)((double)base * tpsFactor));
    }

    @SubscribeEvent
    public static void onServerAboutToStart(ServerAboutToStartEvent event) {
        try {
            LOGGER.info("Initializing KneafCore NeoForge event integration...");
            MinecraftServer server = event.getServer();
            PerformanceConfig config = PerformanceConfig.load();
            optimizationsEnabled = config.isEnabled();
            benchmarkMode = config.isProfilingEnabled();
            if (optimizationsEnabled) {
                LOGGER.info("KneafCore optimizations enabled - targeting { }ms tick time", (Object)(NeoForgeEventIntegration.getTargetTickTimeNanos() / 1000000L));
                NeoForgeEventIntegration.initializePredictiveChunkLoading(server);
                NeoForgeEventIntegration.initializeVillagerProcessing(server);
                NeoForgeEventIntegration.initializeMemoryManagement(server);
                LOGGER.info("KneafCore NeoForge integration initialized successfully");
            } else {
                LOGGER.warn("KneafCore optimizations are disabled in configuration");
            }
        }
        catch (Exception e) {
            LOGGER.error("Failed to initialize KneafCore NeoForge integration", (Throwable)e);
            optimizationsEnabled = false;
        }
    }

    @SubscribeEvent
    public static void onServerStarted(ServerStartedEvent event) {
        if (!optimizationsEnabled) {
            return;
        }
        try {
            MinecraftServer server = event.getServer();
            benchmarkStartTime = System.nanoTime();
            LOGGER.info("KneafCore server started - performance monitoring active");
            NeoForgeEventIntegration.startMemoryManagementThread(server);
            NeoForgeEventIntegration.runInitialBenchmark(server);
        }
        catch (Exception e) {
            LOGGER.error("Error during server started event", (Throwable)e);
        }
    }

    @SubscribeEvent
    public static void onServerTick(ServerTickEvent.Post event) {
        if (!optimizationsEnabled) {
            return;
        }
        long tickStartTime = System.nanoTime();
        try {
            MinecraftServer server = event.getServer();
            CompletableFuture.runAsync(() -> NeoForgeEventIntegration.processPredictiveChunkLoading(server)).exceptionally(ex -> {
                LOGGER.warn("Predictive chunk loading failed asynchronously", ex);
                return null;
            });
            if (entityProcessor == null) {
                entityProcessor = new EntityProcessor();
            }
            entityProcessor.onServerTickAsync(server).exceptionally(ex -> {
                LOGGER.warn("Entity processing failed asynchronously", ex);
                entityProcessor.onServerTick(server);
                return null;
            });
            CompletableFuture.runAsync(() -> NeoForgeEventIntegration.processSIMDOperations(server)).exceptionally(ex -> {
                LOGGER.warn("SIMD operations failed asynchronously", ex);
                return null;
            });
            NeoForgeEventIntegration.updatePerformanceMetrics(server, tickStartTime);
            NeoForgeEventIntegration.checkPerformanceIssues(server, tickStartTime);
        }
        catch (Exception e) {
            LOGGER.error("Error during server tick processing", (Throwable)e);
        }
    }

    @SubscribeEvent
    public static void onChunkLoad(ChunkEvent.Load event) {
        if (!optimizationsEnabled) {
            return;
        }
        if (event.getLevel() instanceof ServerLevel) {
            LevelChunk chunk = (LevelChunk)event.getChunk();
            ChunkPos chunkPos = chunk.getPos();
            PREDICTED_CHUNKS.remove(chunkPos);
        }
    }

    @SubscribeEvent
    public static void onServerStopping(ServerStoppingEvent event) {
        if (!optimizationsEnabled) {
            return;
        }
        try {
            LOGGER.info("KneafCore server stopping - final performance report:");
            NeoForgeEventIntegration.generateFinalReport(event.getServer());
        }
        catch (Exception e) {
            LOGGER.error("Error during server stopping event", (Throwable)e);
        }
    }

    private static void initializePredictiveChunkLoading(MinecraftServer server) {
        LOGGER.info("Initializing predictive chunk loading optimization...");
        for (ServerPlayer player : server.getPlayerList().getPlayers()) {
            NeoForgeEventIntegration.updatePlayerMovementPrediction(player);
        }
    }

    private static void initializeVillagerProcessing(MinecraftServer server) {
        LOGGER.info("Initializing villager processing optimization...");
        for (ServerLevel level : server.getAllLevels()) {
            List villagers = level.getEntitiesOfClass(Villager.class, level.getWorldBorder().getCollisionShape().bounds());
            for (Villager villager : villagers) {
                VILLAGER_PROCESSING_TRACKER.put(villager.getId(), new VillagerProcessingData(villager.getId()));
            }
        }
        LOGGER.info("Prepared optimization for { } villagers", (Object)VILLAGER_PROCESSING_TRACKER.size());
    }

    private static void initializeMemoryManagement(MinecraftServer server) {
        LOGGER.info("Initializing memory management optimization...");
        Runtime runtime = Runtime.getRuntime();
        long maxMemory = runtime.maxMemory();
        long warningThreshold = (long)((double)maxMemory * 0.8);
        long criticalThreshold = (long)((double)maxMemory * 0.9);
        LOGGER.info("Memory thresholds configured - Warning: { }MB, Critical: { }MB", (Object)(warningThreshold / 0x100000L), (Object)(criticalThreshold / 0x100000L));
    }

    private static void processPredictiveChunkLoading(MinecraftServer server) {
        block5: {
            try {
                for (ServerPlayer player : server.getPlayerList().getPlayers()) {
                    NeoForgeEventIntegration.updatePlayerMovementPrediction(player);
                }
                Set<ChunkPos> chunksToGenerate = NeoForgeEventIntegration.getPredictedChunks(server);
                if (chunksToGenerate.isEmpty()) break block5;
                double tps = PerformanceManager.getAverageTPS();
                int adaptiveRadius = PerformanceConstants.getAdaptivePredictiveRadius(tps);
                int adaptiveMax = PerformanceConstants.getAdaptiveMaxPredictiveChunksPerTick(tps);
                ArrayList<ChunkPos> chunkList = new ArrayList<ChunkPos>(chunksToGenerate);
                chunkList.sort(Comparator.comparingInt(p -> p.x).thenComparingInt(p -> p.z));
                int generated = 0;
                HashSet<ChunkPos> attempted = new HashSet<ChunkPos>();
                for (ChunkPos chunkPos : chunkList) {
                    if (generated < adaptiveMax) {
                        int actuallyGenerated;
                        if (Math.abs(chunkPos.x) > adaptiveRadius * 256 || Math.abs(chunkPos.z) > adaptiveRadius * 256 || attempted.contains(chunkPos)) continue;
                        attempted.add(chunkPos);
                        if (RustPerformance.isChunkGenerated(chunkPos.x, chunkPos.z) || (actuallyGenerated = RustPerformance.preGenerateNearbyChunks(chunkPos.x, chunkPos.z, 0)) <= 0) continue;
                        generated += actuallyGenerated;
                        continue;
                    }
                    break;
                }
            }
            catch (Exception e) {
                LOGGER.warn("Predictive chunk loading failed", (Throwable)e);
            }
        }
    }

    private static void processVillagerOptimizations(MinecraftServer server) {
        try {
            PerformanceManager.onServerTick(server);
        }
        catch (Exception e) {
            LOGGER.warn("Villager optimization failed", (Throwable)e);
        }
    }

    private static void processSIMDOperations(MinecraftServer server) {
        try {
            ArrayList<float[]> entityPositions = new ArrayList<float[]>();
            for (ServerLevel level : server.getAllLevels()) {
                for (Entity entity : level.getEntitiesOfClass(Entity.class, level.getWorldBorder().getCollisionShape().bounds())) {
                    if (!(entity instanceof LivingEntity)) continue;
                    entityPositions.add(new float[]{(float)entity.getX(), (float)entity.getY(), (float)entity.getZ()});
                }
            }
            if (!entityPositions.isEmpty()) {
                LOGGER.debug("Processed { } entities with SIMD operations", (Object)entityPositions.size());
            }
        }
        catch (Exception e) {
            LOGGER.warn("SIMD operations failed", (Throwable)e);
        }
    }

    private static void updatePlayerMovementPrediction(ServerPlayer player) {
        PlayerMovementData data = PLAYER_MOVEMENT_TRACKER.computeIfAbsent(player.getUUID(), k -> new PlayerMovementData());
        data.updatePosition(player.getX(), player.getY(), player.getZ());
        data.updateLookAngle(player.getYRot(), player.getXRot());
        List<double[]> predictedPositions = data.predictFuturePositions(5);
        for (double[] pos : predictedPositions) {
            ChunkPos chunkPos = new ChunkPos((int)pos[0] >> 4, (int)pos[2] >> 4);
            PREDICTED_CHUNKS.add(chunkPos);
        }
    }

    private static Set<ChunkPos> getPredictedChunks(MinecraftServer server) {
        HashSet<ChunkPos> chunksToGenerate = new HashSet<ChunkPos>();
        for (ChunkPos predictedPos : PREDICTED_CHUNKS) {
            ServerLevel level = server.overworld();
            if (level.hasChunk(predictedPos.x, predictedPos.z)) continue;
            chunksToGenerate.add(predictedPos);
        }
        int radius = NeoForgeEventIntegration.getPredictiveChunkRadius();
        if (chunksToGenerate.size() > radius * radius) {
            return chunksToGenerate.stream().limit(radius * radius).collect(Collectors.toSet());
        }
        return chunksToGenerate;
    }

    private static boolean shouldOptimizeVillager(Villager villager, VillagerProcessingData data) {
        long currentTime = System.currentTimeMillis();
        return currentTime - data.getLastProcessedTime() > 1000L && data.getErrorCount() < 5;
    }

    private static boolean shouldProcessVillagerThisTick(VillagerProcessingData data) {
        long currentTime = System.currentTimeMillis();
        return currentTime - data.getLastProcessedTime() > 500L && data.getErrorCount() < 3;
    }

    private static VillagerData convertToVillagerData(Villager villager) {
        return new VillagerData(villager.getId(), villager.getX(), villager.getY(), villager.getZ(), 0.0, villager.getVillagerData().getProfession().toString(), villager.getVillagerData().getLevel(), villager.getVillagerData().getProfession() != VillagerProfession.NONE, false, false, 0L, 100, 1);
    }

    private static void updatePerformanceMetrics(MinecraftServer server, long tickStartTime) {
        long tickTime = System.nanoTime() - tickStartTime;
        TICK_COUNTER.incrementAndGet();
        TOTAL_TICK_TIME.addAndGet(tickTime);
        MAX_TICK_TIME.updateAndGet(current -> Math.max(current, tickTime));
        MIN_TICK_TIME.updateAndGet(current -> Math.min(current, tickTime));
        if (TICK_COUNTER.get() % 100L == 0L) {
            long avgTickTime = TOTAL_TICK_TIME.get() / TICK_COUNTER.get();
            String line = String.format("Performance - Avg: %dms, Max: %dms, Min: %dms, TPS: %.2f", avgTickTime / 1000000L, MAX_TICK_TIME.get() / 1000000L, MIN_TICK_TIME.get() == Long.MAX_VALUE ? 0L : MIN_TICK_TIME.get() / 1000000L, NeoForgeEventIntegration.calculateTPS(avgTickTime));
            PerformanceManager.broadcastPerformanceLine(server, line);
        }
    }

    private static void checkPerformanceIssues(MinecraftServer server, long tickStartTime) {
        long tickTime = System.nanoTime() - tickStartTime;
        if (tickTime > NeoForgeEventIntegration.getCriticalTickTimeNanos()) {
            LOGGER.warn("CRITICAL: Tick time { }ms exceeds threshold", (Object)(tickTime / 1000000L));
            NeoForgeEventIntegration.triggerEmergencyOptimizations(server);
        } else if (tickTime > NeoForgeEventIntegration.getTargetTickTimeNanos()) {
            LOGGER.debug("Tick time { }ms above target", (Object)(tickTime / 1000000L));
        }
    }

    private static void triggerEmergencyOptimizations(MinecraftServer server) {
        try {
            System.gc();
            NeoForgeEventIntegration.reduceEntityProcessing(server);
            PREDICTED_CHUNKS.clear();
            LOGGER.info("Emergency optimizations triggered");
        }
        catch (Exception e) {
            LOGGER.error("Emergency optimizations failed", (Throwable)e);
        }
    }

    private static void reduceEntityProcessing(MinecraftServer server) {
        for (ServerLevel level : server.getAllLevels()) {
            for (Entity entity : level.getEntitiesOfClass(Entity.class, level.getWorldBorder().getCollisionShape().bounds())) {
                if (entity instanceof LivingEntity && entity.tickCount % 10 == 0) continue;
            }
        }
    }

    private static void startMemoryManagementThread(MinecraftServer server) {
        Thread memoryThread = new Thread(() -> {
            while (server.isRunning()) {
                try {
                    long currentTime = System.currentTimeMillis();
                    if (currentTime - lastMemoryCheck > 30000L) {
                        NeoForgeEventIntegration.checkMemoryUsage(server);
                        lastMemoryCheck = currentTime;
                    }
                    Thread.sleep(1000L);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    break;
                }
                catch (Exception e) {
                    LOGGER.error("Memory management thread error", (Throwable)e);
                }
            }
        }, "KneafCore-MemoryManager");
        memoryThread.setDaemon(true);
        memoryThread.start();
        LOGGER.info("Memory management thread started");
    }

    private static void checkMemoryUsage(MinecraftServer server) {
        long freeMemory;
        Runtime runtime = Runtime.getRuntime();
        long totalMemory = runtime.totalMemory();
        long usedMemory = totalMemory - (freeMemory = runtime.freeMemory());
        double memoryUsagePercent = (double)usedMemory / (double)totalMemory * 100.0;
        if (memoryUsagePercent > 85.0) {
            NeoForgeEventIntegration.optimizeMemoryUsage(server);
        }
    }

    private static void optimizeMemoryUsage(MinecraftServer server) {
        try {
            PREDICTED_CHUNKS.clear();
            System.gc();
            VILLAGER_PROCESSING_TRACKER.values().forEach(data -> data.reduceProcessingFrequency());
        }
        catch (Exception e) {
            LOGGER.error("Memory optimization failed", (Throwable)e);
        }
    }

    private static void runInitialBenchmark(MinecraftServer server) {
        if (!benchmarkMode) {
            return;
        }
        LOGGER.info("Running initial performance benchmark...");
        try {
            long startTime = System.nanoTime();
            NeoForgeEventIntegration.testChunkProcessing(server);
            NeoForgeEventIntegration.testVillagerProcessing(server);
            NeoForgeEventIntegration.testSIMDOperations();
            long benchmarkTime = System.nanoTime() - startTime;
            LOGGER.info("Initial benchmark completed in { }ms", (Object)(benchmarkTime / 1000000L));
        }
        catch (Exception e) {
            LOGGER.error("Initial benchmark failed", (Throwable)e);
        }
    }

    private static void testChunkProcessing(MinecraftServer server) {
        try {
            ArrayList<ChunkPos> testChunks = new ArrayList<ChunkPos>();
            for (int x = -5; x <= 5; ++x) {
                for (int z = -5; z <= 5; ++z) {
                    testChunks.add(new ChunkPos(x, z));
                }
            }
            LOGGER.info("Chunk processing benchmark completed for { } chunks", (Object)testChunks.size());
        }
        catch (Exception e) {
            LOGGER.warn("Chunk processing benchmark failed", (Throwable)e);
        }
    }

    private static void testVillagerProcessing(MinecraftServer server) {
        try {
            ArrayList<VillagerData> testVillagers = new ArrayList<VillagerData>();
            for (int i = 0; i < 100; ++i) {
                VillagerData data = new VillagerData(i, i * 10, 64.0, i * 10, 0.0, "farmer", 1, true, false, false, 0L, 100, 1);
                testVillagers.add(data);
            }
            LOGGER.info("Villager processing benchmark completed for { } villagers", (Object)testVillagers.size());
        }
        catch (Exception e) {
            LOGGER.warn("Villager processing benchmark failed", (Throwable)e);
        }
    }

    private static void testSIMDOperations() {
        try {
            ArrayList<float[]> testPositions = new ArrayList<float[]>();
            for (int i = 0; i < 1000; ++i) {
                testPositions.add(new float[]{(float)(Math.random() * 1000.0), 64.0f, (float)(Math.random() * 1000.0)});
            }
            LOGGER.info("SIMD operations benchmark completed for { } positions", (Object)testPositions.size());
        }
        catch (Exception e) {
            LOGGER.warn("SIMD operations benchmark failed", (Throwable)e);
        }
    }

    private static void generateFinalReport(MinecraftServer server) {
        try {
            long totalTicks = TICK_COUNTER.get();
            if (totalTicks == 0L) {
                return;
            }
            long avgTickTime = TOTAL_TICK_TIME.get() / totalTicks;
            double avgTPS = NeoForgeEventIntegration.calculateTPS(avgTickTime);
            PerformanceManager.broadcastPerformanceLine(server, "=== KneafCore Final Performance Report ===");
            PerformanceManager.broadcastPerformanceLine(server, String.format("Total Ticks: %d", totalTicks));
            PerformanceManager.broadcastPerformanceLine(server, String.format("Average Tick Time: %dms", avgTickTime / 1000000L));
            PerformanceManager.broadcastPerformanceLine(server, String.format("Maximum Tick Time: %dms", MAX_TICK_TIME.get() / 1000000L));
            PerformanceManager.broadcastPerformanceLine(server, String.format("Minimum Tick Time: %dms", MIN_TICK_TIME.get() == Long.MAX_VALUE ? 0L : MIN_TICK_TIME.get() / 1000000L));
            PerformanceManager.broadcastPerformanceLine(server, String.format("Average TPS: %.1f", avgTPS));
            PerformanceManager.broadcastPerformanceLine(server, String.format("Target Achieved: %s", avgTickTime <= NeoForgeEventIntegration.getTargetTickTimeNanos() ? "YES" : "NO"));
            if (avgTickTime > NeoForgeEventIntegration.getTargetTickTimeNanos()) {
                long improvementNeeded = (avgTickTime - NeoForgeEventIntegration.getTargetTickTimeNanos()) / 1000000L;
                PerformanceManager.broadcastPerformanceLine(server, String.format("Improvement Needed: %dms", improvementNeeded));
            }
            PerformanceManager.broadcastPerformanceLine(server, "=========================================");
        }
        catch (Exception e) {
            LOGGER.error("Failed to generate final report", (Throwable)e);
        }
    }

    private static double calculateTPS(long avgTickTime) {
        return 1000.0 / ((double)avgTickTime / 1000000.0);
    }

    static {
        lastMemoryCheck = 0L;
    }

    private static class VillagerProcessingData {
        private final int villagerId;
        private long lastProcessedTime = 0L;
        private int errorCount = 0;
        private int processingInterval = 500;

        public VillagerProcessingData(int villagerId) {
            this.villagerId = villagerId;
        }

        public long getLastProcessedTime() {
            return this.lastProcessedTime;
        }

        public int getErrorCount() {
            return this.errorCount;
        }

        public void markProcessed() {
            this.lastProcessedTime = System.currentTimeMillis();
            this.errorCount = 0;
        }

        public void incrementErrorCount() {
            ++this.errorCount;
        }

        public void reduceProcessingFrequency() {
            this.processingInterval = Math.min(this.processingInterval * 2, 5000);
        }
    }

    private static class PlayerMovementData {
        private double lastX;
        private double lastY;
        private double lastZ;
        private float lastYaw;
        private float lastPitch;
        private long lastUpdateTime;
        private final List<double[]> movementHistory = new ArrayList<double[]>();

        private PlayerMovementData() {
        }

        public void updatePosition(double x, double y, double z) {
            this.lastX = x;
            this.lastY = y;
            this.lastZ = z;
            this.lastUpdateTime = System.currentTimeMillis();
            this.movementHistory.add(new double[]{x, y, z});
            if (this.movementHistory.size() > 10) {
                this.movementHistory.remove(0);
            }
        }

        public void updateLookAngle(float yaw, float pitch) {
            this.lastYaw = yaw;
            this.lastPitch = pitch;
        }

        public List<double[]> predictFuturePositions(int steps) {
            ArrayList<double[]> predictions = new ArrayList<double[]>();
            if (this.movementHistory.size() < 2) {
                predictions.add(new double[]{this.lastX, this.lastY, this.lastZ});
                return predictions;
            }
            double[] lastPos = this.movementHistory.get(this.movementHistory.size() - 1);
            double[] secondLastPos = this.movementHistory.get(this.movementHistory.size() - 2);
            double dx = lastPos[0] - secondLastPos[0];
            double dy = lastPos[1] - secondLastPos[1];
            double dz = lastPos[2] - secondLastPos[2];
            for (int i = 1; i <= steps; ++i) {
                predictions.add(new double[]{lastPos[0] + dx * (double)i, lastPos[1] + dy * (double)i, lastPos[2] + dz * (double)i});
            }
            return predictions;
        }
    }
}

