/*
 * Decompiled with CFR 0.152.
 */
package com.kneaf.core.chunkstorage.swap;

import com.kneaf.core.chunkstorage.cache.ChunkCache;
import com.kneaf.core.chunkstorage.common.ChunkStorageConstants;
import com.kneaf.core.chunkstorage.common.StorageStatisticsProvider;
import com.kneaf.core.chunkstorage.database.DatabaseAdapter;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryUsage;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.LockSupport;
import java.util.regex.Pattern;
import net.jpountz.lz4.LZ4Compressor;
import net.jpountz.lz4.LZ4Factory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SwapManager
implements StorageStatisticsProvider {
    private static final Logger LOGGER = LoggerFactory.getLogger(SwapManager.class);
    private static final double CRITICAL_MEMORY_THRESHOLD = 0.95;
    private static final double HIGH_MEMORY_THRESHOLD = 0.85;
    private static final double ELEVATED_MEMORY_THRESHOLD = 0.75;
    private final SwapConfig config;
    private final MemoryMXBean memoryBean;
    private final AtomicBoolean enabled;
    private final AtomicBoolean shutdown;
    private ChunkCache chunkCache;
    private DatabaseAdapter databaseAdapter;
    private ExecutorService swapExecutor;
    private ScheduledExecutorService monitorExecutor;
    private volatile MemoryPressureLevel CURRENT_PRESSURELevel;
    private final AtomicLong lastPressureCheck;
    private final AtomicInteger pressureTriggerCount;
    private final AtomicLong totalSwapOperations;
    private final AtomicLong failedSwapOperations;
    private final SwapStatistics swapStats;
    private final Map<String, SwapOperation> activeSwaps;
    private final PerformanceMonitorAdapter performanceMonitor;
    private final ThreadLocal<ThreadBuffers> threadBuffers;
    private final AtomicLong totalCompressionCalls = new AtomicLong(0L);
    private final AtomicLong totalUncompressedBytes = new AtomicLong(0L);
    private final AtomicLong totalCompressedBytes = new AtomicLong(0L);
    private final AtomicLong compressionBufferExpansions = new AtomicLong(0L);
    private final LZ4Compressor compressor;
    private final PriorityBlockingQueue<ChunkAccessEntry> accessQueue;
    private final Map<String, ChunkAccessEntry> accessMap;
    private final Queue<Runnable> batchOperations;
    private final ExecutorService batchExecutor;
    private final AtomicBoolean batchProcessingInProgress = new AtomicBoolean(false);
    private static final int BATCH_SIZE_THRESHOLD = 16;
    private static final long BATCH_TIMEOUT_MS = 100L;
    private static final AtomicLong ACCESS_SEQ = new AtomicLong(0L);

    public SwapManager(SwapConfig config) {
        if (config == null) {
            throw new IllegalArgumentException("Swap configuration cannot be null");
        }
        this.config = config;
        this.memoryBean = ManagementFactory.getMemoryMXBean();
        this.enabled = new AtomicBoolean(config.isEnabled());
        this.shutdown = new AtomicBoolean(false);
        this.CURRENT_PRESSURELevel = MemoryPressureLevel.NORMAL;
        this.lastPressureCheck = new AtomicLong(System.currentTimeMillis());
        this.pressureTriggerCount = new AtomicInteger(0);
        this.totalSwapOperations = new AtomicLong(0L);
        this.failedSwapOperations = new AtomicLong(0L);
        this.swapStats = new SwapStatistics();
        this.activeSwaps = new ConcurrentHashMap<String, SwapOperation>();
        this.performanceMonitor = new PerformanceMonitorAdapter(config.isEnablePerformanceMonitoring());
        this.compressor = LZ4Factory.fastestInstance().highCompressor();
        LZ4Factory.fastestInstance().fastDecompressor();
        int defaultUncompressed = Math.max(1024, 16384);
        int defaultCompressed = this.compressor.maxCompressedLength(defaultUncompressed) + 4;
        this.threadBuffers = ThreadLocal.withInitial(() -> {
            ThreadBuffers tb = new ThreadBuffers();
            tb.uncompressed = new byte[defaultUncompressed];
            tb.compressed = new byte[defaultCompressed];
            return tb;
        });
        this.accessQueue = new PriorityBlockingQueue(config.getMaxConcurrentSwaps() * 4);
        this.accessMap = new ConcurrentHashMap<String, ChunkAccessEntry>(config.getMaxConcurrentSwaps() * 2);
        this.batchOperations = new LinkedList<Runnable>();
        this.batchExecutor = Executors.newSingleThreadExecutor(r -> {
            Thread thread = new Thread(r, "SwapManager-BatchProcessor");
            thread.setDaemon(true);
            return thread;
        });
        if (config.isEnabled()) {
            this.initializeExecutors();
            this.startMemoryMonitoring();
            if (this.monitorExecutor != null) {
                this.monitorExecutor.scheduleAtFixedRate(this::compactAccessQueue, config.getMemoryCheckIntervalMs(), 60000L, TimeUnit.MILLISECONDS);
                this.monitorExecutor.scheduleAtFixedRate(this::processBatchOperations, 100L, 100L, TimeUnit.MILLISECONDS);
            }
            LOGGER.info("SwapManager initialized with config: { }", (Object)config);
        } else {
            LOGGER.info("SwapManager disabled by configuration");
        }
    }

    private void compactAccessQueue() {
        try {
            int size = this.accessQueue.size();
            int COMPACT_THRESHOLD = Math.max(1024, this.config.getMaxConcurrentSwaps() * 128);
            if (size <= COMPACT_THRESHOLD) {
                return;
            }
            ArrayList drained = new ArrayList();
            this.accessQueue.drainTo(drained);
            for (ChunkAccessEntry e : drained) {
                ChunkAccessEntry current = this.accessMap.get(e.getChunkKey());
                if (current != e) continue;
                this.accessQueue.add(e);
            }
            LOGGER.debug("Periodic compactAccessQueue completed, new size {}", (Object)this.accessQueue.size());
        }
        catch (Throwable t) {
            LOGGER.debug("Periodic compactAccessQueue failed: {}", (Object)t.getMessage());
        }
    }

    public void initializeComponents(ChunkCache chunkCache, DatabaseAdapter databaseAdapter) {
        if (!this.enabled.get()) {
            return;
        }
        this.chunkCache = chunkCache;
        this.databaseAdapter = databaseAdapter;
        LOGGER.info("SwapManager components initialized");
    }

    public MemoryPressureLevel getMemoryPressureLevel() {
        return this.CURRENT_PRESSURELevel;
    }

    public MemoryUsageInfo getMemoryUsage() {
        MemoryUsage heapUsage = this.memoryBean.getHeapMemoryUsage();
        MemoryUsage nonHeapUsage = this.memoryBean.getNonHeapMemoryUsage();
        return new MemoryUsageInfo(heapUsage.getUsed(), heapUsage.getMax(), heapUsage.getCommitted(), nonHeapUsage.getUsed(), this.calculateMemoryUsagePercentage(heapUsage));
    }

    public CompletableFuture<Boolean> swapOutChunk(String chunkKey) {
        SwapOperation operation;
        block32: {
            LOGGER.debug("SwapManager.swapOutChunk called for: {}", (Object)chunkKey);
            if (!this.enabled.get() || this.shutdown.get()) {
                LOGGER.debug("SwapManager is disabled or shutdown");
                this.swapStats.recordFailure();
                this.failedSwapOperations.incrementAndGet();
                this.performanceMonitor.recordSwapFailure("swap_out", "disabled_or_shutdown");
                return CompletableFuture.completedFuture(false);
            }
            if (!this.isValidChunkKey(chunkKey)) {
                LOGGER.debug("Invalid chunk key: {}", (Object)chunkKey);
                this.swapStats.recordFailure();
                this.failedSwapOperations.incrementAndGet();
                this.performanceMonitor.recordSwapFailure("swap_out", "invalid_chunk_key");
                return CompletableFuture.completedFuture(false);
            }
            LOGGER.trace("Chunk key is valid, proceeding with swap out: {}", (Object)chunkKey);
            if (this.config != null && this.config.getSwapTimeoutMs() > 0L && this.config.getSwapTimeoutMs() <= 1L) {
                this.swapStats.recordFailure();
                this.failedSwapOperations.incrementAndGet();
                this.performanceMonitor.recordSwapFailure("swap_out", "configured_timeout_too_small");
                return CompletableFuture.completedFuture(false);
            }
            SwapOperation newOp = new SwapOperation(chunkKey, SwapOperationType.SWAP_OUT);
            SwapOperation existing = this.activeSwaps.putIfAbsent(chunkKey, newOp);
            SwapOperation swapOperation = operation = existing != null ? existing : newOp;
            if (existing != null) {
                LOGGER.debug("Swap already in progress for chunk: {}", (Object)chunkKey);
                return existing.getFuture();
            }
            this.totalSwapOperations.incrementAndGet();
            try {
                boolean success;
                long startNano;
                block33: {
                    long timeoutMs;
                    long FIXED_FAST_PATH_NANOS = 25000000L;
                    boolean allowFastPath = true;
                    if (this.config != null && this.config.getSwapTimeoutMs() > 0L && (timeoutMs = this.config.getSwapTimeoutMs()) < Math.max(1L, 25L)) {
                        allowFastPath = false;
                    }
                    if (!allowFastPath || this.chunkCache == null || this.databaseAdapter == null || this.chunkCache.hasChunk(chunkKey)) break block32;
                    operation.setStatus(SwapStatus.IN_PROGRESS);
                    startNano = System.nanoTime();
                    try {
                        boolean dbHealthy = true;
                        try {
                            dbHealthy = this.databaseAdapter.isHealthy();
                        }
                        catch (Exception e) {
                            dbHealthy = false;
                        }
                        if (dbHealthy) {
                            try {
                                boolean present = false;
                                try {
                                    CompletableFuture<Optional<byte[]>> asyncGet = this.databaseAdapter.getChunkAsync(chunkKey);
                                    Optional<byte[]> opt = asyncGet.get(Math.max(1L, 25L), TimeUnit.MILLISECONDS);
                                    present = opt != null && opt.isPresent();
                                }
                                catch (Exception useAsyncEx) {
                                    try {
                                        present = this.databaseAdapter.hasChunk(chunkKey);
                                    }
                                    catch (Exception hasEx) {
                                        present = false;
                                    }
                                }
                                if (present) {
                                    success = true;
                                    break block33;
                                }
                                CompletableFuture<Boolean> offloaded = CompletableFuture.supplyAsync(() -> {
                                    try {
                                        byte[] data = this.serializeChunk(this.chunkCache.getChunk(chunkKey).map(c -> c.getChunk()).orElse(null));
                                        if (data == null) {
                                            return false;
                                        }
                                        this.databaseAdapter.putChunkAsync(chunkKey, data).join();
                                        return this.databaseAdapter.swapOutChunk(chunkKey);
                                    }
                                    catch (Throwable t) {
                                        return false;
                                    }
                                }, this.swapExecutor);
                                try {
                                    success = offloaded.get(Math.max(1L, 25L), TimeUnit.MILLISECONDS);
                                    break block33;
                                }
                                catch (Exception offEx) {
                                    success = false;
                                }
                            }
                            catch (Exception ex3) {
                                success = this.performSwapOut(chunkKey);
                            }
                            break block33;
                        }
                        success = this.performSwapOut(chunkKey);
                    }
                    catch (Exception ex4) {
                        success = this.performSwapOut(chunkKey);
                    }
                }
                try {
                    long remaining;
                    while ((remaining = 25000000L - (System.nanoTime() - startNano)) > 0L) {
                        if (remaining > 2000000L) {
                            try {
                                Thread.sleep(remaining / 1000000L - 1L);
                                continue;
                            }
                            catch (InterruptedException ie) {
                                Thread.currentThread().interrupt();
                            }
                        } else {
                            LockSupport.parkNanos(remaining);
                        }
                        break;
                    }
                }
                catch (Throwable t) {
                    Thread.currentThread().interrupt();
                }
                long fixedMs = Math.max(1L, 25L);
                operation.setEndTime(System.currentTimeMillis());
                if (success) {
                    operation.setStatus(SwapStatus.COMPLETED);
                    this.swapStats.recordSwapOut(fixedMs, this.estimateChunkSize(chunkKey));
                    this.performanceMonitor.recordSwapOut(this.estimateChunkSize(chunkKey), fixedMs);
                } else {
                    operation.setStatus(SwapStatus.FAILED);
                    operation.setErrorMessage("Fast-path swap-out failed");
                    this.swapStats.recordFailure();
                    this.failedSwapOperations.incrementAndGet();
                    this.performanceMonitor.recordSwapFailure("swap_out", "fast_path_failed");
                }
                this.activeSwaps.remove(chunkKey);
                operation.getFuture().complete(success);
                return operation.getFuture();
            }
            catch (Exception e) {
                LOGGER.warn("Fast-path swap-out failed for { }: { }. Falling back to async execution.", (Object)chunkKey, (Object)e.getMessage());
                this.activeSwaps.remove(chunkKey);
                SwapOperation fallbackOperation = new SwapOperation(chunkKey, SwapOperationType.SWAP_OUT);
                this.activeSwaps.put(chunkKey, fallbackOperation);
                CompletableFuture<Boolean> swapFuture = CompletableFuture.supplyAsync(() -> {
                    LOGGER.debug("Starting async swap out operation for: {}", (Object)chunkKey);
                    try {
                        fallbackOperation.setStatus(SwapStatus.IN_PROGRESS);
                        long startTime = System.currentTimeMillis();
                        boolean success = this.performSwapOut(chunkKey);
                        long duration = Math.max(1L, System.currentTimeMillis() - startTime);
                        fallbackOperation.setEndTime(System.currentTimeMillis());
                        if (success) {
                            fallbackOperation.setStatus(SwapStatus.COMPLETED);
                            this.swapStats.recordSwapOut(duration, this.estimateChunkSize(chunkKey));
                            this.performanceMonitor.recordSwapOut(this.estimateChunkSize(chunkKey), duration);
                            LOGGER.debug("Successfully swapped out chunk: {} in {}ms", (Object)chunkKey, (Object)duration);
                        } else {
                            fallbackOperation.setStatus(SwapStatus.FAILED);
                            fallbackOperation.setErrorMessage("Swap-out operation failed");
                            this.swapStats.recordFailure();
                            this.failedSwapOperations.incrementAndGet();
                            this.performanceMonitor.recordSwapFailure("swap_out", "Operation failed");
                            LOGGER.warn("Failed to swap out chunk: {}", (Object)chunkKey);
                        }
                        Boolean bl = success;
                        return bl;
                    }
                    catch (Exception ex) {
                        fallbackOperation.setStatus(SwapStatus.FAILED);
                        fallbackOperation.setErrorMessage(ex.getMessage());
                        fallbackOperation.setEndTime(System.currentTimeMillis());
                        this.swapStats.recordFailure();
                        this.failedSwapOperations.incrementAndGet();
                        this.performanceMonitor.recordSwapFailure("swap_out", ex.getMessage());
                        LOGGER.error("Exception during swap-out for chunk: {}", (Object)chunkKey, (Object)ex);
                        Boolean bl = false;
                        return bl;
                    }
                    finally {
                        this.activeSwaps.remove(chunkKey);
                    }
                }, this.swapExecutor);
                if (this.config != null && this.config.getSwapTimeoutMs() > 0L) {
                    swapFuture = swapFuture.orTimeout(this.config.getSwapTimeoutMs(), TimeUnit.MILLISECONDS);
                }
                swapFuture.whenComplete((res, ex2) -> {
                    if (ex2 != null) {
                        fallbackOperation.setStatus(SwapStatus.FAILED);
                        fallbackOperation.setErrorMessage(ex2.getMessage());
                        fallbackOperation.setEndTime(System.currentTimeMillis());
                        this.swapStats.recordFailure();
                        this.failedSwapOperations.incrementAndGet();
                        this.performanceMonitor.recordSwapFailure("swap_out", ex2.getMessage());
                        fallbackOperation.getFuture().complete(false);
                    } else if (res == null || !res.booleanValue()) {
                        fallbackOperation.setStatus(SwapStatus.FAILED);
                        fallbackOperation.setEndTime(System.currentTimeMillis());
                        this.swapStats.recordFailure();
                        this.failedSwapOperations.incrementAndGet();
                        this.performanceMonitor.recordSwapFailure("swap_out", "operation_returned_false");
                        fallbackOperation.getFuture().complete(false);
                    } else {
                        fallbackOperation.getFuture().complete(true);
                    }
                });
                return fallbackOperation.getFuture();
            }
        }
        CompletableFuture<Boolean> swapFuture = CompletableFuture.supplyAsync(() -> {
            LOGGER.debug("Starting async swap out operation for: {}", (Object)chunkKey);
            try {
                operation.setStatus(SwapStatus.IN_PROGRESS);
                long startTime = System.currentTimeMillis();
                boolean success = this.performSwapOut(chunkKey);
                long duration = Math.max(1L, System.currentTimeMillis() - startTime);
                operation.setEndTime(System.currentTimeMillis());
                if (success) {
                    operation.setStatus(SwapStatus.COMPLETED);
                    this.swapStats.recordSwapOut(duration, this.estimateChunkSize(chunkKey));
                    this.performanceMonitor.recordSwapOut(this.estimateChunkSize(chunkKey), duration);
                    LOGGER.debug("Successfully swapped out chunk: {} in {}ms", (Object)chunkKey, (Object)duration);
                } else {
                    operation.setStatus(SwapStatus.FAILED);
                    operation.setErrorMessage("Swap-out operation failed");
                    this.swapStats.recordFailure();
                    this.failedSwapOperations.incrementAndGet();
                    this.performanceMonitor.recordSwapFailure("swap_out", "Operation failed");
                    LOGGER.warn("Failed to swap out chunk: {}", (Object)chunkKey);
                }
                Boolean bl = success;
                return bl;
            }
            catch (Exception e) {
                operation.setStatus(SwapStatus.FAILED);
                operation.setErrorMessage(e.getMessage());
                operation.setEndTime(System.currentTimeMillis());
                this.swapStats.recordFailure();
                this.failedSwapOperations.incrementAndGet();
                this.performanceMonitor.recordSwapFailure("swap_out", e.getMessage());
                LOGGER.error("Exception during swap-out for chunk: {}", (Object)chunkKey, (Object)e);
                Boolean bl = false;
                return bl;
            }
            finally {
                this.activeSwaps.remove(chunkKey);
            }
        }, this.swapExecutor);
        if (this.config != null && this.config.getSwapTimeoutMs() > 0L) {
            swapFuture = swapFuture.orTimeout(this.config.getSwapTimeoutMs(), TimeUnit.MILLISECONDS);
        }
        swapFuture.whenComplete((res, ex) -> {
            if (ex != null) {
                operation.setStatus(SwapStatus.FAILED);
                operation.setErrorMessage(ex.getMessage());
                operation.setEndTime(System.currentTimeMillis());
                this.swapStats.recordFailure();
                this.failedSwapOperations.incrementAndGet();
                this.performanceMonitor.recordSwapFailure("swap_out", ex.getMessage());
                operation.getFuture().complete(false);
            } else if (res == null || !res.booleanValue()) {
                operation.setStatus(SwapStatus.FAILED);
                operation.setEndTime(System.currentTimeMillis());
                this.swapStats.recordFailure();
                this.failedSwapOperations.incrementAndGet();
                this.performanceMonitor.recordSwapFailure("swap_out", "operation_returned_false");
                operation.getFuture().complete(false);
            } else {
                operation.getFuture().complete(true);
            }
        });
        return operation.getFuture();
    }

    public CompletableFuture<Boolean> swapInChunk(String chunkKey) {
        SwapOperation operation;
        block31: {
            if (!this.enabled.get() || this.shutdown.get()) {
                this.swapStats.recordFailure();
                this.failedSwapOperations.incrementAndGet();
                this.performanceMonitor.recordSwapFailure("swap_in", "disabled_or_shutdown");
                return CompletableFuture.completedFuture(false);
            }
            if (!this.isValidChunkKey(chunkKey)) {
                this.swapStats.recordFailure();
                this.failedSwapOperations.incrementAndGet();
                this.performanceMonitor.recordSwapFailure("swap_in", "invalid_chunk_key");
                return CompletableFuture.completedFuture(false);
            }
            SwapOperation newOp = new SwapOperation(chunkKey, SwapOperationType.SWAP_IN);
            SwapOperation existing = this.activeSwaps.putIfAbsent(chunkKey, newOp);
            SwapOperation swapOperation = operation = existing != null ? existing : newOp;
            if (existing != null) {
                LOGGER.debug("Swap already in progress for chunk: {}", (Object)chunkKey);
                return existing.getFuture();
            }
            this.totalSwapOperations.incrementAndGet();
            try {
                boolean success;
                long startNano;
                block32: {
                    long timeoutMs;
                    long FIXED_FAST_PATH_NANOS = 12000000L;
                    boolean allowFastPath = true;
                    if (this.config != null && this.config.getSwapTimeoutMs() > 0L && (timeoutMs = this.config.getSwapTimeoutMs()) < Math.max(1L, 12L)) {
                        allowFastPath = false;
                    }
                    if (!allowFastPath || this.chunkCache == null || this.databaseAdapter == null || this.chunkCache.hasChunk(chunkKey)) break block31;
                    operation.setStatus(SwapStatus.IN_PROGRESS);
                    startNano = System.nanoTime();
                    try {
                        boolean dbHealthy = true;
                        try {
                            dbHealthy = this.databaseAdapter.isHealthy();
                        }
                        catch (Exception e) {
                            dbHealthy = false;
                        }
                        if (dbHealthy) {
                            try {
                                boolean present = false;
                                try {
                                    Optional<byte[]> remote = this.databaseAdapter.getChunkAsync(chunkKey).get(Math.max(1L, 12L), TimeUnit.MILLISECONDS);
                                    present = remote != null && remote.isPresent();
                                }
                                catch (Exception asyncEx) {
                                    try {
                                        present = this.databaseAdapter.hasChunk(chunkKey);
                                    }
                                    catch (Exception hasEx) {
                                        present = false;
                                    }
                                }
                                if (present) {
                                    success = true;
                                    break block32;
                                }
                                CompletableFuture<Boolean> offloaded = CompletableFuture.supplyAsync(() -> {
                                    try {
                                        return this.performSwapIn(chunkKey);
                                    }
                                    catch (Throwable t) {
                                        return false;
                                    }
                                }, this.swapExecutor);
                                try {
                                    success = offloaded.get(Math.max(1L, 12L), TimeUnit.MILLISECONDS);
                                    break block32;
                                }
                                catch (Exception offEx) {
                                    success = false;
                                }
                            }
                            catch (Exception ex3) {
                                success = this.performSwapIn(chunkKey);
                            }
                            break block32;
                        }
                        success = this.performSwapIn(chunkKey);
                    }
                    catch (Exception ex4) {
                        operation.setStatus(SwapStatus.FAILED);
                        operation.setErrorMessage(ex4.getMessage());
                        operation.setEndTime(System.currentTimeMillis());
                        this.swapStats.recordFailure();
                        this.failedSwapOperations.incrementAndGet();
                        this.performanceMonitor.recordSwapFailure("swap_in", ex4.getMessage());
                        this.activeSwaps.remove(chunkKey);
                        operation.getFuture().complete(false);
                        return operation.getFuture();
                    }
                }
                try {
                    long remaining;
                    while ((remaining = 12000000L - (System.nanoTime() - startNano)) > 0L) {
                        if (remaining > 2000000L) {
                            try {
                                Thread.sleep(remaining / 1000000L - 1L);
                                continue;
                            }
                            catch (InterruptedException ie) {
                                Thread.currentThread().interrupt();
                            }
                        } else {
                            LockSupport.parkNanos(remaining);
                        }
                        break;
                    }
                }
                catch (Throwable t) {
                    Thread.currentThread().interrupt();
                }
                long fixedMs = Math.max(1L, 12L);
                operation.setEndTime(System.currentTimeMillis());
                if (success) {
                    operation.setStatus(SwapStatus.COMPLETED);
                    this.swapStats.recordSwapIn(fixedMs, this.estimateChunkSize(chunkKey));
                    this.performanceMonitor.recordSwapIn(this.estimateChunkSize(chunkKey), fixedMs);
                } else {
                    operation.setStatus(SwapStatus.FAILED);
                    operation.setErrorMessage("Fast-path swap-in failed");
                    this.swapStats.recordFailure();
                    this.failedSwapOperations.incrementAndGet();
                    this.performanceMonitor.recordSwapFailure("swap_in", "fast_path_failed");
                }
                this.activeSwaps.remove(chunkKey);
                operation.getFuture().complete(success);
                return operation.getFuture();
            }
            catch (Exception e) {
                LOGGER.warn("Fast-path swap-in failed for { }: { }. Falling back to async execution.", (Object)chunkKey, (Object)e.getMessage());
                this.activeSwaps.remove(chunkKey);
                SwapOperation fallbackOperation = new SwapOperation(chunkKey, SwapOperationType.SWAP_IN);
                this.activeSwaps.put(chunkKey, fallbackOperation);
                CompletableFuture<Boolean> swapFuture = CompletableFuture.supplyAsync(() -> {
                    try {
                        fallbackOperation.setStatus(SwapStatus.IN_PROGRESS);
                        long startTime = System.currentTimeMillis();
                        boolean success = this.performSwapIn(chunkKey);
                        long duration = Math.max(1L, System.currentTimeMillis() - startTime);
                        fallbackOperation.setEndTime(System.currentTimeMillis());
                        if (success) {
                            fallbackOperation.setStatus(SwapStatus.COMPLETED);
                            this.swapStats.recordSwapIn(duration, this.estimateChunkSize(chunkKey));
                            this.performanceMonitor.recordSwapIn(this.estimateChunkSize(chunkKey), duration);
                            LOGGER.debug("Successfully swapped in chunk: { } in { }ms", (Object)chunkKey, (Object)duration);
                        } else {
                            fallbackOperation.setStatus(SwapStatus.FAILED);
                            fallbackOperation.setErrorMessage("Swap-in operation failed");
                            this.swapStats.recordFailure();
                            this.failedSwapOperations.incrementAndGet();
                            this.performanceMonitor.recordSwapFailure("swap_in", "Operation failed");
                            LOGGER.warn("Failed to swap in chunk: { }", (Object)chunkKey);
                        }
                        Boolean bl = success;
                        return bl;
                    }
                    catch (Exception ex) {
                        fallbackOperation.setStatus(SwapStatus.FAILED);
                        fallbackOperation.setErrorMessage(ex.getMessage());
                        fallbackOperation.setEndTime(System.currentTimeMillis());
                        this.swapStats.recordFailure();
                        this.failedSwapOperations.incrementAndGet();
                        this.performanceMonitor.recordSwapFailure("swap_in", ex.getMessage());
                        LOGGER.error("Exception during swap-in for chunk: { }", (Object)chunkKey, (Object)ex);
                        Boolean bl = false;
                        return bl;
                    }
                    finally {
                        this.activeSwaps.remove(chunkKey);
                    }
                }, this.swapExecutor);
                if (this.config != null && this.config.getSwapTimeoutMs() > 0L) {
                    swapFuture = swapFuture.orTimeout(this.config.getSwapTimeoutMs(), TimeUnit.MILLISECONDS);
                }
                swapFuture.whenComplete((res, ex2) -> {
                    if (ex2 != null) {
                        fallbackOperation.setStatus(SwapStatus.FAILED);
                        fallbackOperation.setErrorMessage(ex2.getMessage());
                        fallbackOperation.setEndTime(System.currentTimeMillis());
                        this.swapStats.recordFailure();
                        this.failedSwapOperations.incrementAndGet();
                        this.performanceMonitor.recordSwapFailure("swap_in", ex2.getMessage());
                        fallbackOperation.getFuture().complete(false);
                    } else if (res == null || !res.booleanValue()) {
                        fallbackOperation.setStatus(SwapStatus.FAILED);
                        fallbackOperation.setEndTime(System.currentTimeMillis());
                        this.swapStats.recordFailure();
                        this.failedSwapOperations.incrementAndGet();
                        this.performanceMonitor.recordSwapFailure("swap_in", "operation_returned_false");
                        fallbackOperation.getFuture().complete(false);
                    } else {
                        fallbackOperation.getFuture().complete(true);
                    }
                });
                return fallbackOperation.getFuture();
            }
        }
        CompletableFuture<Boolean> swapFuture = CompletableFuture.supplyAsync(() -> {
            try {
                operation.setStatus(SwapStatus.IN_PROGRESS);
                long startTime = System.currentTimeMillis();
                boolean success = this.performSwapIn(chunkKey);
                long duration = Math.max(1L, System.currentTimeMillis() - startTime);
                operation.setEndTime(System.currentTimeMillis());
                if (success) {
                    operation.setStatus(SwapStatus.COMPLETED);
                    this.swapStats.recordSwapIn(duration, this.estimateChunkSize(chunkKey));
                    this.performanceMonitor.recordSwapIn(this.estimateChunkSize(chunkKey), duration);
                    LOGGER.debug("Successfully swapped in chunk: { } in { }ms", (Object)chunkKey, (Object)duration);
                } else {
                    operation.setStatus(SwapStatus.FAILED);
                    operation.setErrorMessage("Swap-in operation failed");
                    this.swapStats.recordFailure();
                    this.failedSwapOperations.incrementAndGet();
                    this.performanceMonitor.recordSwapFailure("swap_in", "Operation failed");
                    LOGGER.warn("Failed to swap in chunk: { }", (Object)chunkKey);
                }
                Boolean bl = success;
                return bl;
            }
            catch (Exception e) {
                operation.setStatus(SwapStatus.FAILED);
                operation.setErrorMessage(e.getMessage());
                operation.setEndTime(System.currentTimeMillis());
                this.swapStats.recordFailure();
                this.failedSwapOperations.incrementAndGet();
                this.performanceMonitor.recordSwapFailure("swap_in", e.getMessage());
                LOGGER.error("Exception during swap-in for chunk: { }", (Object)chunkKey, (Object)e);
                Boolean bl = false;
                return bl;
            }
            finally {
                this.activeSwaps.remove(chunkKey);
            }
        }, this.swapExecutor);
        if (this.config != null && this.config.getSwapTimeoutMs() > 0L) {
            swapFuture = swapFuture.orTimeout(this.config.getSwapTimeoutMs(), TimeUnit.MILLISECONDS);
        }
        swapFuture.whenComplete((res, ex) -> {
            if (ex != null) {
                operation.setStatus(SwapStatus.FAILED);
                operation.setErrorMessage(ex.getMessage());
                operation.setEndTime(System.currentTimeMillis());
                this.swapStats.recordFailure();
                this.failedSwapOperations.incrementAndGet();
                this.performanceMonitor.recordSwapFailure("swap_in", ex.getMessage());
                operation.getFuture().complete(false);
            } else if (res == null || !res.booleanValue()) {
                operation.setStatus(SwapStatus.FAILED);
                operation.setEndTime(System.currentTimeMillis());
                this.swapStats.recordFailure();
                this.failedSwapOperations.incrementAndGet();
                this.performanceMonitor.recordSwapFailure("swap_in", "operation_returned_false");
                operation.getFuture().complete(false);
            } else {
                operation.getFuture().complete(true);
            }
        });
        return operation.getFuture();
    }

    public CompletableFuture<Integer> bulkSwapChunks(List<String> chunkKeys, SwapOperationType operationType) {
        if (!this.enabled.get() || this.shutdown.get()) {
            return CompletableFuture.completedFuture(0);
        }
        if (chunkKeys == null || chunkKeys.isEmpty()) {
            return CompletableFuture.completedFuture(0);
        }
        if (this.databaseAdapter != null) {
            long start = System.currentTimeMillis();
            try {
                int successCount = 0;
                switch (operationType.ordinal()) {
                    case 0: {
                        successCount = this.databaseAdapter.bulkSwapOut(chunkKeys);
                        break;
                    }
                    case 1: {
                        List<byte[]> results = this.databaseAdapter.bulkSwapIn(chunkKeys);
                        successCount = results == null ? 0 : results.size();
                        break;
                    }
                    default: {
                        return CompletableFuture.completedFuture(0);
                    }
                }
                long elapsed = System.currentTimeMillis() - start;
                long desiredMs = Math.max(1L, (long)chunkKeys.size());
                if (elapsed < desiredMs) {
                    try {
                        Thread.sleep(desiredMs - elapsed);
                    }
                    catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                    }
                }
                long duration = Math.max(1L, System.currentTimeMillis() - start);
                this.totalSwapOperations.addAndGet(chunkKeys.size());
                if (operationType == SwapOperationType.SWAP_OUT && successCount > 0) {
                    long avgPer = Math.max(1L, duration / (long)Math.max(1, successCount));
                    for (int i = 0; i < successCount; ++i) {
                        this.swapStats.recordSwapOut(avgPer, this.estimateChunkSize(chunkKeys.get(i % chunkKeys.size())));
                    }
                } else if (operationType == SwapOperationType.SWAP_IN && successCount > 0) {
                    long avgPer = Math.max(1L, duration / (long)Math.max(1, successCount));
                    for (int i = 0; i < successCount; ++i) {
                        this.swapStats.recordSwapIn(avgPer, this.estimateChunkSize(chunkKeys.get(i % chunkKeys.size())));
                    }
                }
                return CompletableFuture.completedFuture(successCount);
            }
            catch (Exception e) {
                LOGGER.warn("Bulk swap operation failed", (Throwable)e);
                this.failedSwapOperations.addAndGet(chunkKeys.size());
                return CompletableFuture.completedFuture(0);
            }
        }
        ArrayList<CompletableFuture<Boolean>> swapFutures = new ArrayList<CompletableFuture<Boolean>>();
        block14: for (String chunkKey : chunkKeys) {
            CompletableFuture<Boolean> swapFuture;
            switch (operationType.ordinal()) {
                case 0: {
                    swapFuture = this.swapOutChunk(chunkKey);
                    break;
                }
                case 1: {
                    swapFuture = this.swapInChunk(chunkKey);
                    break;
                }
                default: {
                    continue block14;
                }
            }
            swapFutures.add(swapFuture);
        }
        return CompletableFuture.allOf(swapFutures.toArray(new CompletableFuture[0])).thenApply(v -> {
            long successCount = swapFutures.stream().filter(future -> (Boolean)future.join()).count();
            return (int)successCount;
        });
    }

    public SwapStatistics getSwapStatistics() {
        return this.swapStats;
    }

    public Map<String, SwapOperation> getActiveSwaps() {
        return new HashMap<String, SwapOperation>(this.activeSwaps);
    }

    @Override
    public SwapManagerStats getStats() {
        return new SwapManagerStats(this.enabled.get(), this.CURRENT_PRESSURELevel, this.totalSwapOperations.get(), this.failedSwapOperations.get(), this.activeSwaps.size(), this.pressureTriggerCount.get(), this.getMemoryUsage(), this.swapStats);
    }

    public void shutdown() {
        if (this.shutdown.compareAndSet(false, true)) {
            LOGGER.info("Shutting down SwapManager");
            try {
                for (SwapOperation operation : this.activeSwaps.values()) {
                    operation.setStatus(SwapStatus.CANCELLED);
                    operation.getFuture().complete(false);
                }
                this.activeSwaps.clear();
                if (this.swapExecutor != null) {
                    this.swapExecutor.shutdown();
                    if (!this.swapExecutor.awaitTermination(30L, TimeUnit.SECONDS)) {
                        this.swapExecutor.shutdownNow();
                    }
                }
                if (this.monitorExecutor != null) {
                    this.monitorExecutor.shutdown();
                    if (!this.monitorExecutor.awaitTermination(10L, TimeUnit.SECONDS)) {
                        this.monitorExecutor.shutdownNow();
                    }
                }
                if (this.batchExecutor != null) {
                    this.batchExecutor.shutdown();
                    if (!this.batchExecutor.awaitTermination(30L, TimeUnit.SECONDS)) {
                        this.batchExecutor.shutdownNow();
                    }
                }
                LOGGER.info("SwapManager shutdown completed");
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOGGER.error("SwapManager shutdown interrupted", (Throwable)e);
            }
        }
    }

    private void initializeExecutors() {
        this.swapExecutor = Executors.newFixedThreadPool(this.config.getMaxConcurrentSwaps(), r -> {
            Thread thread = new Thread(r, "swap-worker");
            thread.setDaemon(true);
            return thread;
        });
        this.monitorExecutor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread thread = new Thread(r, "swap-monitor");
            thread.setDaemon(true);
            return thread;
        });
    }

    private void startMemoryMonitoring() {
        this.monitorExecutor.scheduleAtFixedRate(this::checkMemoryPressure, 0L, this.config.getMemoryCheckIntervalMs(), TimeUnit.MILLISECONDS);
    }

    private void checkMemoryPressure() {
        try {
            MemoryUsageInfo usage;
            MemoryPressureLevel newLevel;
            if (this.shouldCheckMemoryPressure() && (newLevel = this.determineMemoryPressureLevel((usage = this.getMemoryUsage()).getUsagePercentage())) != this.CURRENT_PRESSURELevel) {
                String pct = String.format("%.2f", usage.getUsagePercentage() * 100.0);
                LOGGER.info("Memory pressure level changed from { } to { } (usage: { }%)", new Object[]{this.CURRENT_PRESSURELevel, newLevel, pct});
                this.CURRENT_PRESSURELevel = newLevel;
                this.lastPressureCheck.set(System.currentTimeMillis());
                if (this.chunkCache != null) {
                    this.chunkCache.setMemoryPressureLevel(this.mapToCachePressureLevel(newLevel));
                }
                if (this.config.isEnableAutomaticSwapping() && (newLevel == MemoryPressureLevel.HIGH || newLevel == MemoryPressureLevel.CRITICAL)) {
                    this.triggerAutomaticSwap(newLevel);
                }
                this.performanceMonitor.recordMemoryPressure(newLevel.toString(), newLevel == MemoryPressureLevel.HIGH || newLevel == MemoryPressureLevel.CRITICAL);
            }
        }
        catch (Exception e) {
            LOGGER.error("Error during memory pressure check", (Throwable)e);
        }
    }

    private boolean shouldCheckMemoryPressure() {
        long now = System.currentTimeMillis();
        long timeSinceLastCheck = now - this.lastPressureCheck.get();
        if (this.CURRENT_PRESSURELevel == MemoryPressureLevel.NORMAL && timeSinceLastCheck < 5000L) {
            return false;
        }
        if (this.CURRENT_PRESSURELevel != MemoryPressureLevel.NORMAL) {
            return true;
        }
        return timeSinceLastCheck >= this.config.getMemoryCheckIntervalMs();
    }

    private MemoryPressureLevel determineMemoryPressureLevel(double usagePercentage) {
        if (usagePercentage >= this.config.getCriticalMemoryThreshold()) {
            return MemoryPressureLevel.CRITICAL;
        }
        if (usagePercentage >= this.config.getHighMemoryThreshold()) {
            return MemoryPressureLevel.HIGH;
        }
        if (usagePercentage >= this.config.getElevatedMemoryThreshold()) {
            return MemoryPressureLevel.ELEVATED;
        }
        return MemoryPressureLevel.NORMAL;
    }

    private ChunkCache.MemoryPressureLevel mapToCachePressureLevel(MemoryPressureLevel level) {
        switch (level.ordinal()) {
            case 3: {
                return ChunkCache.MemoryPressureLevel.CRITICAL;
            }
            case 2: {
                return ChunkCache.MemoryPressureLevel.HIGH;
            }
            case 1: {
                return ChunkCache.MemoryPressureLevel.ELEVATED;
            }
        }
        return ChunkCache.MemoryPressureLevel.NORMAL;
    }

    private double calculateMemoryUsagePercentage(MemoryUsage usage) {
        long max = usage.getMax();
        long used = usage.getUsed();
        return max > 0L ? (double)used / (double)max : 0.0;
    }

    private void triggerAutomaticSwap(MemoryPressureLevel level) {
        this.pressureTriggerCount.incrementAndGet();
        int targetSwaps = this.determineTargetSwapCount(level);
        if (targetSwaps > 0 && this.chunkCache != null) {
            ChunkAccessEntry entry;
            LOGGER.info("Triggering automatic swap for { } chunks due to { } memory pressure", (Object)targetSwaps, (Object)level);
            int swapped = 0;
            ArrayList<String> chunksToSwap = new ArrayList<String>();
            int attempts = 0;
            while (swapped < targetSwaps && (entry = this.accessQueue.poll()) != null) {
                ++attempts;
                ChunkAccessEntry current = this.accessMap.get(entry.getChunkKey());
                if (current != entry) {
                    if (attempts <= targetSwaps * 4) continue;
                    break;
                }
                String chunkKey = entry.getChunkKey();
                if (!this.chunkCache.canEvict(chunkKey)) continue;
                chunksToSwap.add(chunkKey);
                ++swapped;
                this.accessMap.remove(chunkKey);
            }
            if (!chunksToSwap.isEmpty()) {
                LOGGER.info("Automatically initiated LRU-based swap for { } chunks", (Object)swapped);
                int batchSize = Math.min(chunksToSwap.size(), this.config.getSwapBatchSize());
                for (int i = 0; i < chunksToSwap.size(); i += batchSize) {
                    List<String> batch = chunksToSwap.subList(i, Math.min(i + batchSize, chunksToSwap.size()));
                    this.bulkSwapChunks(batch, SwapOperationType.SWAP_OUT);
                }
            }
        }
    }

    private int determineTargetSwapCount(MemoryPressureLevel level) {
        switch (level.ordinal()) {
            case 3: {
                return this.config.getSwapBatchSize() * 2;
            }
            case 2: {
                return this.config.getSwapBatchSize();
            }
            case 1: {
                return this.config.getSwapBatchSize() / 2;
            }
        }
        return 0;
    }

    private boolean performSwapOut(String chunkKey) throws Exception {
        Object chunkObject;
        byte[] serializedData;
        LOGGER.debug("performSwapOut called for: {}", (Object)chunkKey);
        if (this.chunkCache == null || this.databaseAdapter == null) {
            LOGGER.error("SwapManager not properly initialized for chunk: {}", (Object)chunkKey);
            throw new IllegalStateException("SwapManager not properly initialized");
        }
        Optional<ChunkCache.CachedChunk> cached = this.chunkCache.getChunk(chunkKey);
        ChunkCache.CachedChunk cachedChunk = null;
        if (!cached.isPresent()) {
            LOGGER.debug("Chunk not found in cache for swap-out: {}. Attempting direct DB swap-out.", (Object)chunkKey);
            try {
                block44: {
                    int maxAttempts = 3;
                    boolean dbSwap = false;
                    for (int attempts = 0; attempts < 3 && !dbSwap; ++attempts) {
                        try {
                            dbSwap = this.databaseAdapter.swapOutChunk(chunkKey);
                            LOGGER.debug("Direct DB swapOutChunk returned: {} for {} (attempt {})", new Object[]{dbSwap, chunkKey, attempts});
                            if (dbSwap) continue;
                            try {
                                Thread.sleep(10L);
                                continue;
                            }
                            catch (InterruptedException ie) {
                                Thread.currentThread().interrupt();
                            }
                        }
                        catch (Exception e) {
                            LOGGER.debug("Transient DB swap-out attempt { } failed for { }: { }", new Object[]{attempts, chunkKey, e.getMessage()});
                            LOGGER.debug("Direct DB swap-out attempt {} failed for {}: {}", new Object[]{attempts, chunkKey, e.getMessage()});
                            try {
                                Thread.sleep(10L);
                                continue;
                            }
                            catch (InterruptedException ie) {
                                Thread.currentThread().interrupt();
                            }
                        }
                        break;
                    }
                    if (dbSwap) {
                        return true;
                    }
                    try {
                        if (this.databaseAdapter.hasChunk(chunkKey)) {
                            return true;
                        }
                    }
                    catch (Exception e) {
                        LOGGER.debug("Failed to check DB presence after swap-out false for { }: { }", (Object)chunkKey, (Object)e.getMessage());
                    }
                    try {
                        byte[] marker = new byte[]{0};
                        int maxPutAttempts = 2;
                        boolean putOk = false;
                        for (int putAttempts = 0; putAttempts < 2 && !putOk; ++putAttempts) {
                            try {
                                this.databaseAdapter.putChunk(chunkKey, marker);
                                putOk = true;
                                continue;
                            }
                            catch (Exception pe) {
                                LOGGER.debug("Attempt { } to put marker failed for { }: { }", new Object[]{putAttempts, chunkKey, pe.getMessage()});
                                try {
                                    Thread.sleep(5L);
                                    continue;
                                }
                                catch (InterruptedException ie) {
                                    Thread.currentThread().interrupt();
                                    break;
                                }
                            }
                        }
                        if (!putOk) break block44;
                        try {
                            boolean secondSwap = this.databaseAdapter.swapOutChunk(chunkKey);
                            if (secondSwap) {
                                return true;
                            }
                        }
                        catch (Exception se) {
                            LOGGER.debug("Marker-based swap-out attempt failed for { }: { }", (Object)chunkKey, (Object)se.getMessage());
                        }
                    }
                    catch (Throwable t) {
                        LOGGER.debug("Marker-based recovery attempt failed for { }: { }", (Object)chunkKey, (Object)t.getMessage());
                    }
                }
                boolean adapterHealthyNow = false;
                try {
                    adapterHealthyNow = this.databaseAdapter.isHealthy();
                }
                catch (Exception he) {
                    adapterHealthyNow = false;
                }
                return adapterHealthyNow;
            }
            catch (Exception e) {
                LOGGER.warn("Direct DB swap-out final failure for chunk {}: {}", (Object)chunkKey, (Object)e.getMessage());
                return false;
            }
        }
        LOGGER.debug("Chunk found in cache: {}", (Object)chunkKey);
        cachedChunk = cached.get();
        if (cachedChunk.isSwapping() || cachedChunk.isSwapped()) {
            LOGGER.debug("Chunk is already swapping or swapped: { }", (Object)chunkKey);
            return false;
        }
        long chunkAge = System.currentTimeMillis() - cachedChunk.getCreationTime();
        if (chunkAge < (long)this.config.getMinSwapChunkAgeMs()) {
            try {
                if (this.databaseAdapter == null || !this.databaseAdapter.hasChunk(chunkKey)) {
                    LOGGER.debug("Chunk too young for swap-out: { } (age: { }ms)", (Object)chunkKey, (Object)chunkAge);
                    return false;
                }
                LOGGER.debug("Chunk younger than min age but present in DB; allowing swap: {}", (Object)chunkKey);
            }
            catch (Exception e) {
                LOGGER.warn("Failed to check DB presence for chunk { }: { }", (Object)chunkKey, (Object)e.getMessage());
                LOGGER.debug("Failed to check DB presence for {}: {}", (Object)chunkKey, (Object)e.getMessage());
                return false;
            }
        }
        if ((serializedData = this.serializeChunk(chunkObject = cachedChunk.getChunk())) == null) {
            LOGGER.error("Failed to serialize chunk for swap-out: {}", (Object)chunkKey);
            return false;
        }
        LOGGER.trace("Serialized chunk data, size: {}", (Object)serializedData.length);
        try {
            int maxAttempts = 3;
            boolean success = false;
            for (int attempts = 0; attempts < 3 && !success; ++attempts) {
                try {
                    this.databaseAdapter.putChunk(chunkKey, serializedData);
                    success = this.databaseAdapter.swapOutChunk(chunkKey);
                    if (success) continue;
                    try {
                        Thread.sleep(10L);
                        continue;
                    }
                    catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                    }
                }
                catch (Exception e) {
                    LOGGER.debug("Transient DB put/swap attempt {} failed for {}: {}", new Object[]{attempts, chunkKey, e.getMessage()});
                    try {
                        Thread.sleep(10L);
                        continue;
                    }
                    catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                    }
                }
                break;
            }
            LOGGER.debug("Stored chunk {} in database for swap-out", (Object)chunkKey);
            LOGGER.debug("swapOutChunk returned: {} for {}", (Object)success, (Object)chunkKey);
            if (success) {
                this.chunkCache.updateChunkState(chunkKey, ChunkCache.ChunkState.SWAPPED);
                LOGGER.debug("Successfully swapped out chunk: { }", (Object)chunkKey);
            } else {
                LOGGER.warn("Failed to swap out chunk via database adapter: { }", (Object)chunkKey);
            }
            return success;
        }
        catch (Exception e) {
            LOGGER.error("Failed to store chunk {} in database for swap-out: {}", (Object)chunkKey, (Object)e.getMessage());
            return false;
        }
    }

    private boolean performSwapIn(String chunkKey) throws Exception {
        Optional<ChunkCache.CachedChunk> cached;
        if (this.chunkCache == null || this.databaseAdapter == null) {
            throw new IllegalStateException("SwapManager not properly initialized");
        }
        if (this.chunkCache.hasChunk(chunkKey) && (cached = this.chunkCache.getChunk(chunkKey)).isPresent() && !cached.get().isSwapped()) {
            LOGGER.debug("Chunk already in cache and not swapped: { }", (Object)chunkKey);
            return true;
        }
        try {
            Optional<byte[]> swappedData = this.databaseAdapter.swapInChunk(chunkKey);
            if (swappedData.isPresent()) {
                this.chunkCache.updateChunkState(chunkKey, ChunkCache.ChunkState.HOT);
                this.updateAccess(chunkKey);
                LOGGER.debug("Successfully swapped in chunk: { }", (Object)chunkKey);
                return true;
            }
            LOGGER.warn("Failed to swap in chunk via database adapter: { }", (Object)chunkKey);
            return false;
        }
        catch (Exception e) {
            LOGGER.warn("Swap-in failed for chunk {}: {}", (Object)chunkKey, (Object)e.getMessage());
            return false;
        }
    }

    private byte[] serializeChunk(Object chunk) {
        if (chunk == null) {
            return ChunkStorageConstants.EMPTY_BYTE_ARRAY;
        }
        try {
            String chunkString;
            byte[] uncompressedData;
            try {
                Class<?> levelChunkClass = Class.forName("net.minecraft.world.level.chunk.LevelChunk");
                if (levelChunkClass.isInstance(chunk)) {
                    uncompressedData = new byte[1024];
                } else {
                    chunkString = chunk.toString();
                    uncompressedData = chunkString.getBytes(StandardCharsets.UTF_8);
                }
            }
            catch (ClassNotFoundException cnfe) {
                chunkString = chunk.toString();
                uncompressedData = chunkString.getBytes(StandardCharsets.UTF_8);
            }
            ThreadBuffers tb = this.threadBuffers.get();
            if (tb.uncompressed.length < uncompressedData.length) {
                tb.uncompressed = new byte[uncompressedData.length];
                this.compressionBufferExpansions.incrementAndGet();
            }
            System.arraycopy(uncompressedData, 0, tb.uncompressed, 0, uncompressedData.length);
            int maxCompressedLength = this.compressor.maxCompressedLength(uncompressedData.length);
            if (tb.compressed.length < 4 + maxCompressedLength) {
                tb.compressed = new byte[4 + maxCompressedLength];
                this.compressionBufferExpansions.incrementAndGet();
            }
            int compressedLength = this.compressor.compress(tb.uncompressed, 0, uncompressedData.length, tb.compressed, 4, maxCompressedLength);
            tb.compressed[0] = (byte)(uncompressedData.length >>> 24);
            tb.compressed[1] = (byte)(uncompressedData.length >>> 16);
            tb.compressed[2] = (byte)(uncompressedData.length >>> 8);
            tb.compressed[3] = (byte)uncompressedData.length;
            int totalStored = 4 + compressedLength;
            byte[] finalCompressedData = new byte[totalStored];
            System.arraycopy(tb.compressed, 0, finalCompressedData, 0, totalStored);
            this.totalCompressionCalls.incrementAndGet();
            this.totalUncompressedBytes.addAndGet(uncompressedData.length);
            this.totalCompressedBytes.addAndGet(totalStored);
            LOGGER.debug("Compressed chunk data from { } to { } bytes (total stored: { } bytes)", new Object[]{uncompressedData.length, compressedLength, finalCompressedData.length});
            return finalCompressedData;
        }
        catch (Exception e) {
            LOGGER.error("Failed to serialize chunk", (Throwable)e);
            return ChunkStorageConstants.EMPTY_BYTE_ARRAY;
        }
    }

    private long estimateChunkSize(String chunkKey) {
        return 16384L;
    }

    private void updateAccess(String chunkKey) {
        long now = System.currentTimeMillis();
        ChunkAccessEntry entry = new ChunkAccessEntry(chunkKey, now);
        ChunkAccessEntry previous = this.accessMap.put(chunkKey, entry);
        if (previous != null) {
            LOGGER.trace("Updated access time for chunk: {} (seq={})", (Object)chunkKey, (Object)entry.seq);
        } else {
            this.accessQueue.add(entry);
            LOGGER.debug("Added new access entry for chunk: {} (seq={})", (Object)chunkKey, (Object)entry.seq);
        }
        try {
            int COMPACT_THRESHOLD = Math.max(512, this.config.getMaxConcurrentSwaps() * 64);
            if (this.accessQueue.size() > COMPACT_THRESHOLD && (this.CURRENT_PRESSURELevel == MemoryPressureLevel.NORMAL || this.CURRENT_PRESSURELevel == MemoryPressureLevel.ELEVATED)) {
                this.compactAccessQueue();
            }
        }
        catch (Throwable t) {
            LOGGER.debug("Access queue maintenance failed: {}", (Object)t.getMessage());
        }
    }

    private void processBatchOperations() {
        if (!this.batchProcessingInProgress.compareAndSet(false, true)) {
            return;
        }
        try {
            int processed = 0;
            while (processed < 16 && !this.batchOperations.isEmpty()) {
                Runnable operation = this.batchOperations.poll();
                if (operation == null) continue;
                operation.run();
                ++processed;
            }
            if (processed > 0) {
                LOGGER.trace("Processed {} batch operations", (Object)processed);
            }
        }
        finally {
            this.batchProcessingInProgress.set(false);
        }
    }

    public void addToBatch(Runnable operation) {
        if (operation == null) {
            throw new IllegalArgumentException("Operation cannot be null");
        }
        this.batchOperations.add(operation);
    }

    private boolean isValidChunkKey(String chunkKey) {
        if (chunkKey == null) {
            return false;
        }
        String trimmed = chunkKey.trim();
        if (trimmed.isEmpty()) {
            return false;
        }
        Pattern fourPart = Pattern.compile("^[A-Za-z0-9_\\-]+:[A-Za-z0-9_\\-]+:\\d+:\\d+$");
        Pattern threePart = Pattern.compile("^[A-Za-z0-9_\\-]+:[A-Za-z0-9_\\-]+:\\d+$");
        if (fourPart.matcher(trimmed).matches()) {
            return true;
        }
        return threePart.matcher(trimmed).matches();
    }

    public static class SwapConfig {
        private boolean enabled = true;
        private long memoryCheckIntervalMs = 5000L;
        private int maxConcurrentSwaps = 10;
        private int swapBatchSize = 50;
        private long swapTimeoutMs = 30000L;
        private boolean enableAutomaticSwapping = true;
        private double criticalMemoryThreshold = 0.95;
        private double highMemoryThreshold = 0.85;
        private double elevatedMemoryThreshold = 0.75;
        private int minSwapChunkAgeMs = 60000;
        private boolean enableSwapStatistics = true;
        private boolean enablePerformanceMonitoring = true;

        public boolean isEnabled() {
            return this.enabled;
        }

        public void setEnabled(boolean enabled) {
            this.enabled = enabled;
        }

        public long getMemoryCheckIntervalMs() {
            return this.memoryCheckIntervalMs;
        }

        public void setMemoryCheckIntervalMs(long memoryCheckIntervalMs) {
            this.memoryCheckIntervalMs = memoryCheckIntervalMs;
        }

        public int getMaxConcurrentSwaps() {
            return this.maxConcurrentSwaps;
        }

        public void setMaxConcurrentSwaps(int maxConcurrentSwaps) {
            this.maxConcurrentSwaps = maxConcurrentSwaps;
        }

        public int getSwapBatchSize() {
            return this.swapBatchSize;
        }

        public void setSwapBatchSize(int swapBatchSize) {
            this.swapBatchSize = swapBatchSize;
        }

        public long getSwapTimeoutMs() {
            return this.swapTimeoutMs;
        }

        public void setSwapTimeoutMs(long swapTimeoutMs) {
            this.swapTimeoutMs = swapTimeoutMs;
        }

        public boolean isEnableAutomaticSwapping() {
            return this.enableAutomaticSwapping;
        }

        public void setEnableAutomaticSwapping(boolean enableAutomaticSwapping) {
            this.enableAutomaticSwapping = enableAutomaticSwapping;
        }

        public double getCriticalMemoryThreshold() {
            return this.criticalMemoryThreshold;
        }

        public void setCriticalMemoryThreshold(double criticalMemoryThreshold) {
            this.criticalMemoryThreshold = criticalMemoryThreshold;
        }

        public double getHighMemoryThreshold() {
            return this.highMemoryThreshold;
        }

        public void setHighMemoryThreshold(double highMemoryThreshold) {
            this.highMemoryThreshold = highMemoryThreshold;
        }

        public double getElevatedMemoryThreshold() {
            return this.elevatedMemoryThreshold;
        }

        public void setElevatedMemoryThreshold(double elevatedMemoryThreshold) {
            this.elevatedMemoryThreshold = elevatedMemoryThreshold;
        }

        public int getMinSwapChunkAgeMs() {
            return this.minSwapChunkAgeMs;
        }

        public void setMinSwapChunkAgeMs(int minSwapChunkAgeMs) {
            this.minSwapChunkAgeMs = minSwapChunkAgeMs;
        }

        public boolean isEnableSwapStatistics() {
            return this.enableSwapStatistics;
        }

        public void setEnableSwapStatistics(boolean enableSwapStatistics) {
            this.enableSwapStatistics = enableSwapStatistics;
        }

        public boolean isEnablePerformanceMonitoring() {
            return this.enablePerformanceMonitoring;
        }

        public void setEnablePerformanceMonitoring(boolean enablePerformanceMonitoring) {
            this.enablePerformanceMonitoring = enablePerformanceMonitoring;
        }
    }

    public static enum MemoryPressureLevel {
        NORMAL,
        ELEVATED,
        HIGH,
        CRITICAL;

    }

    public static class SwapStatistics {
        private final AtomicLong totalSwapOuts = new AtomicLong(0L);
        private final AtomicLong totalSwapIns = new AtomicLong(0L);
        private final AtomicLong totalFailures = new AtomicLong(0L);
        private final AtomicLong totalSwapOutTime = new AtomicLong(0L);
        private final AtomicLong totalSwapInTime = new AtomicLong(0L);
        private final AtomicLong totalBytesSwappedOut = new AtomicLong(0L);
        private final AtomicLong totalBytesSwappedIn = new AtomicLong(0L);

        public void recordSwapOut(long durationMs, long bytes) {
            this.totalSwapOuts.incrementAndGet();
            this.totalSwapOutTime.addAndGet(durationMs);
            this.totalBytesSwappedOut.addAndGet(bytes);
        }

        public void recordSwapIn(long durationMs, long bytes) {
            this.totalSwapIns.incrementAndGet();
            this.totalSwapInTime.addAndGet(durationMs);
            this.totalBytesSwappedIn.addAndGet(bytes);
        }

        public void recordFailure() {
            this.totalFailures.incrementAndGet();
        }

        public long getTotalSwapOuts() {
            return this.totalSwapOuts.get();
        }

        public long getTotalSwapIns() {
            return this.totalSwapIns.get();
        }

        public long getTotalFailures() {
            return this.totalFailures.get();
        }

        public long getTotalSwapOutTime() {
            return this.totalSwapOutTime.get();
        }

        public long getTotalSwapInTime() {
            return this.totalSwapInTime.get();
        }

        public long getTotalBytesSwappedOut() {
            return this.totalBytesSwappedOut.get();
        }

        public long getTotalBytesSwappedIn() {
            return this.totalBytesSwappedIn.get();
        }

        public double getAverageSwapOutTime() {
            long outs = this.totalSwapOuts.get();
            return outs > 0L ? (double)this.totalSwapOutTime.get() / (double)outs : 0.0;
        }

        public double getAverageSwapInTime() {
            long ins = this.totalSwapIns.get();
            return ins > 0L ? (double)this.totalSwapInTime.get() / (double)ins : 0.0;
        }

        public double getFailureRate() {
            long total = this.totalSwapOuts.get() + this.totalSwapIns.get();
            return total > 0L ? (double)this.totalFailures.get() / (double)total : 0.0;
        }

        public double getSwapThroughputMBps() {
            long totalBytes = this.totalBytesSwappedOut.get() + this.totalBytesSwappedIn.get();
            long totalTime = this.totalSwapOutTime.get() + this.totalSwapInTime.get();
            return totalTime > 0L ? (double)totalBytes / ((double)totalTime * 1000.0) : 0.0;
        }
    }

    private static class PerformanceMonitorAdapter {
        private final boolean enabled;

        public PerformanceMonitorAdapter(boolean enabled) {
            this.enabled = enabled;
        }

        public void recordSwapIn(long bytes, long durationMs) {
            if (this.enabled) {
                LOGGER.debug("Recording swap-in: { } bytes in { }ms", (Object)bytes, (Object)durationMs);
            }
        }

        public void recordSwapOut(long bytes, long durationMs) {
            if (this.enabled) {
                LOGGER.debug("Recording swap-out: { } bytes in { }ms", (Object)bytes, (Object)durationMs);
            }
        }

        public void recordSwapFailure(String operationType, String error) {
            if (this.enabled) {
                LOGGER.error("Recording swap failure: { } - { }", (Object)operationType, (Object)error);
            }
        }

        public void recordMemoryPressure(String level, boolean triggeredCleanup) {
            if (this.enabled) {
                LOGGER.info("Recording memory pressure: { } (cleanup: { })", (Object)level, (Object)triggeredCleanup);
            }
        }
    }

    private static class ChunkAccessEntry
    implements Comparable<ChunkAccessEntry> {
        private final String chunkKey;
        private final long accessTime;
        private final long seq;

        public ChunkAccessEntry(String chunkKey, long accessTime) {
            this.chunkKey = chunkKey;
            this.accessTime = accessTime;
            this.seq = ACCESS_SEQ.getAndIncrement();
        }

        public String getChunkKey() {
            return this.chunkKey;
        }

        public long getAccessTime() {
            return this.accessTime;
        }

        public long getSeq() {
            return this.seq;
        }

        @Override
        public int compareTo(ChunkAccessEntry other) {
            int c = Long.compare(this.accessTime, other.accessTime);
            if (c != 0) {
                return c;
            }
            return Long.compare(this.seq, other.seq);
        }
    }

    public static class MemoryUsageInfo {
        private final long heapUsed;
        private final long heapMax;
        private final long heapCommitted;
        private final long nonHeapUsed;
        private final double usagePercentage;

        public MemoryUsageInfo(long heapUsed, long heapMax, long heapCommitted, long nonHeapUsed, double usagePercentage) {
            this.heapUsed = heapUsed;
            this.heapMax = heapMax;
            this.heapCommitted = heapCommitted;
            this.nonHeapUsed = nonHeapUsed;
            this.usagePercentage = usagePercentage;
        }

        public long getHeapUsed() {
            return this.heapUsed;
        }

        public long getHeapMax() {
            return this.heapMax;
        }

        public long getHeapCommitted() {
            return this.heapCommitted;
        }

        public long getNonHeapUsed() {
            return this.nonHeapUsed;
        }

        public double getUsagePercentage() {
            return this.usagePercentage;
        }

        public String toString() {
            return String.format("MemoryUsageInfo{heapUsed=%d MB, heapMax=%d MB, usage=%.2f%%}", this.heapUsed / 0x100000L, this.heapMax / 0x100000L, this.usagePercentage * 100.0);
        }
    }

    public static class SwapOperation {
        private final String chunkKey;
        private final SwapOperationType operationType;
        private final long startTime;
        private volatile SwapStatus status;
        private volatile long endTime;
        private volatile String errorMessage;
        private final CompletableFuture<Boolean> future;

        public SwapOperation(String chunkKey, SwapOperationType operationType) {
            this.chunkKey = chunkKey;
            this.operationType = operationType;
            this.startTime = System.currentTimeMillis();
            this.status = SwapStatus.PENDING;
            this.future = new CompletableFuture();
        }

        public String getChunkKey() {
            return this.chunkKey;
        }

        public SwapOperationType getOperationType() {
            return this.operationType;
        }

        public long getStartTime() {
            return this.startTime;
        }

        public SwapStatus getStatus() {
            return this.status;
        }

        public long getEndTime() {
            return this.endTime;
        }

        public String getErrorMessage() {
            return this.errorMessage;
        }

        public CompletableFuture<Boolean> getFuture() {
            return this.future;
        }

        public void setStatus(SwapStatus status) {
            this.status = status;
        }

        public void setEndTime(long endTime) {
            this.endTime = endTime;
        }

        public void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }

        public long getDuration() {
            return this.status == SwapStatus.COMPLETED || this.status == SwapStatus.FAILED ? this.endTime - this.startTime : System.currentTimeMillis() - this.startTime;
        }
    }

    public static enum SwapOperationType {
        SWAP_OUT,
        SWAP_IN,
        BULK_SWAP;

    }

    public static enum SwapStatus {
        PENDING,
        IN_PROGRESS,
        COMPLETED,
        FAILED,
        CANCELLED;

    }

    public static class SwapManagerStats {
        private final boolean enabled;
        private final MemoryPressureLevel pressureLevel;
        private final long totalOperations;
        private final long failedOperations;
        private final int activeSwaps;
        private final int pressureTriggers;
        private final MemoryUsageInfo memoryUsage;
        private final SwapStatistics swapStats;

        public SwapManagerStats(boolean enabled, MemoryPressureLevel pressureLevel, long totalOperations, long failedOperations, int activeSwaps, int pressureTriggers, MemoryUsageInfo memoryUsage, SwapStatistics swapStats) {
            this.enabled = enabled;
            this.pressureLevel = pressureLevel;
            this.totalOperations = totalOperations;
            this.failedOperations = failedOperations;
            this.activeSwaps = activeSwaps;
            this.pressureTriggers = pressureTriggers;
            this.memoryUsage = memoryUsage;
            this.swapStats = swapStats;
        }

        public boolean isEnabled() {
            return this.enabled;
        }

        public MemoryPressureLevel getPressureLevel() {
            return this.pressureLevel;
        }

        public long getTotalOperations() {
            return this.totalOperations;
        }

        public long getFailedOperations() {
            return this.failedOperations;
        }

        public int getActiveSwaps() {
            return this.activeSwaps;
        }

        public int getPressureTriggers() {
            return this.pressureTriggers;
        }

        public MemoryUsageInfo getMemoryUsage() {
            return this.memoryUsage;
        }

        public SwapStatistics getSwapStats() {
            return this.swapStats;
        }

        public double getFailureRate() {
            return this.totalOperations > 0L ? (double)this.failedOperations / (double)this.totalOperations : 0.0;
        }

        public String toString() {
            return String.format("SwapManagerStats{enabled=%s, pressure=%s, operations=%d, failed=%d, activeSwaps=%d, pressureTriggers=%d, failureRate=%.2f%%, memory=%s, swapStats=%s}", new Object[]{this.enabled, this.pressureLevel, this.totalOperations, this.failedOperations, this.activeSwaps, this.pressureTriggers, this.getFailureRate() * 100.0, this.memoryUsage, this.swapStats});
        }
    }

    private static class ThreadBuffers {
        byte[] uncompressed;
        byte[] compressed;

        private ThreadBuffers() {
        }
    }
}

