/*
 * Decompiled with CFR 0.152.
 */
package com.kneaf.core.performance;

import com.kneaf.core.performance.RustPerformance;
import com.kneaf.core.performance.core.PerformanceConstants;
import com.kneaf.core.performance.monitoring.PerformanceManager;
import com.mojang.logging.LogUtils;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import net.minecraft.core.BlockPos;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.level.ChunkPos;
import net.neoforged.bus.api.SubscribeEvent;
import net.neoforged.fml.common.EventBusSubscriber;
import net.neoforged.neoforge.event.level.ChunkEvent;
import net.neoforged.neoforge.event.tick.ServerTickEvent;
import org.slf4j.Logger;

@EventBusSubscriber(modid="kneafcore")
public class ChunkGeneratorOptimizer {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static final Map<UUID, PlayerMovementData> PLAYER_MOVEMENT_DATA = new ConcurrentHashMap<UUID, PlayerMovementData>();
    private static final AtomicInteger PREDICTIVE_CHUNKS_GENERATED = new AtomicInteger(0);
    private static final AtomicInteger TOTAL_PREDICTIVE_ATTEMPTS = new AtomicInteger(0);
    private static final double HIGH_SPEED_THRESHOLD = 10.0;
    private static final int FAST_EXPLORATION_DELTA_THRESHOLD = 16;

    private static int getPlayerMovementHistorySize() {
        return PerformanceConstants.getAdaptivePlayerMovementHistorySize(PerformanceManager.getAverageTPS());
    }

    private static double getMinVelocityThreshold() {
        return PerformanceConstants.getAdaptiveMinVelocityThreshold(PerformanceManager.getAverageTPS());
    }

    private static long getPredictionValidityMs() {
        return PerformanceConstants.getAdaptivePredictionValidityMs(PerformanceManager.getAverageTPS());
    }

    private ChunkGeneratorOptimizer() {
    }

    @SubscribeEvent
    public static void onChunkLoad(ChunkEvent.Load event) {
        if (event.getLevel() instanceof ServerLevel) {
            int chunkX = event.getChunk().getPos().x;
            int chunkZ = event.getChunk().getPos().z;
            RustPerformance.preGenerateNearbyChunks(chunkX, chunkZ, 1);
        }
    }

    @SubscribeEvent
    public static void onServerTick(ServerTickEvent.Post event) {
        ServerLevel overworld = event.getServer().overworld();
        if (overworld != null) {
            ChunkGeneratorOptimizer.performPredictiveChunkLoading(overworld);
        }
    }

    private static void performPredictiveChunkLoading(ServerLevel level) {
        if (level == null) {
            return;
        }
        int chunksGeneratedThisTick = 0;
        for (ServerPlayer player : level.players()) {
            BlockPos predictedPos;
            UUID playerId = player.getUUID();
            BlockPos currentPos = player.blockPosition();
            PlayerMovementData movementData = PLAYER_MOVEMENT_DATA.computeIfAbsent(playerId, k -> new PlayerMovementData());
            movementData.addPosition(currentPos);
            double velocity = movementData.getCurrentVelocity();
            boolean isHighSpeed = velocity > 10.0;
            boolean isFastExploration = false;
            if (movementData.recentPositions.size() >= 2) {
                ArrayList<BlockPos> posList = new ArrayList<BlockPos>(movementData.recentPositions);
                ChunkPos currentChunk = new ChunkPos((BlockPos)posList.get(posList.size() - 1));
                ChunkPos prevChunk = new ChunkPos((BlockPos)posList.get(posList.size() - 2));
                int deltaX = Math.abs(currentChunk.x - prevChunk.x);
                int deltaZ = Math.abs(currentChunk.z - prevChunk.z);
                boolean bl = isFastExploration = deltaX > 16 || deltaZ > 16;
            }
            if ((predictedPos = movementData.predictNextPosition()) == null || movementData.isPredictionValid()) continue;
            ChunkPos predictedChunk = new ChunkPos(predictedPos);
            int tpsAdaptiveRadius = PerformanceConstants.getAdaptivePredictiveRadius(PerformanceManager.getAverageTPS());
            int tpsAdaptiveMax = PerformanceConstants.getAdaptiveMaxPredictiveChunksPerTick(PerformanceManager.getAverageTPS());
            int adjustedRadius = tpsAdaptiveRadius;
            int adjustedMax = tpsAdaptiveMax;
            if (isHighSpeed) {
                adjustedRadius = Math.max(1, tpsAdaptiveRadius / 2);
                adjustedMax = Math.max(1, tpsAdaptiveMax / 2);
            } else if (isFastExploration) {
                adjustedRadius = Math.min(tpsAdaptiveRadius * 2, 32);
                adjustedMax = Math.min(tpsAdaptiveMax * 2, 100);
            }
            for (int dx = -adjustedRadius; dx <= adjustedRadius && chunksGeneratedThisTick < adjustedMax; ++dx) {
                for (int dz = -adjustedRadius; dz <= adjustedRadius && chunksGeneratedThisTick < adjustedMax; ++dz) {
                    int generated;
                    int chunkX = predictedChunk.x + dx;
                    int chunkZ = predictedChunk.z + dz;
                    if (RustPerformance.isChunkGenerated(chunkX, chunkZ) || (generated = RustPerformance.preGenerateNearbyChunks(chunkX, chunkZ, 0)) <= 0) continue;
                    ++chunksGeneratedThisTick;
                    PREDICTIVE_CHUNKS_GENERATED.incrementAndGet();
                    LOGGER.debug("Predictively generated chunk at ({ }, { }) for player { }", new Object[]{chunkX, chunkZ, player.getName().getString()});
                }
            }
            movementData.updatePrediction(predictedPos);
            TOTAL_PREDICTIVE_ATTEMPTS.incrementAndGet();
        }
        ChunkGeneratorOptimizer.cleanupOldPlayerData();
    }

    private static void cleanupOldPlayerData() {
        long currentTime = System.currentTimeMillis();
        PLAYER_MOVEMENT_DATA.entrySet().removeIf(entry -> {
            PlayerMovementData data = (PlayerMovementData)entry.getValue();
            return currentTime - data.lastPredictionTime > ChunkGeneratorOptimizer.getPredictionValidityMs() * 2L;
        });
    }

    public static Map<String, Object> getPredictiveStats() {
        HashMap<String, Object> Stats = new HashMap<String, Object>();
        Stats.put("PREDICTIVE_CHUNKS_GENERATED", PREDICTIVE_CHUNKS_GENERATED.get());
        Stats.put("TOTAL_PREDICTIVE_ATTEMPTS", TOTAL_PREDICTIVE_ATTEMPTS.get());
        Stats.put("activePlayersTracked", PLAYER_MOVEMENT_DATA.size());
        if (TOTAL_PREDICTIVE_ATTEMPTS.get() > 0) {
            double successRate = (double)PREDICTIVE_CHUNKS_GENERATED.get() / (double)TOTAL_PREDICTIVE_ATTEMPTS.get() * 100.0;
            Stats.put("predictiveSuccessRate", String.format("%.2f%%", successRate));
        } else {
            Stats.put("predictiveSuccessRate", "0.00%");
        }
        return Stats;
    }

    public static void cleanupPredictiveData() {
        PLAYER_MOVEMENT_DATA.clear();
        PREDICTIVE_CHUNKS_GENERATED.set(0);
        TOTAL_PREDICTIVE_ATTEMPTS.set(0);
        LOGGER.info("Predictive loading data cleaned up");
    }

    public static void shutdown() {
        ChunkGeneratorOptimizer.cleanupPredictiveData();
        LOGGER.info("ChunkGeneratorOptimizer shutdown complete");
    }

    private static class PlayerMovementData {
        private final Queue<BlockPos> recentPositions = new ArrayDeque<BlockPos>(ChunkGeneratorOptimizer.getPlayerMovementHistorySize());
        private final Queue<Long> timestamps = new ArrayDeque<Long>(ChunkGeneratorOptimizer.getPlayerMovementHistorySize());
        private long lastPredictionTime = 0L;
        private double currentVelocity = 0.0;

        private PlayerMovementData() {
        }

        public void addPosition(BlockPos pos) {
            long currentTime = System.currentTimeMillis();
            if (this.recentPositions.size() >= ChunkGeneratorOptimizer.getPlayerMovementHistorySize()) {
                this.recentPositions.poll();
                this.timestamps.poll();
            }
            if (!this.recentPositions.isEmpty() && !this.timestamps.isEmpty()) {
                BlockPos prevPos = this.recentPositions.peek();
                long prevTime = this.timestamps.peek();
                double timeDelta = (double)(currentTime - prevTime) / 1000.0;
                if (timeDelta > 0.0) {
                    double distance = Math.sqrt(Math.pow(pos.getX() - prevPos.getX(), 2.0) + Math.pow(pos.getY() - prevPos.getY(), 2.0) + Math.pow(pos.getZ() - prevPos.getZ(), 2.0));
                    this.currentVelocity = distance / timeDelta;
                }
            }
            this.recentPositions.offer(pos);
            this.timestamps.offer(currentTime);
        }

        public BlockPos predictNextPosition() {
            if (this.recentPositions.size() < 2) {
                return null;
            }
            ArrayList<ChunkPos> chunkPositions = new ArrayList<ChunkPos>();
            for (BlockPos pos : this.recentPositions) {
                chunkPositions.add(new ChunkPos(pos));
            }
            ChunkPos currentChunk = (ChunkPos)chunkPositions.get(chunkPositions.size() - 1);
            ChunkPos previousChunk = (ChunkPos)chunkPositions.get(chunkPositions.size() - 2);
            int deltaX = currentChunk.x - previousChunk.x;
            int deltaZ = currentChunk.z - previousChunk.z;
            double minVel = ChunkGeneratorOptimizer.getMinVelocityThreshold();
            if ((double)Math.abs(deltaX) < minVel && (double)Math.abs(deltaZ) < minVel) {
                return null;
            }
            int predictedX = currentChunk.x + deltaX;
            int predictedZ = currentChunk.z + deltaZ;
            return new BlockPos(predictedX * 16, 64, predictedZ * 16);
        }

        public boolean isPredictionValid() {
            return System.currentTimeMillis() - this.lastPredictionTime < ChunkGeneratorOptimizer.getPredictionValidityMs();
        }

        public void updatePrediction(BlockPos predictedPos) {
            this.lastPredictionTime = System.currentTimeMillis();
        }

        public double getCurrentVelocity() {
            return this.currentVelocity;
        }
    }
}

