/*
 * Decompiled with CFR 0.152.
 */
package com.kneaf.core.performance.core;

import com.kneaf.core.KneafCore;
import com.kneaf.core.data.entity.EntityData;
import com.kneaf.core.data.entity.PlayerData;
import com.kneaf.core.data.entity.VillagerData;
import com.kneaf.core.performance.bridge.NativeIntegrationManager;
import com.kneaf.core.performance.core.BatchProcessor;
import com.kneaf.core.performance.core.EntityProcessor;
import com.kneaf.core.performance.core.PerformanceConstants;
import com.kneaf.core.performance.core.PerformanceMonitor;
import com.kneaf.core.performance.monitoring.PerformanceManager;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

public class PerformanceOptimizer {
    private final PerformanceMonitor monitor;
    private final NativeIntegrationManager NATIVE_MANAGER;
    private final EntityProcessor entityProcessor;
    private final BatchProcessor batchProcessor;
    private static final double BASE_TARGET_TICK_TIME_MS = 50.0;
    private final AtomicLong totalOptimizationsApplied = new AtomicLong(0L);
    private final AtomicLong totalEntitiesProcessed = new AtomicLong(0L);
    private final AtomicLong totalItemsProcessed = new AtomicLong(0L);
    private final AtomicLong totalMobsProcessed = new AtomicLong(0L);
    private final AtomicLong totalBlocksProcessed = new AtomicLong(0L);
    private final Map<String, OptimizationStats> optimizationStats = new ConcurrentHashMap<String, OptimizationStats>();
    private volatile OptimizationLevel currentOptimizationLevel = OptimizationLevel.NORMAL;
    private volatile long lastOptimizationLevelChange = System.currentTimeMillis();
    private final AtomicReference<OptimizationLevel> previousOptimizationLevel = new AtomicReference<OptimizationLevel>(OptimizationLevel.NORMAL);
    private final AtomicInteger optimizationChangeCounter = new AtomicInteger(0);

    public PerformanceOptimizer(PerformanceMonitor monitor, NativeIntegrationManager NATIVE_MANAGER, EntityProcessor entityProcessor, BatchProcessor batchProcessor) {
        this.monitor = monitor;
        this.NATIVE_MANAGER = NATIVE_MANAGER;
        this.entityProcessor = entityProcessor;
        this.batchProcessor = batchProcessor;
    }

    private int getMaxEntitiesPerTick() {
        double tps = PerformanceManager.getAverageTPS();
        double tickDelayMs = PerformanceManager.getLastTickDurationMs();
        return PerformanceConstants.getAdaptiveMaxEntities(tps, tickDelayMs);
    }

    private int getMaxItemsPerTick() {
        double tps = PerformanceManager.getAverageTPS();
        return PerformanceConstants.getAdaptiveMaxItems(tps);
    }

    private int getMaxMobsPerTick() {
        double tps = PerformanceManager.getAverageTPS();
        return PerformanceConstants.getAdaptiveMaxMobs(tps);
    }

    private int getMaxBlocksPerTick() {
        double tps = PerformanceManager.getAverageTPS();
        return PerformanceConstants.getAdaptiveMaxBlocks(tps);
    }

    private double getTargetTickTimeMs() {
        double tps = PerformanceManager.getAverageTPS();
        return PerformanceConstants.getAdaptiveTargetTickTimeMs(tps);
    }

    private double getSmoothedLoadFactor(double rawLoadFactor) {
        return 0.3 * rawLoadFactor + 0.7 * this.getPreviousLoadFactor();
    }

    private double getPreviousLoadFactor() {
        return 0.5;
    }

    private boolean shouldChangeOptimizationLevel(OptimizationLevel newLevel, OptimizationLevel currentLevel) {
        if (this.optimizationChangeCounter.get() > 5) {
            return false;
        }
        if (newLevel.ordinal() > currentLevel.ordinal()) {
            return true;
        }
        return this.optimizationChangeCounter.get() < 3;
    }

    private int getOptimizationThreshold() {
        double tps = PerformanceManager.getAverageTPS();
        return PerformanceConstants.getAdaptiveOptimizationThreshold(tps);
    }

    public CompletableFuture<List<Long>> optimizeEntities(List<EntityData> entities, List<PlayerData> players) {
        long startTime = System.currentTimeMillis();
        try {
            OptimizationLevel level = this.determineOptimizationLevel(entities.size(), players.size());
            List<EntityData> entitiesToProcess = this.applyEntityLimit(entities, level);
            if (this.shouldUseBatchProcessing(entitiesToProcess.size())) {
                return this.batchProcessor.submitLongListRequest("entities", new EntityProcessor.EntityInput(entitiesToProcess, players)).thenApply(result -> {
                    this.totalEntitiesProcessed.addAndGet(entitiesToProcess.size());
                    this.totalOptimizationsApplied.incrementAndGet();
                    this.updateOptimizationStats("entities", entitiesToProcess.size(), result.size(), System.currentTimeMillis() - startTime);
                    return result;
                });
            }
            List<Long> result2 = this.entityProcessor.processEntities(entitiesToProcess, players);
            this.totalEntitiesProcessed.addAndGet(entitiesToProcess.size());
            this.totalOptimizationsApplied.incrementAndGet();
            this.updateOptimizationStats("entities", entitiesToProcess.size(), result2.size(), System.currentTimeMillis() - startTime);
            return CompletableFuture.completedFuture(result2);
        }
        catch (Exception e) {
            KneafCore.LOGGER.error("Error optimizing entities", (Throwable)e);
            return CompletableFuture.failedFuture(new RuntimeException("Failed to optimize " + entities.size() + " entities", e));
        }
    }

    public List<Long> optimizeVillagers(List<VillagerData> villagers, int centerX, int centerZ, int radius) {
        long startTime = System.currentTimeMillis();
        try {
            List<VillagerData> villagersToProcess = this.applyVillagerSpatialFilter(villagers, centerX, centerZ, radius);
            List<Long> result = this.NATIVE_MANAGER.isNativeAvailable() && villagersToProcess.size() >= this.getOptimizationThreshold() ? this.processVillagersNative(villagersToProcess) : this.processVillagersDirect(villagersToProcess);
            this.updateOptimizationStats("villagers", villagers.size(), result.size(), System.currentTimeMillis() - startTime);
            KneafCore.LOGGER.debug("Optimized { } villagers, { } processed", (Object)villagers.size(), (Object)result.size());
            return result;
        }
        catch (Exception e) {
            KneafCore.LOGGER.error("Error optimizing villagers", (Throwable)e);
            return villagers.stream().map(v -> v.hashCode()).toList();
        }
    }

    public void optimizeMemory() {
        long startTime = System.currentTimeMillis();
        try {
            Runtime runtime = Runtime.getRuntime();
            long totalMemory = runtime.totalMemory();
            long freeMemory = runtime.freeMemory();
            long usedMemory = totalMemory - freeMemory;
            long maxMemory = runtime.maxMemory();
            double memoryUsagePercent = (double)usedMemory / (double)totalMemory * 100.0;
            double heapUsagePercent = (double)usedMemory / (double)maxMemory * 100.0;
            double tps = PerformanceManager.getAverageTPS();
            double memoryThreshold = PerformanceConstants.getAdaptiveMemoryUsageThreshold(tps);
            if (memoryUsagePercent > memoryThreshold + 20.0) {
                this.applyCriticalMemoryOptimization(runtime, usedMemory, memoryUsagePercent);
            } else if (memoryUsagePercent > memoryThreshold) {
                this.applyModerateMemoryOptimization(runtime, usedMemory, memoryUsagePercent);
            } else {
                this.applyMaintenanceMemoryOptimization();
            }
            this.logMemoryOptimizationResults(runtime, usedMemory, freeMemory, memoryUsagePercent, heapUsagePercent);
        }
        catch (Exception e) {
            KneafCore.LOGGER.error("Error during memory optimization", (Throwable)e);
        }
    }

    private void applyCriticalMemoryOptimization(Runtime runtime, long usedMemory, double memoryUsagePercent) {
        System.gc();
        try {
            Thread.sleep(10L);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        System.runFinalization();
        System.gc();
        this.clearAllInternalCaches();
        this.getOptimizationStatistics();
        KneafCore.LOGGER.warn("CRITICAL Memory optimization applied. Usage was { }% - reduced entity processing limits", (Object)String.format("%.1f", memoryUsagePercent));
    }

    private void applyModerateMemoryOptimization(Runtime runtime, long usedMemory, double memoryUsagePercent) {
        System.gc();
        this.clearInternalCaches();
        KneafCore.LOGGER.info("Memory optimization applied. Usage was { }%", (Object)String.format("%.1f", memoryUsagePercent));
    }

    private void applyMaintenanceMemoryOptimization() {
        this.clearNonCriticalCaches();
        KneafCore.LOGGER.debug("Routine memory maintenance completed");
    }

    private void clearAllInternalCaches() {
        this.clearInternalCaches();
    }

    private void clearNonCriticalCaches() {
    }

    private void logMemoryOptimizationResults(Runtime runtime, long usedMemory, long freeMemory, double memoryUsagePercent, double heapUsagePercent) {
        KneafCore.LOGGER.debug("Memory optimization results: Used={}MB, Free={}MB, Usage={}%, HeapUsage={}%", new Object[]{usedMemory / 0x100000L, freeMemory / 0x100000L, String.format("%.1f", memoryUsagePercent), String.format("%.1f", heapUsagePercent)});
    }

    public OptimizationStatistics getOptimizationStatistics() {
        HashMap<String, OptimizationStats> Stats = new HashMap<String, OptimizationStats>(this.optimizationStats);
        return new OptimizationStatistics(this.totalOptimizationsApplied.get(), this.totalEntitiesProcessed.get(), this.totalItemsProcessed.get(), this.totalMobsProcessed.get(), this.totalBlocksProcessed.get(), this.currentOptimizationLevel, Stats);
    }

    public void resetStatistics() {
        this.totalOptimizationsApplied.set(0L);
        this.totalEntitiesProcessed.set(0L);
        this.totalItemsProcessed.set(0L);
        this.totalMobsProcessed.set(0L);
        this.totalBlocksProcessed.set(0L);
        this.optimizationStats.clear();
    }

    private OptimizationLevel determineOptimizationLevel(int entityCount, int playerCount) {
        double avgTickTime = this.getTargetTickTimeMs();
        double memoryUsage = this.getMemoryUsagePercent();
        double loadFactor = (double)(entityCount + playerCount * 10) / 1000.0;
        OptimizationLevel newLevel = avgTickTime > this.getTargetTickTimeMs() * 1.5 || memoryUsage > 85.0 || loadFactor > 2.0 ? OptimizationLevel.AGGRESSIVE : (avgTickTime > this.getTargetTickTimeMs() * 1.2 || memoryUsage > 70.0 || loadFactor > 1.5 ? OptimizationLevel.HIGH : (avgTickTime > this.getTargetTickTimeMs() * 1.1 || memoryUsage > 60.0 || loadFactor > 1.0 ? OptimizationLevel.MEDIUM : OptimizationLevel.NORMAL));
        if (newLevel != this.currentOptimizationLevel) {
            long currentTime = System.currentTimeMillis();
            if (this.shouldChangeOptimizationLevel(newLevel, this.currentOptimizationLevel) && currentTime - this.lastOptimizationLevelChange > 2000L) {
                this.currentOptimizationLevel = newLevel;
                this.lastOptimizationLevelChange = currentTime;
                this.optimizationChangeCounter.incrementAndGet();
                KneafCore.LOGGER.debug("Optimization level changed to {} (from {}) - tickTime={}ms, memory={}%, load={}", new Object[]{newLevel, this.previousOptimizationLevel.get(), String.format("%.1f", avgTickTime), String.format("%.1f", memoryUsage), String.format("%.1f", loadFactor)});
            }
        }
        this.previousOptimizationLevel.set(newLevel);
        return this.currentOptimizationLevel;
    }

    private List<EntityData> applyEntityLimit(List<EntityData> entities, OptimizationLevel level) {
        if (entities.size() <= this.getMaxEntitiesPerTick()) {
            return entities;
        }
        int limit = switch (level.ordinal()) {
            default -> throw new MatchException(null, null);
            case 3 -> Math.max(10, Math.min(this.getMaxEntitiesPerTick() / 2, 50));
            case 2 -> Math.max(20, Math.min(this.getMaxEntitiesPerTick() * 2 / 3, 100));
            case 1 -> Math.max(30, Math.min(this.getMaxEntitiesPerTick() * 3 / 4, 150));
            case 0 -> this.getMaxEntitiesPerTick();
        };
        return entities.parallelStream().sorted((a, b) -> Integer.compare(this.getEntityPriority((EntityData)b), this.getEntityPriority((EntityData)a))).limit(limit).toList();
    }

    private int getEntityPriority(EntityData entity) {
        int priority = 0;
        double distance = entity.getDistance();
        priority = distance < 10.0 ? (priority += 100) : (distance < 30.0 ? (priority += 75) : (distance < 100.0 ? (priority += 50) : (distance < 300.0 ? (priority += 25) : (priority += 10))));
        String typeStr = entity.getType();
        if (typeStr.contains("Player")) {
            priority += 50;
        } else if (typeStr.contains("Mob")) {
            priority += 30;
        } else if (typeStr.contains("Animal")) {
            priority += 20;
        } else if (typeStr.contains("Item")) {
            priority += 15;
        } else if (typeStr.contains("Villager")) {
            priority += 25;
        }
        return priority += 10;
    }

    private List<VillagerData> applyVillagerSpatialFilter(List<VillagerData> villagers, int centerX, int centerZ, int radius) {
        if (villagers.size() <= this.getOptimizationThreshold()) {
            return villagers;
        }
        int radiusSquared = radius * radius;
        return villagers.stream().filter(v -> {
            int dz;
            int[] position = new int[]{v.hashCode(), v.hashCode(), v.hashCode()};
            int dx = position[0] - centerX;
            return dx * dx + (dz = position[1] - centerZ) * dz <= radiusSquared;
        }).limit(this.getMaxEntitiesPerTick()).toList();
    }

    private boolean shouldUseBatchProcessing(int entityCount) {
        return entityCount >= this.getOptimizationThreshold() && this.NATIVE_MANAGER.isNativeAvailable();
    }

    private List<Long> processVillagersNative(List<VillagerData> villagers) {
        return villagers.stream().map(v -> v.hashCode()).toList();
    }

    private List<Long> processVillagersDirect(List<VillagerData> villagers) {
        return villagers.stream().map(v -> v.hashCode()).toList();
    }

    private double getMemoryUsagePercent() {
        Runtime runtime = Runtime.getRuntime();
        long totalMemory = runtime.totalMemory();
        long freeMemory = runtime.freeMemory();
        long usedMemory = totalMemory - freeMemory;
        return (double)usedMemory / (double)totalMemory * 100.0;
    }

    private void updateOptimizationStats(String type, int inputCount, int optimizedCount, long processingTime) {
        OptimizationStats Stats = this.optimizationStats.computeIfAbsent(type, k -> new OptimizationStats());
        Stats.addSample(inputCount, optimizedCount, processingTime);
    }

    private void clearInternalCaches() {
    }

    public static enum OptimizationLevel {
        NORMAL(1.0),
        MEDIUM(1.5),
        HIGH(2.0),
        AGGRESSIVE(3.0);

        private final double multiplier;

        private OptimizationLevel(double multiplier) {
            this.multiplier = multiplier;
        }

        public double getMultiplier() {
            return this.multiplier;
        }
    }

    public static class OptimizationStatistics {
        private final long totalOptimizations;
        private final long totalEntitiesProcessed;
        private final long totalItemsProcessed;
        private final long totalMobsProcessed;
        private final long totalBlocksProcessed;
        private final OptimizationLevel currentLevel;
        private final Map<String, OptimizationStats> typeStats;

        public OptimizationStatistics(long totalOptimizations, long totalEntitiesProcessed, long totalItemsProcessed, long totalMobsProcessed, long totalBlocksProcessed, OptimizationLevel currentLevel, Map<String, OptimizationStats> typeStats) {
            this.totalOptimizations = totalOptimizations;
            this.totalEntitiesProcessed = totalEntitiesProcessed;
            this.totalItemsProcessed = totalItemsProcessed;
            this.totalMobsProcessed = totalMobsProcessed;
            this.totalBlocksProcessed = totalBlocksProcessed;
            this.currentLevel = currentLevel;
            this.typeStats = new HashMap<String, OptimizationStats>(typeStats);
        }

        public long getTotalOptimizations() {
            return this.totalOptimizations;
        }

        public long getTotalEntitiesProcessed() {
            return this.totalEntitiesProcessed;
        }

        public long getTotalItemsProcessed() {
            return this.totalItemsProcessed;
        }

        public long getTotalMobsProcessed() {
            return this.totalMobsProcessed;
        }

        public long getTotalBlocksProcessed() {
            return this.totalBlocksProcessed;
        }

        public OptimizationLevel getCurrentLevel() {
            return this.currentLevel;
        }

        public Map<String, OptimizationStats> getTypeStats() {
            return new HashMap<String, OptimizationStats>(this.typeStats);
        }
    }

    public static class OptimizationStats {
        private final AtomicInteger sampleCount = new AtomicInteger(0);
        private final AtomicInteger totalInput = new AtomicInteger(0);
        private final AtomicInteger totalOptimized = new AtomicInteger(0);
        private final AtomicLong totalProcessingTime = new AtomicLong(0L);

        private long getVillagerId(VillagerData villager) {
            return villager.hashCode();
        }

        private int[] getVillagerPosition(VillagerData villager) {
            return new int[]{0, 0};
        }

        public void addSample(int inputCount, int optimizedCount, long processingTime) {
            this.sampleCount.incrementAndGet();
            this.totalInput.addAndGet(inputCount);
            this.totalOptimized.addAndGet(optimizedCount);
            this.totalProcessingTime.addAndGet(processingTime);
        }

        public double getAverageOptimizationRate() {
            int input = this.totalInput.get();
            int optimized = this.totalOptimized.get();
            return input > 0 ? (double)optimized / (double)input : 0.0;
        }

        public double getAverageProcessingTime() {
            int samples = this.sampleCount.get();
            return samples > 0 ? (double)this.totalProcessingTime.get() / (double)samples : 0.0;
        }

        public int getSampleCount() {
            return this.sampleCount.get();
        }
    }
}

