/*
 * Decompiled with CFR 0.152.
 */
package com.kneaf.core.chunkstorage;

import com.kneaf.core.chunkstorage.cache.ChunkCache;
import com.kneaf.core.chunkstorage.common.ChunkStorageConfig;
import com.kneaf.core.chunkstorage.database.AbstractDatabaseAdapter;
import com.kneaf.core.chunkstorage.database.DatabaseAdapter;
import com.kneaf.core.chunkstorage.database.InMemoryDatabaseAdapter;
import com.kneaf.core.chunkstorage.database.RustDatabaseAdapter;
import com.kneaf.core.chunkstorage.serialization.ChunkSerializer;
import com.kneaf.core.chunkstorage.serialization.NbtChunkSerializer;
import com.kneaf.core.chunkstorage.swap.SwapManager;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import net.minecraft.world.level.chunk.LevelChunk;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChunkStorageManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(ChunkStorageManager.class);
    private final ChunkSerializer serializer;
    private final DatabaseAdapter database;
    private final ChunkCache cache;
    private final ExecutorService asyncExecutor;
    private final String worldName;
    private final boolean enabled;
    private final SwapManager swapManager;
    private volatile boolean shutdown = false;

    public ChunkStorageManager(final String worldName, ChunkStorageConfig config) {
        if (worldName == null || worldName.isEmpty()) {
            throw new IllegalArgumentException("World name cannot be null or empty");
        }
        if (config == null) {
            throw new IllegalArgumentException("Config cannot be null");
        }
        this.worldName = worldName;
        this.enabled = config.isEnabled();
        if (this.enabled) {
            ChunkCache.EvictionPolicy evictionPolicy;
            NbtChunkSerializer tempSerializer = null;
            try {
                if (NbtChunkSerializer.isMinecraftAvailable()) {
                    tempSerializer = new NbtChunkSerializer();
                    LOGGER.info("Using NbtChunkSerializer for chunk serialization");
                } else {
                    LOGGER.warn("Minecraft classes not available - serializer will be null");
                    tempSerializer = null;
                }
            }
            catch (Exception e) {
                LOGGER.warn("Failed to initialize NbtChunkSerializer, serializer will be null: { }", (Object)e.getMessage());
                tempSerializer = null;
            }
            this.serializer = tempSerializer;
            AbstractDatabaseAdapter tempDatabase = null;
            if (config.isUseRustDatabase()) {
                try {
                    LOGGER.info("Attempting to create RustDatabaseAdapter with type: { }, checksums: { }", (Object)config.getDatabaseType(), (Object)config.isEnableChecksums());
                    LOGGER.info("RustDatabaseAdapter.isNativeLibraryAvailable(): { }", (Object)RustDatabaseAdapter.isNativeLibraryAvailable());
                    tempDatabase = new RustDatabaseAdapter(config.getDatabaseType(), config.isEnableChecksums());
                    LOGGER.info("Successfully created Rust database adapter");
                }
                catch (Exception e) {
                    LOGGER.error("Failed to initialize Rust database adapter, falling back to in-memory database", (Throwable)e);
                    LOGGER.warn("Exception details: { }: { }", (Object)e.getClass().getSimpleName(), (Object)e.getMessage());
                    tempDatabase = new InMemoryDatabaseAdapter("in-memory-" + worldName);
                    LOGGER.info("Using in-memory database adapter as fallback");
                }
            } else {
                tempDatabase = new InMemoryDatabaseAdapter("in-memory-" + worldName);
                LOGGER.info("Using in-memory database adapter");
            }
            this.database = tempDatabase;
            switch (config.getEvictionPolicy().toLowerCase()) {
                case "lru": {
                    evictionPolicy = new ChunkCache.LRUEvictionPolicy();
                    break;
                }
                case "distance": {
                    evictionPolicy = new ChunkCache.DistanceEvictionPolicy(0, 0);
                    break;
                }
                case "hybrid": {
                    evictionPolicy = new ChunkCache.HybridEvictionPolicy();
                    break;
                }
                default: {
                    evictionPolicy = new ChunkCache.LRUEvictionPolicy();
                    LOGGER.warn("Unknown eviction policy '{ }', defaulting to LRU", (Object)config.getEvictionPolicy());
                }
            }
            this.cache = new ChunkCache(config.getCacheCapacity(), evictionPolicy);
            ThreadFactory threadFactory = new ThreadFactory(){
                private final AtomicInteger threadNumber = new AtomicInteger(1);

                @Override
                public Thread newThread(Runnable r) {
                    Thread thread = new Thread(r, "chunk-storage-" + worldName + "-" + this.threadNumber.getAndIncrement());
                    thread.setDaemon(true);
                    return thread;
                }
            };
            this.asyncExecutor = Executors.newFixedThreadPool(config.getAsyncThreadpoolSize(), threadFactory);
            SwapManager tempSwapManager = null;
            if (config.isEnableSwapManager() && config.isUseRustDatabase() && this.database instanceof RustDatabaseAdapter) {
                try {
                    SwapManager.SwapConfig swapConfig = this.createSwapConfig(config);
                    tempSwapManager = new SwapManager(swapConfig);
                    LOGGER.info("Initialized SwapManager for world '{ }'", (Object)worldName);
                    tempSwapManager.initializeComponents(this.cache, (RustDatabaseAdapter)this.database);
                }
                catch (Exception e) {
                    LOGGER.warn("Failed to initialize SwapManager, disabling swap functionality: { }", (Object)e.getMessage());
                    tempSwapManager = null;
                }
            } else {
                LOGGER.info("SwapManager disabled for world '{ }'", (Object)worldName);
            }
            this.swapManager = tempSwapManager;
            LOGGER.info("Initialized ChunkStorageManager for world '{ }' with cache capacity { } and { } eviction policy", new Object[]{worldName, config.getCacheCapacity(), config.getEvictionPolicy()});
        } else {
            this.serializer = null;
            this.database = null;
            this.cache = null;
            this.asyncExecutor = null;
            this.swapManager = null;
            LOGGER.info("ChunkStorageManager disabled for world '{ }'", (Object)worldName);
        }
    }

    public CompletableFuture<Void> saveChunk(Object chunk) {
        if (!this.enabled || this.shutdown || this.serializer == null) {
            return CompletableFuture.completedFuture(null);
        }
        String chunkKey = this.createChunkKey(chunk);
        try {
            byte[] serializedData = this.serializer.serialize(chunk);
            ChunkCache.CachedChunk evicted = null;
            try {
                if (this.isMinecraftLevelChunk(chunk)) {
                    Object levelChunk = Class.forName("net.minecraft.world.level.chunk.LevelChunk").cast(chunk);
                    evicted = this.cache.putChunk(chunkKey, (LevelChunk)levelChunk);
                } else {
                    LOGGER.debug("Chunk object is not a LevelChunk, skipping cache for { }", (Object)chunkKey);
                }
            }
            catch (Exception e) {
                LOGGER.debug("Failed to cache chunk { }: { }", (Object)chunkKey, (Object)e.getMessage());
            }
            CompletableFuture<Object> evictedSave = CompletableFuture.completedFuture(null);
            if (evicted != null && evicted.isDirty()) {
                Object evictedChunk = evicted.getChunk();
                String evictedKey = this.createChunkKey(evictedChunk);
                evictedSave = this.saveChunkToDatabase(evictedKey, evictedChunk);
            }
            CompletableFuture<Void> currentSave = this.saveChunkToDatabase(chunkKey, serializedData);
            return CompletableFuture.allOf(evictedSave, currentSave);
        }
        catch (Exception e) {
            LOGGER.error("Failed to save chunk { }", (Object)chunkKey, (Object)e);
            return CompletableFuture.failedFuture(e);
        }
    }

    /*
     * Exception decompiling
     */
    public Optional<Object> loadChunk(Object level, int chunkX, int chunkZ) {
        /*
         * This method has failed to decompile.  When submitting a bug report, please provide this stack trace, and (if you hold appropriate legal rights) the relevant class file.
         * 
         * org.benf.cfr.reader.util.ConfusedCFRException: Tried to end blocks [9[CATCHBLOCK], 0[TRYBLOCK]], but top level block is 5[TRYBLOCK]
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.processEndingBlocks(Op04StructuredStatement.java:435)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.buildNestedBlocks(Op04StructuredStatement.java:484)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op03SimpleStatement.createInitialStructuredBlock(Op03SimpleStatement.java:736)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisInner(CodeAnalyser.java:850)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisOrWrapFail(CodeAnalyser.java:278)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysis(CodeAnalyser.java:201)
         *     at org.benf.cfr.reader.entities.attributes.AttributeCode.analyse(AttributeCode.java:94)
         *     at org.benf.cfr.reader.entities.Method.analyse(Method.java:531)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseMid(ClassFile.java:1055)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseTop(ClassFile.java:942)
         *     at org.benf.cfr.reader.Driver.doJarVersionTypes(Driver.java:257)
         *     at org.benf.cfr.reader.Driver.doJar(Driver.java:139)
         *     at org.benf.cfr.reader.CfrDriverImpl.analyse(CfrDriverImpl.java:76)
         *     at org.benf.cfr.reader.Main.main(Main.java:54)
         */
        throw new IllegalStateException("Decompilation failed");
    }

    public CompletableFuture<Optional<Object>> loadChunkAsync(Object level, int chunkX, int chunkZ) {
        if (!this.enabled || this.shutdown) {
            return CompletableFuture.completedFuture(Optional.empty());
        }
        return CompletableFuture.supplyAsync(() -> this.loadChunk(level, chunkX, chunkZ), this.asyncExecutor);
    }

    public boolean deleteChunk(Object level, int chunkX, int chunkZ) {
        if (!this.enabled || this.shutdown) {
            return false;
        }
        String chunkKey = this.createChunkKey(level, chunkX, chunkZ);
        try {
            boolean deleted;
            ChunkCache.CachedChunk removedFromCache = this.cache.removeChunk(chunkKey);
            boolean removedFromDb = this.database.deleteChunk(chunkKey);
            boolean bl = deleted = removedFromCache != null || removedFromDb;
            if (deleted && LOGGER.isDebugEnabled()) {
                LOGGER.debug("Deleted chunk { } from storage", (Object)chunkKey);
            }
            return deleted;
        }
        catch (Exception e) {
            LOGGER.error("Failed to delete chunk { }", (Object)chunkKey, (Object)e);
            return false;
        }
    }

    public StorageStats getStats() {
        if (!this.enabled) {
            return StorageStats.builder().enabled(false).totalChunksInDb(0L).cachedChunks(0).avgReadLatencyMs(0L).avgWriteLatencyMs(0L).cacheHitRate(0.0).overallHitRate(0.0).status("disabled").swapEnabled(false).memoryPressureLevel("NORMAL").swapOperationsTotal(0L).swapOperationsFailed(0L).build();
        }
        try {
            Object dbStatsObj = this.database.getStats();
            ChunkCache.CacheStats cacheStatsObj = this.cache.getStats();
            SwapManager.MemoryPressureLevel pressureLevel = SwapManager.MemoryPressureLevel.NORMAL;
            long swapOperationsTotal = 0L;
            long swapOperationsFailed = 0L;
            boolean swapEnabled = false;
            if (this.swapManager != null) {
                SwapManager.SwapManagerStats swapStats = this.swapManager.getStats();
                pressureLevel = swapStats.getPressureLevel();
                swapOperationsTotal = swapStats.getTotalOperations();
                swapOperationsFailed = swapStats.getFailedOperations();
                swapEnabled = swapStats.isEnabled();
            }
            long totalChunks = 0L;
            long readLatency = 0L;
            long writeLatency = 0L;
            int cachedChunks = 0;
            double cacheHitRate = 0.0;
            if (dbStatsObj != null) {
                try {
                    totalChunks = (Long)dbStatsObj.getClass().getMethod("getTotalChunks", new Class[0]).invoke(dbStatsObj, new Object[0]);
                    readLatency = (Long)dbStatsObj.getClass().getMethod("getReadLatencyMs", new Class[0]).invoke(dbStatsObj, new Object[0]);
                    writeLatency = (Long)dbStatsObj.getClass().getMethod("getWriteLatencyMs", new Class[0]).invoke(dbStatsObj, new Object[0]);
                }
                catch (Exception e) {
                    LOGGER.warn("Failed to extract database statistics", (Throwable)e);
                }
            }
            if (cacheStatsObj != null) {
                try {
                    cachedChunks = (Integer)cacheStatsObj.getClass().getMethod("getCacheSize", new Class[0]).invoke((Object)cacheStatsObj, new Object[0]);
                    cacheHitRate = (Double)cacheStatsObj.getClass().getMethod("getHitRate", new Class[0]).invoke((Object)cacheStatsObj, new Object[0]);
                }
                catch (Exception e) {
                    LOGGER.warn("Failed to extract cache statistics", (Throwable)e);
                }
            }
            return StorageStats.builder().enabled(true).totalChunksInDb(totalChunks).cachedChunks(cachedChunks).avgReadLatencyMs(readLatency).avgWriteLatencyMs(writeLatency).cacheHitRate(cacheHitRate).overallHitRate(cacheHitRate).status("storage-manager").swapEnabled(swapEnabled).memoryPressureLevel(pressureLevel.toString()).swapOperationsTotal(swapOperationsTotal).swapOperationsFailed(swapOperationsFailed).build();
        }
        catch (Exception e) {
            LOGGER.error("Failed to get storage Stats", (Throwable)e);
            return StorageStats.builder().enabled(false).totalChunksInDb(0L).cachedChunks(0).avgReadLatencyMs(0L).avgWriteLatencyMs(0L).cacheHitRate(0.0).overallHitRate(0.0).status("error").swapEnabled(false).memoryPressureLevel("NORMAL").swapOperationsTotal(0L).swapOperationsFailed(0L).build();
        }
    }

    public void performMaintenance() {
        if (!this.enabled || this.shutdown) {
            return;
        }
        try {
            LOGGER.info("Performing storage maintenance for world '{ }'", (Object)this.worldName);
            this.database.performMaintenance();
            this.cache.resetStats();
            LOGGER.info("Storage maintenance completed for world '{ }'", (Object)this.worldName);
        }
        catch (Exception e) {
            LOGGER.error("Failed to perform storage maintenance for world '{ }'", (Object)this.worldName, (Object)e);
        }
    }

    public void shutdown() {
        if (!this.enabled || this.shutdown) {
            return;
        }
        this.shutdown = true;
        try {
            LOGGER.info("Shutting down ChunkStorageManager for world '{ }'", (Object)this.worldName);
            if (this.swapManager != null) {
                this.swapManager.shutdown();
            }
            this.shutdownExecutor();
            this.database.close();
            this.cache.clear();
            LOGGER.info("ChunkStorageManager shutdown completed for world '{ }'", (Object)this.worldName);
        }
        catch (Exception e) {
            LOGGER.error("Error during shutdown of ChunkStorageManager for world '{ }'", (Object)this.worldName, (Object)e);
        }
    }

    private void shutdownExecutor() {
        this.asyncExecutor.shutdown();
        try {
            if (!this.asyncExecutor.awaitTermination(30L, TimeUnit.SECONDS)) {
                this.asyncExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.asyncExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    public boolean isHealthy() {
        if (!this.enabled || this.shutdown) {
            return false;
        }
        try {
            return this.database.isHealthy() && !this.asyncExecutor.isShutdown();
        }
        catch (Exception e) {
            LOGGER.error("Health check failed for world '{ }'", (Object)this.worldName, (Object)e);
            return false;
        }
    }

    public void createBackup(String backupPath) {
        if (!this.enabled || this.shutdown) {
            return;
        }
        try {
            LOGGER.info("Creating backup for world '{ }' at '{ }'", (Object)this.worldName, (Object)backupPath);
            this.database.createBackup(backupPath);
            LOGGER.info("Backup completed for world '{ }'", (Object)this.worldName);
        }
        catch (Exception e) {
            LOGGER.error("Failed to create backup for world '{ }' at '{ }'", new Object[]{this.worldName, backupPath, e});
        }
    }

    public void clearCache() {
        if (!this.enabled || this.shutdown) {
            return;
        }
        try {
            try {
                Field coreField = this.getClass().getDeclaredField("core");
                coreField.setAccessible(true);
                Object coreObj = coreField.get(this);
                if (coreObj != null) {
                    Method m = coreObj.getClass().getMethod("clearCache", new Class[0]);
                    m.invoke(coreObj, new Object[0]);
                    return;
                }
            }
            catch (NoSuchFieldException coreField) {}
        }
        catch (Exception e) {
            LOGGER.warn("Failed to clear cache via core delegation: { }", (Object)e.getMessage());
        }
    }

    public void setCacheCapacity(int capacity) {
        if (!this.enabled || this.shutdown) {
            return;
        }
        try {
            boolean applied = this.cache.setMaxCapacity(capacity);
            if (applied) {
                LOGGER.info("Cache capacity changed to { } for world '{ }'", (Object)capacity, (Object)this.worldName);
            } else {
                LOGGER.warn("Requested cache capacity change to { } for world '{ }' was rejected", (Object)capacity, (Object)this.worldName);
            }
        }
        catch (Exception e) {
            LOGGER.warn("Failed to change cache capacity for world '{ }': { }", (Object)this.worldName, (Object)e.getMessage());
        }
    }

    public void setEvictionPolicy(String policy) {
        if (!this.enabled || this.shutdown) {
            return;
        }
        try {
            boolean applied = this.cache.setEvictionPolicy(policy);
            if (applied) {
                LOGGER.info("Eviction policy changed to { } for world '{ }'", (Object)policy, (Object)this.worldName);
            } else {
                LOGGER.warn("Requested eviction policy change to { } for world '{ }' was rejected", (Object)policy, (Object)this.worldName);
            }
        }
        catch (Exception e) {
            LOGGER.warn("Failed to change eviction policy for world '{ }': { }", (Object)this.worldName, (Object)e.getMessage());
        }
    }

    private CompletableFuture<Void> saveChunkToDatabase(String chunkKey, byte[] data) {
        return this.database.putChunkAsync(chunkKey, data).exceptionally(throwable -> {
            LOGGER.error("Failed to save chunk { } to database", (Object)chunkKey, throwable);
            return null;
        });
    }

    private CompletableFuture<Void> saveChunkToDatabase(String chunkKey, Object chunk) {
        if (this.serializer == null) {
            LOGGER.warn("Cannot serialize chunk { } - serializer is null", (Object)chunkKey);
            return CompletableFuture.completedFuture(null);
        }
        try {
            byte[] data = this.serializer.serialize(chunk);
            return this.saveChunkToDatabase(chunkKey, data);
        }
        catch (Exception e) {
            LOGGER.error("Failed to serialize chunk { } for database storage", (Object)chunkKey, (Object)e);
            return CompletableFuture.failedFuture(e);
        }
    }

    private String createChunkKey(Object chunk) {
        try {
            Object chunkPos = chunk.getClass().getMethod("getPos", new Class[0]).invoke(chunk, new Object[0]);
            int x = (Integer)chunkPos.getClass().getMethod("x", new Class[0]).invoke(chunkPos, new Object[0]);
            int z = (Integer)chunkPos.getClass().getMethod("z", new Class[0]).invoke(chunkPos, new Object[0]);
            return String.format("%s:%d:%d", this.worldName, x, z);
        }
        catch (Exception e) {
            LOGGER.error("Failed to create chunk key from chunk object", (Throwable)e);
            return String.format("%s:unknown", this.worldName);
        }
    }

    private String createChunkKey(Object level, int chunkX, int chunkZ) {
        try {
            Object dimension = level.getClass().getMethod("dimension", new Class[0]).invoke(level, new Object[0]);
            Object location = dimension.getClass().getMethod("location", new Class[0]).invoke(dimension, new Object[0]);
            String dimensionName = location.toString();
            return String.format("%s:%d:%d", dimensionName, chunkX, chunkZ);
        }
        catch (Exception e) {
            LOGGER.error("Failed to create chunk key from level object", (Throwable)e);
            return String.format("unknown:%d:%d", chunkX, chunkZ);
        }
    }

    private boolean isMinecraftLevelChunk(Object obj) {
        try {
            return obj != null && Class.forName("net.minecraft.world.level.chunk.LevelChunk").isInstance(obj);
        }
        catch (ClassNotFoundException e) {
            return false;
        }
    }

    private boolean isMinecraftCompoundTag(Object obj) {
        try {
            return obj != null && Class.forName("net.minecraft.nbt.CompoundTag").isInstance(obj);
        }
        catch (ClassNotFoundException e) {
            return false;
        }
    }

    public CompletableFuture<Boolean> swapOutChunk(String chunkKey) {
        if (!this.enabled || this.shutdown || this.swapManager == null) {
            return CompletableFuture.completedFuture(false);
        }
        return this.swapManager.swapOutChunk(chunkKey);
    }

    public CompletableFuture<Boolean> swapInChunk(String chunkKey) {
        if (!this.enabled || this.shutdown || this.swapManager == null) {
            return CompletableFuture.completedFuture(false);
        }
        return this.swapManager.swapInChunk(chunkKey);
    }

    public SwapManager.SwapManagerStats getSwapStats() {
        if (!this.enabled || this.swapManager == null) {
            return new SwapManager.SwapManagerStats(false, SwapManager.MemoryPressureLevel.NORMAL, 0L, 0L, 0, 0, new SwapManager.MemoryUsageInfo(0L, 0L, 0L, 0L, 0.0), new SwapManager.SwapStatistics());
        }
        return this.swapManager.getStats();
    }

    public SwapManager.MemoryPressureLevel getMemoryPressureLevel() {
        if (!this.enabled || this.swapManager == null) {
            return SwapManager.MemoryPressureLevel.NORMAL;
        }
        return this.swapManager.getMemoryPressureLevel();
    }

    public SwapManager.MemoryUsageInfo getMemoryUsage() {
        if (!this.enabled || this.swapManager == null) {
            return new SwapManager.MemoryUsageInfo(0L, 0L, 0L, 0L, 0.0);
        }
        return this.swapManager.getMemoryUsage();
    }

    private SwapManager.SwapConfig createSwapConfig(ChunkStorageConfig config) {
        SwapManager.SwapConfig swapConfig = new SwapManager.SwapConfig();
        swapConfig.setEnabled(config.isEnableSwapManager());
        swapConfig.setMemoryCheckIntervalMs(config.getSwapMemoryCheckIntervalMs());
        swapConfig.setMaxConcurrentSwaps(config.getMaxConcurrentSwaps());
        swapConfig.setSwapBatchSize(config.getSwapBatchSize());
        swapConfig.setSwapTimeoutMs(config.getSwapTimeoutMs());
        swapConfig.setEnableAutomaticSwapping(config.isEnableAutomaticSwapping());
        swapConfig.setCriticalMemoryThreshold(config.getCriticalMemoryThreshold());
        swapConfig.setHighMemoryThreshold(config.getHighMemoryThreshold());
        swapConfig.setElevatedMemoryThreshold(config.getElevatedMemoryThreshold());
        swapConfig.setMinSwapChunkAgeMs(config.getMinSwapChunkAgeMs());
        swapConfig.setEnableSwapStatistics(config.isEnableSwapStatistics());
        swapConfig.setEnablePerformanceMonitoring(true);
        return swapConfig;
    }

    public static class StorageStats {
        private final boolean enabled;
        private final long totalChunksInDb;
        private final int cachedChunks;
        private final long avgReadLatencyMs;
        private final long avgWriteLatencyMs;
        private final double cacheHitRate;
        private final double overallHitRate;
        private final String status;
        private final boolean swapEnabled;
        private final String memoryPressureLevel;
        private final long swapOperationsTotal;
        private final long swapOperationsFailed;

        private StorageStats(Builder builder) {
            this.enabled = builder.enabled;
            this.totalChunksInDb = builder.totalChunksInDb;
            this.cachedChunks = builder.cachedChunks;
            this.avgReadLatencyMs = builder.avgReadLatencyMs;
            this.avgWriteLatencyMs = builder.avgWriteLatencyMs;
            this.cacheHitRate = builder.cacheHitRate;
            this.overallHitRate = builder.overallHitRate;
            this.status = builder.status;
            this.swapEnabled = builder.swapEnabled;
            this.memoryPressureLevel = builder.memoryPressureLevel;
            this.swapOperationsTotal = builder.swapOperationsTotal;
            this.swapOperationsFailed = builder.swapOperationsFailed;
        }

        public static Builder builder() {
            return new Builder();
        }

        public boolean isEnabled() {
            return this.enabled;
        }

        public long getTotalChunksInDb() {
            return this.totalChunksInDb;
        }

        public int getCachedChunks() {
            return this.cachedChunks;
        }

        public long getAvgReadLatencyMs() {
            return this.avgReadLatencyMs;
        }

        public long getAvgWriteLatencyMs() {
            return this.avgWriteLatencyMs;
        }

        public double getCacheHitRate() {
            return this.cacheHitRate;
        }

        public double getOverallHitRate() {
            return this.overallHitRate;
        }

        public String getStatus() {
            return this.status;
        }

        public boolean isSwapEnabled() {
            return this.swapEnabled;
        }

        public String getMemoryPressureLevel() {
            return this.memoryPressureLevel;
        }

        public long getSwapOperationsTotal() {
            return this.swapOperationsTotal;
        }

        public long getSwapOperationsFailed() {
            return this.swapOperationsFailed;
        }

        public String toString() {
            return String.format("StorageStats{enabled=%s, dbChunks=%d, cached=%d, readLatency=%d ms, writeLatency=%d ms, cacheHitRate=%.2f%%, overallHitRate=%.2f%%, status=%s, swapEnabled=%s, pressure=%s, swapOps=%d, swapFailed=%d}", this.enabled, this.totalChunksInDb, this.cachedChunks, this.avgReadLatencyMs, this.avgWriteLatencyMs, this.cacheHitRate * 100.0, this.overallHitRate * 100.0, this.status, this.swapEnabled, this.memoryPressureLevel, this.swapOperationsTotal, this.swapOperationsFailed);
        }

        public static class Builder {
            private boolean enabled;
            private long totalChunksInDb;
            private int cachedChunks;
            private long avgReadLatencyMs;
            private long avgWriteLatencyMs;
            private double cacheHitRate;
            private double overallHitRate;
            private String status;
            private boolean swapEnabled;
            private String memoryPressureLevel;
            private long swapOperationsTotal;
            private long swapOperationsFailed;

            public Builder enabled(boolean enabled) {
                this.enabled = enabled;
                return this;
            }

            public Builder totalChunksInDb(long totalChunksInDb) {
                this.totalChunksInDb = totalChunksInDb;
                return this;
            }

            public Builder cachedChunks(int cachedChunks) {
                this.cachedChunks = cachedChunks;
                return this;
            }

            public Builder avgReadLatencyMs(long avgReadLatencyMs) {
                this.avgReadLatencyMs = avgReadLatencyMs;
                return this;
            }

            public Builder avgWriteLatencyMs(long avgWriteLatencyMs) {
                this.avgWriteLatencyMs = avgWriteLatencyMs;
                return this;
            }

            public Builder cacheHitRate(double cacheHitRate) {
                this.cacheHitRate = cacheHitRate;
                return this;
            }

            public Builder overallHitRate(double overallHitRate) {
                this.overallHitRate = overallHitRate;
                return this;
            }

            public Builder status(String status) {
                this.status = status;
                return this;
            }

            public Builder swapEnabled(boolean swapEnabled) {
                this.swapEnabled = swapEnabled;
                return this;
            }

            public Builder memoryPressureLevel(String memoryPressureLevel) {
                this.memoryPressureLevel = memoryPressureLevel;
                return this;
            }

            public Builder swapOperationsTotal(long swapOperationsTotal) {
                this.swapOperationsTotal = swapOperationsTotal;
                return this;
            }

            public Builder swapOperationsFailed(long swapOperationsFailed) {
                this.swapOperationsFailed = swapOperationsFailed;
                return this;
            }

            public StorageStats build() {
                return new StorageStats(this);
            }
        }
    }
}

