/*
 * Decompiled with CFR 0.152.
 */
package com.kneaf.core.performance.monitoring;

import com.kneaf.core.performance.monitoring.EntityProcessor;
import com.kneaf.core.performance.monitoring.PerformanceConfig;
import com.kneaf.core.performance.monitoring.PerformanceMetricsLogger;
import com.mojang.logging.LogUtils;
import java.lang.management.ManagementFactory;
import java.lang.management.OperatingSystemMXBean;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import net.minecraft.server.MinecraftServer;
import org.slf4j.Logger;

public class ThreadPoolManager {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static final PerformanceConfig CONFIG = PerformanceConfig.load();
    private static volatile ThreadPoolExecutor serverTaskExecutor = null;
    private static final AtomicInteger EXECUTOR_INIT_LOCK = new AtomicInteger(0);
    private static volatile ScheduledExecutorService adaptiveScheduler = null;
    private static final AtomicInteger SCHEDULER_INIT_LOCK = new AtomicInteger(0);
    private static final ExecutorMetrics EXECUTOR_METRICS = new ExecutorMetrics();
    private static final AtomicReference<Double> CPU_LOAD_CACHE = new AtomicReference();
    private static volatile long CPU_LOAD_CACHE_TIMESTAMP = 0L;
    private static final long CPU_LOAD_CACHE_TTL_MS = 3000L;

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public ThreadPoolExecutor getExecutor() {
        if (serverTaskExecutor != null && !serverTaskExecutor.isShutdown()) return serverTaskExecutor;
        if (EXECUTOR_INIT_LOCK.compareAndSet(0, 1)) {
            try {
                if (serverTaskExecutor != null && !serverTaskExecutor.isShutdown()) return serverTaskExecutor;
                ThreadPoolManager.createAdvancedThreadPool();
                ThreadPoolManager.startAdaptiveScaling();
                return serverTaskExecutor;
            }
            finally {
                EXECUTOR_INIT_LOCK.set(0);
            }
        }
        try {
            Thread.sleep(1L);
            return this.getExecutor();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        return this.getExecutor();
    }

    private static void createAdvancedThreadPool() {
        int availableProcessors;
        AtomicInteger threadIndex = new AtomicInteger(0);
        ThreadFactory factory = r -> {
            Thread t = new Thread(r, "kneaf-perf-worker-" + threadIndex.getAndIncrement());
            t.setDaemon(true);
            return t;
        };
        int coreThreads = CONFIG.getMinThreadpoolSize();
        int maxThreads = CONFIG.getMaxThreadpoolSize();
        if (CONFIG.isCpuAwareThreadSizing()) {
            availableProcessors = Runtime.getRuntime().availableProcessors();
            double cpuLoad = ThreadPoolManager.getSystemCpuLoad();
            maxThreads = cpuLoad < CONFIG.getCpuLoadThreshold() ? Math.min(maxThreads, availableProcessors) : Math.clamp((long)(availableProcessors / 2), 1, maxThreads);
            coreThreads = Math.min(coreThreads, maxThreads);
        }
        if (CONFIG.isAdaptiveThreadPool()) {
            availableProcessors = Runtime.getRuntime().availableProcessors();
            maxThreads = ThreadPoolManager.clamp(availableProcessors - 1, 1, maxThreads);
            coreThreads = Math.min(coreThreads, maxThreads);
        }
        LinkedBlockingQueue<Runnable> workQueue = CONFIG.isWorkStealingEnabled() ? new LinkedBlockingQueue(CONFIG.getWorkStealingQueueSize()) : new LinkedBlockingQueue<Runnable>();
        serverTaskExecutor = new ThreadPoolExecutor(coreThreads, maxThreads, (long)CONFIG.getThreadPoolKeepAliveSeconds(), TimeUnit.SECONDS, workQueue, factory);
        serverTaskExecutor.allowCoreThreadTimeOut(true);
        ThreadPoolManager.EXECUTOR_METRICS.currentThreadCount = coreThreads;
        ThreadPoolManager.EXECUTOR_METRICS.peakThreadCount = coreThreads;
    }

    private static double getSystemCpuLoad() {
        long now = System.currentTimeMillis();
        Double cached = CPU_LOAD_CACHE.get();
        if (cached != null && now - CPU_LOAD_CACHE_TIMESTAMP < 3000L) {
            return cached;
        }
        try {
            OperatingSystemMXBean osBean = ManagementFactory.getOperatingSystemMXBean();
            double systemLoad = osBean.getSystemLoadAverage();
            if (systemLoad < 0.0) {
                int availableProcessors = osBean.getAvailableProcessors();
                systemLoad = Math.min(1.0, systemLoad / (double)availableProcessors);
            } else {
                int availableProcessors = osBean.getAvailableProcessors();
                systemLoad = Math.min(1.0, systemLoad / (double)availableProcessors);
            }
            CPU_LOAD_CACHE.set(systemLoad);
            CPU_LOAD_CACHE_TIMESTAMP = now;
            return systemLoad;
        }
        catch (Exception e) {
            return 0.0;
        }
    }

    private static void performAdaptiveScaling() {
        if (EXECUTOR_INIT_LOCK.get() == 1) {
            return;
        }
        if (serverTaskExecutor == null || serverTaskExecutor.isShutdown()) {
            return;
        }
        double cpuLoad = ThreadPoolManager.getSystemCpuLoad();
        int currentCore = serverTaskExecutor.getCorePoolSize();
        int currentMax = serverTaskExecutor.getMaximumPoolSize();
        int minThreads = CONFIG.getMinThreadpoolSize();
        int maxThreads = CONFIG.getMaxThreadpoolSize();
        long now = System.currentTimeMillis();
        if (cpuLoad < CONFIG.getCpuLoadThreshold() * 0.7) {
            int newCore = Math.min(currentCore + 1, maxThreads);
            int newMax = Math.min(currentMax + 2, maxThreads);
            if (newCore > currentCore || newMax > currentMax) {
                serverTaskExecutor.setCorePoolSize(newCore);
                serverTaskExecutor.setMaximumPoolSize(newMax);
                ThreadPoolManager.EXECUTOR_METRICS.lastScaleUpTime = now;
                ++ThreadPoolManager.EXECUTOR_METRICS.scaleUpCount;
                ThreadPoolManager.EXECUTOR_METRICS.currentThreadCount = newCore;
                ThreadPoolManager.EXECUTOR_METRICS.peakThreadCount = Math.max(ThreadPoolManager.EXECUTOR_METRICS.peakThreadCount, newCore);
                LOGGER.debug("Scaled up thread pool: core={}, max={}", (Object)newCore, (Object)newMax);
            }
        } else if (cpuLoad > CONFIG.getCpuLoadThreshold() * 1.3) {
            int newCore = Math.max(currentCore - 1, minThreads);
            int newMax = Math.max(currentMax - 1, minThreads);
            if (newCore < currentCore || newMax < currentMax) {
                serverTaskExecutor.setCorePoolSize(newCore);
                serverTaskExecutor.setMaximumPoolSize(newMax);
                ThreadPoolManager.EXECUTOR_METRICS.lastScaleDownTime = now;
                ++ThreadPoolManager.EXECUTOR_METRICS.scaleDownCount;
                ThreadPoolManager.EXECUTOR_METRICS.currentThreadCount = newCore;
                LOGGER.debug("Scaled down thread pool: core={}, max={}", (Object)newCore, (Object)newMax);
            }
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private static void startAdaptiveScaling() {
        if (adaptiveScheduler != null && !adaptiveScheduler.isShutdown()) return;
        if (SCHEDULER_INIT_LOCK.compareAndSet(0, 1)) {
            try {
                if (adaptiveScheduler != null && !adaptiveScheduler.isShutdown()) return;
                adaptiveScheduler = Executors.newScheduledThreadPool(1, r -> {
                    Thread t = new Thread(r, "kneaf-adaptive-scaler");
                    t.setDaemon(true);
                    return t;
                });
                adaptiveScheduler.scheduleAtFixedRate(ThreadPoolManager::performAdaptiveScaling, 10L, 10L, TimeUnit.SECONDS);
                LOGGER.debug("Started adaptive thread scaling scheduler");
                return;
            }
            finally {
                SCHEDULER_INIT_LOCK.set(0);
            }
        }
        try {
            Thread.sleep(1L);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        ThreadPoolManager.startAdaptiveScaling();
    }

    public static int clamp(int v, int min, int max) {
        if (v < min) {
            return min;
        }
        if (v > max) {
            return max;
        }
        return v;
    }

    public static double clamp(double v, double min, double max) {
        if (v < min) {
            return min;
        }
        if (v > max) {
            return max;
        }
        return v;
    }

    public void shutdown() {
        if (serverTaskExecutor != null) {
            EXECUTOR_INIT_LOCK.set(1);
            try {
                serverTaskExecutor.shutdown();
                if (!serverTaskExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    serverTaskExecutor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                serverTaskExecutor.shutdownNow();
            }
            finally {
                serverTaskExecutor = null;
                EXECUTOR_INIT_LOCK.set(0);
            }
        }
        if (adaptiveScheduler != null) {
            SCHEDULER_INIT_LOCK.set(1);
            try {
                adaptiveScheduler.shutdown();
                if (!adaptiveScheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                    adaptiveScheduler.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                adaptiveScheduler.shutdownNow();
            }
            finally {
                adaptiveScheduler = null;
                SCHEDULER_INIT_LOCK.set(0);
            }
        }
    }

    public String getExecutorMetrics() {
        if (serverTaskExecutor == null) {
            return "{\"status\":\"not_initialized\"}";
        }
        ThreadPoolManager.EXECUTOR_METRICS.totalTasksSubmitted = serverTaskExecutor.getTaskCount();
        ThreadPoolManager.EXECUTOR_METRICS.totalTasksCompleted = serverTaskExecutor.getCompletedTaskCount();
        ThreadPoolManager.EXECUTOR_METRICS.currentQueueSize = serverTaskExecutor.getQueue().size();
        ThreadPoolManager.EXECUTOR_METRICS.currentThreadCount = serverTaskExecutor.getPoolSize();
        ThreadPoolManager.EXECUTOR_METRICS.currentUtilization = this.getExecutorUtilization();
        return EXECUTOR_METRICS.toJson();
    }

    public String getExecutorStatus() {
        if (serverTaskExecutor == null) {
            return "Executor not initialized";
        }
        return String.format("ThreadPoolExecutor[core=%d, max=%d, current=%d, active=%d, queue=%d, completed=%d]", serverTaskExecutor.getCorePoolSize(), serverTaskExecutor.getMaximumPoolSize(), serverTaskExecutor.getPoolSize(), serverTaskExecutor.getActiveCount(), serverTaskExecutor.getQueue().size(), serverTaskExecutor.getCompletedTaskCount());
    }

    public boolean isExecutorHealthy() {
        int maxQueueSize;
        if (serverTaskExecutor == null || serverTaskExecutor.isShutdown() || serverTaskExecutor.isTerminated()) {
            return false;
        }
        int queueSize = serverTaskExecutor.getQueue().size();
        int n = maxQueueSize = CONFIG.isWorkStealingEnabled() ? CONFIG.getWorkStealingQueueSize() : 1000;
        if ((double)queueSize > (double)maxQueueSize * 0.9) {
            return false;
        }
        double utilization = this.getExecutorUtilization();
        return serverTaskExecutor.getPoolSize() < serverTaskExecutor.getMaximumPoolSize() || !(utilization > 0.95);
    }

    public int getExecutorQueueSize() {
        if (serverTaskExecutor == null) {
            return 0;
        }
        return serverTaskExecutor.getQueue().size();
    }

    public double getExecutorUtilization() {
        if (serverTaskExecutor == null) {
            return 0.0;
        }
        int activeThreads = serverTaskExecutor.getActiveCount();
        int poolSize = Math.max(1, serverTaskExecutor.getPoolSize());
        return (double)activeThreads / (double)poolSize;
    }

    public void submitAsyncOptimizations(MinecraftServer server, EntityProcessor.EntityDataCollection data, boolean shouldProfile) {
        try {
            this.getExecutor().submit(() -> this.performAsyncOptimization(server, data, shouldProfile));
        }
        catch (Exception e) {
            LOGGER.debug("Executor rejected task; running synchronously", (Throwable)e);
        }
    }

    private void performAsyncOptimization(MinecraftServer server, EntityProcessor.EntityDataCollection data, boolean shouldProfile) {
        try {
            long processingStart = shouldProfile ? System.nanoTime() : 0L;
            EntityProcessor entityProcessor = new EntityProcessor();
            EntityProcessor.OptimizationResults results = entityProcessor.processOptimizations(data);
            if (shouldProfile) {
                long durationMs = (System.nanoTime() - processingStart) / 1000000L;
                PerformanceMetricsLogger.logLine(String.format("PERF: async_processing duration=%.2fms", durationMs));
            }
            server.execute(() -> this.applyOptimizationResults(server, results, shouldProfile));
        }
        catch (Exception e) {
            LOGGER.warn("Error during async processing of optimizations", (Throwable)e);
        }
    }

    private void applyOptimizationResults(MinecraftServer server, EntityProcessor.OptimizationResults results, boolean shouldProfile) {
        try {
            long applicationStart = shouldProfile ? System.nanoTime() : 0L;
            EntityProcessor entityProcessor = new EntityProcessor();
            entityProcessor.applyOptimizations(server, results);
            if (shouldProfile) {
                long durationMs = (System.nanoTime() - applicationStart) / 1000000L;
                PerformanceMetricsLogger.logLine(String.format("PERF: async_application duration=%.2fms", durationMs));
            }
            entityProcessor.removeItems(server, results.itemResult());
        }
        catch (Exception e) {
            LOGGER.warn("Error applying optimizations on server thread", (Throwable)e);
        }
    }

    public static final class ExecutorMetrics {
        public long totalTasksSubmitted = 0L;
        public long totalTasksCompleted = 0L;
        public long totalTasksRejected = 0L;
        public long currentQueueSize = 0L;
        public double currentUtilization = 0.0;
        public int currentThreadCount = 0;
        public int peakThreadCount = 0;
        public long lastScaleUpTime = 0L;
        public long lastScaleDownTime = 0L;
        public int scaleUpCount = 0;
        public int scaleDownCount = 0;

        public String toJson() {
            return String.format("{\"totalTasksSubmitted\":%d,\"totalTasksCompleted\":%d,\"totalTasksRejected\":%d,\"currentQueueSize\":%d,\"currentUtilization\":%.2f,\"currentThreadCount\":%d,\"peakThreadCount\":%d,\"scaleUpCount\":%d,\"scaleDownCount\":%d}", this.totalTasksSubmitted, this.totalTasksCompleted, this.totalTasksRejected, this.currentQueueSize, this.currentUtilization, this.currentThreadCount, this.peakThreadCount, this.scaleUpCount, this.scaleDownCount);
        }
    }
}

