/*
 * Decompiled with CFR 0.152.
 */
package com.kneaf.core.async;

import com.kneaf.core.exceptions.AsyncProcessingException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AsyncProcessor {
    private static final Logger LOGGER = LoggerFactory.getLogger(AsyncProcessor.class);
    private final ExecutorService executor;
    private final long defaultTimeoutMs;
    private final boolean enableLogging;
    private final boolean enableMetrics;
    private final String processorName;
    private final ConcurrentMap<String, AsyncMetrics> metricsMap = new ConcurrentHashMap<String, AsyncMetrics>();
    private final AtomicLong totalOperations = new AtomicLong(0L);
    private final AtomicLong failedOperations = new AtomicLong(0L);

    private AsyncProcessor(AsyncConfig config) {
        this.executor = config.getExecutor() != null ? config.getExecutor() : Executors.newFixedThreadPool(Math.max(1, Runtime.getRuntime().availableProcessors() - 1));
        this.defaultTimeoutMs = config.getDefaultTimeoutMs();
        this.enableLogging = config.isEnableLogging();
        this.enableMetrics = config.isEnableMetrics();
        this.processorName = config.getProcessorName();
    }

    public static AsyncProcessor create() {
        return new AsyncProcessor(new AsyncConfig());
    }

    public static AsyncProcessor create(AsyncConfig config) {
        return new AsyncProcessor(config);
    }

    public <T> CompletableFuture<T> supplyAsync(Supplier<T> supplier) {
        return this.supplyAsync(supplier, this.defaultTimeoutMs);
    }

    public <T> CompletableFuture<T> supplyAsync(Supplier<T> supplier, long timeoutMs) {
        return this.supplyAsync(supplier, timeoutMs, "async-operation");
    }

    public <T> CompletableFuture<T> supplyAsync(Supplier<T> supplier, long timeoutMs, String operationName) {
        this.totalOperations.incrementAndGet();
        long startTime = System.currentTimeMillis();
        if (this.enableLogging) {
            LOGGER.debug("{}: Starting async operation '{}'", (Object)this.processorName, (Object)operationName);
        }
        CompletionStage<Object> future = CompletableFuture.supplyAsync(() -> {
            try {
                Object result = supplier.get();
                long duration = System.currentTimeMillis() - startTime;
                if (this.enableMetrics) {
                    AsyncMetrics metrics = this.metricsMap.computeIfAbsent(operationName, k -> new AsyncMetrics());
                    metrics.recordOperation(duration, true);
                }
                if (this.enableLogging) {
                    LOGGER.debug("{}: Completed async operation '{}' in {}ms", new Object[]{this.processorName, operationName, duration});
                }
                return result;
            }
            catch (Exception e) {
                long duration = System.currentTimeMillis() - startTime;
                this.failedOperations.incrementAndGet();
                if (this.enableMetrics) {
                    AsyncMetrics metrics = this.metricsMap.computeIfAbsent(operationName, k -> new AsyncMetrics());
                    metrics.recordOperation(duration, false);
                }
                if (this.enableLogging) {
                    LOGGER.error("{}: Failed async operation '{}' after {}ms", new Object[]{this.processorName, operationName, duration, e});
                }
                throw new AsyncProcessingException(AsyncProcessingException.AsyncErrorType.SUPPLY_ASYNC_FAILED, operationName, "Async operation '" + operationName + "' failed: " + e.getMessage(), (Throwable)e);
            }
        }, this.executor);
        if (timeoutMs > 0L) {
            future = future.orTimeout(timeoutMs, TimeUnit.MILLISECONDS).exceptionally(throwable -> {
                if (throwable instanceof TimeoutException) {
                    this.failedOperations.incrementAndGet();
                    if (this.enableMetrics) {
                        AsyncMetrics metrics = this.metricsMap.computeIfAbsent(operationName, k -> new AsyncMetrics());
                        metrics.recordOperation(System.currentTimeMillis() - startTime, false);
                    }
                    if (this.enableLogging) {
                        LOGGER.error("{}: Async operation '{}' timed out after {}ms", new Object[]{this.processorName, operationName, timeoutMs});
                    }
                    throw AsyncProcessingException.timeoutExceeded(operationName, timeoutMs, null);
                }
                throw new CompletionException((Throwable)throwable);
            });
        }
        return future;
    }

    public <T> CompletableFuture<T> callAsync(Callable<T> callable) {
        return this.callAsync(callable, this.defaultTimeoutMs);
    }

    public <T> CompletableFuture<T> callAsync(Callable<T> callable, long timeoutMs) {
        return this.callAsync(callable, timeoutMs, "async-callable");
    }

    public <T> CompletableFuture<T> callAsync(Callable<T> callable, long timeoutMs, String operationName) {
        return this.supplyAsync(() -> {
            try {
                return callable.call();
            }
            catch (Exception e) {
                throw new RuntimeException("Callable execution failed", e);
            }
        }, timeoutMs, operationName);
    }

    public CompletableFuture<Void> runAsync(Runnable runnable) {
        return this.runAsync(runnable, this.defaultTimeoutMs);
    }

    public CompletableFuture<Void> runAsync(Runnable runnable, long timeoutMs) {
        return this.runAsync(runnable, timeoutMs, "async-runnable");
    }

    public CompletableFuture<Void> runAsync(Runnable runnable, long timeoutMs, String operationName) {
        return this.supplyAsync(() -> {
            runnable.run();
            return null;
        }, timeoutMs, operationName);
    }

    public <T> CompletableFuture<List<T>> supplyAllAsync(List<Supplier<T>> suppliers) {
        return this.supplyAllAsync(suppliers, this.defaultTimeoutMs);
    }

    public <T> CompletableFuture<List<T>> supplyAllAsync(List<Supplier<T>> suppliers, long timeoutMs) {
        if (suppliers == null || suppliers.isEmpty()) {
            return CompletableFuture.completedFuture(new ArrayList());
        }
        ArrayList futures = new ArrayList();
        for (int i = 0; i < suppliers.size(); ++i) {
            int index = i;
            futures.add(this.supplyAsync(suppliers.get(i), timeoutMs, "batch-operation-" + index));
        }
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).thenApply(v -> {
            ArrayList results = new ArrayList();
            for (CompletableFuture future : futures) {
                try {
                    results.add(future.get());
                }
                catch (Exception e) {
                    throw new AsyncProcessingException(AsyncProcessingException.AsyncErrorType.COMPLETION_EXCEPTION, "batch-operation", "Failed to get result from batch operation", (Throwable)e);
                }
            }
            return results;
        });
    }

    public <T> CompletableFuture<T> supplyWithRetry(Supplier<T> supplier, int maxRetries, long retryDelayMs) {
        return this.supplyWithRetry(supplier, maxRetries, retryDelayMs, "retry-operation");
    }

    public <T> CompletableFuture<T> supplyWithRetry(Supplier<T> supplier, int maxRetries, long retryDelayMs, String operationName) {
        return this.supplyAsync(() -> {
            Exception lastException = null;
            for (int attempt = 1; attempt <= maxRetries; ++attempt) {
                try {
                    if (this.enableLogging && attempt > 1) {
                        LOGGER.debug("{}: Retry attempt {} for operation '{}'", new Object[]{this.processorName, attempt, operationName});
                    }
                    return supplier.get();
                }
                catch (Exception e) {
                    lastException = e;
                    if (attempt >= maxRetries) continue;
                    try {
                        Thread.sleep(retryDelayMs);
                        continue;
                    }
                    catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                        throw AsyncProcessingException.batchRequestInterrupted(operationName, ie);
                    }
                }
            }
            throw new AsyncProcessingException(AsyncProcessingException.AsyncErrorType.EXECUTOR_SHUTDOWN, operationName, "Operation '" + operationName + "' failed after " + maxRetries + " attempts", (Throwable)lastException);
        }, this.defaultTimeoutMs * (long)maxRetries, operationName);
    }

    public <T> CompletableFuture<T> supplyAsyncWithErrorHandler(Supplier<T> supplier, Function<Exception, T> errorHandler) {
        return this.supplyAsyncWithErrorHandler(supplier, errorHandler, this.defaultTimeoutMs);
    }

    public <T> CompletableFuture<T> supplyAsyncWithErrorHandler(Supplier<T> supplier, Function<Exception, T> errorHandler, long timeoutMs) {
        return this.supplyAsyncWithErrorHandler(supplier, errorHandler, timeoutMs, "error-handler-operation");
    }

    public <T> CompletableFuture<T> supplyAsyncWithErrorHandler(Supplier<T> supplier, Function<Exception, T> errorHandler, long timeoutMs, String operationName) {
        return this.supplyAsync(() -> {
            try {
                return supplier.get();
            }
            catch (Exception e) {
                if (this.enableLogging) {
                    LOGGER.warn("{}: Error in operation '{}', applying error handler", new Object[]{this.processorName, operationName, e});
                }
                return errorHandler.apply(e);
            }
        }, timeoutMs, operationName);
    }

    public AsyncMetrics getMetrics(String operationName) {
        return this.metricsMap.getOrDefault(operationName, new AsyncMetrics());
    }

    public ConcurrentMap<String, AsyncMetrics> getAllMetrics() {
        return new ConcurrentHashMap<String, AsyncMetrics>(this.metricsMap);
    }

    public ProcessorStats getProcessorStats() {
        return new ProcessorStats(this.totalOperations.get(), this.failedOperations.get(), this.executor instanceof ThreadPoolExecutor ? ((ThreadPoolExecutor)this.executor).getActiveCount() : -1, this.executor instanceof ThreadPoolExecutor ? ((ThreadPoolExecutor)this.executor).getQueue().size() : -1);
    }

    public void shutdown() {
        if (this.executor != null && !this.executor.isShutdown()) {
            this.executor.shutdown();
            try {
                if (!this.executor.awaitTermination(30L, TimeUnit.SECONDS)) {
                    this.executor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.executor.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
        if (this.enableLogging) {
            LOGGER.info("{}: Shutdown completed. Total operations: {}, Failed: {}", new Object[]{this.processorName, this.totalOperations.get(), this.failedOperations.get()});
        }
    }

    public static class AsyncConfig {
        private ExecutorService executor;
        private long defaultTimeoutMs = 30000L;
        private boolean enableLogging = true;
        private boolean enableMetrics = true;
        private String processorName = "AsyncProcessor";

        public AsyncConfig executor(ExecutorService executor) {
            this.executor = executor;
            return this;
        }

        public AsyncConfig defaultTimeoutMs(long defaultTimeoutMs) {
            this.defaultTimeoutMs = defaultTimeoutMs;
            return this;
        }

        public AsyncConfig enableLogging(boolean enableLogging) {
            this.enableLogging = enableLogging;
            return this;
        }

        public AsyncConfig enableMetrics(boolean enableMetrics) {
            this.enableMetrics = enableMetrics;
            return this;
        }

        public AsyncConfig processorName(String processorName) {
            this.processorName = processorName;
            return this;
        }

        public ExecutorService getExecutor() {
            return this.executor;
        }

        public long getDefaultTimeoutMs() {
            return this.defaultTimeoutMs;
        }

        public boolean isEnableLogging() {
            return this.enableLogging;
        }

        public boolean isEnableMetrics() {
            return this.enableMetrics;
        }

        public String getProcessorName() {
            return this.processorName;
        }
    }

    public static class AsyncMetrics {
        private final AtomicLong operations = new AtomicLong(0L);
        private final AtomicLong failures = new AtomicLong(0L);
        private final AtomicLong totalDuration = new AtomicLong(0L);
        private final AtomicLong minDuration = new AtomicLong(Long.MAX_VALUE);
        private final AtomicLong maxDuration = new AtomicLong(0L);

        public void recordOperation(long duration, boolean success) {
            this.operations.incrementAndGet();
            this.totalDuration.addAndGet(duration);
            if (success) {
                this.updateMinMax(duration);
            } else {
                this.failures.incrementAndGet();
            }
        }

        private void updateMinMax(long duration) {
            long currentMax;
            long currentMin;
            do {
                currentMin = this.minDuration.get();
                currentMax = this.maxDuration.get();
            } while (duration < currentMin && !this.minDuration.compareAndSet(currentMin, duration) || duration > currentMax && !this.maxDuration.compareAndSet(currentMax, duration));
        }

        public long getOperations() {
            return this.operations.get();
        }

        public long getFailures() {
            return this.failures.get();
        }

        public long getTotalDuration() {
            return this.totalDuration.get();
        }

        public long getMinDuration() {
            return this.operations.get() > 0L ? this.minDuration.get() : 0L;
        }

        public long getMaxDuration() {
            return this.maxDuration.get();
        }

        public double getAverageDuration() {
            long ops = this.operations.get();
            return ops > 0L ? (double)this.totalDuration.get() / (double)ops : 0.0;
        }

        public double getFailureRate() {
            long ops = this.operations.get();
            return ops > 0L ? (double)this.failures.get() / (double)ops : 0.0;
        }
    }

    public static class ProcessorStats {
        private final long totalOperations;
        private final long failedOperations;
        private final int activeThreads;
        private final int queueSize;

        public ProcessorStats(long totalOperations, long failedOperations, int activeThreads, int queueSize) {
            this.totalOperations = totalOperations;
            this.failedOperations = failedOperations;
            this.activeThreads = activeThreads;
            this.queueSize = queueSize;
        }

        public long getTotalOperations() {
            return this.totalOperations;
        }

        public long getFailedOperations() {
            return this.failedOperations;
        }

        public int getActiveThreads() {
            return this.activeThreads;
        }

        public int getQueueSize() {
            return this.queueSize;
        }

        public double getFailureRate() {
            return this.totalOperations > 0L ? (double)this.failedOperations / (double)this.totalOperations : 0.0;
        }

        public String toString() {
            return String.format("ProcessorStats{total=%d, failed=%d, failureRate=%.2f%%, activeThreads=%d, queueSize=%d}", this.totalOperations, this.failedOperations, this.getFailureRate() * 100.0, this.activeThreads, this.queueSize);
        }
    }
}

