/*
 * Decompiled with CFR 0.152.
 */
package com.kneaf.core.chunkstorage;

import com.kneaf.core.chunkstorage.AbstractDatabaseAdapter;
import com.kneaf.core.chunkstorage.DatabaseAdapter;
import java.io.BufferedOutputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InMemoryDatabaseAdapter
extends AbstractDatabaseAdapter {
    private static final Logger LOGGER = LoggerFactory.getLogger(InMemoryDatabaseAdapter.class);
    private final Map<String, byte[]> chunkStorage = new ConcurrentHashMap<String, byte[]>();
    private final AtomicLong totalSizeBytes = new AtomicLong(0L);
    private final ReadWriteLock statsLock = new ReentrantReadWriteLock();
    private volatile long readLatencyMs = 0L;
    private volatile long writeLatencyMs = 0L;
    private volatile long lastMaintenanceTime = System.currentTimeMillis();
    private volatile boolean healthy = true;
    private final String databaseType;

    public InMemoryDatabaseAdapter() {
        this("in-memory");
    }

    public InMemoryDatabaseAdapter(String databaseType) {
        this.databaseType = databaseType;
        LOGGER.info("Initialized InMemoryDatabaseAdapter of type: {}", (Object)databaseType);
    }

    @Override
    public void putChunk(String key, byte[] data) throws IOException {
        this.validateKey(key);
        this.validateData(data);
        long startTime = System.nanoTime();
        try {
            byte[] existingData = this.chunkStorage.get(key);
            int existingSize = existingData != null ? existingData.length : 0;
            this.chunkStorage.put(key, (byte[])data.clone());
            this.totalSizeBytes.addAndGet((long)data.length - (long)existingSize);
            long duration = System.nanoTime() - startTime;
            this.writeLatencyMs = duration / 1000000L;
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Stored chunk {} ({} bytes) in {} ms", new Object[]{key, data.length, this.writeLatencyMs});
            }
        }
        catch (Exception e) {
            LOGGER.error("Failed to store chunk {}", (Object)key, (Object)e);
            throw new IOException("Failed to store chunk", e);
        }
    }

    @Override
    public CompletableFuture<Void> putChunkAsync(String key, byte[] data) {
        return CompletableFuture.runAsync(() -> {
            try {
                this.putChunk(key, data);
            }
            catch (IOException e) {
                LOGGER.error("Failed to store chunk asynchronously {}", (Object)key, (Object)e);
                throw new DatabaseOperationException("Failed to store chunk asynchronously", e);
            }
        });
    }

    @Override
    public Optional<byte[]> getChunk(String key) throws IOException {
        this.validateKey(key);
        long startTime = System.nanoTime();
        try {
            byte[] data = this.chunkStorage.get(key);
            long duration = System.nanoTime() - startTime;
            this.readLatencyMs = duration / 1000000L;
            if (LOGGER.isDebugEnabled()) {
                if (data != null) {
                    LOGGER.debug("Retrieved chunk {} ({} bytes) in {} ms", new Object[]{key, data.length, this.readLatencyMs});
                } else {
                    LOGGER.debug("Chunk {} not found (retrieved in {} ms)", (Object)key, (Object)this.readLatencyMs);
                }
            }
            return data != null ? Optional.of((byte[])data.clone()) : Optional.empty();
        }
        catch (Exception e) {
            LOGGER.error("Failed to retrieve chunk {}", (Object)key, (Object)e);
            throw new IOException("Failed to retrieve chunk", e);
        }
    }

    @Override
    public CompletableFuture<Optional<byte[]>> getChunkAsync(String key) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                return this.getChunk(key);
            }
            catch (IOException e) {
                LOGGER.error("Failed to retrieve chunk asynchronously {}", (Object)key, (Object)e);
                throw new DatabaseOperationException("Failed to retrieve chunk asynchronously", e);
            }
        });
    }

    @Override
    public boolean deleteChunk(String key) throws IOException {
        this.validateKey(key);
        try {
            byte[] removedData = this.chunkStorage.remove(key);
            if (removedData != null) {
                this.totalSizeBytes.addAndGet(-removedData.length);
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Deleted chunk {} ({} bytes)", (Object)key, (Object)removedData.length);
                }
                return true;
            }
            return false;
        }
        catch (Exception e) {
            LOGGER.error("Failed to delete chunk {}", (Object)key, (Object)e);
            throw new IOException("Failed to delete chunk", e);
        }
    }

    @Override
    public CompletableFuture<Boolean> deleteChunkAsync(String key) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                return this.deleteChunk(key);
            }
            catch (IOException e) {
                LOGGER.error("Failed to delete chunk asynchronously {}", (Object)key, (Object)e);
                throw new DatabaseOperationException("Failed to delete chunk asynchronously", e);
            }
        });
    }

    @Override
    public boolean hasChunk(String key) throws IOException {
        this.validateKey(key);
        try {
            return this.chunkStorage.containsKey(key);
        }
        catch (Exception e) {
            LOGGER.error("Failed to check chunk existence {}", (Object)key, (Object)e);
            throw new IOException("Failed to check chunk existence", e);
        }
    }

    @Override
    public long getChunkCount() throws IOException {
        try {
            return this.chunkStorage.size();
        }
        catch (Exception e) {
            LOGGER.error("Failed to get chunk count", (Throwable)e);
            throw new IOException("Failed to get chunk count", e);
        }
    }

    @Override
    public DatabaseAdapter.DatabaseStats getStats() throws IOException {
        this.statsLock.readLock().lock();
        try {
            long chunkCount = this.getChunkCount();
            long currentSize = this.totalSizeBytes.get();
            long currentReadLatency = this.readLatencyMs;
            long currentWriteLatency = this.writeLatencyMs;
            long lastMaint = this.lastMaintenanceTime;
            boolean health = this.healthy;
            DatabaseAdapter.DatabaseStats databaseStats = new DatabaseAdapter.DatabaseStats(chunkCount, currentSize, currentReadLatency, currentWriteLatency, lastMaint, health, 0L, 0L, 0L, 0L, 0L);
            return databaseStats;
        }
        catch (Exception e) {
            LOGGER.error("Failed to get database stats", (Throwable)e);
            throw new IOException("Failed to get database stats", e);
        }
        finally {
            this.statsLock.readLock().unlock();
        }
    }

    @Override
    public void performMaintenance() throws IOException {
        try {
            long startTime = System.currentTimeMillis();
            this.healthy = true;
            this.lastMaintenanceTime = startTime;
            if (LOGGER.isInfoEnabled()) {
                long duration = System.currentTimeMillis() - startTime;
                LOGGER.info("Database maintenance completed in {} ms for {} chunks", (Object)duration, (Object)this.getChunkCount());
            }
        }
        catch (Exception e) {
            LOGGER.error("Failed to perform database maintenance", (Throwable)e);
            this.healthy = false;
            throw new IOException("Failed to perform database maintenance", e);
        }
    }

    @Override
    public void close() throws IOException {
        try {
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("Closing InMemoryDatabaseAdapter with {} chunks ({} bytes)", (Object)this.getChunkCount(), (Object)this.totalSizeBytes.get());
            }
            this.chunkStorage.clear();
            this.totalSizeBytes.set(0L);
            this.healthy = false;
        }
        catch (Exception e) {
            LOGGER.error("Failed to close database", (Throwable)e);
            throw new IOException("Failed to close database", e);
        }
    }

    @Override
    public String getDatabaseType() {
        return this.databaseType;
    }

    @Override
    public boolean isHealthy() {
        return this.healthy;
    }

    @Override
    public void createBackup(String backupPath) throws IOException {
        this.validateBackupPath(backupPath);
        File backupFile = new File(backupPath);
        File parentDir = backupFile.getParentFile();
        if (parentDir != null && !parentDir.exists() && !parentDir.mkdirs()) {
            throw new IOException("Failed to create backup directory: " + parentDir.getAbsolutePath());
        }
        try {
            long chunkCount = this.getChunkCount();
            long totalSize = this.totalSizeBytes.get();
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("Creating backup at {} for {} chunks ({} bytes)", new Object[]{backupPath, chunkCount, totalSize});
            }
            File tempFile = new File(backupPath + ".tmp");
            try (DataOutputStream dos = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(tempFile)));){
                dos.writeUTF("KNEAF_DB_BACKUP_v1");
                dos.writeLong(System.currentTimeMillis());
                dos.writeUTF(this.databaseType);
                dos.writeLong(chunkCount);
                dos.writeLong(totalSize);
                dos.writeUTF("InMemoryDB_v1");
                dos.writeInt(this.chunkStorage.size());
                long writtenChunks = 0L;
                long writtenBytes = 0L;
                for (Map.Entry<String, byte[]> entry : this.chunkStorage.entrySet()) {
                    String key = entry.getKey();
                    byte[] data = entry.getValue();
                    dos.writeUTF(key);
                    dos.writeInt(data.length);
                    dos.write(data);
                    writtenBytes += (long)data.length;
                    if (!LOGGER.isDebugEnabled() || ++writtenChunks % 1000L != 0L) continue;
                    LOGGER.debug("Backup progress: {}/{} chunks written", (Object)writtenChunks, (Object)chunkCount);
                }
                dos.writeUTF("BACKUP_END");
                dos.writeLong(writtenChunks);
                dos.writeLong(writtenBytes);
                dos.flush();
                if (LOGGER.isInfoEnabled()) {
                    LOGGER.info("Backup serialization completed: {} chunks ({} bytes) written to temporary file", (Object)writtenChunks, (Object)writtenBytes);
                }
            }
            catch (Exception e) {
                if (tempFile.exists() && !tempFile.delete()) {
                    LOGGER.warn("Failed to clean up temporary backup file: {}", (Object)tempFile.getAbsolutePath());
                }
                throw e;
            }
            if (!tempFile.renameTo(backupFile)) {
                throw new IOException("Failed to rename temporary backup file to: " + backupPath);
            }
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("Backup successfully created at {} ({} chunks, {} bytes)", new Object[]{backupPath, chunkCount, totalSize});
            }
        }
        catch (Exception e) {
            LOGGER.error("Failed to create backup at {}", (Object)backupPath, (Object)e);
            throw new IOException("Failed to create backup", e);
        }
    }

    public static class DatabaseOperationException
    extends RuntimeException {
        public DatabaseOperationException(String message, Throwable cause) {
            super(message, cause);
        }
    }
}

