/*
 * Decompiled with CFR 0.152.
 */
package com.kneaf.core.protocol;

import com.google.gson.Gson;
import com.kneaf.core.exceptions.KneafCoreException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ProtocolProcessor {
    private static final Logger LOGGER = LoggerFactory.getLogger(ProtocolProcessor.class);
    private static final Gson GSON = new Gson();
    private final boolean nativeAvailable;
    private final ProtocolType defaultProtocol;

    public ProtocolProcessor(boolean nativeAvailable, ProtocolType defaultProtocol) {
        this.nativeAvailable = nativeAvailable;
        this.defaultProtocol = defaultProtocol;
    }

    public <T, R> ProtocolResult<R> processWithFallback(T input, String operationName, BinarySerializer<T> binarySerializer, BinaryNativeCaller<byte[]> binaryNativeCaller, BinaryDeserializer<R> binaryDeserializer, JsonInputPreparer<T> jsonInputPreparer, JsonNativeCaller<String> jsonNativeCaller, JsonResultParser<R> jsonResultParser, R fallbackResult) {
        if (this.defaultProtocol == ProtocolType.JSON || !this.nativeAvailable) {
            return this.processJson(input, operationName, jsonInputPreparer, jsonNativeCaller, jsonResultParser, fallbackResult);
        }
        if (this.defaultProtocol == ProtocolType.BINARY) {
            return this.processBinary(input, operationName, binarySerializer, binaryNativeCaller, binaryDeserializer, fallbackResult);
        }
        ProtocolResult<Object> binaryResult = this.processBinary(input, operationName, binarySerializer, binaryNativeCaller, binaryDeserializer, null);
        if (binaryResult.isSuccess()) {
            return binaryResult;
        }
        LOGGER.debug("Binary protocol failed for {}, falling back to JSON: {}", (Object)operationName, (Object)binaryResult.getErrorMessage());
        return this.processJson(input, operationName, jsonInputPreparer, jsonNativeCaller, jsonResultParser, fallbackResult);
    }

    private <T, R> ProtocolResult<R> processBinary(T input, String operationName, BinarySerializer<T> binarySerializer, BinaryNativeCaller<byte[]> binaryNativeCaller, BinaryDeserializer<R> binaryDeserializer, R fallbackResult) {
        if (!this.nativeAvailable) {
            String message = "Native library not available for binary protocol";
            LOGGER.debug("{}: {}", (Object)message, (Object)operationName);
            return fallbackResult != null ? ProtocolResult.success(fallbackResult, ProtocolType.BINARY) : ProtocolResult.failure(message);
        }
        try {
            ByteBuffer inputBuffer = binarySerializer.serialize(input);
            byte[] resultBytes = binaryNativeCaller.callNative(inputBuffer);
            if (resultBytes == null) {
                String message = "Native method returned null";
                LOGGER.debug("{} for {}: {}", new Object[]{message, operationName, operationName});
                return fallbackResult != null ? ProtocolResult.success(fallbackResult, ProtocolType.BINARY) : ProtocolResult.failure(message);
            }
            this.logBinaryResult(operationName, resultBytes);
            R result = binaryDeserializer.deserialize(resultBytes);
            return ProtocolResult.success(result, ProtocolType.BINARY);
        }
        catch (Exception e) {
            String message = String.format("Binary protocol processing failed: %s", e.getMessage());
            LOGGER.debug("{} for {}: {}", new Object[]{message, operationName, e.getMessage()});
            if (fallbackResult != null) {
                return ProtocolResult.success(fallbackResult, ProtocolType.BINARY);
            }
            return ProtocolResult.failure(message);
        }
    }

    private <T, R> ProtocolResult<R> processJson(T input, String operationName, JsonInputPreparer<T> jsonInputPreparer, JsonNativeCaller<String> jsonNativeCaller, JsonResultParser<R> jsonResultParser, R fallbackResult) {
        try {
            Map<String, Object> jsonInputMap = jsonInputPreparer.prepareInput(input);
            String jsonInput = GSON.toJson(jsonInputMap);
            String jsonResult = jsonNativeCaller.callNative(jsonInput);
            if (jsonResult == null) {
                String message = "Native method returned null";
                LOGGER.warn("{} for {} returned null, using fallback", (Object)operationName, (Object)operationName);
                return fallbackResult != null ? ProtocolResult.success(fallbackResult, ProtocolType.JSON) : ProtocolResult.failure(message);
            }
            R result = jsonResultParser.parseResult(jsonResult);
            return ProtocolResult.success(result, ProtocolType.JSON);
        }
        catch (Exception e) {
            String message = String.format("JSON protocol processing failed: %s", e.getMessage());
            LOGGER.warn("{} for {}: {}", new Object[]{message, operationName, e.getMessage(), e});
            if (fallbackResult != null) {
                return ProtocolResult.success(fallbackResult, ProtocolType.JSON);
            }
            return ProtocolResult.failure(message);
        }
    }

    private void logBinaryResult(String operationName, byte[] resultBytes) {
        if (!LOGGER.isDebugEnabled()) {
            return;
        }
        try {
            String prefix = this.bytesPrefixHex(resultBytes, 64);
            LOGGER.debug("[BINARY] Received {} bytes from native for {}: prefix={}", new Object[]{resultBytes.length, operationName, prefix});
            if (resultBytes.length >= 4) {
                ByteBuffer buffer = ByteBuffer.wrap(resultBytes).order(ByteOrder.LITTLE_ENDIAN);
                int int0 = buffer.getInt(0);
                LOGGER.debug("[BINARY] First int: {}", (Object)int0);
                if (resultBytes.length >= 8) {
                    long long0 = buffer.getLong(0);
                    LOGGER.debug("[BINARY] First long: {}", (Object)long0);
                }
            }
        }
        catch (Exception e) {
            LOGGER.debug("[BINARY] Failed to log binary result details: {}", (Object)e.getMessage());
        }
    }

    private String bytesPrefixHex(byte[] data, int maxBytes) {
        if (data == null) {
            return "";
        }
        int len = Math.min(data.length, Math.max(0, maxBytes));
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < len; ++i) {
            sb.append(String.format("%02x", data[i] & 0xFF));
            if (i >= len - 1) continue;
            sb.append(',');
        }
        if (data.length > len) {
            sb.append("...");
        }
        return sb.toString();
    }

    public static ProtocolProcessor createAuto(boolean nativeAvailable) {
        return new ProtocolProcessor(nativeAvailable, ProtocolType.AUTO);
    }

    public static ProtocolProcessor createBinary(boolean nativeAvailable) {
        return new ProtocolProcessor(nativeAvailable, ProtocolType.BINARY);
    }

    public static ProtocolProcessor createJson() {
        return new ProtocolProcessor(false, ProtocolType.JSON);
    }

    public static enum ProtocolType {
        BINARY("Binary"),
        JSON("JSON"),
        AUTO("Auto");

        private final String name;

        private ProtocolType(String name) {
            this.name = name;
        }

        public String getName() {
            return this.name;
        }
    }

    @FunctionalInterface
    public static interface JsonInputPreparer<T> {
        public Map<String, Object> prepareInput(T var1);
    }

    @FunctionalInterface
    public static interface JsonNativeCaller<R> {
        public R callNative(String var1) throws Exception;
    }

    @FunctionalInterface
    public static interface JsonResultParser<R> {
        public R parseResult(String var1) throws Exception;
    }

    public static class ProtocolResult<T> {
        private final T data;
        private final ProtocolType protocolUsed;
        private final boolean success;
        private final String errorMessage;

        private ProtocolResult(T data, ProtocolType protocolUsed, boolean success, String errorMessage) {
            this.data = data;
            this.protocolUsed = protocolUsed;
            this.success = success;
            this.errorMessage = errorMessage;
        }

        public static <T> ProtocolResult<T> success(T data, ProtocolType protocolUsed) {
            return new ProtocolResult<T>(data, protocolUsed, true, null);
        }

        public static <T> ProtocolResult<T> failure(String errorMessage) {
            return new ProtocolResult<Object>(null, null, false, errorMessage);
        }

        public T getData() {
            return this.data;
        }

        public ProtocolType getProtocolUsed() {
            return this.protocolUsed;
        }

        public boolean isSuccess() {
            return this.success;
        }

        public String getErrorMessage() {
            return this.errorMessage;
        }

        public T getDataOrThrow() {
            if (!this.success) {
                throw new KneafCoreException(KneafCoreException.ErrorCategory.PROTOCOL_ERROR, "Protocol processing failed: " + this.errorMessage);
            }
            return this.data;
        }
    }

    @FunctionalInterface
    public static interface BinarySerializer<T> {
        public ByteBuffer serialize(T var1) throws Exception;
    }

    @FunctionalInterface
    public static interface BinaryNativeCaller<R> {
        public R callNative(ByteBuffer var1) throws Exception;
    }

    @FunctionalInterface
    public static interface BinaryDeserializer<R> {
        public R deserialize(byte[] var1) throws Exception;
    }
}

