/*
 * Decompiled with CFR 0.152.
 */
package com.kneaf.core.resource;

import com.kneaf.core.exceptions.KneafCoreException;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ResourceManager
implements Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(ResourceManager.class);
    private final ConcurrentMap<String, ResourceWrapper> resources = new ConcurrentHashMap<String, ResourceWrapper>();
    private final AtomicBoolean shutdown = new AtomicBoolean(false);
    private final String managerName;
    private final List<ResourceLifecycleListener> listeners = new CopyOnWriteArrayList<ResourceLifecycleListener>();

    public ResourceManager(String managerName) {
        this.managerName = managerName;
    }

    public void registerResource(ManagedResource resource) throws Exception {
        if (this.shutdown.get()) {
            throw new IllegalStateException("ResourceManager " + this.managerName + " is shutdown");
        }
        String resourceName = resource.getResourceName();
        ResourceWrapper existing = this.resources.putIfAbsent(resourceName, new ResourceWrapper(resource));
        if (existing != null) {
            throw new IllegalArgumentException("Resource with name '" + resourceName + "' already registered");
        }
        try {
            ResourceWrapper wrapper = (ResourceWrapper)this.resources.get(resourceName);
            wrapper.initialize();
            this.notifyListeners(ResourceLifecycleEvent.Type.REGISTERED, resource);
            LOGGER.info("Registered resource: {} (type: {})", (Object)resourceName, (Object)resource.getResourceType());
        }
        catch (Exception e) {
            this.resources.remove(resourceName);
            this.notifyListeners(ResourceLifecycleEvent.Type.REGISTRATION_FAILED, resource);
            throw new KneafCoreException(KneafCoreException.ErrorCategory.RESOURCE_MANAGEMENT, "registerResource", "Failed to register resource: " + resourceName, e);
        }
    }

    public void startResource(String resourceName) throws Exception {
        ResourceWrapper wrapper = (ResourceWrapper)this.resources.get(resourceName);
        if (wrapper == null) {
            throw new IllegalArgumentException("Resource not found: " + resourceName);
        }
        try {
            wrapper.start();
            this.notifyListeners(ResourceLifecycleEvent.Type.STARTED, wrapper.getResource());
            LOGGER.info("Started resource: {}", (Object)resourceName);
        }
        catch (Exception e) {
            this.notifyListeners(ResourceLifecycleEvent.Type.START_FAILED, wrapper.getResource());
            throw new KneafCoreException(KneafCoreException.ErrorCategory.RESOURCE_MANAGEMENT, "startResource", "Failed to start resource: " + resourceName, e);
        }
    }

    public void stopResource(String resourceName) throws Exception {
        ResourceWrapper wrapper = (ResourceWrapper)this.resources.get(resourceName);
        if (wrapper == null) {
            return;
        }
        try {
            wrapper.stop();
            this.notifyListeners(ResourceLifecycleEvent.Type.STOPPED, wrapper.getResource());
            LOGGER.info("Stopped resource: {}", (Object)resourceName);
        }
        catch (Exception e) {
            this.notifyListeners(ResourceLifecycleEvent.Type.STOP_FAILED, wrapper.getResource());
            throw new KneafCoreException(KneafCoreException.ErrorCategory.RESOURCE_MANAGEMENT, "stopResource", "Failed to stop resource: " + resourceName, e);
        }
    }

    public void startAll() throws Exception {
        ArrayList<Exception> exceptions = new ArrayList<Exception>();
        for (ResourceWrapper wrapper : this.resources.values()) {
            try {
                this.startResource(wrapper.getResource().getResourceName());
            }
            catch (Exception e) {
                exceptions.add(e);
            }
        }
        if (!exceptions.isEmpty()) {
            KneafCoreException aggregated = new KneafCoreException(KneafCoreException.ErrorCategory.RESOURCE_MANAGEMENT, "startAll", "Failed to start " + exceptions.size() + " resources", (Throwable)exceptions.get(0));
            exceptions.subList(1, exceptions.size()).forEach(aggregated::addSuppressed);
            throw aggregated;
        }
    }

    public void stopAll() throws Exception {
        this.shutdown.set(true);
        ArrayList resourceList = new ArrayList(this.resources.values());
        Collections.reverse(resourceList);
        ArrayList<Exception> exceptions = new ArrayList<Exception>();
        for (ResourceWrapper wrapper : resourceList) {
            try {
                this.stopResource(wrapper.getResource().getResourceName());
            }
            catch (Exception e) {
                exceptions.add(e);
            }
        }
        if (!exceptions.isEmpty()) {
            KneafCoreException aggregated = new KneafCoreException(KneafCoreException.ErrorCategory.RESOURCE_MANAGEMENT, "stopAll", "Failed to stop " + exceptions.size() + " resources", (Throwable)exceptions.get(0));
            exceptions.subList(1, exceptions.size()).forEach(aggregated::addSuppressed);
            throw aggregated;
        }
    }

    public ResourceHealthReport checkHealth() {
        HashMap<String, Boolean> healthStatus = new HashMap<String, Boolean>();
        HashMap<String, ResourceState> stateStatus = new HashMap<String, ResourceState>();
        for (Map.Entry entry : this.resources.entrySet()) {
            String name = (String)entry.getKey();
            ResourceWrapper wrapper = (ResourceWrapper)entry.getValue();
            healthStatus.put(name, wrapper.isHealthy());
            stateStatus.put(name, wrapper.getState());
        }
        return new ResourceHealthReport(healthStatus, stateStatus);
    }

    public ManagedResource getResource(String resourceName) {
        ResourceWrapper wrapper = (ResourceWrapper)this.resources.get(resourceName);
        return wrapper != null ? wrapper.getResource() : null;
    }

    public List<String> getResourceNames() {
        return new ArrayList<String>(this.resources.keySet());
    }

    public ResourceStatistics getStatistics() {
        HashMap<String, Object> stats = new HashMap<String, Object>();
        long totalUptime = 0L;
        int healthyCount = 0;
        for (Map.Entry entry : this.resources.entrySet()) {
            String name = (String)entry.getKey();
            ResourceWrapper wrapper = (ResourceWrapper)entry.getValue();
            HashMap<String, Object> resourceStats = new HashMap<String, Object>();
            resourceStats.put("state", wrapper.getState().name());
            resourceStats.put("healthy", wrapper.isHealthy());
            resourceStats.put("initializationTime", wrapper.getInitializationTime());
            resourceStats.put("lastHealthCheckTime", wrapper.getLastHealthCheckTime());
            if (wrapper.isHealthy()) {
                ++healthyCount;
            }
            stats.put(name, resourceStats);
            totalUptime += System.currentTimeMillis() - wrapper.getInitializationTime();
        }
        return new ResourceStatistics(this.resources.size(), healthyCount, totalUptime, stats);
    }

    public void addLifecycleListener(ResourceLifecycleListener listener) {
        this.listeners.add(listener);
    }

    public void removeLifecycleListener(ResourceLifecycleListener listener) {
        this.listeners.remove(listener);
    }

    private void notifyListeners(ResourceLifecycleEvent.Type type, ManagedResource resource) {
        ResourceLifecycleEvent event = new ResourceLifecycleEvent(type, resource, this);
        for (ResourceLifecycleListener listener : this.listeners) {
            try {
                listener.onResourceLifecycleEvent(event);
            }
            catch (Exception e) {
                LOGGER.warn("Lifecycle listener threw exception", (Throwable)e);
            }
        }
    }

    @Override
    public void close() throws IOException {
        try {
            this.stopAll();
        }
        catch (Exception e) {
            throw new IOException("Failed to close ResourceManager", e);
        }
    }

    public String getManagerName() {
        return this.managerName;
    }

    public boolean isShutdown() {
        return this.shutdown.get();
    }

    public static ResourceManager create(String managerName) {
        return new ResourceManager(managerName);
    }

    public static interface ManagedResource {
        public void initialize() throws Exception;

        default public void start() throws Exception {
        }

        public void stop() throws Exception;

        default public boolean isHealthy() {
            return true;
        }

        public String getResourceName();

        default public String getResourceType() {
            return this.getClass().getSimpleName();
        }

        default public Object getMetadata() {
            return null;
        }
    }

    private static class ResourceWrapper {
        private final ManagedResource resource;
        private final AtomicReference<ResourceState> state = new AtomicReference<ResourceState>(ResourceState.INITIALIZED);
        private final AtomicBoolean healthCheckEnabled = new AtomicBoolean(true);
        private final long initializationTime;
        private volatile long lastHealthCheckTime = this.initializationTime = System.currentTimeMillis();

        ResourceWrapper(ManagedResource resource) {
            this.resource = resource;
        }

        void initialize() throws Exception {
            if (!this.state.compareAndSet(ResourceState.INITIALIZED, ResourceState.STARTED)) {
                throw new IllegalStateException("Resource " + this.resource.getResourceName() + " is already initialized");
            }
            this.resource.initialize();
        }

        void start() throws Exception {
            if (this.state.get() != ResourceState.STARTED) {
                throw new IllegalStateException("Resource " + this.resource.getResourceName() + " is not in STARTED state");
            }
            this.resource.start();
        }

        void stop() throws Exception {
            if (!this.state.compareAndSet(ResourceState.STARTED, ResourceState.STOPPING)) {
                return;
            }
            try {
                this.resource.stop();
                this.state.set(ResourceState.STOPPED);
            }
            catch (Exception e) {
                this.state.set(ResourceState.FAILED);
                throw e;
            }
        }

        boolean isHealthy() {
            if (!this.healthCheckEnabled.get()) {
                return true;
            }
            try {
                boolean healthy = this.resource.isHealthy();
                this.lastHealthCheckTime = System.currentTimeMillis();
                return healthy;
            }
            catch (Exception e) {
                LOGGER.warn("Health check failed for resource {}: {}", (Object)this.resource.getResourceName(), (Object)e.getMessage());
                return false;
            }
        }

        ResourceState getState() {
            return this.state.get();
        }

        ManagedResource getResource() {
            return this.resource;
        }

        long getInitializationTime() {
            return this.initializationTime;
        }

        long getLastHealthCheckTime() {
            return this.lastHealthCheckTime;
        }

        void setHealthCheckEnabled(boolean enabled) {
            this.healthCheckEnabled.set(enabled);
        }
    }

    public static class ResourceLifecycleEvent {
        private final Type type;
        private final ManagedResource resource;
        private final ResourceManager source;

        public ResourceLifecycleEvent(Type type, ManagedResource resource, ResourceManager source) {
            this.type = type;
            this.resource = resource;
            this.source = source;
        }

        public Type getType() {
            return this.type;
        }

        public ManagedResource getResource() {
            return this.resource;
        }

        public ResourceManager getSource() {
            return this.source;
        }

        public static enum Type {
            REGISTERED,
            REGISTRATION_FAILED,
            STARTED,
            START_FAILED,
            STOPPED,
            STOP_FAILED;

        }
    }

    public static enum ResourceState {
        INITIALIZED("Initialized"),
        STARTED("Started"),
        STOPPING("Stopping"),
        STOPPED("Stopped"),
        FAILED("Failed");

        private final String description;

        private ResourceState(String description) {
            this.description = description;
        }

        public String getDescription() {
            return this.description;
        }
    }

    public static class ResourceHealthReport {
        private final Map<String, Boolean> healthStatus;
        private final Map<String, ResourceState> stateStatus;

        public ResourceHealthReport(Map<String, Boolean> healthStatus, Map<String, ResourceState> stateStatus) {
            this.healthStatus = Collections.unmodifiableMap(healthStatus);
            this.stateStatus = Collections.unmodifiableMap(stateStatus);
        }

        public Map<String, Boolean> getHealthStatus() {
            return this.healthStatus;
        }

        public Map<String, ResourceState> getStateStatus() {
            return this.stateStatus;
        }

        public boolean isHealthy() {
            return this.healthStatus.values().stream().allMatch(Boolean::booleanValue);
        }

        public boolean isHealthy(String resourceName) {
            return Boolean.TRUE.equals(this.healthStatus.get(resourceName));
        }
    }

    public static class ResourceStatistics {
        private final int totalResources;
        private final int healthyResources;
        private final long totalUptimeMs;
        private final Map<String, Object> detailedStats;

        public ResourceStatistics(int totalResources, int healthyResources, long totalUptimeMs, Map<String, Object> detailedStats) {
            this.totalResources = totalResources;
            this.healthyResources = healthyResources;
            this.totalUptimeMs = totalUptimeMs;
            this.detailedStats = Collections.unmodifiableMap(detailedStats);
        }

        public int getTotalResources() {
            return this.totalResources;
        }

        public int getHealthyResources() {
            return this.healthyResources;
        }

        public long getTotalUptimeMs() {
            return this.totalUptimeMs;
        }

        public Map<String, Object> getDetailedStats() {
            return this.detailedStats;
        }

        public double getHealthPercentage() {
            return this.totalResources > 0 ? (double)this.healthyResources * 100.0 / (double)this.totalResources : 0.0;
        }
    }

    public static interface ResourceLifecycleListener {
        public void onResourceLifecycleEvent(ResourceLifecycleEvent var1);
    }
}

