/*
 * Decompiled with CFR 0.152.
 */
package com.kneaf.core.protocol.utils;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.kneaf.core.protocol.core.ProtocolException;
import com.kneaf.core.protocol.core.ProtocolLogger;
import com.kneaf.core.protocol.core.ProtocolValidator;
import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ProtocolLoggerImpl
implements ProtocolLogger {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"ProtocolLogger");
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().disableHtmlEscaping().create();
    private final String loggerName;
    private final Map<String, ProtocolLogger.ProtocolLogEntry> recentLogs;
    private final int maxRecentLogs;

    public ProtocolLoggerImpl(String loggerName) {
        this(loggerName, 1000);
    }

    public ProtocolLoggerImpl(String loggerName, int maxRecentLogs) {
        this.loggerName = loggerName;
        this.maxRecentLogs = maxRecentLogs;
        this.recentLogs = new ConcurrentHashMap<String, ProtocolLogger.ProtocolLogEntry>();
    }

    @Override
    public void logOperationStart(String operation, String protocolFormat, String traceId, Map<String, Object> metadata) {
        ProtocolLogger.ProtocolLogEntry entry = new ProtocolLogger.ProtocolLogEntry(traceId, Instant.now(), operation, protocolFormat, "INFO", "Operation started", metadata);
        this.logEntry(entry);
        this.storeRecentLog(entry);
    }

    @Override
    public void logOperationComplete(String operation, String protocolFormat, String traceId, long durationMs, boolean success, Map<String, Object> metadata) {
        String level = success ? "INFO" : "WARN";
        String message = success ? "Operation completed successfully" : "Operation completed with issues";
        HashMap<String, Object> completeMetadata = new HashMap<String, Object>(metadata);
        completeMetadata.put("duration", durationMs);
        completeMetadata.put("success", success);
        ProtocolLogger.ProtocolLogEntry entry = new ProtocolLogger.ProtocolLogEntry(traceId, Instant.now(), operation, protocolFormat, level, message, completeMetadata);
        this.logEntry(entry);
        this.storeRecentLog(entry);
    }

    @Override
    public void logValidation(String validationType, ProtocolValidator.ValidationResult result, String traceId, Map<String, Object> metadata) {
        String level = result.isValid() ? "INFO" : "ERROR";
        String message = result.isValid() ? "Validation passed: " + validationType : "Validation failed: " + validationType;
        HashMap<String, Object> validationMetadata = new HashMap<String, Object>(metadata);
        validationMetadata.put("validation_type", validationType);
        validationMetadata.put("validation_result", result.isValid());
        if (result.hasErrors()) {
            validationMetadata.put("validation_errors", result.getErrors());
        }
        if (result.hasWarnings()) {
            validationMetadata.put("validation_warnings", result.getWarnings());
        }
        ProtocolLogger.ProtocolLogEntry entry = new ProtocolLogger.ProtocolLogEntry(traceId, Instant.now(), "validation", null, level, message, validationMetadata);
        this.logEntry(entry);
        this.storeRecentLog(entry);
    }

    @Override
    public void logError(String operation, Exception error, String traceId, Map<String, Object> context) {
        HashMap<String, Object> errorMetadata = new HashMap<String, Object>(context);
        errorMetadata.put("operation", operation);
        errorMetadata.put("error_type", error.getClass().getSimpleName());
        errorMetadata.put("error_message", error.getMessage());
        if (error instanceof ProtocolException) {
            ProtocolException protocolError = (ProtocolException)error;
            errorMetadata.put("errorCode", protocolError.getErrorCode());
            errorMetadata.put("protocol_format", protocolError.getProtocolFormat());
        }
        ProtocolLogger.ProtocolLogEntry entry = new ProtocolLogger.ProtocolLogEntry(traceId, Instant.now(), operation, null, "ERROR", "Operation failed with error", errorMetadata);
        this.logEntry(entry);
        this.storeRecentLog(entry);
        LOGGER.error("Protocol error in operation '{ }': { }", new Object[]{operation, error.getMessage(), error});
    }

    @Override
    public void logWarning(String operation, String warning, String traceId, Map<String, Object> context) {
        HashMap<String, Object> warningMetadata = new HashMap<String, Object>(context);
        warningMetadata.put("operation", operation);
        warningMetadata.put("warning_message", warning);
        ProtocolLogger.ProtocolLogEntry entry = new ProtocolLogger.ProtocolLogEntry(traceId, Instant.now(), operation, null, "WARN", warning, warningMetadata);
        this.logEntry(entry);
        this.storeRecentLog(entry);
    }

    @Override
    public void logMetrics(String operation, Map<String, Number> metrics, String traceId) {
        HashMap<String, Object> metricsMetadata = new HashMap<String, Object>();
        metricsMetadata.put("operation", operation);
        metricsMetadata.put("metrics", metrics);
        ProtocolLogger.ProtocolLogEntry entry = new ProtocolLogger.ProtocolLogEntry(traceId, Instant.now(), operation, null, "INFO", "Performance metrics", metricsMetadata);
        this.logEntry(entry);
        this.storeRecentLog(entry);
    }

    @Override
    public String generateTraceId() {
        return UUID.randomUUID().toString();
    }

    private void logEntry(ProtocolLogger.ProtocolLogEntry entry) {
        String jsonLog = GSON.toJson(Map.of("logger", this.loggerName, "trace_id", entry.getTraceId(), "timestamp", entry.getTimestamp().toString(), "level", entry.getLevel(), "operation", entry.getOperation(), "protocol_format", entry.getProtocolFormat(), "message", entry.getMessage(), "metadata", entry.getMetadata()));
        switch (entry.getLevel()) {
            case "DEBUG": {
                LOGGER.debug(jsonLog);
                break;
            }
            case "INFO": {
                LOGGER.info(jsonLog);
                break;
            }
            case "WARN": {
                LOGGER.warn(jsonLog);
                break;
            }
            case "ERROR": {
                LOGGER.error(jsonLog);
                break;
            }
            default: {
                LOGGER.info(jsonLog);
            }
        }
    }

    private void storeRecentLog(ProtocolLogger.ProtocolLogEntry entry) {
        this.recentLogs.put(entry.getTraceId(), entry);
        if (this.recentLogs.size() > this.maxRecentLogs) {
            String oldestKey = this.recentLogs.keySet().iterator().next();
            this.recentLogs.remove(oldestKey);
        }
    }

    public List<ProtocolLogger.ProtocolLogEntry> getRecentLogs(int limit) {
        return this.recentLogs.values().stream().sorted((a, b) -> b.getTimestamp().compareTo(a.getTimestamp())).limit(limit).toList();
    }

    public ProtocolLogger.ProtocolLogEntry getLogByTraceId(String traceId) {
        return this.recentLogs.get(traceId);
    }

    public List<ProtocolLogger.ProtocolLogEntry> getLogsByOperation(String operation, int limit) {
        return this.recentLogs.values().stream().filter(entry -> operation.equals(entry.getOperation())).sorted((a, b) -> b.getTimestamp().compareTo(a.getTimestamp())).limit(limit).toList();
    }

    public List<ProtocolLogger.ProtocolLogEntry> getErrorLogs(int limit) {
        return this.recentLogs.values().stream().filter(entry -> "ERROR".equals(entry.getLevel())).sorted((a, b) -> b.getTimestamp().compareTo(a.getTimestamp())).limit(limit).toList();
    }

    public Map<String, Object> getStatistics() {
        HashMap<String, Object> Stats = new HashMap<String, Object>();
        Stats.put("logger_name", this.loggerName);
        Stats.put("recent_logs_count", this.recentLogs.size());
        Stats.put("max_recent_logs", this.maxRecentLogs);
        Map<String, Long> levelCounts = this.recentLogs.values().stream().collect(Collectors.groupingBy(ProtocolLogger.ProtocolLogEntry::getLevel, Collectors.counting()));
        Stats.put("level_counts", levelCounts);
        return Stats;
    }

    public void clearRecentLogs() {
        this.recentLogs.clear();
    }
}

