package falseresync.wizcraft.common;

import com.google.common.base.Preconditions;
import falseresync.wizcraft.common.config.WizcraftConfig;
import falseresync.wizcraft.common.data.WizcraftAttachments;
import falseresync.wizcraft.common.data.WizcraftComponents;
import net.fabricmc.fabric.api.event.Event;
import net.fabricmc.fabric.api.event.EventFactory;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_1944;
import javax.annotation.Nullable;

public class ChargeManager {
    public static final Event<WandExpend> WAND_CHARGE_SPENT = EventFactory.createArrayBacked(WandExpend.class, listeners -> (wandStack, cost, user) -> {
        for (WandExpend listener : listeners) {
            listener.onWandChargeSpent(wandStack, cost, user);
        }
    });

    public static final Event<WandOvercharge> WAND_OVERCHARGED = EventFactory.createArrayBacked(WandOvercharge.class, listeners -> (wandStack, excess, user) -> {
        for (WandOvercharge listener : listeners) {
            listener.onWandOvercharged(wandStack, excess, user);
        }
    });

    public ChargeManager() {
        WAND_CHARGE_SPENT.register((wandStack, cost, user) -> {
            if (user != null) {
                // Maybe only compensate the cost? But that would be confusing
                var chargeShells = user.getAttached(WizcraftAttachments.CHARGE_SHELLS);
                if (chargeShells == null) {
                    return;
                }

                var wandCurrent = wandStack.method_57825(WizcraftComponents.WAND_CHARGE, 0);
                var wandMax = wandStack.method_57825(WizcraftComponents.WAND_MAX_CHARGE, 0);
                var compensation = wandMax - wandCurrent;
                var newShells = chargeShells.withChargeChange(-compensation);
                if (newShells != null) {
                    user.setAttached(WizcraftAttachments.CHARGE_SHELLS, newShells);
                    wandStack.method_57368(WizcraftComponents.WAND_CHARGE, 0, it -> it + compensation);
                }
            }
        });

        WAND_OVERCHARGED.register((wandStack, excess, user) -> {
            if (user != null) {
                Wizcraft.getChargeManager().applyShellCharge(user, excess);
            }
        });
    }

    public boolean areShellsFull(class_1657 player) {
        //noinspection DataFlowIssue
        return player.hasAttached(WizcraftAttachments.CHARGE_SHELLS)
                && player.getAttached(WizcraftAttachments.CHARGE_SHELLS).areShellsFull();
    }

    public void applyShellCharge(class_1657 player, int amount) {
        var shells = player.getAttached(WizcraftAttachments.CHARGE_SHELLS);
        if (shells == null) {
            return;
        }
        var newShells = shells.withChargeChange(amount);
        if (newShells != null) {
            player.setAttached(WizcraftAttachments.CHARGE_SHELLS, newShells);
        }
    }

    public boolean isWandFullyCharged(class_1799 wandStack) {
        return wandStack.method_57825(WizcraftComponents.WAND_CHARGE, 0) >= wandStack.method_57825(WizcraftComponents.WAND_MAX_CHARGE, 0);
    }

    public boolean cannotAddAnyCharge(class_1799 wandStack, class_1657 player) {
        return isWandFullyCharged(wandStack) && areShellsFull(player);
    }

    public boolean tryExpendWandCharge(class_1799 wandStack, int cost, @Nullable class_1657 user) {
        if (user != null && (user.method_7337() && Wizcraft.getConfig().infiniteCharge.isCreativeOnly() || Wizcraft.getConfig().infiniteCharge.isAlways())) {
            return true;
        }
        var charge = wandStack.method_57825(WizcraftComponents.WAND_CHARGE, 0);
        if (charge >= cost) {
            wandStack.method_57368(WizcraftComponents.WAND_CHARGE, charge, current -> current - cost);
            ChargeManager.WAND_CHARGE_SPENT.invoker().onWandChargeSpent(wandStack, cost, user);
            return true;
        }
        return false;
    }

    public void chargeWand(class_1799 wandStack, int amount, @Nullable class_1657 user) {
        Preconditions.checkArgument(amount > 0, "Use tryExpendCharge to subtract charge");
        var current = wandStack.method_57825(WizcraftComponents.WAND_CHARGE, 0);
        var max = wandStack.method_57825(WizcraftComponents.WAND_MAX_CHARGE, 0);
        wandStack.method_57368(WizcraftComponents.WAND_CHARGE, 0, it -> Math.min(it + amount, max));
        if (current + amount > max) {
            ChargeManager.WAND_OVERCHARGED.invoker().onWandOvercharged(wandStack, current + amount - max, user);
        }
    }

    public void tryChargeWandPassively(class_1799 wandStack, class_1937 world, class_1657 player) {
        if (Wizcraft.getChargeManager().cannotAddAnyCharge(wandStack, player)) {
            return;
        }

        var config = Wizcraft.getConfig().passiveCharge;
        if (config == WizcraftConfig.PassiveCharge.DISABLED) {
            return;
        }

        var usageCoefficient = class_1799.method_7973(player.method_6047(), wandStack) ? 1f : 0.25f;
        var passiveChargingThreshold = Math.clamp(0.005f * calculateEnvironmentCoefficient(world, player) * config.coefficient * usageCoefficient, 0, 0.1f);

        // At most 10% of the time, i.e. up to 2 times per second
        if (world.field_9229.method_43057() < passiveChargingThreshold) {
            Wizcraft.getChargeManager().chargeWand(wandStack, 1, player);
        }
    }

    public float calculateEnvironmentCoefficient(class_1937 world, class_1657 player) {
        var environmentCoefficient = 1f;
        var worldType = world.method_27983();
        if (worldType == class_1937.field_25180) {
            environmentCoefficient *= 0.1f;
        } else if (worldType == class_1937.field_25181) {
            environmentCoefficient *= 3f;
        } else {
            environmentCoefficient *= world.method_23886() ? 1 : 0.5f;
            environmentCoefficient *= world.method_23753(player.method_24515()).comp_349().method_48163() ? 1 - world.method_8430(1) : 1;
            environmentCoefficient *= world.method_8314(class_1944.field_9284, player.method_24515()) / (world.method_8315() * 0.5f);
        }
        return environmentCoefficient;
    }

    @FunctionalInterface
    public interface WandExpend {
        void onWandChargeSpent(class_1799 wandStack, int cost, @Nullable class_1657 user);
    }

    @FunctionalInterface
    public interface WandOvercharge {
        void onWandOvercharged(class_1799 wandStack, int excess, @Nullable class_1657 user);
    }
}
