package kono.ceu.gtconsolidate.loader.handlers;

import static com.github.gtexpert.core.integration.deda.recipes.DraconicMaterialsRecipe.ABFDurationMultiplier;
import static com.github.gtexpert.core.integration.deda.recipes.DraconicMaterialsRecipe.ABFPyrotheumAmount;
import static gregtech.api.GTValues.*;
import static gregtech.api.unification.ore.OrePrefix.*;

import java.util.ArrayList;
import java.util.List;

import net.minecraftforge.fluids.Fluid;
import net.minecraftforge.fluids.FluidStack;

import org.jetbrains.annotations.NotNull;

import com.github.gtexpert.core.api.unification.material.GTEMaterials;

import gregtech.api.GTValues;
import gregtech.api.GregTechAPI;
import gregtech.api.fluids.store.FluidStorageKeys;
import gregtech.api.recipes.GTRecipeHandler;
import gregtech.api.recipes.RecipeBuilder;
import gregtech.api.recipes.builders.BlastRecipeBuilder;
import gregtech.api.recipes.ingredients.IntCircuitIngredient;
import gregtech.api.unification.OreDictUnifier;
import gregtech.api.unification.material.Material;
import gregtech.api.unification.material.Materials;
import gregtech.api.unification.material.properties.BlastProperty;
import gregtech.api.unification.material.properties.PropertyKey;
import gregtech.api.unification.ore.OrePrefix;
import gregtech.api.unification.stack.MaterialStack;

import gregicality.multiblocks.api.fluids.GCYMFluidStorageKeys;
import gregicality.multiblocks.api.recipes.GCYMRecipeMaps;
import gregicality.multiblocks.api.unification.GCYMMaterialFlags;
import gregicality.multiblocks.api.unification.properties.GCYMPropertyKey;

import kono.ceu.gtconsolidate.api.recipes.GTConsolidateRecipeMaps;
import kono.ceu.gtconsolidate.api.util.Mods;

public class TurboBlastFurnaceLoader {

    public static void generate() {
        if (Mods.GregTechExpertCore.isModLoaded()) {
            List<Material> materials = new ArrayList<>(GregTechAPI.materialManager.getRegisteredMaterials());
            materials.forEach(TurboBlastFurnaceLoader::alloyBlastFurnaceExtended);

            // Pyrotheum
            GTConsolidateRecipeMaps.TURBO_BLAST_RECIPE.recipeBuilder()
                    .input(dust, Materials.Redstone)
                    .input(dust, Materials.Sulfur)
                    .fluidInputs(Materials.Blaze.getFluid(2304))
                    .fluidInputs(Materials.Argon.getFluid(FluidStorageKeys.GAS, 200))
                    .circuitMeta(15)
                    .blastFurnaceTemp(7200)
                    .fluidOutputs(GTEMaterials.Pyrotheum.getFluid(GCYMFluidStorageKeys.MOLTEN, 1000))
                    .EUt(VA[LuV]).duration(10 * 20).buildAndRegister();

            GTConsolidateRecipeMaps.TURBO_BLAST_RECIPE.recipeBuilder()
                    .input(dust, Materials.Redstone)
                    .input(dust, Materials.Sulfur)
                    .fluidInputs(Materials.Blaze.getFluid(2304))
                    .circuitMeta(5)
                    .blastFurnaceTemp(7200)
                    .fluidOutputs(GTEMaterials.Pyrotheum.getFluid(GCYMFluidStorageKeys.MOLTEN, 1000))
                    .EUt(VA[LuV]).duration(60 * 20).buildAndRegister();
        }
    }

    // fix recipe confit: -remove-
    public static void removeConfitRecipe() {
        // Steel dust -> Steel Ingot
        GTRecipeHandler.removeRecipesByInputs(GTConsolidateRecipeMaps.TURBO_BLAST_RECIPE,
                OreDictUnifier.get(dust, Materials.Steel));
        // Iron & Carbone dust -> Steel ingot
        GTRecipeHandler.removeRecipesByInputs(GTConsolidateRecipeMaps.TURBO_BLAST_RECIPE,
                OreDictUnifier.get(dust, Materials.Iron, 4), OreDictUnifier.get(dust, Materials.Carbon));
        // WroughtIron & Carbone dust -> Steel ingot
        GTRecipeHandler.removeRecipesByInputs(GTConsolidateRecipeMaps.TURBO_BLAST_RECIPE,
                OreDictUnifier.get(dust, Materials.WroughtIron, 4), OreDictUnifier.get(dust, Materials.Carbon));
        // Yttrium dust -> Yttrium ingot
        GTRecipeHandler.removeRecipesByInputs(GTConsolidateRecipeMaps.TURBO_BLAST_RECIPE,
                OreDictUnifier.get(dust, Materials.Yttrium));
    }

    // fix recipe confit: re-add
    public static void reAddRecipe() {
        GTConsolidateRecipeMaps.TURBO_BLAST_RECIPE.recipeBuilder()
                .input(dust, Materials.Steel)
                .output(ingot, Materials.Steel)
                .circuitMeta(1)
                .blastFurnaceTemp(1000)
                .duration(800).EUt(VA[MV]).buildAndRegister();

        GTConsolidateRecipeMaps.TURBO_BLAST_RECIPE.recipeBuilder()
                .input(dust, Materials.Iron, 4)
                .input(dust, Materials.Carbon)
                .output(ingot, Materials.Steel, 4)
                .chancedOutput(dust, Materials.Ash, 3333, 0)
                .circuitMeta(1)
                .blastFurnaceTemp(2000)
                .duration(250).EUt(VA[EV]).buildAndRegister();

        GTConsolidateRecipeMaps.TURBO_BLAST_RECIPE.recipeBuilder()
                .input(dust, Materials.WroughtIron, 4)
                .input(dust, Materials.Carbon)
                .output(ingot, Materials.Steel, 4)
                .chancedOutput(dust, Materials.Ash, 3333, 0)
                .circuitMeta(1)
                .blastFurnaceTemp(2000)
                .duration(50).EUt(VA[EV]).buildAndRegister();

        GTConsolidateRecipeMaps.TURBO_BLAST_RECIPE.recipeBuilder()
                .input(dust, Materials.Yttrium)
                .output(ingot, Materials.Yttrium)
                .circuitMeta(1)
                .blastFurnaceTemp(1799)
                .duration(3202).EUt(VA[MV]).buildAndRegister();
    }

    private static void alloyBlastFurnaceExtended(Material material) {
        // Do not generate for disabled materials
        if (material.hasFlag(GCYMMaterialFlags.NO_ALLOY_BLAST_RECIPES)) return;

        // Check if the material has a blast recipe
        if (!material.hasProperty(GCYMPropertyKey.ALLOY_BLAST)) return;

        // Check if the material has a molten fluid
        Fluid molten = material.getFluid(GCYMFluidStorageKeys.MOLTEN);
        if (molten == null) return;

        // Get the vacuum freezer EUt and duration
        BlastProperty property = material.getProperty(PropertyKey.BLAST);

        produce(material, property);
    }

    /**
     * Generates alloy blast recipes for a material
     *
     * @param material      the material to generate for
     * @param blastProperty the blast property of the material
     */
    private static void produce(@NotNull Material material, @NotNull BlastProperty blastProperty) {
        final int componentAmount = material.getMaterialComponents().size();

        // ignore non-alloys
        if (componentAmount < 2) return;

        // get the output fluid
        Fluid molten = material.getFluid(GCYMFluidStorageKeys.MOLTEN);
        if (molten == null) return;

        RecipeBuilder<BlastRecipeBuilder> builder = createBuilder(blastProperty, material);

        int outputAmount = addInputs(material, builder);
        if (outputAmount <= 0) return;

        buildRecipes(blastProperty, molten, outputAmount, componentAmount, builder);
    }

    /**
     * Creates the recipeBuilder with duration and EUt
     *
     * @param property the blast property of the material
     * @param material the material
     * @return the builder
     */
    @SuppressWarnings("MethodMayBeStatic")
    private static @NotNull BlastRecipeBuilder createBuilder(@NotNull BlastProperty property,
                                                             @NotNull Material material) {
        BlastRecipeBuilder builder = GCYMRecipeMaps.ALLOY_BLAST_RECIPES.recipeBuilder();
        // apply the duration override
        int duration = property.getDurationOverride();
        if (duration < 0) duration = Math.max(1, (int) (material.getMass() * property.getBlastTemperature() / 100L));
        builder.duration(duration);

        // apply the EUt override
        int EUt = property.getEUtOverride();
        if (EUt < 0) EUt = GTValues.VA[GTValues.MV];
        builder.EUt(EUt);

        return builder.blastFurnaceTemp(property.getBlastTemperature());
    }

    /**
     * @param material the material to start recipes for
     * @param builder  the recipe builder to append to
     * @return the outputAmount if the recipe is valid, otherwise -1
     */
    private static int addInputs(@NotNull Material material, @NotNull RecipeBuilder<BlastRecipeBuilder> builder) {
        // calculate the output amount and add inputs
        int outputAmount = 0;
        int fluidAmount = 0;
        int dustAmount = 0;
        for (MaterialStack materialStack : material.getMaterialComponents()) {
            final Material msMat = materialStack.material;
            final int msAmount = (int) materialStack.amount;

            if (msMat.hasProperty(PropertyKey.DUST)) {
                if (dustAmount >= 9) return -1; // more than 9 dusts won't fit in the machine
                dustAmount++;
                builder.input(OrePrefix.dust, msMat, msAmount);
            } else if (msMat.hasProperty(PropertyKey.FLUID)) {
                if (fluidAmount >= 2) return -1; // more than 2 fluids won't fit in the machine
                fluidAmount++;
                // assume all fluids have 1000mB/mol, since other quantities should be as an item input
                builder.fluidInputs(msMat.getFluid(1000 * msAmount));
            } else return -1; // no fluid or item prop means no valid recipe
            outputAmount += msAmount;
        }
        return outputAmount;
    }

    /**
     * Builds the alloy blast recipes
     *
     * @param property        the blast property to utilize
     * @param molten          the molten fluid
     * @param outputAmount    the amount of material to output
     * @param componentAmount the amount of different components in the material
     * @param builder         the builder to continue
     */
    private static void buildRecipes(@NotNull BlastProperty property, @NotNull Fluid molten, int outputAmount,
                                     int componentAmount,
                                     @NotNull RecipeBuilder<BlastRecipeBuilder> builder) {
        // add the fluid output with the correct amount
        builder.fluidOutputs(new FluidStack(molten, GTValues.L * outputAmount));

        // apply alloy blast duration reduction: 3/4
        int duration = builder.getDuration() * outputAmount * 3 / 4;

        // build the gas recipe if it exists
        if (property.getGasTier() != null) {
            RecipeBuilder<BlastRecipeBuilder> builderGas = builder.copy();
            builderGas.notConsumable(new IntCircuitIngredient(getGasCircuitNum(componentAmount)))
                    .fluidInputs(GTEMaterials.Pyrotheum.getFluid(GCYMFluidStorageKeys.MOLTEN, ABFPyrotheumAmount))
                    .duration((int) (duration * 0.67 * ABFDurationMultiplier))
                    .buildAndRegister();
        }

        // build the non-gas recipe
        builder.notConsumable(new IntCircuitIngredient(getCircuitNum(componentAmount)))
                .duration(duration)
                .buildAndRegister();
    }

    /**
     * @param componentAmount the amount of different components in the material
     * @return the circuit number for the regular recipe
     */
    private static int getCircuitNum(int componentAmount) {
        return componentAmount;
    }

    /**
     * @param componentAmount the amount of different components in the material
     * @return the circuit number for the gas-boosted recipe
     */
    private static int getGasCircuitNum(int componentAmount) {
        return componentAmount + 11;
    }
}
