/*
 * Decompiled with CFR 0.152.
 */
package com.treecutter.managers;

import com.treecutter.TreeCutterPlugin;
import java.io.File;
import java.io.IOException;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.configuration.file.FileConfiguration;
import org.bukkit.configuration.file.YamlConfiguration;
import org.bukkit.entity.Player;
import org.bukkit.plugin.Plugin;

public class PlayerDataManager {
    private final TreeCutterPlugin plugin;
    private final Map<UUID, PlayerData> playerDataCache;
    private final File dataFile;
    private FileConfiguration dataConfig;

    public PlayerDataManager(TreeCutterPlugin plugin) {
        this.plugin = plugin;
        this.playerDataCache = new ConcurrentHashMap<UUID, PlayerData>();
        if (!plugin.getDataFolder().exists()) {
            plugin.getDataFolder().mkdirs();
        }
        this.dataFile = new File(plugin.getDataFolder(), "playerdata.yml");
        this.loadPlayerData();
        this.startAutoSaveTask();
    }

    private void loadPlayerData() {
        if (!this.dataFile.exists()) {
            try {
                this.dataFile.createNewFile();
            }
            catch (IOException e) {
                this.plugin.getLogger().severe("Could not create player data file: " + e.getMessage());
                return;
            }
        }
        this.dataConfig = YamlConfiguration.loadConfiguration((File)this.dataFile);
        if (this.dataConfig.contains("players")) {
            for (String uuidString : this.dataConfig.getConfigurationSection("players").getKeys(false)) {
                try {
                    UUID playerId = UUID.fromString(uuidString);
                    boolean enabled = this.dataConfig.getBoolean("players." + uuidString + ".tree-cutting-enabled", true);
                    String lastKnownName = this.dataConfig.getString("players." + uuidString + ".last-known-name", "Unknown");
                    long lastSeen = this.dataConfig.getLong("players." + uuidString + ".last-seen", System.currentTimeMillis());
                    PlayerData data = new PlayerData(playerId, enabled, lastKnownName, lastSeen);
                    this.playerDataCache.put(playerId, data);
                    this.plugin.debugLog("Loaded player data for " + lastKnownName + " (" + String.valueOf(playerId) + ") - Enabled: " + enabled);
                }
                catch (IllegalArgumentException e) {
                    this.plugin.getLogger().warning("Invalid UUID in player data: " + uuidString);
                }
            }
        }
        this.plugin.getLogger().info("Loaded " + this.playerDataCache.size() + " player data entries");
    }

    public void savePlayerData() {
        try {
            this.dataConfig.set("players", null);
            for (Map.Entry<UUID, PlayerData> entry : this.playerDataCache.entrySet()) {
                UUID playerId = entry.getKey();
                PlayerData data = entry.getValue();
                String path = "players." + playerId.toString();
                this.dataConfig.set(path + ".tree-cutting-enabled", (Object)data.isTreeCuttingEnabled());
                this.dataConfig.set(path + ".last-known-name", (Object)data.getLastKnownName());
                this.dataConfig.set(path + ".last-seen", (Object)data.getLastSeen());
            }
            this.dataConfig.save(this.dataFile);
            this.plugin.debugLog("Saved " + this.playerDataCache.size() + " player data entries");
        }
        catch (IOException e) {
            this.plugin.getLogger().severe("Could not save player data: " + e.getMessage());
        }
    }

    public void saveAllPlayerData() {
        this.savePlayerData();
    }

    private void startAutoSaveTask() {
        this.plugin.getServer().getScheduler().runTaskTimerAsynchronously((Plugin)this.plugin, this::savePlayerData, 6000L, 6000L);
    }

    public PlayerData getPlayerData(UUID playerId) {
        return this.playerDataCache.computeIfAbsent(playerId, id -> {
            PlayerData data = new PlayerData((UUID)id, true, "Unknown", System.currentTimeMillis());
            this.plugin.debugLog("Created new player data for " + String.valueOf(id));
            return data;
        });
    }

    public PlayerData getPlayerData(Player player) {
        PlayerData data = this.getPlayerData(player.getUniqueId());
        if (!data.getLastKnownName().equals(player.getName())) {
            data.setLastKnownName(player.getName());
            data.setLastSeen(System.currentTimeMillis());
            this.plugin.debugLog("Updated player info for " + player.getName());
        }
        return data;
    }

    public boolean isTreeCuttingEnabled(UUID playerId) {
        return this.getPlayerData(playerId).isTreeCuttingEnabled();
    }

    public boolean isTreeCuttingEnabled(Player player) {
        return this.getPlayerData(player).isTreeCuttingEnabled();
    }

    public void setTreeCuttingEnabled(UUID playerId, boolean enabled) {
        PlayerData data = this.getPlayerData(playerId);
        data.setTreeCuttingEnabled(enabled);
        data.setLastSeen(System.currentTimeMillis());
        this.plugin.debugLog("Set tree cutting for " + String.valueOf(playerId) + " to " + enabled);
        if (this.plugin.isDebugMode()) {
            String debugMsg = this.plugin.getConfig().getString("messages.debug-player-toggle", "&7[TreeCutter Debug] Player %player% toggled tree cutting: %state%").replace("%player%", data.getLastKnownName()).replace("%state%", enabled ? "ON" : "OFF");
            this.plugin.getServer().getOnlinePlayers().stream().filter(p -> p.hasPermission("treecutter.admin")).forEach(admin -> admin.sendMessage(this.plugin.colorize(debugMsg)));
        }
    }

    public void setTreeCuttingEnabled(Player player, boolean enabled) {
        this.setTreeCuttingEnabled(player.getUniqueId(), enabled);
        this.getPlayerData(player.getUniqueId()).setLastKnownName(player.getName());
    }

    public void toggleTreeCutting(UUID playerId) {
        PlayerData data = this.getPlayerData(playerId);
        this.setTreeCuttingEnabled(playerId, !data.isTreeCuttingEnabled());
    }

    public void toggleTreeCutting(Player player) {
        this.toggleTreeCutting(player.getUniqueId());
    }

    public Map<UUID, PlayerData> getAllPlayerData() {
        return new ConcurrentHashMap<UUID, PlayerData>(this.playerDataCache);
    }

    public void cleanupOldData(long maxAgeMillis) {
        long currentTime = System.currentTimeMillis();
        int removed = 0;
        this.playerDataCache.entrySet().removeIf(entry -> {
            boolean isOld;
            PlayerData data = (PlayerData)entry.getValue();
            boolean bl = isOld = currentTime - data.getLastSeen() > maxAgeMillis;
            if (isOld) {
                this.plugin.debugLog("Removed old player data for " + data.getLastKnownName());
            }
            return isOld;
        });
        if (removed > 0) {
            this.plugin.getLogger().info("Cleaned up " + removed + " old player data entries");
            this.savePlayerData();
        }
    }

    public static class PlayerData {
        private final UUID playerId;
        private boolean treeCuttingEnabled;
        private String lastKnownName;
        private long lastSeen;

        public PlayerData(UUID playerId, boolean treeCuttingEnabled, String lastKnownName, long lastSeen) {
            this.playerId = playerId;
            this.treeCuttingEnabled = treeCuttingEnabled;
            this.lastKnownName = lastKnownName;
            this.lastSeen = lastSeen;
        }

        public UUID getPlayerId() {
            return this.playerId;
        }

        public boolean isTreeCuttingEnabled() {
            return this.treeCuttingEnabled;
        }

        public void setTreeCuttingEnabled(boolean enabled) {
            this.treeCuttingEnabled = enabled;
        }

        public String getLastKnownName() {
            return this.lastKnownName;
        }

        public void setLastKnownName(String name) {
            this.lastKnownName = name;
        }

        public long getLastSeen() {
            return this.lastSeen;
        }

        public void setLastSeen(long timestamp) {
            this.lastSeen = timestamp;
        }
    }
}

