/*
 * Decompiled with CFR 0.152.
 */
package com.treecutter.utils;

import com.treecutter.TreeCutterPlugin;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.Location;
import org.bukkit.Material;
import org.bukkit.block.Block;
import org.bukkit.block.BlockFace;
import org.bukkit.plugin.Plugin;

public class TreeDetector {
    private final TreeCutterPlugin plugin;
    private final Map<Location, TreeStructure> treeCache;
    private final Set<Material> allowedLogs;
    private static final BlockFace[] DIRECTIONS = new BlockFace[]{BlockFace.NORTH, BlockFace.SOUTH, BlockFace.EAST, BlockFace.WEST, BlockFace.UP, BlockFace.DOWN, BlockFace.NORTH_EAST, BlockFace.NORTH_WEST, BlockFace.SOUTH_EAST, BlockFace.SOUTH_WEST};

    public TreeDetector(TreeCutterPlugin plugin) {
        this.plugin = plugin;
        this.treeCache = new ConcurrentHashMap<Location, TreeStructure>();
        this.allowedLogs = this.loadAllowedLogs();
        if (plugin.isCacheTreesEnabled()) {
            this.startCacheCleanupTask();
        }
    }

    private Set<Material> loadAllowedLogs() {
        HashSet<Material> logs = new HashSet<Material>();
        List configLogs = this.plugin.getConfig().getStringList("blocks.allowed-logs");
        for (String logName : configLogs) {
            try {
                Material material = Material.valueOf((String)logName.toUpperCase());
                logs.add(material);
                this.plugin.debugLog("Added allowed log type: " + material.name());
            }
            catch (IllegalArgumentException e) {
                this.plugin.getLogger().warning("Invalid log material in config: " + logName);
            }
        }
        if (logs.isEmpty()) {
            logs.addAll(Arrays.asList(Material.OAK_LOG, Material.SPRUCE_LOG, Material.BIRCH_LOG, Material.JUNGLE_LOG, Material.ACACIA_LOG, Material.DARK_OAK_LOG, Material.MANGROVE_LOG, Material.CHERRY_LOG));
            this.plugin.debugLog("Using default log types as fallback");
        }
        return logs;
    }

    public boolean isAllowedLog(Material material) {
        return this.allowedLogs.contains(material);
    }

    public TreeStructure detectTree(Block startBlock) {
        if (!this.isAllowedLog(startBlock.getType())) {
            return null;
        }
        Location startLoc = startBlock.getLocation();
        if (this.plugin.isCacheTreesEnabled() && this.treeCache.containsKey(startLoc)) {
            TreeStructure cached = this.treeCache.get(startLoc);
            if (!cached.isExpired()) {
                this.plugin.debugLog("Using cached tree structure for " + this.locationToString(startLoc));
                return cached;
            }
            this.treeCache.remove(startLoc);
        }
        TreeStructure tree = this.performTreeDetection(startBlock);
        if (this.plugin.isCacheTreesEnabled() && tree != null) {
            for (Location logLoc : tree.getLogLocations()) {
                this.treeCache.put(logLoc, tree);
            }
            this.plugin.debugLog("Cached tree structure with " + tree.getLogCount() + " logs");
        }
        return tree;
    }

    private TreeStructure performTreeDetection(Block startBlock) {
        HashSet<Location> visited = new HashSet<Location>();
        HashSet<Location> logLocations = new HashSet<Location>();
        LinkedList<Block> toCheck = new LinkedList<Block>();
        toCheck.add(startBlock);
        visited.add(startBlock.getLocation());
        int maxLogs = this.plugin.getMaxLogsPerTree();
        int maxDistance = this.plugin.getMaxSearchDistance();
        Location origin = startBlock.getLocation();
        while (!toCheck.isEmpty() && logLocations.size() < maxLogs) {
            Block current = (Block)toCheck.poll();
            Location currentLoc = current.getLocation();
            if (origin.distance(currentLoc) > (double)maxDistance || !this.isAllowedLog(current.getType())) continue;
            logLocations.add(currentLoc);
            for (BlockFace face : DIRECTIONS) {
                Block adjacent = current.getRelative(face);
                Location adjLoc = adjacent.getLocation();
                if (visited.contains(adjLoc) || !this.isAllowedLog(adjacent.getType())) continue;
                visited.add(adjLoc);
                toCheck.add(adjacent);
            }
        }
        if (logLocations.size() < this.plugin.getMinLogsForTree()) {
            this.plugin.debugLog("Detected structure too small to be a tree: " + logLocations.size() + " logs");
            return null;
        }
        this.plugin.debugLog("Detected tree with " + logLocations.size() + " logs");
        return new TreeStructure(logLocations, this.plugin.getCacheDuration());
    }

    public boolean isWorldEnabled(String worldName) {
        List enabledWorlds = this.plugin.getConfig().getStringList("worlds.enabled-worlds");
        List disabledWorlds = this.plugin.getConfig().getStringList("worlds.disabled-worlds");
        if (disabledWorlds.contains(worldName)) {
            return false;
        }
        if (enabledWorlds.isEmpty()) {
            return true;
        }
        return enabledWorlds.contains(worldName);
    }

    private void startCacheCleanupTask() {
        this.plugin.getServer().getScheduler().runTaskTimerAsynchronously((Plugin)this.plugin, () -> {
            int removed = 0;
            Iterator<Map.Entry<Location, TreeStructure>> iterator = this.treeCache.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<Location, TreeStructure> entry = iterator.next();
                if (!entry.getValue().isExpired()) continue;
                iterator.remove();
                ++removed;
            }
            if (removed > 0) {
                this.plugin.debugLog("Cleaned up " + removed + " expired tree cache entries");
            }
        }, 6000L, 6000L);
    }

    private String locationToString(Location loc) {
        return String.format("%s:(%d,%d,%d)", loc.getWorld().getName(), loc.getBlockX(), loc.getBlockY(), loc.getBlockZ());
    }

    public void clearCache() {
        this.treeCache.clear();
        this.plugin.debugLog("Tree cache cleared");
    }

    public int getCacheSize() {
        return this.treeCache.size();
    }

    public static class TreeStructure {
        private final Set<Location> logLocations;
        private final long createdAt;
        private final long expiryTime;

        public TreeStructure(Set<Location> logLocations, int cacheDurationSeconds) {
            this.logLocations = new HashSet<Location>(logLocations);
            this.createdAt = System.currentTimeMillis();
            this.expiryTime = this.createdAt + (long)cacheDurationSeconds * 1000L;
        }

        public Set<Location> getLogLocations() {
            return new HashSet<Location>(this.logLocations);
        }

        public int getLogCount() {
            return this.logLocations.size();
        }

        public boolean isExpired() {
            return System.currentTimeMillis() > this.expiryTime;
        }

        public boolean containsLocation(Location location) {
            return this.logLocations.contains(location);
        }
    }
}

