package settingdust.preloading_tricks.api;

import com.google.common.base.Suppliers;
import settingdust.preloading_tricks.util.ServiceLoaderUtil;

import java.util.ServiceLoader;
import java.util.function.Supplier;

/**
 * Callback interface for preloading phase hooks, used to execute custom logic at key stages of module loading.
 *
 * <p>This interface is implemented based on the Java {@link java.util.ServiceLoader} mechanism,
 * allowing mod providers to intercept the module loading process by registering implementation classes of this interface.</p>
 *
 * <p>Implementation classes of this interface should be registered in the file
 * {@code META-INF/services/settingdust.preloading_tricks.api.PreloadingTricksCallback}
 * with their fully qualified class names.</p>
 *
 * @see java.util.ServiceLoader
 * @see PreloadingTricksModManager
 */
public interface PreloadingTricksCallback {
    /**
     * Supplier that discovers all registered callback implementations through the {@link ServiceLoader} mechanism.
     * Results are memoized for performance optimization.
     */
    Supplier<Iterable<PreloadingTricksCallback>> supplier =
        Suppliers.memoize(() -> ServiceLoaderUtil.findServices(
            PreloadingTricksCallback.class,
            ServiceLoaderUtil.load(PreloadingTricksCallback.class, PreloadingTricksCallback.class.getClassLoader()),
            false
        ));

    /**
     * Global callback invoker responsible for iterating through all registered callback implementations
     * and calling their corresponding methods in order.
     *
     * <p>This object is for internal use; external code should not call it directly.</p>
     */
    PreloadingTricksCallback invoker = new PreloadingTricksCallback() {
        @Override
        public void onSetupLanguageAdapter() {
            for (final var callback : supplier.get()) {
                callback.onSetupLanguageAdapter();
            }
        }

        @Override
        public void onCollectModCandidates() {
            for (final var callback : supplier.get()) {
                callback.onCollectModCandidates();
            }
        }

        @Override
        public void onSetupMods() {
            var oldClassLoader = Thread.currentThread().getContextClassLoader();
            Thread.currentThread().setContextClassLoader(PreloadingTricksModManager.class.getClassLoader());
            for (final var callback : supplier.get()) {
                callback.onSetupMods();
            }
            Thread.currentThread().setContextClassLoader(oldClassLoader);
        }
    };

    /**
     * Called during the language adapter setup phase.
     *
     * <p>This method is triggered when the module loader initializes the language system,
     * and can be used to register custom language providers or modify language-related configurations.</p>
     *
     * <p>Default implementation does nothing.</p>
     */
    default void onSetupLanguageAdapter() {}

    default void onCollectModCandidates() {}

    /**
     * Called during the mod setup phase.
     *
     * <p>This method is triggered when the module loader prepares to load modules,
     * and can be used to modify the module list, inject virtual modules, or perform other pre-loading operations.</p>
     *
     * <p>Default implementation does nothing.</p>
     */
    default void onSetupMods() {}
}
