/**
 * 创建 Java 对象包装器
 * @param {java.lang.Class} javaClass - 要包装的 Java 类
 * @param {Object} impl - 包含自定义方法的实现对象
 * @param {Array} [interfaces=[]] - 需要实现的额外接口数组
 * @returns {Object} - 包装后的 Java 实例
 */
function createJavaWrapper(javaClass, impl, interfaces) {
    // 确保 interfaces 是数组
    // interfaces = asJavaArray(interfaces) || ([]);

    // 添加默认实现的基础方法
    var handler = {
        // 默认 toString 实现
        toString: function () {
            return "JavaWrapper@" + java.lang.System.identityHashCode(this);
        },

        // 默认 equals 实现
        equals: function (other) {
            return this === other;
        },

        // 默认 hashCode 实现
        hashCode: function () {
            return java.lang.System.identityHashCode(this);
        }
    };

    // 合并用户自定义实现
    for (var key in impl) {
        if (impl.hasOwnProperty(key)) {
            handler[key] = impl[key];
        }
    }

    // 添加兜底的 invoke 方法
    handler.invoke = function (proxy, method, args) {
        var methodName = method.getName();

        // 特殊处理 getClass 方法
        if ("getClass" === methodName) {
            return javaClass;
        }

        // 尝试调用已定义的方法
        if (typeof handler[methodName] === "function") {
            return handler[methodName].apply(proxy, args || []);
        }

        // 根据返回类型返回安全默认值
        var returnType = method.getReturnType();
        if (returnType === Void.TYPE) return;
        if (returnType.isPrimitive()) {
            if (returnType === Boolean.TYPE) return false;
            if (returnType === Integer.TYPE) return 0;
            if (returnType === Long.TYPE) return 0;
            if (returnType === Double.TYPE) return 0.0;
            if (returnType === Float.TYPE) return 0.0;
            if (returnType === Character.TYPE) return "\u0000";
            if (returnType === Byte.TYPE) return 0;
            if (returnType === Short.TYPE) return 0;
        }
        return null;
    };

    // 创建 JavaAdapter 实例
    return new JavaAdapter(javaClass, [], handler);
}
