package io.github.startsmercury.visual_snowy_leaves.impl.client;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonParser;
import com.google.gson.stream.JsonWriter;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.JsonOps;
import io.github.startsmercury.visual_snowy_leaves.impl.client.config.Config;
import io.github.startsmercury.visual_snowy_leaves.impl.client.extension.SnowDataAware;
import io.github.startsmercury.visual_snowy_leaves.impl.client.util.Chunks;
import io.github.startsmercury.visual_snowy_leaves.impl.client.util.Reporter;
import it.unimi.dsi.fastutil.objects.ReferenceOpenHashSet;
import net.fabricmc.loader.api.FabricLoader;
import net.fabricmc.loader.api.ModContainer;
import net.fabricmc.loader.api.Version;
import net.fabricmc.loader.api.metadata.ModMetadata;
import net.minecraft.class_10820;
import net.minecraft.class_1087;
import net.minecraft.class_124;
import net.minecraft.class_155;
import net.minecraft.class_156;
import net.minecraft.class_2558;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.*;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.function.Function;
import java.util.stream.Collectors;

public final class VisualSnowyLeavesImpl {
    private Config config;

    private final FabricLoader fabricLoader;

    private final Logger logger;

    private final class_310 minecraft;

    private final Reporter<Class<? extends class_1087.class_10892>> blockStateModelReporter;

    private final Reporter<Class<? extends class_10820>> geometryReporter;

    private Path reportFile;

    public VisualSnowyLeavesImpl(final class_310 minecraft) {
        this.config = Config.DEFAULT;
        this.fabricLoader = FabricLoader.getInstance();
        this.logger = LoggerFactory.getLogger(VslConstants.NAME);
        this.minecraft = minecraft;

        final Function<? super Class<?>, String> classFormatter = Class::getName;
        this.blockStateModelReporter = new Reporter<>(new ReferenceOpenHashSet<>(), classFormatter);
        this.geometryReporter = new Reporter<>(new ReferenceOpenHashSet<>(), classFormatter);

        this.logger.info("{} is initialized!", VslConstants.NAME);
    }

    public Logger getLogger() {
        return this.logger;
    }

    public Config getConfig() {
        return this.config;
    }

    public void setConfig(final Config config) {
        final var oldConfig = this.config;
        this.config = config;

        //noinspection ConstantValue
        if (
            !oldConfig.targetBlockKeys().equals(config.targetBlockKeys())
                // May be true when too early in construction injection
                && this.minecraft.method_1478() != null
        ) {
            this.logger.debug(
                "[{}] Reloading resource packs to modify sprite changes...",
                VslConstants.NAME
            );

            this.minecraft.method_1521();
        }

        final var level = this.minecraft.field_1687;
        if (level == null) {
            this.logger.debug(
                "[{}] Skipping level snowy ticker since there is no level...",
                VslConstants.NAME
            );
            return;
        }

        if (oldConfig.transitionDuration() != config.transitionDuration()) {
            this.logger.debug(
                "[{}] Normalizing snowy progress using ratio and proportion...",
                VslConstants.NAME
            );

            ((SnowDataAware) level).visual_snowy_leaves$getSnowData().onTransitionDurationChange(
                oldConfig.transitionDuration().asTicks(),
                config.transitionDuration().asTicks()
            );
        }

        if (oldConfig.snowyMode() != config.snowyMode()) {
            this.logger.debug(
                "[{}] Snowing mode changed, requesting lazy rebuild to all chunks...",
                VslConstants.NAME
            );

            Chunks.requestRebuildAll(level);
        }
    }

    public void openConfigFile() {
        this.logger.debug("[{}] Opening config...", VslConstants.NAME);
        class_156.method_668().method_672(this.getConfigFile());
    }

    public void reloadConfig() {
        if (this.loadConfig()) {
            this.saveConfig();
        } else {
            this.openConfigFile();
        }
    }

    /**
     * @return {@code false} if loading encountered json syntax exceptions;
     *     {@code true} otherwise.
     */
    private boolean loadConfig() {
        this.logger.debug("[{}] Loading config...", VslConstants.NAME);

        final var path = this.getConfigPath();
        final JsonElement json;

        final ArrayList<CharSequence> lines;
        try (final var lineStream = Files.lines(path)) {
            lines = lineStream
                .filter(line -> !line.endsWith(VslConstants.IGNORE_TAG))
                .collect(Collectors.toCollection(ArrayList::new));
        } catch (final NoSuchFileException cause) {
            this.logger.info("[{}] Config does not exist, using default", VslConstants.NAME);
            return true;
        } catch (final IOException cause) {
            this.logger.warn("[{}] Unable to read config json", VslConstants.NAME, cause);
            return true;
        }

        try{
            json = JsonParser.parseString(String.join("\n", lines));
        } catch (final JsonParseException cause) {
            this.logger.warn("[{}] Invalid config json syntax", VslConstants.NAME, cause);

            final var lineMatcher = VslConstants.LINE_PATTERN.matcher(cause.getMessage());
            var line = 0;

            if (lineMatcher.find()) {
                final var capturedLine = lineMatcher.group(1);
                try {
                    line = Integer.parseInt(capturedLine);
                } catch (final NumberFormatException ignored) {

                }
            }


            final var errorMessageBuilder = new StringBuilder();

            final var columnMatcher = VslConstants.COLUMN_PATTERN.matcher(cause.getMessage());

            if (columnMatcher.find()) {
                try {
                    final var capturedColumn = columnMatcher.group(1);
                    final var column = Integer.parseInt(capturedColumn);

                    if (column >= 2) {
                        errorMessageBuilder.append(" ".repeat(column - 2));
                    }

                    errorMessageBuilder.append("^ ");
                } catch (final NumberFormatException ignored) {

                }
            }

            errorMessageBuilder.append(cause.getMessage())
                .append("\t")
                .append(VslConstants.IGNORE_TAG);
            lines.add(line, errorMessageBuilder);

            try {
                Files.write(path, lines);
            } catch (final IOException cause2) {
                this.logger.warn(
                    "[{}] Unable to update config json with an error message",
                    VslConstants.NAME,
                    cause2
                );
            }

            return false;
        }

        Config.LENIENT_CODEC
            .decode(JsonOps.INSTANCE, json)
            .ifSuccess(result -> this.setConfig(result.getFirst().upgrade()))
            .ifError(result -> this.logger
                .warn("[{}] Unable to decode config: {}", VslConstants.NAME, result.message())
            );

        return true;
    }

    private void saveConfig() {
        this.logger.debug("[{}] Saving config...", VslConstants.NAME);

        final var path = this.fabricLoader.getConfigDir().resolve(VslConstants.CONFIG_NAME);

        final JsonObject json;
        switch (Config.CODEC.encodeStart(JsonOps.INSTANCE, this.config)) {
            case DataResult.Success<JsonElement>(final var value, final var lifecycle):
                json = (JsonObject) value;
                break;
            case DataResult.Error<JsonElement>(
                final var messageSupplier,
                final var partialValue,
                final var lifecycle
            ):
                final var message = messageSupplier.get();
                this.logger.warn("[{}] Unable to encode config: {}", VslConstants.NAME, message);
                return;
        }

        json.addProperty("__message", "Click the config button again to load changes.");

        try (
            final var bufferedWriter = Files.newBufferedWriter(path);
            final var jsonWriter = new JsonWriter(bufferedWriter)
        ) {
            jsonWriter.setIndent("    ");

            GsonHelper.writeValue(jsonWriter, json, Comparator.naturalOrder());

            bufferedWriter.newLine();
        } catch (final IOException cause) {
            this.logger.warn("[{}] Unable to write config json", VslConstants.NAME, cause);
        }
    }

    private Path getConfigPath() {
        return this.fabricLoader.getConfigDir().resolve(VslConstants.CONFIG_NAME);
    }

    public File getConfigFile() {
        return this.getConfigPath().toFile();
    }

    public Reporter<Class<? extends class_1087.class_10892>> getBlockStateModelRecRep() {
        return this.blockStateModelReporter;
    }

    public Reporter<Class<? extends class_10820>> getGeometryRecRep() {
        return this.geometryReporter;
    }

    public void collectReports(final SpriteWhitener whitener) {
        class PathBuf {
            Path inner;
        }

        final PathBuf tempFile = new PathBuf();

        try (final var printWriter = whitener.collectReports(() -> {
            try {
                tempFile.inner = Files.createTempFile(VslConstants.MODID, "reports.txt");

                this.logger.info("[{}] Created new temporary report file", VslConstants.NAME);

                final var template = "[{}] Created new temporary report file at {}";
                this.logger.debug(template, VslConstants.NAME, tempFile);
            } catch (final IOException cause) {
                throw new IOException("Unable to create temporary report file", cause);
            }

            final var writer = new PrintWriter(Files.newBufferedWriter(tempFile.inner));

            writer.print("Minecraft ");
            try {
                writer.println(class_155.method_16673().method_48019());
            } catch (final RuntimeException cause) {
                writer.print('<');
                writer.print(cause.getMessage());
                writer.println('>');
            }

            final var version = this
                .fabricLoader
                .getModContainer(VslConstants.MODID)
                .map(ModContainer::getMetadata)
                .map(ModMetadata::getVersion)
                .map(Version::getFriendlyString)
                .orElse("<unknown>");
            writer.print(VslConstants.NAME);
            writer.print(' ');
            writer.println(version);

            writer.println();

            return writer;
        })) {
            if (printWriter == null) {
                final var template = "[{}] Skipping reporting, there is nothing to report";
                this.logger.info(template, VslConstants.NAME);
                return;
            } else {
                this.logger.info("[{}] Successfully wrote the temporary report", VslConstants.NAME);
                printWriter.println("Submit this to the dedicated Google Forms:");
                printWriter.println();
                for (var i = 0; i < 3; i++) {
                    printWriter.println("> https://forms.gle/UiwbEqhkTnrBC97C7");
                }
            }
        } catch (final IOException cause) {
            this.logger.error("[{}] Unable to write the temporary report", VslConstants.NAME, cause);
            return;
        }

        final var path = this
            .fabricLoader
            .getGameDir()
            .resolve("logs")
            .resolve(VslConstants.MODID)
            .resolve("reports.txt");

        try {
            final var loggingDirectory = path.getParent();
            Files.createDirectory(loggingDirectory);
            this.logger.info("[{}] Successfully created logging subdirectory", VslConstants.NAME);

            final var template = "[{}] Successfully created logging subdirectory at {}";
            this.logger.info(template, VslConstants.NAME, loggingDirectory);
        } catch (final FileAlreadyExistsException ignored) {
            this.logger.info("[{}] Logging subdirectory already exists", VslConstants.NAME);
        } catch (final IOException cause) {
            final var template = "[{}] Unable to create logging subdirectory";
            this.logger.error(template, VslConstants.NAME, cause);
            return;
        }

        try {
            assert tempFile.inner != null;
            Files.move(tempFile.inner, path, StandardCopyOption.REPLACE_EXISTING);
            this.logger.info("[{}] Successfully committed report file changes", VslConstants.NAME);
        } catch (final IOException cause) {
            this.logger.error("[{}] Unable to commit changes to report file", VslConstants.NAME, cause);
            return;
        }

        this.reportFile = path;
    }

    public boolean sendReportNotice() {
        final var reportFile = this.reportFile;
        if (reportFile == null) {
            return false;
        }

        final var player = this.minecraft.field_1724;
        if (player == null) {
            return false;
        }

        final var underlined = class_2561.method_43470("Click this message to view reports.txt")
            .method_27694(arg -> arg.method_30938(true));
        final var message = class_2561.method_43471(
            "["
                + VslConstants.NAME
                + "] Detected unsupported custom classes."
                + " Some models may fail to be snowy. "
        ).method_10852(underlined)
            .method_27694(arg -> arg
                .method_27706(class_124.field_1061)
                .method_10958(new class_2558.class_10607(reportFile))
            );
        player.method_7353(message, false);

        this.reportFile = null;

        return true;
    }
}
