package io.github.startsmercury.visual_snowy_leaves.impl.client;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import io.github.startsmercury.visual_snowy_leaves.impl.client.util.ColorComponent;
import io.github.startsmercury.visual_snowy_leaves.impl.client.util.Reporter;
import io.github.startsmercury.visual_snowy_leaves.mixin.client.tint.BlockColorsAccessor;
import io.github.startsmercury.visual_snowy_leaves.mixin.client.tint.SpriteContentsAccessor;
import it.unimi.dsi.fastutil.ints.AbstractInt2IntMap;
import it.unimi.dsi.fastutil.ints.Int2IntMap;
import it.unimi.dsi.fastutil.ints.Int2IntOpenHashMap;
import it.unimi.dsi.fastutil.ints.Int2ReferenceMap;
import it.unimi.dsi.fastutil.ints.Int2ReferenceOpenHashMap;
import it.unimi.dsi.fastutil.objects.ReferenceOpenHashSet;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import net.minecraft.class_1011;
import net.minecraft.class_10419;
import net.minecraft.class_1058;
import net.minecraft.class_10802;
import net.minecraft.class_10820;
import net.minecraft.class_1087;
import net.minecraft.class_10893;
import net.minecraft.class_1097;
import net.minecraft.class_1100;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_324;
import net.minecraft.class_4730;
import net.minecraft.class_6010;
import net.minecraft.class_783;
import net.minecraft.class_785;
import net.minecraft.class_790;
import net.minecraft.class_7923;
import net.minecraft.class_793;
import net.minecraft.class_8012;
import net.minecraft.class_813;
import net.minecraft.class_819;
import net.minecraft.class_9848;
import org.apache.commons.io.function.IOSupplier;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.system.MemoryUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SpriteWhitener {
    private static final SpriteWhitener EMPTY;

    static {
        @SuppressWarnings({ "rawtypes", "unchecked" })
        final var recRep = new Reporter(Set.of());

        @SuppressWarnings("unchecked")
        final var empty = new SpriteWhitener(
            LoggerFactory.getLogger(Logger.ROOT_LOGGER_NAME),
            Multimaps.forMap(Map.of()),
            Set.of(),
            recRep,
            recRep
        );

        EMPTY = empty;
    }

    public static SpriteWhitener createDefault() {
        return create(class_310.method_1551().getVisualSnowyLeaves());
    }

    public static SpriteWhitener create(final VisualSnowyLeavesImpl visualSnowyLeaves) {
        return new SpriteWhitener(
            visualSnowyLeaves.getLogger(),
            HashMultimap.create(),
            Set.copyOf(visualSnowyLeaves.getConfig().targetBlockKeys()),
            visualSnowyLeaves.getBlockStateModelRecRep(),
            visualSnowyLeaves.getGeometryRecRep()
        );
    }

    public static SpriteWhitener getEmpty() {
        return SpriteWhitener.EMPTY;
    }

    private final Logger logger;

    private final Multimap<class_2960, class_2960> models;

    private final Set<? extends class_2960> targetBlockKeys;

    private final Reporter<Class<? extends class_1087.class_10892>> blockStateModelReporter;

    private final Reporter<Class<? extends class_10820>> geometryReporter;

    private SpriteWhitener(
        final Logger logger,
        final Multimap<class_2960, class_2960> models,
        final Set<? extends class_2960> targetBlockKeys,
        final Reporter<Class<? extends class_1087.class_10892>> blockStateModelReporter,
        final Reporter<Class<? extends class_10820>> geometryReporter
    ) {
        this.logger = logger;
        this.models = models;
        this.targetBlockKeys = targetBlockKeys;
        this.blockStateModelReporter = blockStateModelReporter;
        this.geometryReporter = geometryReporter;
    }

    public void analyzeModels(
        final class_2960 blockKey,
        final class_790 blockModelDefinition
    ) {
        if (!this.targetBlockKeys.contains(blockKey)) {
            return;
        }

        final var multiPartVariants = blockModelDefinition
            .comp_3746()
            .stream()
            .flatMap(it -> it.comp_3052().stream().map(class_819::comp_3765));

        final var variants = blockModelDefinition
            .comp_3813()
            .stream()
            .map(class_790.class_10888::comp_3814)
            .map(Map::values)
            .flatMap(Collection::stream);

        Stream.concat(multiPartVariants, variants)
            .flatMap(this::flattenToVariants)
            .map(class_813::comp_3379)
            .forEach(model -> this.models.put(blockKey, model));
    }

    public Stream<class_813> flattenToVariants(final class_1087.class_10892 unbaked) {
        return switch (unbaked) {
            case class_10893.class_10894(final var variant) -> Stream.of(variant);
            case class_1097.class_10898(final var entries) -> entries
                .method_34994()
                .stream()
                .map(class_6010::comp_2542)
                .flatMap(this::flattenToVariants);
            default -> {
                blockStateModelReporter.report(unbaked.getClass());
                yield Stream.empty();
            }
        };
    }

    public void modifySprites(
        final class_324 blockColors,
        final Map<class_2960, class_1100> modelResources,
        final Function<class_2960, class_1058> atlas
    ) {
        final var blockColorsAccessor = (BlockColorsAccessor) blockColors;

        for (final var blockColor : blockColorsAccessor.getBlockColors()) {
            if (blockColor instanceof final SnowableBlockColor snowable) {
                final var id = blockColorsAccessor.getBlockColors().method_10206(blockColor);
                // It is possible for changes to persist without this:
                blockColors.method_1690(snowable.blockColor(), class_7923.field_41175.method_10200(id));
            }
        }

        for (final var block : this.models.keySet()) {
            modifySpritesOf(blockColors, modelResources, atlas, logger, block);
        }
    }

    private void modifySpritesOf(
        final class_324 blockColors,
        final Map<class_2960, class_1100> modelResources,
        final Function<class_2960, class_1058> atlas,
        final Logger logger,
        final class_2960 blockKey
    ) {
        final var layers = Set.copyOf(this.models.get(blockKey))
            .stream()
            .map(modelResources::get)
            .map(SpriteWhitener::asBlockModelOrElseNull)
            .filter(Objects::nonNull)
            .map(blockModel -> {
                @SuppressWarnings({ "unchecked", "rawtypes" })
                final var models = (List<class_793>) (List) Stream.iterate(
                    (class_1100) blockModel,
                    it -> it != null && it.comp_3744() != null,
                    it -> modelResources.get(it.comp_3744())
                ).toList();

                final var textureSlots = new HashMap<String, class_10419.class_10424>();
                for (final var model : models.reversed()) {
                    textureSlots.putAll(model.comp_3743().comp_3376());
                }

                return models
                    .stream()
                    .map(class_793::comp_3739)
                    .filter(Objects::nonNull)
                    .flatMap(this::flattenToElements)
                    .flatMap(element -> element.comp_3729().values().stream())
                    .collect(Collectors.groupingBy(
                        class_783::comp_2868,
                        Int2ReferenceOpenHashMap::new,
                        Collectors.mapping(
                            face -> {
                                final var material =
                                    resolveSlotContent(textureSlots, textureSlots.get(face.comp_2869().substring(1)));
                                if (material == null) {
                                    return null;
                                }

                                final var sprite = atlas.apply(material.method_24147());
                                if (sprite == null) {
                                    return null;
                                }

                                return (SpriteContentsAccessor) sprite.method_45851();
                            },
                            Collectors.filtering(Objects::nonNull, Collectors.toCollection(ReferenceOpenHashSet::new))
                        )
                    ));
            })
            .map(Int2ReferenceMap::int2ReferenceEntrySet)
            .flatMap(Set::stream)
            .collect(Collectors.groupingBy(
                Int2ReferenceMap.Entry::getIntKey,
                Int2ReferenceOpenHashMap::new,
                Collectors.flatMapping(
                    entry -> entry.getValue().stream(),
                    Collectors.toCollection(ReferenceOpenHashSet::new)
                )
            ));

        final var multipliers = layers.int2ReferenceEntrySet().stream().map(entry -> {
            final var index = entry.getIntKey();
            final var contentsCollection = entry.getValue();

            final var allArgbPixels = contentsCollection
                .stream()
                .map(SpriteContentsAccessor::getOriginalImage)
                .map(class_1011::method_61942)
                .flatMapToInt(IntStream::of)
                .toArray();

            final var _rgbMultiplier = 0xFF_00_00_00 | getArgbOfMaxLightness(allArgbPixels);

            contentsCollection
                .stream()
                .map(SpriteContentsAccessor::getByMipLevel)
                .flatMap(Stream::of)
                .forEach(image -> {
                    final var format = image.method_4318();

                    if (format != class_1011.class_1012.field_4997) {
                        final var template = "function application only works on RGBA images; have %s";
                        final var message = String.format(Locale.ROOT, template, format);
                        throw new IllegalArgumentException(message);
                    }

                    final var pointer = image.method_67769();

                    if (pointer == 0L) {
                        throw new IllegalStateException("Image is not allocated.");
                    }

                    final var pixelCount = image.method_4307() * image.method_4323();
                    final var buffer = MemoryUtil.memIntBuffer(pointer, pixelCount);

                    for (var i = 0; i < pixelCount; ++i) {
                        final var original = class_9848.method_61338(buffer.get(i));
                        final var modified = normalize(original, _rgbMultiplier);
                        buffer.put(i, class_9848.method_61337(modified));
                    }
                });

            return new AbstractInt2IntMap.BasicEntry(index, _rgbMultiplier);
        }).collect(Collectors.toMap(Int2IntMap.Entry::getIntKey, Map.Entry::getValue, (x, y) -> x, Int2IntOpenHashMap::new));
        multipliers.defaultReturnValue(class_8012.field_42973);

        final var optionalBlockHolder = class_7923.field_41175.method_10223(blockKey);
        if (optionalBlockHolder.isEmpty()) {
            logger.warn("[{}] Registry holder of block {} is missing from the registry", VslConstants.NAME, blockKey);
            return;
        }

        final var blockHolder = optionalBlockHolder.get();
        if (!blockHolder.method_40227()) {
            logger.warn("[{}] Registry holder of block {} is not yet bound", VslConstants.NAME, blockKey);
            return;
        }

        multipliers.remove(-1);
        final var block = blockHolder.comp_349();
        final var id = class_7923.field_41175.method_10206(block);
        final var blockColor = Objects.requireNonNullElse(
            ((BlockColorsAccessor) blockColors).getBlockColors().method_10200(id),
            ConstantBlockColor.WHITE
        );
        blockColors.method_1690(SnowableBlockColor.setMultiplier(blockColor, multipliers), block);
    }

    private Stream<class_785> flattenToElements(final class_10820 geometry) {
        return switch (geometry) {
            case class_10802(final var elements) -> elements.stream();
            default -> {
                this.geometryReporter.report(geometry.getClass());
                yield Stream.empty();
            }
        };
    }

    private static @Nullable class_4730 resolveSlotContent(
        final Map<? super String, ? extends class_10419.class_10424> textureSlots,
        class_10419.class_10424 slotContents
    ) {
        while (true) {
            switch (slotContents) {
                case null:
                    return null;
                case class_10419.class_10425(final var material):
                    return material;
                case class_10419.class_10422(final var target):
                    slotContents = textureSlots.get(target);
                    break;
            }
        }
    }

    private static @Nullable class_793 asBlockModelOrElseNull(final class_1100 unbakedModel) {
        return unbakedModel instanceof final class_793 blockModel ? blockModel : null;
    }

    private static int getArgbOfMaxLightness(final int[] argbArray) {
        var best = 0;
        var value = 0;

        for (final var argb : argbArray) {
            final var a = class_9848.method_61320(argb);
            final var r = class_9848.method_61327(argb);
            final var g = class_9848.method_61329(argb);
            final var b = class_9848.method_61331(argb);

            if (a < best) {
                continue;
            }

            final var key = Math.max(Math.max(r, g), b);

            if (a > best || key > value) {
                value = key;
            }

            best = a;
        }

        return class_9848.method_61324(best, value, value, value);
    }

    private int normalize(final int argb, final int _xyz) {
        final var a = class_9848.method_61320(argb);
        final var r = class_9848.method_61327(argb);
        final var g = class_9848.method_61329(argb);
        final var b = class_9848.method_61331(argb);

        final var x = class_9848.method_61327(_xyz);
        final var y = class_9848.method_61329(_xyz);
        final var z = class_9848.method_61331(_xyz);

        final var sr = ColorComponent.div(r, x);
        final var sg = ColorComponent.div(g, y);
        final var sb = ColorComponent.div(b, z);

        return class_9848.method_61324(a, sr, sg, sb);
    }

    public @Nullable PrintWriter collectReports(final IOSupplier<PrintWriter> writerProvider) throws IOException {
        if (!(this.blockStateModelReporter.consumeChanged() | this.geometryReporter.consumeChanged())) {
            return null;
        }

        final var writer = writerProvider.get();

        if (!(
            this.geometryReporter.collectReport(writer, "Unrecognized Unbaked Geometry classes:")
                | this.blockStateModelReporter.collectReport(writer, "Unrecognized Unbaked BlockStateModel classes:"))
        ) {
            return null;
        }

        return writer;
    }
}
