/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.tunnelyrefab.messaging;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import okhttp3.Response;
import org.texboobcat.tunnelyrefab.auth.SupabaseClient;
import org.texboobcat.tunnelyrefab.config.TunnelConfig;
import org.texboobcat.tunnelyrefab.messaging.Message;
import org.texboobcat.tunnelyrefab.messaging.SupabaseRealtimeClient;
import org.texboobcat.tunnelyrefab.ui.ToastNotificationManager;

public class MessageManager {
    private static MessageManager instance;
    private final ScheduledExecutorService scheduler;
    private final AtomicBoolean running;
    private final TunnelConfig config;
    private final Map<String, List<Consumer<Message>>> conversationListeners;
    private final List<Consumer<Integer>> unreadCountListeners;
    private final Map<String, String> lastSeenMessageIds;
    private boolean isFirstPoll = true;
    private static final int POLL_INTERVAL_SECONDS = 10;
    private static final long REMINDER_INTERVAL_MS = 3600000L;
    private long lastReminderTime = 0L;
    private List<String> cachedFriendIds = new ArrayList<String>();
    private long lastFriendsRefresh = 0L;
    private static final long FRIENDS_REFRESH_INTERVAL_MS = 30000L;
    private SupabaseRealtimeClient realtimeClient;
    private boolean useRealtime = true;

    private MessageManager() {
        this.scheduler = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread thread = new Thread(r, "Tunnely-MessagePoller");
            thread.setDaemon(true);
            return thread;
        });
        this.running = new AtomicBoolean(false);
        this.config = TunnelConfig.getInstance();
        this.conversationListeners = new ConcurrentHashMap<String, List<Consumer<Message>>>();
        this.unreadCountListeners = new CopyOnWriteArrayList<Consumer<Integer>>();
        this.lastSeenMessageIds = new ConcurrentHashMap<String, String>();
    }

    public static synchronized MessageManager getInstance() {
        if (instance == null) {
            instance = new MessageManager();
        }
        return instance;
    }

    public void start() {
        if (this.running.get()) {
            System.out.println("[Tunnely Messages] Already running");
            return;
        }
        if (!this.config.isAuthenticated()) {
            System.out.println("[Tunnely Messages] Not authenticated, not starting message system");
            return;
        }
        this.running.set(true);
        if (this.useRealtime && this.tryStartRealtime()) {
            System.out.println("[Tunnely Messages] \u2713 Using Realtime WebSocket (instant notifications)");
            this.scheduler.scheduleAtFixedRate(this::pollUnreadCount, 30L, 30L, TimeUnit.SECONDS);
        } else {
            System.out.println("[Tunnely Messages] Using polling fallback (10-second intervals)");
            this.startPolling();
        }
    }

    private boolean tryStartRealtime() {
        try {
            String userId = this.config.getSupabaseUserId();
            if (userId == null || userId.isEmpty()) {
                return false;
            }
            String supabaseUrl = this.config.getSupabaseUrl();
            String apiKey = this.config.getSupabaseAnonKey();
            String accessToken = this.config.getUserJwtToken();
            if (supabaseUrl == null || apiKey == null || accessToken == null) {
                System.err.println("[Tunnely Messages] Missing Supabase credentials for Realtime");
                return false;
            }
            this.realtimeClient = new SupabaseRealtimeClient(supabaseUrl, apiKey, accessToken, userId, this::onRealtimeMessage);
            this.realtimeClient.connect();
            for (int attempts = 0; attempts < 20 && !this.realtimeClient.isSubscribed(); ++attempts) {
                Thread.sleep(100L);
            }
            if (this.realtimeClient.isSubscribed()) {
                return true;
            }
            System.out.println("[Tunnely Messages] Realtime connection timeout, falling back to polling");
            this.realtimeClient.disconnect();
            return false;
        }
        catch (Exception e) {
            System.err.println("[Tunnely Messages] Realtime failed: " + e.getMessage());
            if (this.realtimeClient != null) {
                this.realtimeClient.disconnect();
            }
            return false;
        }
    }

    private void startPolling() {
        System.out.println("[Tunnely Messages] Starting polling mode (every 10 seconds)");
        this.pollMessages();
        this.scheduler.scheduleAtFixedRate(this::pollMessages, 10L, 10L, TimeUnit.SECONDS);
    }

    public void stop() {
        if (!this.running.get()) {
            return;
        }
        this.running.set(false);
        if (this.realtimeClient != null) {
            this.realtimeClient.disconnect();
            this.realtimeClient = null;
        }
        this.conversationListeners.clear();
        this.lastSeenMessageIds.clear();
        this.cachedFriendIds.clear();
        this.lastFriendsRefresh = 0L;
        this.isFirstPoll = true;
        System.out.println("[Tunnely Messages] Stopped");
    }

    public void subscribeToConversation(String friendId, Consumer<Message> listener) {
        String userId = this.config.getSupabaseUserId();
        if (userId == null) {
            return;
        }
        String conversationId = Message.createConversationId(userId, friendId);
        this.conversationListeners.computeIfAbsent(conversationId, k -> new CopyOnWriteArrayList()).add(listener);
        System.out.println("[Tunnely Messages] Subscribed to conversation: " + conversationId);
    }

    public void unsubscribeFromConversation(String friendId) {
        String userId = this.config.getSupabaseUserId();
        if (userId == null) {
            return;
        }
        String conversationId = Message.createConversationId(userId, friendId);
        this.conversationListeners.remove(conversationId);
        this.lastSeenMessageIds.remove(conversationId);
        System.out.println("[Tunnely Messages] Unsubscribed from conversation: " + conversationId);
    }

    public void subscribeToUnreadCount(Consumer<Integer> listener) {
        this.unreadCountListeners.add(listener);
    }

    public void unsubscribeFromUnreadCount(Consumer<Integer> listener) {
        this.unreadCountListeners.remove(listener);
    }

    private void pollMessages() {
        if (!this.running.get()) {
            return;
        }
        try {
            long currentTime;
            SupabaseClient client = new SupabaseClient();
            String userId = this.config.getSupabaseUserId();
            if (userId != null) {
                currentTime = System.currentTimeMillis();
                if (this.cachedFriendIds.isEmpty() || currentTime - this.lastFriendsRefresh >= 30000L) {
                    this.refreshFriendsList(client, userId);
                    this.lastFriendsRefresh = currentTime;
                }
                for (String friendId : this.cachedFriendIds) {
                    String conversationId = Message.createConversationId(userId, friendId);
                    this.pollConversation(client, conversationId);
                }
            }
            if (!this.unreadCountListeners.isEmpty()) {
                int unreadCount = client.getUnreadCount();
                for (Consumer<Integer> listener : this.unreadCountListeners) {
                    listener.accept(unreadCount);
                }
            }
            if ((currentTime = System.currentTimeMillis()) - this.lastReminderTime >= 3600000L) {
                ToastNotificationManager.getInstance().showMessageNotification("Tunnely Tip", "Press U to disable notifications in settings");
                this.lastReminderTime = currentTime;
            }
            if (this.isFirstPoll) {
                this.isFirstPoll = false;
            }
        }
        catch (Exception e) {
            System.err.println("[Tunnely Messages] Error polling: " + e.getMessage());
        }
    }

    private void onRealtimeMessage(JsonObject messageRecord) {
        try {
            Message msg = this.parseRealtimeMessage(messageRecord);
            if (msg == null) {
                return;
            }
            String userId = this.config.getSupabaseUserId();
            if (userId == null) {
                return;
            }
            if (!msg.getRecipientId().equals(userId)) {
                return;
            }
            System.out.println("[Tunnely Messages] New message via Realtime from: " + msg.getSenderId());
            String conversationId = msg.getConversationId();
            List<Consumer<Message>> listeners = this.conversationListeners.get(conversationId);
            if (listeners != null) {
                for (Consumer<Message> listener : listeners) {
                    listener.accept(msg);
                }
            }
            if (!msg.getSenderId().equals(userId)) {
                String friendId = msg.getSenderId();
                this.showMessageToast(msg, friendId);
            }
        }
        catch (Exception e) {
            System.err.println("[Tunnely Messages] Error handling realtime message: " + e.getMessage());
        }
    }

    private Message parseRealtimeMessage(JsonObject record) {
        try {
            String id = record.get("id").getAsString();
            String senderId = record.get("sender_id").getAsString();
            String recipientId = record.get("recipient_id").getAsString();
            String content = record.get("content").getAsString();
            String sentAt = record.get("sent_at").getAsString();
            String readAt = record.has("read_at") && !record.get("read_at").isJsonNull() ? record.get("read_at").getAsString() : null;
            String conversationId = record.get("conversation_id").getAsString();
            String decryptedContent = content;
            return new Message(id, senderId, recipientId, decryptedContent, sentAt, readAt, conversationId);
        }
        catch (Exception e) {
            System.err.println("[Tunnely Messages] Error parsing realtime message: " + e.getMessage());
            return null;
        }
    }

    private void pollUnreadCount() {
        if (!this.running.get()) {
            return;
        }
        try {
            if (!this.unreadCountListeners.isEmpty()) {
                SupabaseClient client = new SupabaseClient();
                int unreadCount = client.getUnreadCount();
                for (Consumer<Integer> listener : this.unreadCountListeners) {
                    listener.accept(unreadCount);
                }
            }
        }
        catch (Exception e) {
            System.err.println("[Tunnely Messages] Error polling unread count: " + e.getMessage());
        }
    }

    private void refreshFriendsList(SupabaseClient client, String userId) {
        try {
            System.out.println("[Tunnely Messages] Refreshing friends list cache...");
            SupabaseClient.FriendListResult friendsResult = client.getFriends();
            if (friendsResult.success && friendsResult.data != null) {
                JsonArray friends = JsonParser.parseString((String)friendsResult.data).getAsJsonArray();
                this.cachedFriendIds.clear();
                for (JsonElement friendElement : friends) {
                    JsonObject friendObj = friendElement.getAsJsonObject();
                    String friendUserId = friendObj.get("user_id").getAsString();
                    String friendFriendId = friendObj.get("friend_id").getAsString();
                    String friendId = friendUserId.equals(userId) ? friendFriendId : friendUserId;
                    this.cachedFriendIds.add(friendId);
                }
                System.out.println("[Tunnely Messages] Cached " + this.cachedFriendIds.size() + " friends");
            }
        }
        catch (Exception e) {
            System.err.println("[Tunnely Messages] Error refreshing friends list: " + e.getMessage());
        }
    }

    private void pollConversation(SupabaseClient client, String conversationId) {
        try {
            JsonArray messages;
            String userId = this.config.getSupabaseUserId();
            if (userId == null) {
                return;
            }
            String[] parts = conversationId.split("_");
            String friendId = parts[0].equals(userId) ? parts[1] : parts[0];
            SupabaseClient.MessagesResult result = client.getMessages(friendId, 5);
            if (result.success && result.data != null && (messages = JsonParser.parseString((String)result.data).getAsJsonArray()).size() > 0) {
                JsonObject latestMsg = messages.get(0).getAsJsonObject();
                String latestId = latestMsg.get("id").getAsString();
                String lastSeenId = this.lastSeenMessageIds.get(conversationId);
                if (lastSeenId == null || !latestId.equals(lastSeenId)) {
                    Object msg;
                    JsonElement element;
                    JsonObject msgObj;
                    Object msgId;
                    ArrayList<Message> newMessages = new ArrayList<Message>();
                    Iterator iterator = messages.iterator();
                    while (iterator.hasNext() && !((String)(msgId = (msgObj = (element = (JsonElement)iterator.next()).getAsJsonObject()).get("id").getAsString())).equals(lastSeenId)) {
                        Message message = this.parseMessage(msgObj);
                        if (message == null) continue;
                        newMessages.add(message);
                    }
                    List<Consumer<Message>> listeners = this.conversationListeners.get(conversationId);
                    if (listeners != null) {
                        for (int i = newMessages.size() - 1; i >= 0; --i) {
                            msg = (Message)newMessages.get(i);
                            for (Consumer consumer : listeners) {
                                consumer.accept(msg);
                            }
                        }
                    }
                    if (this.isFirstPoll && !newMessages.isEmpty()) {
                        Message mostRecentUnreadFromFriend = null;
                        for (Message msg3 : newMessages) {
                            if (msg3.getSenderId().equals(userId) || msg3.isRead()) continue;
                            mostRecentUnreadFromFriend = msg3;
                            break;
                        }
                        if (mostRecentUnreadFromFriend != null) {
                            this.showMessageToast(mostRecentUnreadFromFriend, friendId);
                        }
                    } else {
                        for (int i = newMessages.size() - 1; i >= 0; --i) {
                            msg = (Message)newMessages.get(i);
                            if (((Message)msg).getSenderId().equals(userId) || ((Message)msg).isRead()) continue;
                            this.showMessageToast((Message)msg, friendId);
                        }
                    }
                    this.lastSeenMessageIds.put(conversationId, latestId);
                }
            }
        }
        catch (Exception e) {
            System.err.println("[Tunnely Messages] Error polling conversation " + conversationId + ": " + e.getMessage());
        }
    }

    private Message parseMessage(JsonObject json) {
        try {
            String id = json.get("id").getAsString();
            String senderId = json.get("sender_id").getAsString();
            String recipientId = json.get("recipient_id").getAsString();
            String content = json.get("content").getAsString();
            String sentAt = json.get("sent_at").getAsString();
            String readAt = json.has("read_at") && !json.get("read_at").isJsonNull() ? json.get("read_at").getAsString() : null;
            String conversationId = json.get("conversation_id").getAsString();
            return new Message(id, senderId, recipientId, content, sentAt, readAt, conversationId);
        }
        catch (Exception e) {
            return null;
        }
    }

    private void showMessageToast(Message message, String friendId) {
        try {
            SupabaseClient client = new SupabaseClient();
            String endpoint = "/rest/v1/user_profiles?id=eq." + friendId + "&select=minecraft_name";
            try (Response response = client.makeAuthenticatedRequest(endpoint, "GET", null);){
                String responseBody;
                JsonArray profiles;
                if (response.isSuccessful() && response.body() != null && (profiles = JsonParser.parseString((String)(responseBody = response.body().string())).getAsJsonArray()).size() > 0) {
                    JsonObject profile = profiles.get(0).getAsJsonObject();
                    String senderName = profile.has("minecraft_name") ? profile.get("minecraft_name").getAsString() : "Friend";
                    String decryptedContent = message.getContent();
                    ToastNotificationManager.getInstance().showMessageNotification(senderName, decryptedContent);
                }
            }
        }
        catch (Exception e) {
            System.err.println("[Tunnely Messages] Failed to show toast: " + e.getMessage());
        }
    }

    public void shutdown() {
        this.stop();
        this.scheduler.shutdown();
        try {
            if (!this.scheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.scheduler.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.scheduler.shutdownNow();
        }
        System.out.println("[Tunnely Messages] Shutdown complete");
    }

    public boolean isRunning() {
        return this.running.get();
    }
}

