/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.tunnelyrefab.worlds;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class WorldScannerUtil {
    private static final Set<String> BLOCKED_EXTENSIONS = Set.of(".exe", ".msi", ".scr", ".com", ".pif", ".bat", ".cmd", ".vbs", ".vbe", ".js", ".jse", ".wsf", ".wsh", ".ps1", ".psm1", ".hta", ".cpl", ".msc", ".gadget", ".application", ".msp", ".inf", ".reg", ".app", ".deb", ".rpm", ".bin", ".run", ".sh", ".bash", ".csh", ".ksh", ".jar", ".class", ".war", ".ear", ".py", ".pyc", ".pyo", ".rb", ".pl", ".php", ".asp", ".aspx", ".jsp", ".dll", ".so", ".dylib", ".a", ".lib", ".ocx", ".zip", ".rar", ".7z", ".tar", ".gz", ".bz2", ".xz", ".z", ".lz", ".lzma", ".cab", ".arj", ".ace", ".iso", ".dmg", ".sys", ".drv", ".ini", ".lnk", ".url", ".db", ".sql", ".mdb", ".accdb", ".xlsm", ".docm", ".pptm", ".dotm", ".xltm", ".potm", ".vhd", ".vhdx", ".vmdk", ".qcow2");
    private static final Set<String> ALLOWED_EXTENSIONS = Set.of(".mca", ".mcc", ".mcr", ".dat", ".dat_old", ".dat_mcr", ".nbt", ".snbt", ".schem", ".schematic", ".litematic", ".nbtstructure", ".txt", ".json", ".toml", ".yaml", ".yml", ".properties", ".cfg", ".png", ".jpg", ".jpeg", ".log", ".mcfunction", "");
    private static final Set<String> EXCLUDED_ITEMS = Set.of("session.lock", "advancements", "playerdata", "stats", "usercache.json", "banned-players.json", "banned-ips.json", "ops.json", "whitelist.json");
    private static final Map<byte[], String> MAGIC_BYTES = new LinkedHashMap<byte[], String>();
    private static final Set<String> SUSPICIOUS_STRINGS;
    private static final Pattern OBFUSCATED_CODE;
    private static final long MAX_UNCOMPRESSED_SIZE = 0x80000000L;
    private static final int MAX_FILE_COUNT = 100000;
    private static final int MAX_DEPTH = 50;
    private static final long MAX_INDIVIDUAL_FILE_SIZE = 524288000L;
    private static final int MAGIC_BYTES_CHECK_SIZE = 8192;

    public static ScanResult scanWorld(Path worldPath) {
        if (!Files.exists(worldPath, new LinkOption[0]) || !Files.isDirectory(worldPath, new LinkOption[0])) {
            return new ScanResult(false, Collections.singletonList("World path does not exist or is not a directory"), Collections.emptyList(), 0L, 0);
        }
        ArrayList<String> blockedFiles = new ArrayList<String>();
        ArrayList<String> warnings = new ArrayList<String>();
        long totalSize = 0L;
        int fileCount = 0;
        try {
            SecurityScanVisitor visitor = new SecurityScanVisitor(worldPath, blockedFiles, warnings);
            Files.walkFileTree(worldPath, EnumSet.noneOf(FileVisitOption.class), 50, visitor);
            totalSize = visitor.getTotalSize();
            fileCount = visitor.getFileCount();
            if (totalSize > 0x80000000L) {
                blockedFiles.add("World exceeds maximum size: " + WorldScannerUtil.formatSize(totalSize) + " > " + WorldScannerUtil.formatSize(0x80000000L));
            }
            if (fileCount > 100000) {
                blockedFiles.add("World exceeds maximum file count: " + fileCount + " > 100000");
            }
        }
        catch (IOException e) {
            return new ScanResult(false, Collections.singletonList("Failed to scan world: " + e.getMessage()), Collections.emptyList(), 0L, 0);
        }
        boolean safe = blockedFiles.isEmpty();
        return new ScanResult(safe, blockedFiles, warnings, totalSize, fileCount);
    }

    public static boolean isFileAllowed(String filename) {
        String lowerName = filename.toLowerCase();
        for (String blockedExt : BLOCKED_EXTENSIONS) {
            if (!lowerName.endsWith(blockedExt)) continue;
            return false;
        }
        String extension = WorldScannerUtil.getFileExtension(lowerName);
        return ALLOWED_EXTENSIONS.contains(extension);
    }

    private static String deepScanFile(Path filePath) {
        try {
            String suspiciousContent;
            long fileSize = Files.size(filePath);
            if (fileSize > 524288000L) {
                return "File too large: " + WorldScannerUtil.formatSize(fileSize);
            }
            if (fileSize == 0L) {
                return null;
            }
            int bytesToRead = (int)Math.min(fileSize, 8192L);
            byte[] fileHeader = new byte[bytesToRead];
            try (InputStream inputStream = Files.newInputStream(filePath, new OpenOption[0]);){
                inputStream.read(fileHeader);
            }
            catch (IOException e) {
                return "Failed to read file for analysis";
            }
            String magicByteMatch = WorldScannerUtil.checkMagicBytes(fileHeader);
            if (magicByteMatch != null) {
                return "Executable file detected: " + magicByteMatch;
            }
            String filename = filePath.getFileName().toString().toLowerCase();
            if (WorldScannerUtil.isTextFile(filename) && (suspiciousContent = WorldScannerUtil.scanTextContent(fileHeader)) != null) {
                return "Suspicious code detected: " + suspiciousContent;
            }
            if (WorldScannerUtil.isPotentialPolyglot(fileHeader, filename)) {
                return "Potential polyglot file (multiple format signatures)";
            }
            return null;
        }
        catch (IOException e) {
            return "Failed to scan file: " + e.getMessage();
        }
    }

    private static String checkMagicBytes(byte[] header) {
        for (Map.Entry<byte[], String> entry : MAGIC_BYTES.entrySet()) {
            byte[] magic = entry.getKey();
            if (header.length < magic.length) continue;
            boolean matches = true;
            for (int i = 0; i < magic.length; ++i) {
                if (header[i] == magic[i]) continue;
                matches = false;
                break;
            }
            if (!matches) continue;
            return entry.getValue();
        }
        return null;
    }

    private static String scanTextContent(byte[] content) {
        try {
            String text = new String(content, "UTF-8");
            String lowerText = text.toLowerCase();
            for (String suspicious : SUSPICIOUS_STRINGS) {
                if (!lowerText.contains(suspicious.toLowerCase())) continue;
                return "Contains suspicious code: " + suspicious;
            }
            if (OBFUSCATED_CODE.matcher(text).find()) {
                return "Contains obfuscated/encoded code";
            }
            if (WorldScannerUtil.containsExcessiveBase64(text)) {
                return "Contains suspicious base64 encoding";
            }
            return null;
        }
        catch (Exception e) {
            return null;
        }
    }

    private static boolean containsExcessiveBase64(String text) {
        Pattern base64Pattern = Pattern.compile("[A-Za-z0-9+/]{100,}={0,2}");
        int base64Count = 0;
        int totalLength = text.length();
        Matcher matcher = base64Pattern.matcher(text);
        while (matcher.find()) {
            base64Count += matcher.group().length();
        }
        return totalLength > 200 && (double)base64Count / (double)totalLength > 0.3;
    }

    private static boolean isPotentialPolyglot(byte[] header, String filename) {
        byte[] pngSignature;
        if (filename.endsWith(".png") && header.length >= (pngSignature = new byte[]{-119, 80, 78, 71, 13, 10, 26, 10}).length) {
            boolean isPng = true;
            for (int i = 0; i < pngSignature.length; ++i) {
                if (header[i] == pngSignature[i]) continue;
                isPng = false;
                break;
            }
            if (!isPng) {
                return true;
            }
        }
        return filename.endsWith(".json") && header.length >= 2 && header[0] == 77 && header[1] == 90;
    }

    private static boolean isTextFile(String filename) {
        return filename.endsWith(".txt") || filename.endsWith(".json") || filename.endsWith(".yml") || filename.endsWith(".yaml") || filename.endsWith(".toml") || filename.endsWith(".properties") || filename.endsWith(".cfg") || filename.endsWith(".mcfunction") || filename.endsWith(".snbt");
    }

    public static boolean shouldExclude(String filename) {
        return EXCLUDED_ITEMS.contains(filename);
    }

    private static String getFileExtension(String filename) {
        int lastDot = filename.lastIndexOf(46);
        if (lastDot == -1 || lastDot == filename.length() - 1) {
            return "";
        }
        return filename.substring(lastDot);
    }

    public static boolean isSafePath(Path path) {
        String pathString = path.toString();
        return !pathString.contains("..") && !pathString.startsWith("/") && !pathString.startsWith("\\");
    }

    public static long calculateWorldSize(Path worldPath) {
        try {
            return Files.walk(worldPath, new FileVisitOption[0]).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).mapToLong(path -> {
                try {
                    return Files.size(path);
                }
                catch (IOException e) {
                    return 0L;
                }
            }).sum();
        }
        catch (IOException e) {
            return 0L;
        }
    }

    public static String formatSize(long bytes) {
        if (bytes < 1024L) {
            return bytes + " B";
        }
        int exp = (int)(Math.log(bytes) / Math.log(1024.0));
        String pre = "" + "KMGTPE".charAt(exp - 1);
        return String.format("%.1f %sB", (double)bytes / Math.pow(1024.0, exp), pre);
    }

    static {
        MAGIC_BYTES.put(new byte[]{77, 90}, "Windows executable (MZ/PE)");
        MAGIC_BYTES.put(new byte[]{127, 69, 76, 70}, "Linux executable (ELF)");
        MAGIC_BYTES.put(new byte[]{-2, -19, -6, -50}, "macOS executable (Mach-O 32-bit)");
        MAGIC_BYTES.put(new byte[]{-2, -19, -6, -49}, "macOS executable (Mach-O 64-bit)");
        MAGIC_BYTES.put(new byte[]{-50, -6, -19, -2}, "macOS executable (Mach-O reverse)");
        MAGIC_BYTES.put(new byte[]{-49, -6, -19, -2}, "macOS executable (Mach-O 64-bit reverse)");
        MAGIC_BYTES.put(new byte[]{-54, -2, -70, -66}, "Java class file");
        MAGIC_BYTES.put(new byte[]{80, 75, 3, 4}, "ZIP archive");
        MAGIC_BYTES.put(new byte[]{80, 75, 5, 6}, "ZIP archive (empty)");
        MAGIC_BYTES.put(new byte[]{80, 75, 7, 8}, "ZIP archive (spanned)");
        MAGIC_BYTES.put(new byte[]{82, 97, 114, 33, 26, 7}, "RAR archive");
        MAGIC_BYTES.put(new byte[]{55, 122, -68, -81, 39, 28}, "7-Zip archive");
        MAGIC_BYTES.put(new byte[]{76, 0, 0, 0, 1, 20, 2, 0}, "Windows shortcut (LNK)");
        SUSPICIOUS_STRINGS = Set.of("eval(", "exec(", "system(", "shell_exec", "passthru", "Runtime.getRuntime", "ProcessBuilder", "cmd.exe", "/bin/sh", "powershell", "invoke-expression", "downloadstring", "webclient", "base64_decode", "gzinflate", "str_rot13", "assert(", "CreateObject", "WScript.Shell", "ActiveXObject");
        OBFUSCATED_CODE = Pattern.compile("(?i)(eval|exec|system)\\s*\\(\\s*['\"]\\s*[a-zA-Z0-9+/=]{50,}", 2);
    }

    public static class ScanResult {
        private final boolean safe;
        private final List<String> blockedFiles;
        private final List<String> warnings;
        private final long totalSize;
        private final int fileCount;

        public ScanResult(boolean safe, List<String> blockedFiles, List<String> warnings, long totalSize, int fileCount) {
            this.safe = safe;
            this.blockedFiles = new ArrayList<String>(blockedFiles);
            this.warnings = new ArrayList<String>(warnings);
            this.totalSize = totalSize;
            this.fileCount = fileCount;
        }

        public boolean isSafe() {
            return this.safe;
        }

        public List<String> getBlockedFiles() {
            return Collections.unmodifiableList(this.blockedFiles);
        }

        public List<String> getWarnings() {
            return Collections.unmodifiableList(this.warnings);
        }

        public long getTotalSize() {
            return this.totalSize;
        }

        public int getFileCount() {
            return this.fileCount;
        }

        public String getFormattedSize() {
            return WorldScannerUtil.formatSize(this.totalSize);
        }
    }

    private static class SecurityScanVisitor
    extends SimpleFileVisitor<Path> {
        private final Path rootPath;
        private final List<String> blockedFiles;
        private final List<String> warnings;
        private long totalSize = 0L;
        private int fileCount = 0;

        public SecurityScanVisitor(Path rootPath, List<String> blockedFiles, List<String> warnings) {
            this.rootPath = rootPath;
            this.blockedFiles = blockedFiles;
            this.warnings = warnings;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
            String filename = file.getFileName().toString();
            String relativePath = this.rootPath.relativize(file).toString();
            if (WorldScannerUtil.shouldExclude(filename)) {
                return FileVisitResult.CONTINUE;
            }
            ++this.fileCount;
            this.totalSize += attrs.size();
            if (!WorldScannerUtil.isFileAllowed(filename)) {
                this.blockedFiles.add("Blocked file: " + relativePath + " (dangerous file type)");
                return FileVisitResult.CONTINUE;
            }
            String deepScanResult = WorldScannerUtil.deepScanFile(file);
            if (deepScanResult != null) {
                this.blockedFiles.add("Blocked file: " + relativePath + " (" + deepScanResult + ")");
                return FileVisitResult.CONTINUE;
            }
            if (filename.contains("..") || filename.contains("\\") || filename.contains("/")) {
                this.blockedFiles.add("Suspicious filename: " + relativePath + " (path traversal attempt)");
            }
            String lowerFilename = filename.toLowerCase();
            for (String blockedExt : BLOCKED_EXTENSIONS) {
                if (!lowerFilename.contains(blockedExt + ".")) continue;
                this.blockedFiles.add("Suspicious filename: " + relativePath + " (double extension)");
                break;
            }
            if (attrs.size() > 0x6400000L) {
                this.warnings.add("Large file detected: " + relativePath + " (" + WorldScannerUtil.formatSize(attrs.size()) + ")");
            }
            if (filename.startsWith(".") && !filename.equals(".") && !filename.equals("..")) {
                this.warnings.add("Hidden file detected: " + relativePath);
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException exc) {
            this.warnings.add("Failed to scan file: " + String.valueOf(this.rootPath.relativize(file)) + " (" + exc.getMessage() + ")");
            return FileVisitResult.CONTINUE;
        }

        public long getTotalSize() {
            return this.totalSize;
        }

        public int getFileCount() {
            return this.fileCount;
        }
    }
}

