/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.tunnelyrefab.worlds;

import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.concurrent.TimeUnit;
import org.texboobcat.tunnelyrefab.shaded.okhttp3.OkHttpClient;
import org.texboobcat.tunnelyrefab.shaded.okhttp3.Request;
import org.texboobcat.tunnelyrefab.shaded.okhttp3.Response;
import org.texboobcat.tunnelyrefab.shaded.okhttp3.ResponseBody;
import org.texboobcat.tunnelyrefab.worlds.BandwidthThrottle;
import org.texboobcat.tunnelyrefab.worlds.CompressionProgressCallback;
import org.texboobcat.tunnelyrefab.worlds.DownloadState;

public class ResumableDownloadManager {
    private final OkHttpClient httpClient = new OkHttpClient.Builder().connectTimeout(60L, TimeUnit.SECONDS).readTimeout(300L, TimeUnit.SECONDS).writeTimeout(300L, TimeUnit.SECONDS).build();
    private final BandwidthThrottle throttle;
    private DownloadState currentDownload;
    private volatile boolean isPaused = false;
    private volatile boolean isCancelled = false;

    public ResumableDownloadManager(long maxBytesPerSecond) {
        this.throttle = new BandwidthThrottle(maxBytesPerSecond);
    }

    public DownloadResult startDownload(String worldId, String worldName, String fileUrl, Path outputPath, CompressionProgressCallback callback) {
        try {
            Path tempFile = Files.createTempFile("world_download_", ".tar.gz.part", new FileAttribute[0]);
            Path stateFile = Paths.get(tempFile.toString() + ".state", new String[0]);
            this.currentDownload = new DownloadState(worldId, worldName, fileUrl, tempFile.toString(), stateFile.toString(), outputPath.getFileName().toString());
            return this.performDownload(this.currentDownload, callback);
        }
        catch (Exception e) {
            System.err.println("[ResumableDownloadManager] Failed to start download: " + e.getMessage());
            return new DownloadResult(false, "Download failed: " + e.getMessage());
        }
    }

    public DownloadResult resumeDownload(Path stateFile, CompressionProgressCallback callback) {
        try {
            this.currentDownload = DownloadState.load(stateFile);
            this.currentDownload.setPaused(false);
            System.out.println("[ResumableDownloadManager] Resuming download from " + this.currentDownload.getDownloadedBytesFormatted() + " / " + this.currentDownload.getTotalBytesFormatted());
            return this.performDownload(this.currentDownload, callback);
        }
        catch (Exception e) {
            System.err.println("[ResumableDownloadManager] Failed to resume download: " + e.getMessage());
            return new DownloadResult(false, "Resume failed: " + e.getMessage());
        }
    }

    public void pauseDownload() {
        this.isPaused = true;
        if (this.currentDownload != null) {
            this.currentDownload.setPaused(true);
            try {
                this.currentDownload.save();
                System.out.println("[ResumableDownloadManager] Download paused at " + this.currentDownload.getDownloadedBytesFormatted());
            }
            catch (IOException e) {
                System.err.println("[ResumableDownloadManager] Failed to save pause state: " + e.getMessage());
            }
        }
    }

    public void cancelDownload() {
        this.isCancelled = true;
        if (this.currentDownload != null) {
            try {
                Files.deleteIfExists(Paths.get(this.currentDownload.getTempFilePath(), new String[0]));
                this.currentDownload.cleanup();
                System.out.println("[ResumableDownloadManager] Download cancelled");
            }
            catch (Exception e) {
                System.err.println("[ResumableDownloadManager] Cleanup error: " + e.getMessage());
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private DownloadResult performDownload(DownloadState state, CompressionProgressCallback callback) {
        this.isPaused = false;
        this.isCancelled = false;
        try {
            Path tempFile = Paths.get(state.getTempFilePath(), new String[0]);
            long startPosition = state.getDownloadedBytes();
            Request.Builder requestBuilder = new Request.Builder().url(state.getFileUrl()).get();
            if (startPosition > 0L) {
                requestBuilder.addHeader("Range", "bytes=" + startPosition + "-");
                System.out.println("[ResumableDownloadManager] Requesting resume from byte " + startPosition);
            }
            Request request = requestBuilder.build();
            try (Response response = this.httpClient.newCall(request).execute();){
                if (!response.isSuccessful() && response.code() != 206) {
                    DownloadResult downloadResult = new DownloadResult(false, "Download failed: HTTP " + response.code());
                    return downloadResult;
                }
                ResponseBody body = response.body();
                if (body == null) {
                    DownloadResult downloadResult = new DownloadResult(false, "No response body");
                    return downloadResult;
                }
                long contentLength = body.contentLength();
                if (state.getTotalBytes() == 0L) {
                    state.setTotalBytes(startPosition + contentLength);
                }
                System.out.println("[ResumableDownloadManager] Downloading: " + state.getTotalBytesFormatted() + (String)(this.throttle.isEnabled() ? " at " + this.throttle.getSpeedLimitFormatted() : ""));
                try (InputStream input = body.byteStream();
                     RandomAccessFile output = new RandomAccessFile(tempFile.toFile(), "rw");){
                    int bytesRead;
                    output.seek(startPosition);
                    byte[] buffer = new byte[8192];
                    long lastSaveTime = System.currentTimeMillis();
                    while ((bytesRead = input.read(buffer)) != -1) {
                        long currentTime;
                        if (this.isPaused) {
                            System.out.println("[ResumableDownloadManager] Download paused by user");
                            DownloadResult downloadResult = new DownloadResult(false, "Download paused", state.getTempFilePath());
                            return downloadResult;
                        }
                        if (this.isCancelled) {
                            System.out.println("[ResumableDownloadManager] Download cancelled by user");
                            DownloadResult downloadResult = new DownloadResult(false, "Download cancelled");
                            return downloadResult;
                        }
                        output.write(buffer, 0, bytesRead);
                        state.setDownloadedBytes(state.getDownloadedBytes() + (long)bytesRead);
                        this.throttle.throttle(bytesRead);
                        if (callback != null && state.getTotalBytes() > 0L) {
                            int percentage = state.getProgressPercentage();
                            String message = String.format("Downloading... %s / %s (%d%%) @ %s", state.getDownloadedBytesFormatted(), state.getTotalBytesFormatted(), percentage, this.formatSpeed(state.getDownloadSpeed()));
                            callback.onProgress(percentage, message);
                        }
                        if ((currentTime = System.currentTimeMillis()) - lastSaveTime <= 5000L) continue;
                        state.save();
                        lastSaveTime = currentTime;
                    }
                }
                state.setCompleted(true);
                state.save();
                System.out.println("[ResumableDownloadManager] Download complete: " + state.getTotalBytesFormatted());
                DownloadResult downloadResult = new DownloadResult(true, "Download complete", state.getTempFilePath());
                return downloadResult;
            }
        }
        catch (Exception e) {
            System.err.println("[ResumableDownloadManager] Download error: " + e.getMessage());
            e.printStackTrace();
            if (this.currentDownload == null) return new DownloadResult(false, "Download error: " + e.getMessage());
            if (this.isCancelled) return new DownloadResult(false, "Download error: " + e.getMessage());
            try {
                this.currentDownload.save();
                return new DownloadResult(false, "Download error: " + e.getMessage());
            }
            catch (IOException saveError) {
                System.err.println("[ResumableDownloadManager] Failed to save state: " + saveError.getMessage());
            }
            return new DownloadResult(false, "Download error: " + e.getMessage());
        }
    }

    public DownloadState getCurrentDownload() {
        return this.currentDownload;
    }

    public boolean isPaused() {
        return this.isPaused;
    }

    public boolean isCancelled() {
        return this.isCancelled;
    }

    public BandwidthThrottle getThrottle() {
        return this.throttle;
    }

    private String formatSpeed(long bytesPerSecond) {
        if (bytesPerSecond < 1024L) {
            return bytesPerSecond + " B/s";
        }
        int exp = (int)(Math.log(bytesPerSecond) / Math.log(1024.0));
        String pre = "" + "KMGTPE".charAt(exp - 1);
        return String.format("%.1f %sB/s", (double)bytesPerSecond / Math.pow(1024.0, exp), pre);
    }

    public static class DownloadResult {
        private final boolean success;
        private final String message;
        private final String tempFilePath;

        public DownloadResult(boolean success, String message) {
            this(success, message, null);
        }

        public DownloadResult(boolean success, String message, String tempFilePath) {
            this.success = success;
            this.message = message;
            this.tempFilePath = tempFilePath;
        }

        public boolean isSuccess() {
            return this.success;
        }

        public String getMessage() {
            return this.message;
        }

        public String getTempFilePath() {
            return this.tempFilePath;
        }
    }
}

