/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.tunnelyrefab.tunnel;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URI;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import org.texboobcat.tunnelyrefab.auth.SupabaseClient;
import org.texboobcat.tunnelyrefab.config.PortForwardingConfig;
import org.texboobcat.tunnelyrefab.config.TunnelConfig;
import org.texboobcat.tunnelyrefab.shaded.org.java_websocket.client.WebSocketClient;
import org.texboobcat.tunnelyrefab.shaded.org.java_websocket.drafts.Draft_6455;
import org.texboobcat.tunnelyrefab.shaded.org.java_websocket.handshake.ServerHandshake;
import org.texboobcat.tunnelyrefab.tunnel.HostProxyServer;

public class TunnelClient
extends WebSocketClient {
    private final TunnelConfig config;
    private final Consumer<byte[]> packetHandler;
    private final String authToken;
    private final String clientType;
    private volatile boolean connected = false;
    private volatile boolean handshakeComplete = false;
    private volatile String publicEndpoint = null;
    private final CountDownLatch connectionLatch = new CountDownLatch(1);
    private final CountDownLatch handshakeLatch = new CountDownLatch(1);
    private final CountDownLatch portForwardLatch = new CountDownLatch(1);
    private Runnable disconnectListener = null;
    private HostProxyServer hostProxyServer = null;
    private String databaseSessionId = null;
    private volatile boolean portForwardingComplete = false;
    private volatile String portForwardingError = null;
    private final Map<Integer, Consumer<byte[]>> udpPacketHandlers = new ConcurrentHashMap<Integer, Consumer<byte[]>>();

    public TunnelClient(URI serverUri, String authToken, Consumer<byte[]> packetHandler, String databaseSessionId) {
        super(serverUri, new Draft_6455());
        this.config = TunnelConfig.getInstance();
        this.packetHandler = packetHandler;
        this.authToken = authToken;
        this.clientType = "minecraft-host";
        this.databaseSessionId = databaseSessionId;
        if (serverUri.getScheme().equals("wss")) {
            try {
                SSLContext sslContext = SSLContext.getInstance("TLS");
                sslContext.init(null, TunnelClient.getTrustAllCerts(), new SecureRandom());
                SSLSocketFactory factory = sslContext.getSocketFactory();
                this.setSocketFactory(factory);
                System.out.println("[Tunnely] SSL configured for secure WebSocket connection");
            }
            catch (Exception e) {
                System.err.println("[Tunnely] Failed to configure SSL: " + e.getMessage());
                e.printStackTrace();
            }
        }
    }

    private static TrustManager[] getTrustAllCerts() {
        return new TrustManager[]{new X509TrustManager(){

            @Override
            public X509Certificate[] getAcceptedIssuers() {
                return null;
            }

            @Override
            public void checkClientTrusted(X509Certificate[] certs, String authType) {
            }

            @Override
            public void checkServerTrusted(X509Certificate[] certs, String authType) {
            }
        }};
    }

    @Override
    public void onOpen(ServerHandshake handshake) {
        this.connected = true;
        this.connectionLatch.countDown();
    }

    @Override
    public void onMessage(String message) {
        try {
            System.out.println("[Tunnely DEBUG] Received message from relay: " + message.substring(0, Math.min(200, message.length())));
            JsonObject json = JsonParser.parseString((String)message).getAsJsonObject();
            String type = json.has("type") ? json.get("type").getAsString() : null;
            System.out.println("[Tunnely DEBUG] Message type: " + type);
            if (type == null) {
                this.handleLegacyMessage(message);
                return;
            }
            switch (type) {
                case "HANDSHAKE_CHALLENGE": {
                    this.handleHandshakeChallenge(json);
                    break;
                }
                case "HANDSHAKE_SUCCESS": {
                    this.handshakeComplete = true;
                    this.handshakeLatch.countDown();
                    System.out.println("[Tunnely] Handshake successful, connection secured");
                    break;
                }
                case "ERROR": {
                    String errorMsg = json.has("message") ? json.get("message").getAsString() : "Unknown error";
                    System.err.println("[Tunnely] Server error: " + errorMsg);
                    break;
                }
                case "ENDPOINT": {
                    if (json.has("endpoint")) {
                        this.publicEndpoint = json.get("endpoint").getAsString();
                        System.out.println("[Tunnely] Public endpoint assigned: " + this.publicEndpoint);
                    }
                    break;
                }
                case "CLIENT_DISCONNECTED": {
                    this.handleClientDisconnected(json);
                    break;
                }
                case "PORT_FORWARD_SUCCESS": {
                    System.out.println("[Tunnely DEBUG] ===== RECEIVED PORT_FORWARD_SUCCESS =====");
                    this.handlePortForwardSuccess(json);
                    break;
                }
                case "PORT_FORWARD_ERROR": {
                    System.err.println("[Tunnely DEBUG] ===== RECEIVED PORT_FORWARD_ERROR =====");
                    this.handlePortForwardError(json);
                    break;
                }
                case "UDP_PACKET": {
                    this.handleUdpPacket(json);
                    break;
                }
                default: {
                    if (this.config.isDebugMode()) {
                        System.out.println("[Tunnely] Received message: " + type);
                    }
                    break;
                }
            }
        }
        catch (Exception e) {
            this.handleLegacyMessage(message);
        }
    }

    public void setHostProxyServer(HostProxyServer proxyServer) {
        this.hostProxyServer = proxyServer;
    }

    public void requestPortForwarding(List<PortForwardingConfig> portConfigs) {
        if (!this.connected || !this.handshakeComplete) {
            System.err.println("[Tunnely] Cannot request port forwarding: not connected or handshake incomplete");
            return;
        }
        if (portConfigs == null || portConfigs.isEmpty()) {
            this.portForwardingComplete = true;
            this.portForwardLatch.countDown();
            return;
        }
        try {
            JsonObject request = new JsonObject();
            request.addProperty("type", "PORT_FORWARD_REQUEST");
            request.addProperty("sessionId", this.databaseSessionId);
            JsonArray portsArray = new JsonArray();
            for (PortForwardingConfig config : portConfigs) {
                if (!config.isEnabled()) continue;
                JsonObject portObj = new JsonObject();
                portObj.addProperty("port", (Number)config.getPort());
                portObj.addProperty("protocol", config.getProtocol().toString());
                portObj.addProperty("service", config.getServiceName());
                portsArray.add((JsonElement)portObj);
            }
            request.add("ports", (JsonElement)portsArray);
            System.out.println("[Tunnely] Requesting forwarding for " + portsArray.size() + " additional port(s)");
            System.out.println("[Tunnely DEBUG] ===== PORT FORWARD REQUEST =====");
            System.out.println("[Tunnely DEBUG] Database Session ID: " + this.databaseSessionId);
            System.out.println("[Tunnely DEBUG] WebSocket connected: " + this.isOpen());
            System.out.println("[Tunnely DEBUG] Handshake complete: " + this.handshakeComplete);
            System.out.println("[Tunnely DEBUG] Request JSON: " + request.toString());
            this.send(request.toString());
            System.out.println("[Tunnely DEBUG] Request SENT to relay server");
        }
        catch (Exception e) {
            System.err.println("[Tunnely] Failed to request port forwarding: " + e.getMessage());
            this.portForwardingError = e.getMessage();
            this.portForwardLatch.countDown();
        }
    }

    private void handlePortForwardSuccess(JsonObject json) {
        System.out.println("[Tunnely DEBUG] handlePortForwardSuccess called");
        System.out.println("[Tunnely DEBUG] Success response: " + json.toString());
        this.portForwardingComplete = true;
        this.portForwardLatch.countDown();
        if (json.has("ports")) {
            JsonArray ports = json.getAsJsonArray("ports");
            System.out.println("[Tunnely] Port forwarding established for " + ports.size() + " port(s)");
            for (int i = 0; i < ports.size(); ++i) {
                JsonObject portInfo = ports.get(i).getAsJsonObject();
                int port = portInfo.get("port").getAsInt();
                String protocol = portInfo.get("protocol").getAsString();
                String service = portInfo.has("service") ? portInfo.get("service").getAsString() : "unknown";
                System.out.println("  \u2713 " + service + " (" + port + "/" + protocol + ")");
            }
        }
    }

    private void handlePortForwardError(JsonObject json) {
        System.err.println("[Tunnely DEBUG] handlePortForwardError called");
        System.err.println("[Tunnely DEBUG] Error response: " + json.toString());
        String errorMsg = json.has("message") ? json.get("message").getAsString() : "Unknown error";
        System.err.println("[Tunnely] Port forwarding error: " + errorMsg);
        this.portForwardingError = errorMsg;
        this.portForwardLatch.countDown();
    }

    private void handleUdpPacket(JsonObject json) {
        try {
            if (!json.has("port") || !json.has("data")) {
                return;
            }
            int port = json.get("port").getAsInt();
            String base64Data = json.get("data").getAsString();
            byte[] data = Base64.getDecoder().decode(base64Data);
            Consumer<byte[]> handler = this.udpPacketHandlers.get(port);
            if (handler != null) {
                handler.accept(data);
            }
        }
        catch (Exception e) {
            System.err.println("[Tunnely] Error handling UDP packet: " + e.getMessage());
        }
    }

    public void registerUdpHandler(int port, Consumer<byte[]> handler) {
        this.udpPacketHandlers.put(port, handler);
    }

    public void unregisterUdpHandler(int port) {
        this.udpPacketHandlers.remove(port);
    }

    private void handleClientDisconnected(JsonObject json) {
        String clientId = json.has("clientId") ? json.get("clientId").getAsString() : "unknown";
        int remainingClients = json.has("remainingClients") ? json.get("remainingClients").getAsInt() : 0;
        System.out.println(String.format("[Tunnely Host] Client %s disconnected (%d remaining)", clientId, remainingClients));
        if (this.hostProxyServer != null) {
            System.out.println("[Tunnely Host] Triggering Minecraft connection refresh to remove disconnected player");
            this.hostProxyServer.forceReconnect();
        }
    }

    private void handleLegacyMessage(String message) {
        if (message.startsWith("ENDPOINT:")) {
            this.publicEndpoint = message.substring(9).trim();
            System.out.println("[Tunnely] Public endpoint assigned: " + this.publicEndpoint);
        } else if (message.equals("PING")) {
            this.send("PONG");
        } else if (this.config.isDebugMode()) {
            System.out.println("[Tunnely] Received control message: " + message);
        }
    }

    private void handleHandshakeChallenge(JsonObject json) {
        try {
            SupabaseClient supabaseClient;
            String ticket;
            if (!json.has("challenge")) {
                System.err.println("[Tunnely] Invalid handshake challenge: missing challenge field");
                this.close();
                return;
            }
            String challenge = json.get("challenge").getAsString();
            if (this.config.isDebugMode()) {
                System.out.println("[Tunnely] Received handshake challenge, generating connection ticket...");
            }
            if ((ticket = (supabaseClient = new SupabaseClient()).generateRelayConnectionTicket(challenge)) == null) {
                System.err.println("[Tunnely] Failed to generate connection ticket");
                this.close();
                return;
            }
            String signature = this.computeHmacSignature(challenge, ticket);
            JsonObject response = new JsonObject();
            response.addProperty("type", "HANDSHAKE_RESPONSE");
            response.addProperty("signature", signature);
            response.addProperty("clientType", this.clientType);
            response.addProperty("ticket", ticket);
            this.send(response.toString());
            if (this.config.isDebugMode()) {
                System.out.println("[Tunnely] Handshake response sent with ticket (JWT not exposed to relay)");
            }
        }
        catch (Exception e) {
            System.err.println("[Tunnely] Failed to process handshake challenge: " + e.getMessage());
            if (this.config.isDebugMode()) {
                e.printStackTrace();
            }
            this.close();
        }
    }

    private String computeHmacSignature(String challenge, String secret) throws NoSuchAlgorithmException, InvalidKeyException {
        Mac hmac = Mac.getInstance("HmacSHA256");
        SecretKeySpec secretKey = new SecretKeySpec(secret.getBytes(StandardCharsets.UTF_8), "HmacSHA256");
        hmac.init(secretKey);
        byte[] signatureBytes = hmac.doFinal(challenge.getBytes(StandardCharsets.UTF_8));
        StringBuilder hexString = new StringBuilder();
        for (byte b : signatureBytes) {
            String hex = Integer.toHexString(0xFF & b);
            if (hex.length() == 1) {
                hexString.append('0');
            }
            hexString.append(hex);
        }
        return hexString.toString();
    }

    @Override
    public void onMessage(ByteBuffer bytes) {
        try {
            byte[] incoming = new byte[bytes.remaining()];
            bytes.get(incoming);
            if (this.hostProxyServer != null) {
                int headerLen;
                String clientId = null;
                byte[] compressedPayload = incoming;
                if (incoming.length > 2 && (headerLen = (incoming[0] & 0xFF) << 8 | incoming[1] & 0xFF) > 0 && headerLen < incoming.length - 2 && headerLen < 4096) {
                    try {
                        String headerJson = new String(incoming, 2, headerLen, StandardCharsets.UTF_8);
                        JsonObject header = JsonParser.parseString((String)headerJson).getAsJsonObject();
                        if (header.has("type") && "CLIENT_PACKET".equals(header.get("type").getAsString())) {
                            int payloadOffset;
                            int payloadLen;
                            if (header.has("clientId")) {
                                clientId = header.get("clientId").getAsString();
                            }
                            if ((payloadLen = incoming.length - (payloadOffset = 2 + headerLen)) > 0) {
                                compressedPayload = new byte[payloadLen];
                                System.arraycopy(incoming, payloadOffset, compressedPayload, 0, payloadLen);
                            }
                        }
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                }
                byte[] data = this.decompressData(compressedPayload);
                if (clientId != null) {
                    this.hostProxyServer.handleRelayPacket(clientId, data);
                } else {
                    this.hostProxyServer.handleRelayPacket(data);
                }
                return;
            }
            if (this.packetHandler != null) {
                byte[] compressed = incoming;
                byte[] data = this.decompressData(compressed);
                this.packetHandler.accept(data);
            }
        }
        catch (Exception e) {
            System.err.println("[Tunnely Host] Error forwarding packet: " + e.getMessage());
        }
    }

    @Override
    public void onClose(int code, String reason, boolean remote) {
        this.connected = false;
        if (this.disconnectListener != null) {
            this.disconnectListener.run();
        }
    }

    @Override
    public void onError(Exception ex) {
        System.err.println("[Tunnely] WebSocket error: " + ex.getMessage());
        if (this.config.isDebugMode()) {
            ex.printStackTrace();
        }
    }

    public void sendPacket(byte[] data) {
        if (!(this.connected && this.handshakeComplete && this.isOpen())) {
            return;
        }
        try {
            byte[] compressed = this.compressData(data);
            this.send(compressed);
        }
        catch (IOException e) {
            System.err.println("[Tunnely Host] Error compressing packet: " + e.getMessage());
        }
    }

    public void sendPacketToClient(byte[] data, String clientId) {
        if (!(this.connected && this.handshakeComplete && this.isOpen())) {
            return;
        }
        if (clientId == null || clientId.isEmpty()) {
            this.sendPacket(data);
            return;
        }
        try {
            byte[] compressed = this.compressData(data);
            JsonObject header = new JsonObject();
            header.addProperty("type", "CLIENT_PACKET");
            header.addProperty("clientId", clientId);
            byte[] headerBytes = header.toString().getBytes(StandardCharsets.UTF_8);
            byte[] headerLen = new byte[]{(byte)(headerBytes.length >> 8 & 0xFF), (byte)(headerBytes.length & 0xFF)};
            byte[] frame = new byte[2 + headerBytes.length + compressed.length];
            System.arraycopy(headerLen, 0, frame, 0, 2);
            System.arraycopy(headerBytes, 0, frame, 2, headerBytes.length);
            System.arraycopy(compressed, 0, frame, 2 + headerBytes.length, compressed.length);
            this.send(frame);
        }
        catch (IOException e) {
            System.err.println("[Tunnely Host] Error preparing framed packet: " + e.getMessage());
        }
    }

    public void sendUdpPacket(byte[] data, int port) {
        if (!(this.connected && this.handshakeComplete && this.isOpen())) {
            return;
        }
        try {
            JsonObject packet = new JsonObject();
            packet.addProperty("type", "UDP_PACKET");
            packet.addProperty("port", (Number)port);
            packet.addProperty("data", Base64.getEncoder().encodeToString(data));
            this.send(packet.toString());
        }
        catch (Exception e) {
            System.err.println("[Tunnely Host] Error sending UDP packet: " + e.getMessage());
        }
    }

    public boolean awaitConnection(long timeout, TimeUnit unit) throws InterruptedException {
        return this.connectionLatch.await(timeout, unit);
    }

    public boolean awaitHandshake(long timeout, TimeUnit unit) throws InterruptedException {
        return this.handshakeLatch.await(timeout, unit);
    }

    public boolean awaitPortForwarding(long timeout, TimeUnit unit) throws InterruptedException {
        return this.portForwardLatch.await(timeout, unit);
    }

    public boolean isPortForwardingComplete() {
        return this.portForwardingComplete;
    }

    public String getPortForwardingError() {
        return this.portForwardingError;
    }

    public String getPublicEndpoint() {
        return this.publicEndpoint;
    }

    public boolean isConnected() {
        return this.connected && this.handshakeComplete && this.isOpen();
    }

    public boolean isHandshakeComplete() {
        return this.handshakeComplete;
    }

    public void setDisconnectListener(Runnable listener) {
        this.disconnectListener = listener;
    }

    private byte[] compressData(byte[] data) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try (GZIPOutputStream gzipOut = new GZIPOutputStream(baos);){
            gzipOut.write(data);
        }
        return baos.toByteArray();
    }

    private byte[] decompressData(byte[] compressed) throws IOException {
        ByteArrayInputStream bais = new ByteArrayInputStream(compressed);
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try (GZIPInputStream gzipIn = new GZIPInputStream(bais);){
            int len;
            byte[] buffer = new byte[1024];
            while ((len = gzipIn.read(buffer)) > 0) {
                baos.write(buffer, 0, len);
            }
        }
        return baos.toByteArray();
    }
}

