/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.tunnelyrefab.worlds;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import org.texboobcat.tunnelyrefab.worlds.CompressionProgressCallback;
import org.texboobcat.tunnelyrefab.worlds.WorldScannerUtil;

public class WorldCompressor {
    private static final int BUFFER_SIZE = 65536;
    private static final int COMPRESSION_LEVEL = 9;

    public static CompressResult compress(Path worldPath, Path outputZip, ProgressCallback callback) {
        if (!Files.exists(worldPath, new LinkOption[0]) || !Files.isDirectory(worldPath, new LinkOption[0])) {
            return new CompressResult(false, "World path does not exist or is not a directory", 0L, 0L);
        }
        long startTime = System.currentTimeMillis();
        long originalSize = 0L;
        long compressedSize = 0L;
        AtomicInteger filesCompressed = new AtomicInteger(0);
        try {
            long totalFiles = Files.walk(worldPath, new FileVisitOption[0]).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(p -> !WorldScannerUtil.shouldExclude(p.getFileName().toString())).count();
            Files.createDirectories(outputZip.getParent(), new FileAttribute[0]);
            try (FileOutputStream fos = new FileOutputStream(outputZip.toFile());
                 BufferedOutputStream bos = new BufferedOutputStream(fos);
                 ZipOutputStream zos = new ZipOutputStream(bos);){
                zos.setLevel(9);
                zos.setMethod(8);
                Files.walk(worldPath, new FileVisitOption[0]).forEach(sourcePath -> {
                    if (Files.isDirectory(sourcePath, new LinkOption[0])) {
                        return;
                    }
                    String filename = sourcePath.getFileName().toString();
                    if (WorldScannerUtil.shouldExclude(filename)) {
                        return;
                    }
                    try {
                        Path relativePath = worldPath.relativize((Path)sourcePath);
                        String zipEntryName = relativePath.toString().replace('\\', '/');
                        ZipEntry zipEntry = new ZipEntry(zipEntryName);
                        zipEntry.setTime(Files.getLastModifiedTime(sourcePath, new LinkOption[0]).toMillis());
                        zos.putNextEntry(zipEntry);
                        long fileSize = Files.size(sourcePath);
                        Files.copy(sourcePath, zos);
                        zos.closeEntry();
                        int currentCount = filesCompressed.incrementAndGet();
                        if (callback != null && totalFiles > 0L) {
                            int percentage = (int)((long)(currentCount * 100) / totalFiles);
                            callback.onProgress(percentage, filename);
                        }
                    }
                    catch (IOException e) {
                        System.err.println("[WorldCompressor] Failed to compress file " + filename + ": " + e.getMessage());
                    }
                });
            }
            originalSize = WorldScannerUtil.calculateWorldSize(worldPath);
            compressedSize = Files.size(outputZip);
            long duration = System.currentTimeMillis() - startTime;
            double compressionRatio = originalSize > 0L ? 100.0 * (double)compressedSize / (double)originalSize : 0.0;
            String message = String.format("Compressed %d files from %s to %s (%.1f%%) in %.1f seconds", filesCompressed.get(), WorldScannerUtil.formatSize(originalSize), WorldScannerUtil.formatSize(compressedSize), compressionRatio, (double)duration / 1000.0);
            if (callback != null) {
                callback.onComplete(true, message);
            }
            System.out.println("[WorldCompressor] " + message);
            return new CompressResult(true, message, originalSize, compressedSize);
        }
        catch (IOException e) {
            String errorMsg = "Failed to compress world: " + e.getMessage();
            System.err.println("[WorldCompressor] " + errorMsg);
            e.printStackTrace();
            if (callback != null) {
                callback.onComplete(false, errorMsg);
            }
            return new CompressResult(false, errorMsg, originalSize, compressedSize);
        }
    }

    public static CompressResult compress(Path worldPath, Path outputZip) {
        return WorldCompressor.compress(worldPath, outputZip, null);
    }

    public static DecompressResult decompress(Path zipFile, Path outputDir, ProgressCallback callback) {
        if (!Files.exists(zipFile, new LinkOption[0]) || !Files.isRegularFile(zipFile, new LinkOption[0])) {
            return new DecompressResult(false, "ZIP file does not exist or is not a file", 0);
        }
        long startTime = System.currentTimeMillis();
        int filesExtracted = 0;
        try {
            int totalEntries = 0;
            try (ZipFile zf = new ZipFile(zipFile.toFile());){
                totalEntries = zf.size();
            }
            Files.createDirectories(outputDir, new FileAttribute[0]);
            try (FileInputStream fis = new FileInputStream(zipFile.toFile());
                 BufferedInputStream bis = new BufferedInputStream(fis);
                 ZipInputStream zis = new ZipInputStream(bis);){
                ZipEntry entry;
                byte[] buffer = new byte[65536];
                while ((entry = zis.getNextEntry()) != null) {
                    String entryName = entry.getName();
                    if (entryName.contains("..") || entryName.startsWith("/") || entryName.startsWith("\\")) {
                        System.err.println("[WorldCompressor] Blocked suspicious path: " + entryName);
                        continue;
                    }
                    Path outputPath = outputDir.resolve(entryName);
                    if (entry.isDirectory()) {
                        Files.createDirectories(outputPath, new FileAttribute[0]);
                    } else {
                        Files.createDirectories(outputPath.getParent(), new FileAttribute[0]);
                        try (FileOutputStream fos = new FileOutputStream(outputPath.toFile());
                             BufferedOutputStream bos = new BufferedOutputStream(fos);){
                            int bytesRead;
                            while ((bytesRead = zis.read(buffer)) != -1) {
                                bos.write(buffer, 0, bytesRead);
                            }
                        }
                        if (entry.getTime() > 0L) {
                            Files.setLastModifiedTime(outputPath, FileTime.fromMillis(entry.getTime()));
                        }
                        ++filesExtracted;
                        if (callback != null && totalEntries > 0) {
                            int percentage = filesExtracted * 100 / totalEntries;
                            callback.onProgress(percentage, entry.getName());
                        }
                    }
                    zis.closeEntry();
                }
            }
            long duration = System.currentTimeMillis() - startTime;
            String message = String.format("Extracted %d files in %.1f seconds", filesExtracted, (double)duration / 1000.0);
            if (callback != null) {
                callback.onComplete(true, message);
            }
            System.out.println("[WorldCompressor] " + message);
            return new DecompressResult(true, message, filesExtracted);
        }
        catch (IOException e) {
            String errorMsg = "Failed to decompress world: " + e.getMessage();
            System.err.println("[WorldCompressor] " + errorMsg);
            e.printStackTrace();
            if (callback != null) {
                callback.onComplete(false, errorMsg);
            }
            return new DecompressResult(false, errorMsg, filesExtracted);
        }
    }

    public static DecompressResult decompress(Path zipFile, Path outputDir) {
        return WorldCompressor.decompress(zipFile, outputDir, null);
    }

    public static class CompressResult {
        private final boolean success;
        private final String message;
        private final long originalSize;
        private final long compressedSize;

        public CompressResult(boolean success, String message, long originalSize, long compressedSize) {
            this.success = success;
            this.message = message;
            this.originalSize = originalSize;
            this.compressedSize = compressedSize;
        }

        public boolean isSuccess() {
            return this.success;
        }

        public String getMessage() {
            return this.message;
        }

        public long getOriginalSize() {
            return this.originalSize;
        }

        public long getCompressedSize() {
            return this.compressedSize;
        }

        public double getCompressionRatio() {
            return this.originalSize > 0L ? 100.0 * (double)this.compressedSize / (double)this.originalSize : 0.0;
        }
    }

    @Deprecated
    public static interface ProgressCallback
    extends CompressionProgressCallback {
    }

    public static class DecompressResult {
        private final boolean success;
        private final String message;
        private final int filesExtracted;

        public DecompressResult(boolean success, String message, int filesExtracted) {
            this.success = success;
            this.message = message;
            this.filesExtracted = filesExtracted;
        }

        public boolean isSuccess() {
            return this.success;
        }

        public String getMessage() {
            return this.message;
        }

        public int getFilesExtracted() {
            return this.filesExtracted;
        }
    }
}

