package com.cleanroommc.groovyscript.api.documentation.annotations;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.lang.reflect.Method;

/**
 * Marks the given Method as the initializing method for a Recipe Builder, indicating it returns a Recipe Builder class.<br>
 * A Recipe Builder is a class which follows the Builder design pattern, where each step of the builder returns the builder class,
 * allowing chaining of methods to quickly and cleanly create complicated objects which may or may not require some values.
 *
 * <ul>
 *     <li>{@link #title()} is a localization key that is autogenerated to be
 *     <code>
 *         groovyscript.wiki.{@link com.cleanroommc.groovyscript.compat.mods.GroovyContainer#getModId() {modId}}.{@link com.cleanroommc.groovyscript.registry.VirtualizedRegistry#getName() {name}}.{@link Method#getName() {methodName}}.title
 *     </code>
 *     to name the Recipe Builder.
 *     </li>
 *     <li>{@link #description()} is a localization key that is autogenerated to be
 *     <code>
 *         groovyscript.wiki.{@link com.cleanroommc.groovyscript.compat.mods.GroovyContainer#getModId() {modId}}.{@link com.cleanroommc.groovyscript.registry.VirtualizedRegistry#getName() {name}}.{@link Method#getName() {methodName}}.description
 *     </code>
 *     to describe the method used to create the Recipe Builder.
 *     </li>
 *     <li>{@link #method()} either contains nothing if annotated on a method or contains a
 *     string that targets the desired method in conjunction with {@link MethodOverride}.
 *     To target a method, if only a single method has the given name, excluding
 *     bridge, non-public, Object, or methods annotated with {@link com.cleanroommc.groovyscript.api.GroovyBlacklist},
 *     the target may be the method name.
 *     Otherwise, the target must be the name and full descriptor of the method.</li>
 *     <li>{@link #clazz()} is the class being targeted by the recipe builder. By default this is the return value of the method.</li>
 *     <li>{@link #example()} is an array of {@link Example}s In situations where either a single {@link Example} with multiple lines or
 *     multiple {@link Example}s could be used, using multiple {@link Example}s is preferable.</li>
 *     <li>{@link #override()} allows creating overrides for the {@link Property}, {@link RecipeBuilderMethodDescription}, and {@link RecipeBuilderRegistrationMethod}
 *     annotated in the builder using {@link RecipeBuilderOverride}.</li>
 *     <li>{@link #priority()} is an integer that influences the sorting of the {@link RecipeBuilderDescription} relative to other {@link RecipeBuilderDescription}s.</li>
 * </ul>
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.METHOD)
public @interface RecipeBuilderDescription {

    /**
     * The localization key for the name of the builder, if different from Recipe Builder.
     * <br>
     * This should be used when there is more than one way to create a Recipe Builder
     * for the given class.
     * <br>
     * Generates the title via minecraft's localization files via
     * <code>{@link net.minecraft.client.resources.I18n#format(String, Object...) I18n.format(description())}</code>
     * <br>
     * If this is empty, will fall back to generating a description based on
     * <code>groovyscript.wiki.{@link com.cleanroommc.groovyscript.compat.mods.GroovyContainer#getModId() {modId}}.{@link com.cleanroommc.groovyscript.registry.VirtualizedRegistry#getName() {name}}.{@link Method#getName() {methodName}}.title</code>.
     * Then, if that does not have a lang key defined, it will use the global default for recipe builders of
     * <code>groovyscript.wiki.recipe_builder.title</code>.
     *
     * @return localization key for method title
     */
    String title() default "";

    /**
     * The localization key for a description of the compat.
     * <br>
     * This should always be used when the creation method accepts parameters.
     * In most cases this should be changed so that each parameter is an individual builder method,
     * but in some situations this is not possible, and so must be documented accordingly.
     * <br>
     * When a builder method with one or more parameters does not have a localization key set,
     * if it does not use a lang key, it will log a missing key in the {@code groovy.log} file.
     * <br>
     * Generates a description via minecraft's localization files via
     * <code>{@link net.minecraft.client.resources.I18n#format(String, Object...) I18n.format(description())}</code>
     * <br>
     * If this is empty, will fall back to generating a description based on
     * <code>groovyscript.wiki.{@link com.cleanroommc.groovyscript.compat.mods.GroovyContainer#getModId() {modId}}.{@link com.cleanroommc.groovyscript.registry.VirtualizedRegistry#getName() {name}}.{@link Method#getName() {methodName}}.description</code>.
     * Then, if that does not have a lang key defined, it will use the global default for recipe builders of
     * <code>groovyscript.wiki.recipe_builder.description</code>.
     *
     * @return localization key for method description
     */
    String description() default "";

    /**
     * If this {@link RecipeBuilderDescription} annotation is attached to a method, this element is set to the name of the method they are attached to.
     * When annotated on a method directly, this should not be set, as it has no functionality.
     *
     * @return any number of target methods, if not annotated to a method directly.
     * @see MethodOverride
     */
    String[] method() default {};

    /**
     * The builder class. By default, this will use the return class of the target method.
     * If this is undesired, specify the real class here.
     *
     * @return the class used by the builder, if different from the return type of the target method
     */
    Class<?> clazz() default void.class;

    /**
     * An array of examples, which will then be formatted and generated for both the wiki and the text script files.
     *
     * @return array of examples
     * @see Example
     */
    Example[] example() default {};

    /**
     * An override to other {@link Property} declarations, and only used for {@link Property} instances which are unique to the Recipe Builder
     * obtained via the given method.
     *
     * @return array of requirements unique to the recipe builder being accessed via this method
     * @see Property
     * @deprecated use {@link #override()} and {@link RecipeBuilderOverride#requirement()}
     */
    @Deprecated
    Property[] requirement() default {};

    /**
     * Allows overriding the {@link Property}, {@link RecipeBuilderMethodDescription}, and {@link RecipeBuilderRegistrationMethod} annotations
     * for the builder class.
     * This should only be used in situations where custom data that does not apply to the class normally
     * is used.
     *
     * @return overrides for the Recipe Builder annotations
     * @see Property
     * @see RecipeBuilderMethodDescription
     * @see RecipeBuilderRegistrationMethod
     */
    RecipeBuilderOverride override() default @RecipeBuilderOverride;

    /**
     * Priority of the method, relative to other Recipe Builder methods in the same class.
     * Priorities sort entries such that lowest is first, with ties being broken via alphabetical sorting of the method name.
     *
     * @return the method priority (relative to other recipe builder priorities in the same class)
     */
    int priority() default 1000;
}
